/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.util;

import com.google.common.annotations.VisibleForTesting;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Arrays;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.db.RowIndexEntry;
import org.apache.cassandra.io.FSReadError;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.IndexSummary;
import org.apache.cassandra.io.util.FileDataInput;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.io.util.MappedFileDataInput;
import org.apache.cassandra.io.util.RandomAccessReader;
import org.apache.cassandra.io.util.SegmentedFile;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MmappedSegmentedFile
extends SegmentedFile {
    private static final Logger logger = LoggerFactory.getLogger(MmappedSegmentedFile.class);
    public static long MAX_SEGMENT_SIZE = Integer.MAX_VALUE;
    private final SegmentedFile.Segment[] segments;

    public MmappedSegmentedFile(String path, long length, SegmentedFile.Segment[] segments) {
        super(new Cleanup(path, segments), path, length);
        this.segments = segments;
    }

    private MmappedSegmentedFile(MmappedSegmentedFile copy) {
        super(copy);
        this.segments = copy.segments;
    }

    @Override
    public MmappedSegmentedFile sharedCopy() {
        return new MmappedSegmentedFile(this);
    }

    private SegmentedFile.Segment floor(long position) {
        assert (0L <= position && position < this.length) : String.format("%d >= %d in %s", position, this.length, this.path);
        SegmentedFile.Segment seg = new SegmentedFile.Segment(position, null);
        int idx = Arrays.binarySearch(this.segments, seg);
        assert (idx != -1) : String.format("Bad position %d for segments %s in %s", position, Arrays.toString(this.segments), this.path);
        if (idx < 0) {
            idx = -(idx + 2);
        }
        return this.segments[idx];
    }

    @Override
    public FileDataInput getSegment(long position) {
        SegmentedFile.Segment segment = this.floor(position);
        if (segment.right != null) {
            return new MappedFileDataInput((MappedByteBuffer)segment.right, this.path, (Long)segment.left, (int)(position - (Long)segment.left));
        }
        RandomAccessReader file = RandomAccessReader.open(new File(this.path));
        file.seek(position);
        return file;
    }

    public static boolean maybeRepair(CFMetaData metadata, Descriptor descriptor, IndexSummary indexSummary, SegmentedFile.Builder ibuilder, SegmentedFile.Builder dbuilder) {
        boolean mayNeedRepair = false;
        if (ibuilder instanceof Builder) {
            mayNeedRepair = ((Builder)ibuilder).mayNeedRepair(descriptor.filenameFor(Component.PRIMARY_INDEX));
        }
        if (dbuilder instanceof Builder) {
            mayNeedRepair |= ((Builder)dbuilder).mayNeedRepair(descriptor.filenameFor(Component.DATA));
        }
        if (mayNeedRepair) {
            MmappedSegmentedFile.forceRepair(metadata, descriptor, indexSummary, ibuilder, dbuilder);
        }
        return mayNeedRepair;
    }

    private static void forceRepair(CFMetaData metadata, Descriptor descriptor, IndexSummary indexSummary, SegmentedFile.Builder ibuilder, SegmentedFile.Builder dbuilder) {
        if (ibuilder instanceof Builder) {
            ((Builder)ibuilder).boundaries.clear();
        }
        if (dbuilder instanceof Builder) {
            ((Builder)dbuilder).boundaries.clear();
        }
        try (RandomAccessFile raf = new RandomAccessFile(descriptor.filenameFor(Component.PRIMARY_INDEX), "r");){
            long iprev = 0L;
            long dprev = 0L;
            for (int i = 0; i < indexSummary.size(); ++i) {
                long icur = indexSummary.getPosition(i);
                raf.seek(icur);
                ByteBufferUtil.readWithShortLength(raf);
                RowIndexEntry rie = metadata.comparator.rowIndexEntrySerializer().deserialize(raf, descriptor.version);
                long dcur = rie.position;
                if (Math.max(icur - iprev, dcur - dprev) > MAX_SEGMENT_SIZE) {
                    raf.seek(iprev);
                    while (raf.getFilePointer() < icur) {
                        ibuilder.addPotentialBoundary(raf.getFilePointer());
                        ByteBufferUtil.readWithShortLength(raf);
                        rie = metadata.comparator.rowIndexEntrySerializer().deserialize(raf, descriptor.version);
                        dbuilder.addPotentialBoundary(rie.position);
                    }
                }
                ibuilder.addPotentialBoundary(icur);
                dbuilder.addPotentialBoundary(dcur);
                iprev = icur;
                dprev = dcur;
            }
        }
        catch (IOException e) {
            logger.error("Failed to recalculate boundaries for {}; mmap access may degrade to buffered for this file", (Object)descriptor);
        }
    }

    public static class Builder
    extends SegmentedFile.Builder {
        private final Boundaries boundaries = new Boundaries();

        public long[] boundaries() {
            return this.boundaries.truncate();
        }

        boolean mayNeedRepair(String path) {
            long length = new File(path).length();
            this.boundaries.addCandidate(length);
            long[] boundaries = this.boundaries.truncate();
            long prev = 0L;
            for (long boundary : boundaries) {
                if (boundary - prev > MAX_SEGMENT_SIZE) {
                    return true;
                }
                prev = boundary;
            }
            return false;
        }

        @Override
        public void addPotentialBoundary(long boundary) {
            this.boundaries.addCandidate(boundary);
        }

        @Override
        public SegmentedFile complete(String path, long overrideLength, boolean isFinal) {
            assert (!isFinal || overrideLength <= 0L);
            long length = overrideLength > 0L ? overrideLength : new File(path).length();
            return new MmappedSegmentedFile(path, length, this.createSegments(path, length, isFinal));
        }

        private SegmentedFile.Segment[] createSegments(String path, long length, boolean isFinal) {
            RandomAccessFile raf;
            try {
                raf = new RandomAccessFile(path, "r");
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
            long[] boundaries = this.boundaries.finish(length, isFinal);
            int segcount = boundaries.length - 1;
            SegmentedFile.Segment[] segments = new SegmentedFile.Segment[segcount];
            try {
                for (int i = 0; i < segcount; ++i) {
                    long start = boundaries[i];
                    long size = boundaries[i + 1] - start;
                    MappedByteBuffer segment = size <= MAX_SEGMENT_SIZE ? raf.getChannel().map(FileChannel.MapMode.READ_ONLY, start, size) : null;
                    segments[i] = new SegmentedFile.Segment(start, segment);
                }
            }
            catch (IOException e) {
                throw new FSReadError((Throwable)e, path);
            }
            finally {
                FileUtils.closeQuietly(raf);
            }
            return segments;
        }

        @Override
        public void serializeBounds(DataOutput out) throws IOException {
            super.serializeBounds(out);
            long[] boundaries = this.boundaries.truncate();
            out.writeInt(boundaries.length);
            for (long boundary : boundaries) {
                out.writeLong(boundary);
            }
        }

        @Override
        public void deserializeBounds(DataInput in) throws IOException {
            super.deserializeBounds(in);
            int size = in.readInt();
            long[] boundaries = new long[size];
            for (int i = 0; i < size; ++i) {
                boundaries[i] = in.readLong();
            }
            this.boundaries.init(boundaries, size);
        }

        @VisibleForTesting
        public static class Boundaries {
            private long[] boundaries;
            private int fixedCount;

            public Boundaries() {
                this(new long[8], 1);
            }

            public Boundaries(long[] boundaries, int fixedCount) {
                this.init(boundaries, fixedCount);
            }

            void init(long[] boundaries, int fixedCount) {
                this.boundaries = boundaries;
                this.fixedCount = fixedCount;
            }

            public void addCandidate(long candidate) {
                this.boundaries = Boundaries.ensureCapacity(this.boundaries, this.fixedCount);
                this.fixedCount = Boundaries.addCandidate(this.boundaries, this.fixedCount, candidate);
            }

            private static int addCandidate(long[] boundaries, int fixedCount, long candidate) {
                long delta = candidate - boundaries[fixedCount - 1];
                assert (delta >= 0L);
                if (delta != 0L) {
                    if (delta <= MAX_SEGMENT_SIZE) {
                        boundaries[fixedCount] = candidate;
                    } else if (boundaries[fixedCount] == 0L) {
                        boundaries[fixedCount++] = candidate;
                    } else {
                        boundaries[++fixedCount] = candidate;
                    }
                }
                return fixedCount;
            }

            private static long[] ensureCapacity(long[] boundaries, int fixedCount) {
                if (fixedCount + 1 >= boundaries.length) {
                    return Arrays.copyOf(boundaries, Math.max(fixedCount + 2, boundaries.length * 2));
                }
                return boundaries;
            }

            void clear() {
                this.fixedCount = 1;
                Arrays.fill(this.boundaries, 0L);
            }

            public long[] truncate() {
                return Arrays.copyOf(this.boundaries, this.fixedCount);
            }

            public long[] finish(long length, boolean isFinal) {
                assert (length > 0L);
                this.boundaries = Boundaries.ensureCapacity(this.boundaries, this.fixedCount);
                int fixedCount = this.fixedCount;
                long[] boundaries = (long[])this.boundaries.clone();
                while (boundaries[fixedCount - 1] >= length) {
                    boundaries[fixedCount--] = 0L;
                }
                if (boundaries[fixedCount] >= length) {
                    boundaries[fixedCount] = 0L;
                }
                if (boundaries[fixedCount = Boundaries.addCandidate(boundaries, fixedCount, length)] != 0L) {
                    ++fixedCount;
                }
                boundaries = Arrays.copyOf(boundaries, fixedCount);
                if (isFinal) {
                    this.boundaries = boundaries;
                    this.fixedCount = fixedCount;
                }
                return boundaries;
            }
        }
    }

    private static final class Cleanup
    extends SegmentedFile.Cleanup {
        final SegmentedFile.Segment[] segments;

        protected Cleanup(String path, SegmentedFile.Segment[] segments) {
            super(path);
            this.segments = segments;
        }

        @Override
        public void tidy() {
            if (!FileUtils.isCleanerAvailable()) {
                return;
            }
            try {
                for (SegmentedFile.Segment segment : this.segments) {
                    if (segment.right == null) continue;
                    FileUtils.clean((MappedByteBuffer)segment.right);
                }
                logger.debug("All segments have been unmapped successfully");
            }
            catch (Exception e) {
                JVMStabilityInspector.inspectThrowable(e);
                logger.error("Error while unmapping segments", (Throwable)e);
            }
        }
    }
}

