/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.AbstractReconnectionHandler;
import com.datastax.driver.core.AuthProvider;
import com.datastax.driver.core.BusyConnectionException;
import com.datastax.driver.core.CloseFuture;
import com.datastax.driver.core.ClusterNameMismatchException;
import com.datastax.driver.core.Configuration;
import com.datastax.driver.core.Connection;
import com.datastax.driver.core.ConnectionException;
import com.datastax.driver.core.ControlConnection;
import com.datastax.driver.core.ConvictionPolicy;
import com.datastax.driver.core.DefaultResultSetFuture;
import com.datastax.driver.core.ExceptionCatchingRunnable;
import com.datastax.driver.core.Host;
import com.datastax.driver.core.HostConnectionPool;
import com.datastax.driver.core.HostDistance;
import com.datastax.driver.core.KeyspaceMetadata;
import com.datastax.driver.core.LatencyTracker;
import com.datastax.driver.core.MD5Digest;
import com.datastax.driver.core.Message;
import com.datastax.driver.core.Metadata;
import com.datastax.driver.core.Metrics;
import com.datastax.driver.core.MetricsOptions;
import com.datastax.driver.core.PlainTextAuthProvider;
import com.datastax.driver.core.PoolingOptions;
import com.datastax.driver.core.PreparedStatement;
import com.datastax.driver.core.ProtocolEvent;
import com.datastax.driver.core.ProtocolOptions;
import com.datastax.driver.core.QueryOptions;
import com.datastax.driver.core.Requests;
import com.datastax.driver.core.Responses;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.ResultSetFuture;
import com.datastax.driver.core.SSLOptions;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.SessionManager;
import com.datastax.driver.core.SocketOptions;
import com.datastax.driver.core.SystemProperties;
import com.datastax.driver.core.UnsupportedProtocolVersionException;
import com.datastax.driver.core.exceptions.AuthenticationException;
import com.datastax.driver.core.exceptions.DriverInternalError;
import com.datastax.driver.core.exceptions.NoHostAvailableException;
import com.datastax.driver.core.policies.AddressTranslater;
import com.datastax.driver.core.policies.CloseableLoadBalancingPolicy;
import com.datastax.driver.core.policies.LoadBalancingPolicy;
import com.datastax.driver.core.policies.Policies;
import com.datastax.driver.core.policies.ReconnectionPolicy;
import com.datastax.driver.core.policies.RetryPolicy;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.MapMaker;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.common.util.concurrent.Uninterruptibles;
import java.io.Closeable;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Cluster
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(Cluster.class);
    @VisibleForTesting
    static final int NEW_NODE_DELAY_SECONDS = SystemProperties.getInt("com.datastax.driver.NEW_NODE_DELAY_SECONDS", 1);
    private static final int NON_BLOCKING_EXECUTOR_SIZE = SystemProperties.getInt("com.datastax.driver.NON_BLOCKING_EXECUTOR_SIZE", Runtime.getRuntime().availableProcessors());
    private static final ResourceBundle driverProperties = ResourceBundle.getBundle("com.datastax.driver.core.Driver");
    private static final AtomicInteger CLUSTER_ID = new AtomicInteger(0);
    private static final int DEFAULT_THREAD_KEEP_ALIVE = 30;
    private static final int NOTIF_LOCK_TIMEOUT_SECONDS = SystemProperties.getInt("com.datastax.driver.NOTIF_LOCK_TIMEOUT_SECONDS", 60);
    final Manager manager;

    protected Cluster(String name, List<InetSocketAddress> contactPoints, Configuration configuration) {
        this(name, contactPoints, configuration, Collections.emptySet());
    }

    protected Cluster(Initializer initializer) {
        this(initializer.getClusterName(), Cluster.checkNotEmpty(initializer.getContactPoints()), initializer.getConfiguration(), initializer.getInitialListeners());
    }

    private static List<InetSocketAddress> checkNotEmpty(List<InetSocketAddress> contactPoints) {
        if (contactPoints.isEmpty()) {
            throw new IllegalArgumentException("Cannot build a cluster without contact points");
        }
        return contactPoints;
    }

    private Cluster(String name, List<InetSocketAddress> contactPoints, Configuration configuration, Collection<Host.StateListener> listeners) {
        this.manager = new Manager(name, contactPoints, configuration, listeners);
    }

    public Cluster init() {
        this.manager.init();
        return this;
    }

    public static Cluster buildFrom(Initializer initializer) {
        return new Cluster(initializer);
    }

    public static Builder builder() {
        return new Builder();
    }

    public static String getDriverVersion() {
        return driverProperties.getString("driver.version");
    }

    public Session newSession() {
        return this.manager.newSession();
    }

    public Session connect() {
        this.init();
        Session session = this.manager.newSession();
        session.init();
        return session;
    }

    public Session connect(String keyspace) {
        long timeout = this.getConfiguration().getSocketOptions().getConnectTimeoutMillis();
        Session session = this.connect();
        try {
            ResultSetFuture future = session.executeAsync("USE " + keyspace);
            Uninterruptibles.getUninterruptibly((Future)((Object)future), (long)timeout, (TimeUnit)TimeUnit.MILLISECONDS);
            return session;
        }
        catch (TimeoutException e) {
            throw new DriverInternalError(String.format("No responses after %d milliseconds while setting current keyspace. This should not happen, unless you have setup a very low connection timeout.", timeout));
        }
        catch (ExecutionException e) {
            throw DefaultResultSetFuture.extractCauseFromExecutionException(e);
        }
        catch (RuntimeException e) {
            session.close();
            throw e;
        }
    }

    public String getClusterName() {
        return this.manager.clusterName;
    }

    public Metadata getMetadata() {
        this.manager.init();
        return this.manager.metadata;
    }

    public Configuration getConfiguration() {
        return this.manager.configuration;
    }

    public Metrics getMetrics() {
        return this.manager.metrics;
    }

    public Cluster register(Host.StateListener listener) {
        this.manager.listeners.add(listener);
        return this;
    }

    public Cluster unregister(Host.StateListener listener) {
        this.manager.listeners.remove(listener);
        return this;
    }

    public Cluster register(LatencyTracker tracker) {
        this.manager.trackers.add(tracker);
        return this;
    }

    public Cluster unregister(LatencyTracker tracker) {
        this.manager.trackers.remove(tracker);
        return this;
    }

    public CloseFuture closeAsync() {
        return this.manager.close();
    }

    @Override
    public void close() {
        try {
            this.closeAsync().get();
        }
        catch (ExecutionException e) {
            throw DefaultResultSetFuture.extractCauseFromExecutionException(e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    public boolean isClosed() {
        return this.manager.closeFuture.get() != null;
    }

    private static ThreadFactory threadFactory(String nameFormat) {
        return new ThreadFactoryBuilder().setNameFormat(nameFormat).build();
    }

    static long timeSince(long startNanos, TimeUnit destUnit) {
        return destUnit.convert(System.nanoTime() - startNanos, TimeUnit.NANOSECONDS);
    }

    private static String generateClusterName() {
        return "cluster" + CLUSTER_ID.incrementAndGet();
    }

    private static ListeningExecutorService makeExecutor(int threads, String name) {
        ThreadPoolExecutor executor = new ThreadPoolExecutor(threads, threads, 30L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(), Cluster.threadFactory(name));
        executor.allowCoreThreadTimeOut(true);
        return MoreExecutors.listeningDecorator((ExecutorService)executor);
    }

    static /* synthetic */ ThreadFactory access$300(String x0) {
        return Cluster.threadFactory(x0);
    }

    static class ConnectionReaper {
        private static final int INTERVAL_MS = 15000;
        private final ScheduledExecutorService executor = Executors.newScheduledThreadPool(1, Cluster.access$300("Reaper-%d"));
        private final Map<Connection, Long> connections = new ConcurrentHashMap<Connection, Long>();
        private volatile boolean shutdown;
        private final Runnable reaperTask = new Runnable(){

            @Override
            public void run() {
                long now = System.currentTimeMillis();
                Iterator iterator = ConnectionReaper.this.connections.entrySet().iterator();
                while (iterator.hasNext()) {
                    boolean terminated;
                    Map.Entry entry = iterator.next();
                    Connection connection = (Connection)entry.getKey();
                    Long terminateTime = (Long)entry.getValue();
                    if (terminateTime > now || !(terminated = connection.tryTerminate(true))) continue;
                    iterator.remove();
                }
            }
        };

        ConnectionReaper() {
            this.executor.scheduleWithFixedDelay(this.reaperTask, 15000L, 15000L, TimeUnit.MILLISECONDS);
        }

        void register(Connection connection, long terminateTime) {
            if (this.shutdown) {
                logger.warn("Connection registered after reaper shutdown: {}", (Object)connection);
                connection.tryTerminate(true);
            } else {
                this.connections.put(connection, terminateTime);
            }
        }

        void shutdown() {
            this.shutdown = true;
            this.executor.shutdownNow();
            this.reaperTask.run();
        }
    }

    class Manager
    implements Connection.DefaultResponseHandler {
        final String clusterName;
        private boolean isInit;
        private volatile boolean isFullyInit;
        final List<InetSocketAddress> contactPoints;
        final Set<SessionManager> sessions = new CopyOnWriteArraySet<SessionManager>();
        final Metadata metadata;
        final Configuration configuration;
        final Metrics metrics;
        final Connection.Factory connectionFactory;
        final ControlConnection controlConnection;
        final ConvictionPolicy.Factory convictionPolicyFactory = new ConvictionPolicy.Simple.Factory();
        final ScheduledExecutorService reconnectionExecutor = Executors.newScheduledThreadPool(2, Cluster.access$300("Reconnection-%d"));
        final ScheduledExecutorService scheduledTasksExecutor = Executors.newScheduledThreadPool(1, Cluster.access$300("Scheduled Tasks-%d"));
        final ListeningExecutorService executor;
        final ListeningExecutorService blockingExecutor;
        final ConnectionReaper reaper;
        final AtomicReference<CloseFuture> closeFuture = new AtomicReference();
        final ConcurrentMap<MD5Digest, PreparedStatement> preparedQueries = new MapMaker().weakValues().makeMap();
        final Set<Host.StateListener> listeners;
        final Set<LatencyTracker> trackers = new CopyOnWriteArraySet<LatencyTracker>();

        private Manager(String clusterName, List<InetSocketAddress> contactPoints, Configuration configuration, Collection<Host.StateListener> listeners) {
            logger.debug("Starting new cluster with contact points " + contactPoints);
            this.clusterName = clusterName == null ? Cluster.generateClusterName() : clusterName;
            this.configuration = configuration;
            this.configuration.register(this);
            this.executor = Cluster.makeExecutor(NON_BLOCKING_EXECUTOR_SIZE, "Cassandra Java Driver worker-%d");
            this.blockingExecutor = Cluster.makeExecutor(2, "Cassandra Java Driver blocking tasks worker-%d");
            this.reaper = new ConnectionReaper();
            this.metadata = new Metadata(this);
            this.contactPoints = contactPoints;
            this.connectionFactory = new Connection.Factory(this, configuration);
            this.controlConnection = new ControlConnection(this);
            this.metrics = configuration.getMetricsOptions() == null ? null : new Metrics(this);
            this.listeners = new CopyOnWriteArraySet<Host.StateListener>(listeners);
        }

        synchronized void init() {
            if (this.isClosed()) {
                throw new IllegalStateException("Can't use this Cluster instance because it was previously closed");
            }
            if (this.isInit) {
                return;
            }
            this.isInit = true;
            for (InetSocketAddress address : this.contactPoints) {
                this.metadata.add(address);
            }
            try {
                while (true) {
                    try {
                        HashSet contactPointHosts = Sets.newHashSet(this.metadata.allHosts());
                        this.controlConnection.connect();
                        if (this.connectionFactory.protocolVersion < 0) {
                            this.connectionFactory.protocolVersion = 2;
                        }
                        HashSet downContactPointHosts = Sets.newHashSet();
                        for (Host host : contactPointHosts) {
                            if (host.state != Host.State.DOWN) continue;
                            downContactPointHosts.add(host);
                        }
                        contactPointHosts.removeAll(downContactPointHosts);
                        this.loadBalancingPolicy().init(Cluster.this, contactPointHosts);
                        for (Host host : downContactPointHosts) {
                            this.loadBalancingPolicy().onDown(host);
                            for (Host.StateListener listener : this.listeners) {
                                listener.onDown(host);
                            }
                        }
                        for (Host host : this.metadata.allHosts()) {
                            if (host.state == Host.State.DOWN) continue;
                            logger.info("New Cassandra host {} added", (Object)host);
                            if (this.connectionFactory.protocolVersion == 2 && !this.supportsProtocolV2(host)) {
                                this.logUnsupportedVersionProtocol(host);
                                continue;
                            }
                            if (!contactPointHosts.contains(host)) {
                                this.loadBalancingPolicy().onAdd(host);
                            }
                            host.setUp();
                            for (Host.StateListener listener : this.listeners) {
                                listener.onAdd(host);
                            }
                        }
                        this.isFullyInit = true;
                        return;
                    }
                    catch (UnsupportedProtocolVersionException e) {
                        assert (this.connectionFactory.protocolVersion < 1);
                        if (e.versionUnsupported <= 1) {
                            throw new DriverInternalError("Got a node that don't even support the protocol version 1, this makes no sense", e);
                        }
                        logger.debug("{}: retrying with version {}", (Object)e.getMessage(), (Object)(e.versionUnsupported - 1));
                        this.connectionFactory.protocolVersion = e.versionUnsupported - 1;
                        continue;
                    }
                    break;
                }
            }
            catch (NoHostAvailableException e) {
                this.close();
                throw e;
            }
        }

        int protocolVersion() {
            return this.connectionFactory.protocolVersion;
        }

        Cluster getCluster() {
            return Cluster.this;
        }

        LoadBalancingPolicy loadBalancingPolicy() {
            return this.configuration.getPolicies().getLoadBalancingPolicy();
        }

        ReconnectionPolicy reconnectionPolicy() {
            return this.configuration.getPolicies().getReconnectionPolicy();
        }

        InetSocketAddress translateAddress(InetAddress address) {
            InetSocketAddress sa = new InetSocketAddress(address, this.connectionFactory.getPort());
            InetSocketAddress translated = this.configuration.getPolicies().getAddressTranslater().translate(sa);
            return translated == null ? sa : translated;
        }

        private Session newSession() {
            SessionManager session = new SessionManager(Cluster.this);
            this.sessions.add(session);
            return session;
        }

        boolean removeSession(Session session) {
            return this.sessions.remove(session);
        }

        void reportLatency(Host host, long latencyNanos) {
            for (LatencyTracker tracker : this.trackers) {
                tracker.update(host, latencyNanos);
            }
        }

        boolean isClosed() {
            return this.closeFuture.get() != null;
        }

        private CloseFuture close() {
            LoadBalancingPolicy loadBalancingPolicy;
            CloseFuture future = this.closeFuture.get();
            if (future != null) {
                return future;
            }
            logger.debug("Shutting down");
            this.shutdownNow(this.reconnectionExecutor);
            this.shutdownNow(this.scheduledTasksExecutor);
            this.shutdownNow((ExecutorService)this.blockingExecutor);
            this.executor.shutdown();
            if (this.metrics != null) {
                this.metrics.shutdown();
            }
            if ((loadBalancingPolicy = this.loadBalancingPolicy()) instanceof CloseableLoadBalancingPolicy) {
                ((CloseableLoadBalancingPolicy)loadBalancingPolicy).close();
            }
            ArrayList<CloseFuture> futures = new ArrayList<CloseFuture>(this.sessions.size() + 1);
            futures.add(this.controlConnection.closeAsync());
            for (Session session : this.sessions) {
                futures.add(session.closeAsync());
            }
            future = new ClusterCloseFuture(futures);
            return this.closeFuture.compareAndSet(null, future) ? future : this.closeFuture.get();
        }

        private void shutdownNow(ExecutorService executor) {
            List<Runnable> pendingTasks = executor.shutdownNow();
            for (Runnable pendingTask : pendingTasks) {
                if (!(pendingTask instanceof FutureTask)) continue;
                ((FutureTask)pendingTask).cancel(false);
            }
        }

        void logUnsupportedVersionProtocol(Host host) {
            logger.warn("Detected added or restarted Cassandra host {} but ignoring it since it does not support the version 2 of the native protocol which is currently in use. If you want to force the use of the version 1 of the native protocol, use Cluster.Builder#usingProtocolVersion() when creating the Cluster instance.", (Object)host);
        }

        void logClusterNameMismatch(Host host, String expectedClusterName, String actualClusterName) {
            logger.warn("Detected added or restarted Cassandra host {} but ignoring it since its cluster name '{}' does not match the one currently known ({})", new Object[]{host, actualClusterName, expectedClusterName});
        }

        public ListenableFuture<?> triggerOnUp(final Host host) {
            return this.executor.submit((Runnable)new ExceptionCatchingRunnable(){

                @Override
                public void runMayThrow() throws InterruptedException, ExecutionException {
                    Manager.this.onUp(host);
                }
            });
        }

        private void onUp(Host host) throws InterruptedException, ExecutionException {
            this.onUp(host, this.blockingExecutor);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Loose catch block
         */
        private void onUp(final Host host, ListeningExecutorService poolCreationExecutor) throws InterruptedException, ExecutionException {
            block18: {
                logger.debug("Host {} is UP", (Object)host);
                if (this.isClosed()) {
                    return;
                }
                if (this.connectionFactory.protocolVersion == 2 && !this.supportsProtocolV2(host)) {
                    this.logUnsupportedVersionProtocol(host);
                    return;
                }
                boolean locked = host.notificationsLock.tryLock(NOTIF_LOCK_TIMEOUT_SECONDS, TimeUnit.SECONDS);
                if (!locked) {
                    logger.warn("Could not acquire notifications lock within {} seconds, ignoring UP notification for {}", (Object)NOTIF_LOCK_TIMEOUT_SECONDS, (Object)host);
                    return;
                }
                try {
                    if (host.state == Host.State.UP) {
                        return;
                    }
                    Future scheduledAttempt = host.reconnectionAttempt.getAndSet(null);
                    if (scheduledAttempt != null) {
                        logger.debug("Cancelling reconnection attempt since node is UP");
                        scheduledAttempt.cancel(false);
                    }
                    try {
                        this.prepareAllQueries(host);
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                    catch (UnsupportedProtocolVersionException e) {
                        this.logUnsupportedVersionProtocol(host);
                        host.notificationsLock.unlock();
                        return;
                    }
                    catch (ClusterNameMismatchException e) {
                        this.logClusterNameMismatch(host, e.expectedClusterName, e.actualClusterName);
                        host.notificationsLock.unlock();
                        return;
                    }
                    for (SessionManager s : this.sessions) {
                        s.removePool(host);
                    }
                    this.loadBalancingPolicy().onUp(host);
                    this.controlConnection.onUp(host);
                    logger.trace("Adding/renewing host pools for newly UP host {}", (Object)host);
                    ArrayList<ListenableFuture<Boolean>> futures = new ArrayList<ListenableFuture<Boolean>>(this.sessions.size());
                    for (SessionManager s : this.sessions) {
                        futures.add(s.forceRenewPool(host, poolCreationExecutor));
                    }
                    ListenableFuture f = Futures.allAsList(futures);
                    Futures.addCallback((ListenableFuture)f, (FutureCallback)new FutureCallback<List<Boolean>>(){

                        public void onSuccess(List<Boolean> poolCreationResults) {
                            if (Iterables.any(poolCreationResults, (Predicate)Predicates.equalTo((Object)false))) {
                                logger.debug("Connection pool cannot be created, not marking {} UP", (Object)host);
                                return;
                            }
                            host.setUp();
                            for (Host.StateListener listener : Manager.this.listeners) {
                                listener.onUp(host);
                            }
                        }

                        public void onFailure(Throwable t) {
                            if (!(t instanceof InterruptedException)) {
                                logger.error("Unexpected error while marking node UP: while this shouldn't happen, this shouldn't be critical", t);
                            }
                        }
                    });
                    f.get();
                    for (SessionManager s : this.sessions) {
                        s.updateCreatedPools(this.blockingExecutor);
                    }
                    {
                        break block18;
                        catch (Throwable throwable) {
                            throw throwable;
                        }
                    }
                }
                finally {
                    host.notificationsLock.unlock();
                }
            }
        }

        public ListenableFuture<?> triggerOnDown(Host host) {
            return this.triggerOnDown(host, false);
        }

        public ListenableFuture<?> triggerOnDown(final Host host, final boolean isHostAddition) {
            return this.executor.submit((Runnable)new ExceptionCatchingRunnable(){

                @Override
                public void runMayThrow() throws InterruptedException, ExecutionException {
                    Manager.this.onDown(host, isHostAddition, false);
                }
            });
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void onSuspected(final Host host) {
            logger.debug("Host {} is Suspected", (Object)host);
            if (this.isClosed()) {
                return;
            }
            if (this.loadBalancingPolicy().distance(host) == HostDistance.IGNORED) {
                this.triggerOnDown(host);
                return;
            }
            Host host2 = host;
            synchronized (host2) {
                if (!host.setSuspected() || host.reconnectionAttempt.get() != null) {
                    return;
                }
                host.initialReconnectionAttempt.set(this.executor.submit((Runnable)new ExceptionCatchingRunnable(){

                    @Override
                    public void runMayThrow() throws InterruptedException, ExecutionException {
                        boolean success;
                        try {
                            Manager.this.connectionFactory.open(host).closeAsync();
                            Manager.this.onUp(host, MoreExecutors.sameThreadExecutor());
                            success = host.state == Host.State.UP;
                        }
                        catch (Exception e) {
                            success = false;
                        }
                        if (!success) {
                            Manager.this.onDown(host, false, true);
                        }
                    }
                }));
                this.loadBalancingPolicy().onSuspected(host);
            }
            this.controlConnection.onSuspected(host);
            for (SessionManager s : this.sessions) {
                s.onSuspected(host);
            }
            for (Host.StateListener listener : this.listeners) {
                listener.onSuspected(host);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void onDown(Host host, boolean isHostAddition, boolean isSuspectedVerification) throws InterruptedException, ExecutionException {
            logger.debug("Host {} is DOWN", (Object)host);
            if (this.isClosed()) {
                return;
            }
            boolean locked = host.notificationsLock.tryLock(NOTIF_LOCK_TIMEOUT_SECONDS, TimeUnit.SECONDS);
            if (!locked) {
                logger.warn("Could not acquire notifications lock within {} seconds, ignoring DOWN notification for {}", (Object)NOTIF_LOCK_TIMEOUT_SECONDS, (Object)host);
                return;
            }
            try {
                if (!isSuspectedVerification && host.state == Host.State.SUSPECT) {
                    logger.debug("Aborting onDown because a reconnection is running on SUSPECT host {}", (Object)host);
                    return;
                }
                if (host.reconnectionAttempt.get() != null) {
                    logger.debug("Aborting onDown because a reconnection is running on DOWN host {}", (Object)host);
                    return;
                }
                HostDistance distance = this.loadBalancingPolicy().distance(host);
                boolean wasUp = host.isUp();
                host.setDown();
                this.loadBalancingPolicy().onDown(host);
                this.controlConnection.onDown(host);
                for (SessionManager s : this.sessions) {
                    s.onDown(host);
                }
                if (wasUp) {
                    for (Host.StateListener listener : this.listeners) {
                        listener.onDown(host);
                    }
                }
                if (distance == HostDistance.IGNORED) {
                    return;
                }
                logger.debug("{} is down, scheduling connection retries", (Object)host);
                this.startPeriodicReconnectionAttempt(host, isHostAddition);
            }
            finally {
                host.notificationsLock.unlock();
            }
        }

        void startPeriodicReconnectionAttempt(final Host host, final boolean isHostAddition) {
            new AbstractReconnectionHandler(this.reconnectionExecutor, this.reconnectionPolicy().newSchedule(), host.reconnectionAttempt){

                @Override
                protected Connection tryReconnect() throws ConnectionException, InterruptedException, UnsupportedProtocolVersionException, ClusterNameMismatchException {
                    return Manager.this.connectionFactory.open(host);
                }

                @Override
                protected void onReconnection(Connection connection) {
                    block6: {
                        connection.closeAsync();
                        if (Manager.this.controlConnection.refreshNodeInfo(host)) {
                            logger.debug("Successful reconnection to {}, setting host UP", (Object)host);
                            try {
                                if (isHostAddition) {
                                    Manager.this.onAdd(host);
                                    break block6;
                                }
                                Manager.this.onUp(host);
                            }
                            catch (InterruptedException e) {
                                Thread.currentThread().interrupt();
                            }
                            catch (Exception e) {
                                logger.error("Unexpected error while setting node up", (Throwable)e);
                            }
                        } else {
                            logger.debug("Not enough info for {}, ignoring host", (Object)host);
                        }
                    }
                }

                @Override
                protected boolean onConnectionException(ConnectionException e, long nextDelayMs) {
                    if (logger.isDebugEnabled()) {
                        logger.debug("Failed reconnection to {} ({}), scheduling retry in {} milliseconds", new Object[]{host, e.getMessage(), nextDelayMs});
                    }
                    return true;
                }

                @Override
                protected boolean onUnknownException(Exception e, long nextDelayMs) {
                    logger.error(String.format("Unknown error during reconnection to %s, scheduling retry in %d milliseconds", host, nextDelayMs), (Throwable)e);
                    return true;
                }

                @Override
                protected boolean onAuthenticationException(AuthenticationException e, long nextDelayMs) {
                    logger.error(String.format("Authentication error during reconnection to %s, scheduling retry in %d milliseconds", host, nextDelayMs), (Throwable)e);
                    return true;
                }
            }.start();
        }

        void startSingleReconnectionAttempt(final Host host) {
            if (this.isClosed() || host.isUp()) {
                return;
            }
            logger.debug("Scheduling one-time reconnection to {}", (Object)host);
            new AbstractReconnectionHandler(this.reconnectionExecutor, this.reconnectionPolicy().newSchedule(), host.reconnectionAttempt, 0L){

                @Override
                protected Connection tryReconnect() throws ConnectionException, InterruptedException, UnsupportedProtocolVersionException, ClusterNameMismatchException {
                    return Manager.this.connectionFactory.open(host);
                }

                @Override
                protected void onReconnection(Connection connection) {
                    connection.closeAsync();
                    if (Manager.this.controlConnection.refreshNodeInfo(host)) {
                        logger.debug("Successful reconnection to {}, setting host UP", (Object)host);
                        try {
                            Manager.this.onUp(host);
                        }
                        catch (InterruptedException e) {
                            Thread.currentThread().interrupt();
                        }
                        catch (Exception e) {
                            logger.error("Unexpected error while setting node up", (Throwable)e);
                        }
                    } else {
                        logger.debug("Not enough info for {}, ignoring host", (Object)host);
                    }
                }

                @Override
                protected boolean onConnectionException(ConnectionException e, long nextDelayMs) {
                    if (logger.isDebugEnabled()) {
                        logger.debug("Failed one-time reconnection to {} ({})", (Object)host, (Object)e.getMessage());
                    }
                    return false;
                }

                @Override
                protected boolean onUnknownException(Exception e, long nextDelayMs) {
                    logger.error(String.format("Unknown error during one-time reconnection to %s", host), (Throwable)e);
                    return false;
                }

                @Override
                protected boolean onAuthenticationException(AuthenticationException e, long nextDelayMs) {
                    logger.error(String.format("Authentication error during one-time reconnection to %s", host), (Throwable)e);
                    return false;
                }
            }.start();
        }

        public ListenableFuture<?> triggerOnAdd(final Host host) {
            return this.executor.submit((Runnable)new ExceptionCatchingRunnable(){

                @Override
                public void runMayThrow() throws InterruptedException, ExecutionException {
                    Manager.this.onAdd(host);
                }
            });
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Loose catch block
         */
        private void onAdd(final Host host) throws InterruptedException, ExecutionException {
            block17: {
                if (this.isClosed()) {
                    return;
                }
                logger.info("New Cassandra host {} added", (Object)host);
                if (this.connectionFactory.protocolVersion == 2 && !this.supportsProtocolV2(host)) {
                    this.logUnsupportedVersionProtocol(host);
                    return;
                }
                boolean locked = host.notificationsLock.tryLock(NOTIF_LOCK_TIMEOUT_SECONDS, TimeUnit.SECONDS);
                if (!locked) {
                    logger.warn("Could not acquire notifications lock within {} seconds, ignoring ADD notification for {}", (Object)NOTIF_LOCK_TIMEOUT_SECONDS, (Object)host);
                    return;
                }
                try {
                    this.loadBalancingPolicy().onAdd(host);
                    if (this.loadBalancingPolicy().distance(host) == HostDistance.IGNORED) {
                        host.setUp();
                        for (Host.StateListener listener : this.listeners) {
                            listener.onAdd(host);
                        }
                        return;
                    }
                    try {
                        this.prepareAllQueries(host);
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                    catch (UnsupportedProtocolVersionException e) {
                        this.logUnsupportedVersionProtocol(host);
                        host.notificationsLock.unlock();
                        return;
                    }
                    catch (ClusterNameMismatchException e) {
                        this.logClusterNameMismatch(host, e.expectedClusterName, e.actualClusterName);
                        host.notificationsLock.unlock();
                        return;
                    }
                    this.controlConnection.onAdd(host);
                    ArrayList<ListenableFuture<Boolean>> futures = new ArrayList<ListenableFuture<Boolean>>(this.sessions.size());
                    for (SessionManager s : this.sessions) {
                        futures.add(s.maybeAddPool(host, this.blockingExecutor));
                    }
                    ListenableFuture f = Futures.allAsList(futures);
                    Futures.addCallback((ListenableFuture)f, (FutureCallback)new FutureCallback<List<Boolean>>(){

                        public void onSuccess(List<Boolean> poolCreationResults) {
                            if (Iterables.any(poolCreationResults, (Predicate)Predicates.equalTo((Object)false))) {
                                logger.debug("Connection pool cannot be created, not marking {} UP", (Object)host);
                                return;
                            }
                            host.setUp();
                            for (Host.StateListener listener : Manager.this.listeners) {
                                listener.onAdd(host);
                            }
                        }

                        public void onFailure(Throwable t) {
                            if (!(t instanceof InterruptedException)) {
                                logger.error("Unexpected error while adding node: while this shouldn't happen, this shouldn't be critical", t);
                            }
                        }
                    });
                    f.get();
                    for (SessionManager s : this.sessions) {
                        s.updateCreatedPools(this.blockingExecutor);
                    }
                    {
                        break block17;
                        catch (Throwable throwable) {
                            throw throwable;
                        }
                    }
                }
                finally {
                    host.notificationsLock.unlock();
                }
            }
        }

        public ListenableFuture<?> triggerOnRemove(final Host host) {
            return this.executor.submit((Runnable)new ExceptionCatchingRunnable(){

                @Override
                public void runMayThrow() throws InterruptedException, ExecutionException {
                    Manager.this.onRemove(host);
                }
            });
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void onRemove(Host host) throws InterruptedException, ExecutionException {
            if (this.isClosed()) {
                return;
            }
            boolean locked = host.notificationsLock.tryLock(NOTIF_LOCK_TIMEOUT_SECONDS, TimeUnit.SECONDS);
            if (!locked) {
                logger.warn("Could not acquire notifications lock within {} seconds, ignoring REMOVE notification for {}", (Object)NOTIF_LOCK_TIMEOUT_SECONDS, (Object)host);
                return;
            }
            try {
                host.setDown();
                logger.debug("Removing host {}", (Object)host);
                this.loadBalancingPolicy().onRemove(host);
                this.controlConnection.onRemove(host);
                for (SessionManager s : this.sessions) {
                    s.onRemove(host);
                }
                for (Host.StateListener listener : this.listeners) {
                    listener.onRemove(host);
                }
            }
            finally {
                host.notificationsLock.unlock();
            }
        }

        public boolean signalConnectionFailure(Host host, ConnectionException exception, boolean isHostAddition, boolean markSuspected) {
            if (!this.isFullyInit || this.isClosed()) {
                return true;
            }
            boolean isDown = host.signalConnectionFailure(exception);
            if (isDown) {
                if (isHostAddition || !markSuspected) {
                    this.triggerOnDown(host, isHostAddition);
                } else {
                    this.onSuspected(host);
                }
            }
            return isDown;
        }

        private boolean supportsProtocolV2(Host newHost) {
            return newHost.getCassandraVersion() == null || newHost.getCassandraVersion().getMajor() >= 2;
        }

        public void removeHost(Host host, boolean isInitialConnection) {
            if (host == null) {
                return;
            }
            if (this.metadata.remove(host)) {
                if (isInitialConnection) {
                    logger.warn("You listed {} in your contact points, but it could not be reached at startup", (Object)host);
                } else {
                    logger.info("Cassandra host {} removed", (Object)host);
                    this.triggerOnRemove(host);
                }
            }
        }

        public void ensurePoolsSizing() {
            for (SessionManager session : this.sessions) {
                for (HostConnectionPool pool : session.pools.values()) {
                    pool.ensureCoreConnections();
                }
            }
        }

        public PreparedStatement addPrepared(PreparedStatement stmt) {
            PreparedStatement previous = this.preparedQueries.putIfAbsent(stmt.getPreparedId().id, stmt);
            if (previous != null) {
                logger.warn("Re-preparing already prepared query {}. Please note that preparing the same query more than once is generally an anti-pattern and will likely affect performance. Consider preparing the statement only once.", (Object)stmt.getQueryString());
                return previous;
            }
            return stmt;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void prepareAllQueries(Host host) throws InterruptedException, UnsupportedProtocolVersionException, ClusterNameMismatchException {
            if (this.preparedQueries.isEmpty()) {
                return;
            }
            logger.debug("Preparing {} prepared queries on newly up node {}", (Object)this.preparedQueries.size(), (Object)host);
            try {
                Connection connection = this.connectionFactory.open(host);
                try {
                    try {
                        ControlConnection.waitForSchemaAgreement(connection, this);
                    }
                    catch (ExecutionException e) {
                        // empty catch block
                    }
                    HashMultimap perKeyspace = HashMultimap.create();
                    for (PreparedStatement ps : this.preparedQueries.values()) {
                        String keyspace = ps.getQueryKeyspace() == null ? "" : ps.getQueryKeyspace();
                        perKeyspace.put((Object)keyspace, (Object)ps.getQueryString());
                    }
                    for (String keyspace : perKeyspace.keySet()) {
                        if (!keyspace.isEmpty()) {
                            connection.setKeyspace(keyspace);
                        }
                        ArrayList<Connection.Future> futures = new ArrayList<Connection.Future>(this.preparedQueries.size());
                        for (String query : perKeyspace.get((Object)keyspace)) {
                            futures.add(connection.write(new Requests.Prepare(query)));
                        }
                        for (Connection.Future future : futures) {
                            try {
                                future.get();
                            }
                            catch (ExecutionException e) {
                                logger.debug("Unexpected error while preparing queries on new/newly up host", (Throwable)e);
                            }
                        }
                    }
                }
                finally {
                    connection.closeAsync();
                }
            }
            catch (ConnectionException e) {
            }
            catch (AuthenticationException e) {
            }
            catch (BusyConnectionException busyConnectionException) {
                // empty catch block
            }
        }

        public void submitSchemaRefresh(final String keyspace, final String table) {
            logger.trace("Submitting schema refresh");
            this.executor.submit((Runnable)new ExceptionCatchingRunnable(){

                @Override
                public void runMayThrow() throws InterruptedException, ExecutionException {
                    Manager.this.controlConnection.refreshSchema(keyspace, table);
                }
            });
        }

        public void refreshSchemaAndSignal(final Connection connection, final DefaultResultSetFuture future, final ResultSet rs, final String keyspace, final String table) {
            if (logger.isDebugEnabled()) {
                logger.debug("Refreshing schema for {}{}", (Object)(keyspace == null ? "" : keyspace), (Object)(table == null ? "" : '.' + table));
            }
            this.executor.submit(new Runnable(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public void run() {
                    try {
                        if (!ControlConnection.waitForSchemaAgreement(connection, Manager.this)) {
                            logger.warn("No schema agreement from live replicas after {} s. The schema may not be up to date on some nodes.", (Object)Manager.this.configuration.getProtocolOptions().getMaxSchemaAgreementWaitSeconds());
                        }
                        ControlConnection.refreshSchema(connection, keyspace, table, Manager.this, false);
                    }
                    catch (Exception e) {
                        logger.error("Error during schema refresh ({}). The schema from Cluster.getMetadata() might appear stale. Asynchronously submitting job to fix.", (Object)e.getMessage());
                        Manager.this.submitSchemaRefresh(keyspace, table);
                    }
                    finally {
                        future.setResult(rs);
                    }
                }
            });
        }

        @Override
        public void handle(Message.Response response) {
            if (!(response instanceof Responses.Event)) {
                logger.error("Received an unexpected message from the server: {}", (Object)response);
                return;
            }
            ProtocolEvent event = ((Responses.Event)response).event;
            logger.debug("Received event {}, scheduling delivery", (Object)response);
            block0 : switch (event.type) {
                case TOPOLOGY_CHANGE: {
                    ProtocolEvent.TopologyChange tpc = (ProtocolEvent.TopologyChange)event;
                    InetSocketAddress tpAddr = this.translateAddress(tpc.node.getAddress());
                    switch (tpc.change) {
                        case NEW_NODE: {
                            final Host newHost = this.metadata.add(tpAddr);
                            if (newHost == null) break;
                            this.scheduledTasksExecutor.schedule(new ExceptionCatchingRunnable(){

                                @Override
                                public void runMayThrow() throws InterruptedException, ExecutionException {
                                    if (Manager.this.controlConnection.refreshNodeInfo(newHost)) {
                                        Manager.this.onAdd(newHost);
                                    } else {
                                        logger.debug("Not enough info for {}, ignoring host", (Object)newHost);
                                    }
                                }
                            }, (long)NEW_NODE_DELAY_SECONDS, TimeUnit.SECONDS);
                            break;
                        }
                        case REMOVED_NODE: {
                            this.removeHost(this.metadata.getHost(tpAddr), false);
                            break;
                        }
                        case MOVED_NODE: {
                            this.executor.submit((Runnable)new ExceptionCatchingRunnable(){

                                @Override
                                public void runMayThrow() {
                                    Manager.this.controlConnection.refreshNodeListAndTokenMap();
                                }
                            });
                        }
                    }
                    break;
                }
                case STATUS_CHANGE: {
                    ProtocolEvent.StatusChange stc = (ProtocolEvent.StatusChange)event;
                    InetSocketAddress stAddr = this.translateAddress(stc.node.getAddress());
                    switch (stc.status) {
                        case UP: {
                            final Host hostUp = this.metadata.getHost(stAddr);
                            if (hostUp == null) {
                                final Host h = this.metadata.add(stAddr);
                                if (h == null) {
                                    return;
                                }
                                this.scheduledTasksExecutor.schedule(new ExceptionCatchingRunnable(){

                                    @Override
                                    public void runMayThrow() throws InterruptedException, ExecutionException {
                                        if (Manager.this.controlConnection.refreshNodeInfo(h)) {
                                            Manager.this.onAdd(h);
                                        } else {
                                            logger.debug("Not enough info for {}, ignoring host", (Object)h);
                                        }
                                    }
                                }, (long)NEW_NODE_DELAY_SECONDS, TimeUnit.SECONDS);
                                break;
                            }
                            this.executor.submit((Runnable)new ExceptionCatchingRunnable(){

                                @Override
                                public void runMayThrow() throws InterruptedException, ExecutionException {
                                    if (Manager.this.controlConnection.refreshNodeInfo(hostUp)) {
                                        Manager.this.onUp(hostUp);
                                    } else {
                                        logger.debug("Not enough info for {}, ignoring host", (Object)hostUp);
                                    }
                                }
                            });
                            break;
                        }
                        case DOWN: {
                            Host hostDown = this.metadata.getHost(stAddr);
                            if (hostDown == null) break;
                            this.triggerOnDown(hostDown);
                        }
                    }
                    break;
                }
                case SCHEMA_CHANGE: {
                    ProtocolEvent.SchemaChange scc = (ProtocolEvent.SchemaChange)event;
                    switch (scc.change) {
                        case CREATED: {
                            if (scc.table.isEmpty()) {
                                this.submitSchemaRefresh(scc.keyspace, null);
                                break block0;
                            }
                            this.submitSchemaRefresh(scc.keyspace, scc.table);
                            break block0;
                        }
                        case DROPPED: {
                            if (scc.table.isEmpty()) {
                                Cluster.this.manager.metadata.removeKeyspace(scc.keyspace);
                                break block0;
                            }
                            KeyspaceMetadata keyspace = Cluster.this.manager.metadata.getKeyspace(scc.keyspace);
                            if (keyspace == null) {
                                logger.warn("Received a DROPPED notification for {}.{}, but this keyspace is unknown in our metadata", (Object)scc.keyspace, (Object)scc.table);
                                break block0;
                            }
                            keyspace.removeTable(scc.table);
                            break block0;
                        }
                        case UPDATED: {
                            if (scc.table.isEmpty()) {
                                this.submitSchemaRefresh(scc.keyspace, null);
                                break block0;
                            }
                            this.submitSchemaRefresh(scc.keyspace, scc.table);
                        }
                    }
                }
            }
        }

        void refreshConnectedHosts() {
            Host ccHost = this.controlConnection.connectedHost();
            if (ccHost == null || this.loadBalancingPolicy().distance(ccHost) != HostDistance.LOCAL) {
                this.controlConnection.reconnect();
            }
            for (SessionManager s : this.sessions) {
                s.updateCreatedPools(this.executor);
            }
        }

        void refreshConnectedHost(Host host) {
            Host ccHost = this.controlConnection.connectedHost();
            if (ccHost == null || ccHost.equals(host) && this.loadBalancingPolicy().distance(ccHost) != HostDistance.LOCAL) {
                this.controlConnection.reconnect();
            }
            for (SessionManager s : this.sessions) {
                s.updateCreatedPools(host, this.executor);
            }
        }

        private class ClusterCloseFuture
        extends CloseFuture.Forwarding {
            ClusterCloseFuture(List<CloseFuture> futures) {
                super(futures);
            }

            @Override
            public CloseFuture force() {
                Manager.this.shutdownNow((ExecutorService)Manager.this.executor);
                return super.force();
            }

            @Override
            protected void onFuturesDone() {
                new Thread("Shutdown-checker"){

                    @Override
                    public void run() {
                        try {
                            Manager.this.reconnectionExecutor.awaitTermination(Long.MAX_VALUE, TimeUnit.SECONDS);
                            Manager.this.scheduledTasksExecutor.awaitTermination(Long.MAX_VALUE, TimeUnit.SECONDS);
                            Manager.this.executor.awaitTermination(Long.MAX_VALUE, TimeUnit.SECONDS);
                            Manager.this.blockingExecutor.awaitTermination(Long.MAX_VALUE, TimeUnit.SECONDS);
                            Manager.this.connectionFactory.shutdown();
                            Manager.this.reaper.shutdown();
                            ClusterCloseFuture.this.set(null);
                        }
                        catch (InterruptedException e) {
                            Thread.currentThread().interrupt();
                            ClusterCloseFuture.this.setException(e);
                        }
                    }
                }.start();
            }
        }
    }

    public static class Builder
    implements Initializer {
        private String clusterName;
        private final List<InetSocketAddress> addresses = new ArrayList<InetSocketAddress>();
        private final List<InetAddress> rawAddresses = new ArrayList<InetAddress>();
        private int port = 9042;
        private int maxSchemaAgreementWaitSeconds = 10;
        private int protocolVersion = -1;
        private AuthProvider authProvider = AuthProvider.NONE;
        private LoadBalancingPolicy loadBalancingPolicy;
        private ReconnectionPolicy reconnectionPolicy;
        private RetryPolicy retryPolicy;
        private AddressTranslater addressTranslater;
        private ProtocolOptions.Compression compression = ProtocolOptions.Compression.NONE;
        private SSLOptions sslOptions = null;
        private boolean metricsEnabled = true;
        private boolean jmxEnabled = true;
        private PoolingOptions poolingOptions;
        private SocketOptions socketOptions;
        private QueryOptions queryOptions;
        private Collection<Host.StateListener> listeners;

        @Override
        public String getClusterName() {
            return this.clusterName;
        }

        @Override
        public List<InetSocketAddress> getContactPoints() {
            if (this.rawAddresses.isEmpty()) {
                return this.addresses;
            }
            ArrayList<InetSocketAddress> allAddresses = new ArrayList<InetSocketAddress>(this.addresses);
            for (InetAddress address : this.rawAddresses) {
                allAddresses.add(new InetSocketAddress(address, this.port));
            }
            return allAddresses;
        }

        public Builder withClusterName(String name) {
            this.clusterName = name;
            return this;
        }

        public Builder withPort(int port) {
            this.port = port;
            return this;
        }

        public Builder withMaxSchemaAgreementWaitSeconds(int maxSchemaAgreementWaitSeconds) {
            if (maxSchemaAgreementWaitSeconds <= 0) {
                throw new IllegalArgumentException("Max schema agreement wait must be greater than zero");
            }
            this.maxSchemaAgreementWaitSeconds = maxSchemaAgreementWaitSeconds;
            return this;
        }

        public Builder withProtocolVersion(int version) {
            if (version >= 0 && version != 1 && version != 2) {
                throw new IllegalArgumentException(String.format("Unsupported protocol version %d; valid values are 1, 2 or negative (for auto-detect).", version));
            }
            this.protocolVersion = version;
            return this;
        }

        public Builder addContactPoint(String address) {
            if (address == null) {
                throw new NullPointerException();
            }
            try {
                this.rawAddresses.add(InetAddress.getByName(address));
                return this;
            }
            catch (UnknownHostException e) {
                throw new IllegalArgumentException(e.getMessage());
            }
        }

        public Builder addContactPoints(String ... addresses) {
            for (String address : addresses) {
                this.addContactPoint(address);
            }
            return this;
        }

        public Builder addContactPoints(InetAddress ... addresses) {
            Collections.addAll(this.rawAddresses, addresses);
            return this;
        }

        public Builder addContactPoints(Collection<InetAddress> addresses) {
            this.rawAddresses.addAll(addresses);
            return this;
        }

        public Builder addContactPointsWithPorts(Collection<InetSocketAddress> addresses) {
            this.addresses.addAll(addresses);
            return this;
        }

        public Builder withLoadBalancingPolicy(LoadBalancingPolicy policy) {
            this.loadBalancingPolicy = policy;
            return this;
        }

        public Builder withReconnectionPolicy(ReconnectionPolicy policy) {
            this.reconnectionPolicy = policy;
            return this;
        }

        public Builder withRetryPolicy(RetryPolicy policy) {
            this.retryPolicy = policy;
            return this;
        }

        public Builder withAddressTranslater(AddressTranslater translater) {
            this.addressTranslater = translater;
            return this;
        }

        public Builder withCredentials(String username, String password) {
            this.authProvider = new PlainTextAuthProvider(username, password);
            return this;
        }

        public Builder withAuthProvider(AuthProvider authProvider) {
            this.authProvider = authProvider;
            return this;
        }

        public Builder withCompression(ProtocolOptions.Compression compression) {
            this.compression = compression;
            return this;
        }

        public Builder withoutMetrics() {
            this.metricsEnabled = false;
            return this;
        }

        public Builder withSSL() {
            this.sslOptions = new SSLOptions();
            return this;
        }

        public Builder withSSL(SSLOptions sslOptions) {
            this.sslOptions = sslOptions;
            return this;
        }

        public Builder withInitialListeners(Collection<Host.StateListener> listeners) {
            this.listeners = listeners;
            return this;
        }

        public Builder withoutJMXReporting() {
            this.jmxEnabled = false;
            return this;
        }

        public Builder withPoolingOptions(PoolingOptions options) {
            this.poolingOptions = options;
            return this;
        }

        public Builder withSocketOptions(SocketOptions options) {
            this.socketOptions = options;
            return this;
        }

        public Builder withQueryOptions(QueryOptions options) {
            this.queryOptions = options;
            return this;
        }

        @Override
        public Configuration getConfiguration() {
            Policies policies = new Policies(this.loadBalancingPolicy == null ? Policies.defaultLoadBalancingPolicy() : this.loadBalancingPolicy, this.reconnectionPolicy == null ? Policies.defaultReconnectionPolicy() : this.reconnectionPolicy, this.retryPolicy == null ? Policies.defaultRetryPolicy() : this.retryPolicy, this.addressTranslater == null ? Policies.defaultAddressTranslater() : this.addressTranslater);
            return new Configuration(policies, new ProtocolOptions(this.port, this.protocolVersion, this.maxSchemaAgreementWaitSeconds, this.sslOptions, this.authProvider).setCompression(this.compression), this.poolingOptions == null ? new PoolingOptions() : this.poolingOptions, this.socketOptions == null ? new SocketOptions() : this.socketOptions, this.metricsEnabled ? new MetricsOptions(this.jmxEnabled) : null, this.queryOptions == null ? new QueryOptions() : this.queryOptions);
        }

        @Override
        public Collection<Host.StateListener> getInitialListeners() {
            return this.listeners == null ? Collections.emptySet() : this.listeners;
        }

        public Cluster build() {
            return Cluster.buildFrom(this);
        }
    }

    public static interface Initializer {
        public String getClusterName();

        public List<InetSocketAddress> getContactPoints();

        public Configuration getConfiguration();

        public Collection<Host.StateListener> getInitialListeners();
    }
}

