/*
   derived from aspi.c - from chris@alderan.sdata.de

   mbarth2193@aol.com
*/
#include <windows.h>
#include <stdio.h>
#include <fcntl.h>
#include <io.h>
#include <string.h>
#include <malloc.h>
#include <ctype.h>
#include <errno.h>


#define EX_SYSTEM -1
#ifndef TAPE_ID
#define TAPE_ID "0"
#endif

#include "ntapi.h"

// #ifndef CTCTRL
// #include "tar.h"
// #endif

void tapeinit (void);

/*
 * The handle of our tape device
 */
HANDLE hTape;

#define TARGET hTape
#define BLOCKSIZE 512


#ifndef CTCTRL

/* Flags to for use with the streamer (detected at tapeopen()) */
static int tapefile, no_rewind;

/* This holds the max., and min. recordsize, the streamer can deal with */
static blocklimit lim;
static long recsize;    /* the record size to use */
static int logrec;      /* log2(recordsize) */
static long apos;       /* this holds the current archive file pointer */
static int write_file_mark_on_close;
static int end_of_medium;

/*
 * The next two functions are called when you open a file.
 * They check if the filename specifies the scsi-tape.
 * The filenames used for that are "/dev/ct" - "rewind cartridge tape"
 * and "/dev/nrct" - "no rewind cartridge tape" (no rewind on close)
 * I just choose those names "/dev/ct" and "/dev/nrct" because the
 * filenames are unusual to exist on a MSDOS filesystem and also because
 * I'm used to them from my ix386. Anyway, there are no other dependencies
 * on those names in the whole program. So, if you prefer to change the
 * names to "/dev/rmt0" or whatever, just change them !
 * As you can see in aspicreate() and aspiopen(), I still open a
 * MSDOS-output-dummy-file. "nul:" is kind of /dev/null for MSDOS
 * and I had to do this because the MSDOS code from tar still
 * does some operations on this file like setmode(stream) etc. and
 * I was too lazy to change the code in tar. So, with this they
 * may do setmode()'s on the null-device as long as they want :-)
 */

/*
 * Create an archive file
 */

tapecreat (char *path, int mode)
{
  end_of_medium = 0;

  if (!strcmp (path, "/dev/ct"))
  {
     apos = 0l;
     no_rewind = 0;
     tapeinit();
     write_file_mark_on_close = 1;
     return (tapefile = _creat ("NUL", mode));
  }
  else
     if (!strcmp (path,"/dev/nrct"))
     {
        apos = 0l;
        no_rewind = 1;
        tapeinit();
        write_file_mark_on_close = 1;
        return (tapefile = _creat ("NUL", mode));
     }

  tapefile = -1;
  return creat (path, mode);
}

/*
 * Open an archive file
 */

tapeopen (const char *path, int oflag, int mode, const char *command)
{
  end_of_medium = 0;

  if (!strcmp (path, "/dev/ct"))
  {
    apos      = 0l;
    no_rewind = 0;
    tapeinit();
    if ((oflag & O_WRONLY) || (oflag & O_RDWR))
       write_file_mark_on_close = 1;
    else
       write_file_mark_on_close = 0;

    return (tapefile = open ("NUL", oflag, mode));
  }
  else
   if (!strcmp (path,"/dev/nrct"))
   {
      apos      = 0l;
      no_rewind = 1;
      tapeinit();
      if ((oflag & O_WRONLY) || (oflag & O_RDWR))
         write_file_mark_on_close = 1;
      else
         write_file_mark_on_close = 0;

      return (tapefile = open ("NUL", oflag, mode));
   }

  tapefile = -1;
  return open (path, oflag, mode);
}


/*
 * So, all the other functions now just check if (fileds == tapefile)
 * and do redirect the operation to the tape. Otherwise they just
 * hand the request to the original function (e.g aspiread()->read() etc.).
 */


/*
 * Read data from an archive file
 */

taperead (int fileds, char *buf, int nbyte)
{
  unsigned int *u;
  DWORD bytes_read, Tstatus;

  if (fileds == tapefile)
  {
#ifdef FIXED
     u = (unsigned int *) &nbyte;    /* Hack,  I know */
     if (nbyte % (int) recsize)
     {
        fprintf (stderr, "taperead: Illegal blocklen for tape: %u\n", nbyte);
        tapeclose (fileds);
        exit(EX_SYSTEM);
     }

     Tstatus = APIRead (TARGET, buf, (long) *u, logrec, 0, 1, &bytes_read);
#else
     Tstatus = APIRead (TARGET, buf, (long) nbyte, 0, 0, 0, &bytes_read);
#endif
     if (Tstatus == ERROR_END_OF_MEDIA)
     {
        errno = ENOSPC;
        apos += (long) bytes_read;
        return (int) bytes_read;
     }

     if (!Tstatus)
     {                                 /* If any error ... */
        Tstatus = GetLastError ();
        fprintf (stderr, "taperead: %lu\n", Tstatus);
        tapeclose (fileds);
        exit (EX_SYSTEM);
     }
     else
      if (Tstatus > 1)                /* read doesn't return TRUE */
      {
         fprintf (stderr, "taperead: ");
         APIerror (Tstatus);
         tapeclose (fileds);
         exit (EX_SYSTEM);
      }
#ifdef FIXED
     apos += (long) *u;
#else
     apos += nbyte;
#endif
     return nbyte;
  }
  return read (fileds, buf, nbyte);
}

/*
 * Write data to an archive file
 */

tapewrite (int fileds, char *buf, int nbyte)
{
  unsigned int *u;
  DWORD bytes_written, Tstatus;

  if (fileds == tapefile)
  {
     if (end_of_medium)
     {
        /*
         * This only happens when an end_of_medium condition was detected
         * at the previous aspiwrite() call but all bytes were written to
         * the tape anyway. In this case we don't make anymore attempts to
         * write to the tape but return an ENOSPC error and set the size
         * of transfered bytes to 0
         */
        errno = ENOSPC;
        return 0;
     }

#ifdef FIXED
     u = (unsigned int *) &nbyte;     /* Hack , i know */

     if (nbyte % (int) recsize)
     {
        fprintf (stderr, "tapewrite: Illegal blocklen for tape: %u\n", nbyte);
        aspiclose(fileds);
        exit (EX_SYSTEM);
     }

     Tstatus = APIWrite (TARGET, buf, (long) *u, logrec, 1, &bytes_written);
#else
     Tstatus = APIWrite (TARGET, buf, (long) nbyte, 0, 0, &bytes_written);
#endif

     if (Tstatus == ERROR_END_OF_MEDIA)
     {
        errno = ENOSPC;
        end_of_medium = 1;
        apos += (long) bytes_written;
        return (int) bytes_written;
     }

     if (!Tstatus)
     {                    /* any other error we can't recover */
        Tstatus = GetLastError ();
        fprintf (stderr, "tapewrite: %lu\n", Tstatus);
        tapeclose (fileds);
        exit (EX_SYSTEM);
     }
     else
      if (Tstatus > 1)                /* write doesn't return TRUE */
      {
         fprintf (stderr, "tapewrite: ");
         APIerror (Tstatus);
         tapeclose (fileds);
         exit (EX_SYSTEM);
      }
#ifdef FIXED
     apos += (long) *u;
#else
     apos += (long) nbyte;
#endif
     return nbyte;
  }

  return write (fileds, buf, nbyte);
}

/*
 * close an archive file
 */

tapeclose (int fileds)
{
  int i;

  if (fileds == tapefile)
  {
    /* first, write a filemark ! */
    if (write_file_mark_on_close && (!end_of_medium))
       if ((i = APIWriteFilemarks (TARGET, 1l, 0, 0)))
          fprintf (stderr,"tapeclose: Error writing filemark\n");

    /* then rewind the tape if we have to */
    if (!no_rewind)
    {
       if ((i = APIRewind (TARGET, 0)))
          fprintf (stderr,"tapeclose: Error rewinding the tape\n");
    }
    else
     if (!write_file_mark_on_close)
     {
       /*
        * This means we've been reading an archive on a "/dev/nrct" tape.
        * In this case we have seek over the filemark at the end of the
        * archive.
        */
        if ((i = APISpace(TARGET, 1, 1l)))
           fprintf (stderr, "tapeclose: Error seeking filemark\n");
     }
    apos = 0l;
  }
  CloseHandle (hTape);
  return _close (fileds);
}

/*
 * Seek on the archive file
 */

long tapelseek (int fileds, long offset, int whence)
{
  int i;
  long newpos;

  if (fileds == tapefile)
  {
     /*
      * NOTE: seeking backwards is not supported by every streamer.
      * The Archive Viper 2150S does, but I don't know about others.
      */
      switch (whence)
      {
           case SEEK_SET: newpos = offset;
                          break;
           case SEEK_CUR: newpos = apos + offset;
                          break;

           default: fprintf (stderr, "tapelseek: mode %d not supported\n", whence);
                    tapeclose(fileds);
                    exit(EX_SYSTEM);
                    break;
      }

      if (newpos % recsize)
      {
         fprintf (stderr,
                  "tapelseek: can't seek to a non-block-boundary position (%ld)\n",
                   newpos);
         tapeclose (fileds);
         exit(EX_SYSTEM);
      }

      if ((i = APISpace (TARGET, 0, (newpos-apos) >> logrec)))
      {
         fprintf (stderr, "tapelseek: APISpace retuned %d\n", i);
         tapeclose (fileds);
         exit (EX_SYSTEM);
      }

      apos = newpos;
      return newpos;
  }
  return _lseek(fileds, offset, whence);
}

#endif  /* CTCTRL */

/*
 * Here we start with all that tape-api stuff !!!
 */

/*
 * Initialize Tape-API and the tape device
 */

void tapeinit (void)
{
  DWORD l, varlen, Tstatus;
  TAPE_GET_DRIVE_PARAMETERS tapedrive;
  TAPE_SET_DRIVE_PARAMETERS settapepar;
  TAPE_SET_MEDIA_PARAMETERS setmediapar;
  char *getenv(), *tapeid;
  char devname[30] = "\\\\.\\Tape";
  int retry;

 /*
  * Now find out on which device we have to operate
  */
  tapeid = getenv ("TAPEID");
  if (!tapeid)
     tapeid = TAPE_ID;

  if (!isdigit (*tapeid))
  {
     fprintf (stderr, "tapeinit: Illegal TAPEID: \"%s\"\n", tapeid);
     exit(1);
  }
  else
     strcat (devname, tapeid);

  hTape = CreateFile (devname,
                      GENERIC_READ | GENERIC_WRITE,
                      0,
                      0,
                      OPEN_EXISTING,
                      0,
                      NULL);

  if (hTape == INVALID_HANDLE_VALUE)
  {
     fprintf (stderr,
             "tapeinit: Can't open device \"%s\"\n", devname);
     exit (EX_SYSTEM);
  }

#ifndef CTCTRL
/*
 *  Now get the max. and min. recordsize for the streamer
 */

  for (retry = 10; retry; retry--)
  {
     varlen = sizeof (tapedrive);
     if ((Tstatus = GetTapeParameters (hTape,
                                       GET_TAPE_DRIVE_INFORMATION,
                                       &varlen,
                                       &tapedrive)) == NO_ERROR)
        break;
  }

  if (!retry)
  {
     fprintf (stderr,"tapeinit: Can't read blocklimits. %d\n", Tstatus);
     CloseHandle (hTape);
     exit (EX_SYSTEM);
  }
  else
  {
     lim.max = tapedrive.MaximumBlockSize;
     lim.min = tapedrive.MinimumBlockSize;

     /* if compression is supported - use it */
     if (tapedrive.FeaturesLow & TAPE_DRIVE_COMPRESSION)
     {
        memset (&settapepar, 0, sizeof (settapepar));
        settapepar.Compression        = TRUE;
        settapepar.DataPadding        = tapedrive.DataPadding;
        settapepar.ReportSetmarks     = tapedrive.ReportSetmarks;
        settapepar.EOTWarningZoneSize = tapedrive.EOTWarningZoneSize;
        Tstatus = SetTapeParameters (hTape,
                                     SET_TAPE_DRIVE_INFORMATION,
                                     &settapepar);
     }
	 
	 Tstatus = GetTapeStatus (hTape);
	 if (Tstatus != NO_ERROR)
	 {
        fprintf (stderr,"GetTapeStatus: %d\n", Tstatus);
        CloseHandle (hTape);
        exit (EX_SYSTEM);
     }

	 setmediapar.BlockSize = BLOCKSIZE;
	 Tstatus = SetTapeParameters (hTape,
                                  SET_TAPE_MEDIA_INFORMATION,
                                  &setmediapar);
	 if (Tstatus != NO_ERROR)
	 {
	    fprintf (stderr,"SetMediaParameter: %d\n", Tstatus);
        CloseHandle (hTape);
        exit (EX_SYSTEM);
     }
          


  
  }

  if (lim.max == lim.min)
     recsize = lim.max;
  else
     if ((lim.min <= 512) && (lim.max >= 512))
        recsize = 512;
     else
        recsize = lim.min;

  l      = recsize;
  logrec = 0;

  while ((l >>= 1))
      ++logrec;    /* logrec = log2(recsize) */

#endif /* CTCTRL */
}

int APIErase (HANDLE hTape, int immediate, int lng)
{
  DWORD Tstatus, erasemod;

  switch (lng)
  {
      case 0: erasemod = TAPE_ERASE_SHORT;
              break;

      case 1: erasemod = TAPE_ERASE_LONG;
              break;
  }

  if ((Tstatus = EraseTape (hTape,
                            erasemod,
                            (BOOL) immediate)) != NO_ERROR)
     APIerror (Tstatus);


  return(Tstatus);
}


int APILoad (HANDLE hTape, int immediate, int load, int retention, int eot)
{
  DWORD Tstatus;

  if (retention)
     Tstatus = PrepareTape (hTape,
                            TAPE_TENSION,
                            FALSE);
  else
   if (!load)
      Tstatus = PrepareTape (hTape,
                             TAPE_UNLOAD,
                             FALSE);
   else
      if (load)
         Tstatus = PrepareTape (hTape,
                                TAPE_LOAD,
                                FALSE);

  if (Tstatus == ERROR_INVALID_FUNCTION)
     return (0);
  else
     return (Tstatus);
}


int APIRead (HANDLE hTape, char *buf, DWORD len, int l2bf, int sili, int fixed, DWORD *bytesread)
{
  DWORD length, Tstatus;

  length = (fixed) ? len >> l2bf : len;

  Tstatus = ReadFile (hTape,
                      buf,
                      length,
                      bytesread,
                      NULL);
  return (Tstatus);
}

int APIReadBlockLimits (HANDLE hTape, blocklimit *lim)
{
  DWORD varlen, Tstatus;
  TAPE_GET_DRIVE_PARAMETERS tapedrive;

  varlen = sizeof (tapedrive);

  if ((Tstatus = GetTapeParameters (hTape,
                                    GET_TAPE_DRIVE_INFORMATION,
                                    &varlen,
                                    &tapedrive)) != NO_ERROR)
  {
     APIerror (Tstatus);
  }
  else
  {
     lim->max =  tapedrive.MaximumBlockSize;
     lim->min =  tapedrive.MinimumBlockSize;
  }

  return (Tstatus);
}

int APIRewind (HANDLE hTape, int immediate)
{
  DWORD Tstatus;

  if ((Tstatus = SetTapePosition (hTape,
                                  TAPE_REWIND,
                                  0,
                                  0,
                                  0,
                                  (BOOL) immediate)) != NO_ERROR)
     APIerror (Tstatus);

  return (Tstatus);
}


int APISpace (HANDLE hTape, int code, long count)
{
  long lcount, posmode, Tstatus;

  lcount = count;

  switch (code)
  {
      case 0: posmode = TAPE_SPACE_RELATIVE_BLOCKS;
              break;

      case 1: posmode = TAPE_SPACE_FILEMARKS;
              break;

      case 3: posmode = TAPE_SPACE_END_OF_DATA;
              break;

  }

  if ((Tstatus = SetTapePosition (hTape,
                                  posmode,
                                  1,
                                  lcount,
                                  0,
                                  FALSE)) != NO_ERROR)
     APIerror (Tstatus);


  return(Tstatus);
}


int APIWrite (HANDLE hTape, char *buf, DWORD len, int l2bf, int fixed, DWORD *byteswritten)
{
  DWORD length, Tstatus;

  length = (fixed) ? len >> l2bf : len;

  Tstatus = WriteFile (hTape,
                       buf,
                       length,
                       byteswritten,
                       NULL);
  return (Tstatus);
}



int APIWriteFilemarks (HANDLE hTape, DWORD len, int setmark, int immediate)
{
  DWORD length, Tstatus;

  length = len;

  if ((Tstatus = WriteTapemark (hTape,
                               TAPE_FILEMARKS,
                               length,
                               (BOOL) immediate)) != NO_ERROR)
     APIerror (Tstatus);

  return (Tstatus);
}

APIerror (DWORD Tstatus)
{
    switch (Tstatus)
    {
         case ERROR_BEGINNING_OF_MEDIA:
              fprintf (stderr, " Beginnung of media\n");
              break;

         case ERROR_NO_DATA_DETECTED:
              fprintf (stderr, " No data detected\n");
              break;

         case ERROR_END_OF_MEDIA:
              fprintf (stderr, " End of media\n");
              break;

         case ERROR_MEDIA_CHANGED:
              fprintf (stderr, "  Media changed\n");
              break;

         case ERROR_UNABLE_TO_LOCK_MEDIA:
              fprintf (stderr, "  Unable to lock media\n");
              break;

         case ERROR_UNABLE_TO_UNLOAD_MEDIA:
              fprintf (stderr, "  Unable to unload media\n");
              break;

         case ERROR_NO_MEDIA_IN_DRIVE :
              fprintf (stderr, "  No media in drive\n");
              break;

         case ERROR_BUS_RESET:
              fprintf (stderr, " Bus reset\n");
              break;

         default: fprintf (stderr, "Tape API returned %lu\n", Tstatus);
    }
    return (0);
}

#ifdef TEST_API
void main (int argc, char *argv[])
{
  int i, j;
  DWORD l;
  char *buf;
  DWORD bytes_proc, Tstatus, varlen;
  TAPE_GET_DRIVE_PARAMETERS tapedrive;
  TAPE_GET_MEDIA_PARAMETERS tapemedia;
  TAPE_SET_DRIVE_PARAMETERS settapepar;

  tapeinit();

  printf ("\nrecsize: %ld, log2(recsize)= %d\n", recsize, logrec);

  if ((buf = malloc(32768)) == 0)
  {
     printf ("can't alloc buf\n");
     exit (0);
  }
  else
     memset (buf, 0, 32768);

  printf ("\nBlocklimits, max:%ld min:%ld\n", lim.max, lim.min);

  // varlen = 0;
  Tstatus = GetTapeParameters (hTape,
                               GET_TAPE_DRIVE_INFORMATION,
                               &varlen,
                               &tapedrive);
  if (Tstatus == NO_ERROR)
  {
     printf ("\nTape drive -Information\n\n");

     printf ("ECC %d\n", tapedrive.ECC);
     printf ("Compression %d\n",            tapedrive.Compression);
     printf ("DataPadding %d\n",            tapedrive.DataPadding);
     printf ("ReportSetmarks %d\n",         tapedrive.ReportSetmarks);
     printf ("DefaultBlockSize %ld\n",      tapedrive.DefaultBlockSize);
     printf ("MaximumBlockSize %ld\n",      tapedrive.MaximumBlockSize);
     printf ("MinimumBlockSize %ld\n",      tapedrive.MinimumBlockSize);
     printf ("MaximumPartitionCount %ld\n", tapedrive.MaximumPartitionCount);
     printf ("EOTWarningZoneSize %ld\n",    tapedrive.EOTWarningZoneSize);

     if (tapedrive.FeaturesLow & TAPE_DRIVE_COMPRESSION)
        printf ("TAPE_DRIVE_COMPRESSION\n");
     if (tapedrive.FeaturesLow & TAPE_DRIVE_ECC)
        printf ("TAPE_DRIVE_ECC\n");
     if (tapedrive.FeaturesLow & TAPE_DRIVE_ERASE_BOP_ONLY)
        printf ("TAPE_DRIVE_ERASE_BOP_ONLY\n");
     if (tapedrive.FeaturesLow & TAPE_DRIVE_ERASE_LONG)
        printf ("TAPE_DRIVE_ERASE_LONG\n");
     if (tapedrive.FeaturesLow & TAPE_DRIVE_ERASE_IMMEDIATE)
        printf ("TAPE_DRIVE_ERASE_IMMEDIATE\n");
     if (tapedrive.FeaturesLow & TAPE_DRIVE_ERASE_SHORT)
        printf ("TAPE_DRIVE_ERASE_SHORT\n");
     if (tapedrive.FeaturesLow & TAPE_DRIVE_FIXED)
         printf ("TAPE_DRIVE_FIXED\n");
     if (tapedrive.FeaturesLow & TAPE_DRIVE_FIXED_BLOCK)
        printf ("TAPE_DRIVE_FIXED_BLOCK\n");
     if (tapedrive.FeaturesLow & TAPE_DRIVE_INITIATOR)
        printf ("TAPE_DRIVE_INITIATOR\n");
     if (tapedrive.FeaturesLow & TAPE_DRIVE_PADDING)
        printf ("TAPE_DRIVE_PADDING\n");
     if (tapedrive.FeaturesLow & TAPE_DRIVE_GET_ABSOLUTE_BLK)
        printf ("TAPE_DRIVE_GET_ABSOLUTE_BLK\n");
     if (tapedrive.FeaturesLow & TAPE_DRIVE_REPORT_SMKS)
        printf ("TAPE_DRIVE_REPORT_SMKS\n");
     if (tapedrive.FeaturesLow & TAPE_DRIVE_SELECT)
        printf ("TAPE_DRIVE_SELCET\n");
     if (tapedrive.FeaturesLow & TAPE_DRIVE_SET_EOT_WZ_SIZE)
        printf ("TAPE_DRIVE_SET_EOT_WZ_SIZEd\n");
     if (tapedrive.FeaturesLow & TAPE_DRIVE_TAPE_CAPACITY)
        printf ("TAPE_DRIVE_TAPE_CAPACITY\n");
     if (tapedrive.FeaturesLow & TAPE_DRIVE_TAPE_REMAINING)
        printf ("TAPE_DRIVE_TAPE_REMAINING\n");
     if (tapedrive.FeaturesLow & TAPE_DRIVE_VARIABLE_BLOCK)
        printf ("TAPE_DRIVE_VARIABLE_BLOCK\n");
     if (tapedrive.FeaturesLow & TAPE_DRIVE_WRITE_PROTECT)
        printf ("TAPE_DRIVE_WRITE_PROTECT\n");

     if (tapedrive.FeaturesHigh & TAPE_DRIVE_ABS_BLK_IMMED)
         printf ("TAPE_DRIVE_ABS_BLK_IMMED\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_ABSOLUTE_BLK)
        printf ("TAPE_DRIVE_ABSOLUTE_BLK\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_END_OF_DATA)
        printf ("TAPE_DRIVE_END_OF_DATA\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_FILEMARKS)
        printf ("TAPE_DRIVE_FILEMARKS\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_LOAD_UNLOAD)
        printf ("TAPE_DRIVE_LOAD_UNLOAD\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_LOAD_UNLD_IMMED)
        printf ("TAPE_DRIVE_LOAD_UNLD_IMMED\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_LOCK_UNLOCK)
         printf ("TAPE_DRIVE_LOCK_UNLOCK\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_LOCK_UNLK_IMMED)
        printf ("TAPE_DRIVE_LOCK_UNLK_IMMED\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_LOG_BLK_IMMED)
        printf ("TAPE_DRIVE_LOG_BLK_IMMED\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_LOGICAL_BLK)
        printf ("TAPE_DRIVE_LOGICAL_BLK\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_RELATIVE_BLKS)
        printf ("TAPE_DRIVE_RELATIVE_BLKS\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_REVERSE_POSITION)
        printf ("TAPE_DRIVE_REVERSE_POSITION\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_REWIND_IMMEDIATE)
        printf ("TAPE_DRIVE_REWIND_IMMEDIATE\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_SEQUENTIAL_FMKS)
        printf ("TAPE_DRIVE_SEQUENTIAL_FMKS\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_SEQUENTIAL_SMKS)
        printf ("TAPE_DRIVE_SEQUENTIAL_SMKS\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_SET_BLOCK_SIZE)
        printf ("TAPE_DRIVE_SET_BLOCK_SIZE\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_SET_COMPRESSION)
        printf ("TAPE_DRIVE_SET_COMPRESSION\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_SET_ECC)
        printf ("TAPE_DRIVE_SET_ECC\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_SET_PADDING)
         printf ("TAPE_DRIVE_SET_PADDING\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_SET_REPORT_SMKS)
        printf ("TAPE_DRIVE_SET_REPORT_SMKS\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_SETMARKS)
        printf ("TAPE_DRIVE_SETMARKS\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_SPACE_IMMEDIATE)
        printf ("TAPE_DRIVE_SPACE_IMMEDIATE\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_TENSION)
        printf ("TAPE_DRIVE_TENSION\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_TENSION_IMMED)
        printf ("TAPE_DRIVE_TENSION_IMMED\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_WRITE_FILEMARKS)
        printf ("TAPE_DRIVE_WRITE_FILEMARKS\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_WRITE_LONG_FMKS)
        printf ("TAPE_DRIVE_WRITE_LONG_FMKS\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_WRITE_MARK_IMMED)
        printf ("TAPE_DRIVE_WRITE_MARK_IMMED\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_WRITE_SETMARKS)
        printf ("TAPE_DRIVE_WRITE_SETMARKS\n");
     if (tapedrive.FeaturesHigh & TAPE_DRIVE_WRITE_SHORT_FMKS)
        printf ("TAPE_DRIVE_WRITE_SHORT_FMKS\n");
  }
  else
     printf ("GetTapeParameters (Drive) returned %ld\n", Tstatus);

  // varlen = 0;
  Tstatus = GetTapeParameters (hTape,
                               GET_TAPE_MEDIA_INFORMATION,
                               &varlen,
                               &tapemedia);
  if (Tstatus == NO_ERROR)
  {
     printf ("\nMedia -Information\n\n");

	 printf ("Blocksize %d\n", tapemedia.BlockSize);
	 printf ("Partitioncount %d\n", tapemedia.PartitionCount);

  }

/*
  memset (&settapepar, 0, sizeof (settapepar));
  settapepar.Compression = TRUE;
  settapepar.ReportSetmarks = TRUE;

  Tstatus = SetTapeParameters (hTape,
                               SET_TAPE_DRIVE_INFORMATION,
                               &settapepar);
  printf ("Set parameter returned %d\n", Tstatus);
  i = APILoad (TARGET, 0, 1, 0);
  printf ("\nLoad returned %d\n -->", i);
*/

#if 0
  printf ("seeking to tape file 2...\n");

  if ((i = APISpace (TARGET, 1, (DWORD) 1)))
     printf ("Error %d\n", i );
  else
     printf("OK\n");

  printf ("seeking to tape block 400 on file 2...\n");
  if ((i = APISpace (TARGET, 0, (DWORD) 400)))
     printf ("Error %d\n", i );
  else
     printf ("OK\n");

  printf ("Writing two sectors..(this should return an error)...\n");
  i = APIWrite (TARGET, buf, 1024l, 9, 0, &bytes_proc);
  printf ("Write returned %d\n", i);

  printf ("seeking back two sectors...\n");
  if ((i = APISpace(TARGET, 0, -2l)))
     printf ("Error %d\n", i);
  else
     printf ("OK\n");

  printf ("reading back the two sectors...\n");
  i = APIRead (TARGET, buf, 1024l, 9, 0, 0, &bytes_proc);
  printf ("Read returned %d\n", i);

  printf ("rewinding the tape...\n");
  i = APIRewind (TARGET, 0);
  printf ("Rewind returned %d\n", i);

  printf ("seeking to the end of the data...\n");
  if ((i = APISpace (TARGET, 3, 0l)) )
     printf ("Error %d\n", i );
  else
     printf("OK\n");

  printf ("Writing two sectors...\n");
  i = APIWrite (TARGET, buf, 1024l, 9, 0, &bytes_proc);
  printf ("Write returned %d\n", i);

  printf ("Writing two sectors...\n");
  i = APIWrite (TARGET, buf, 1024l, 9, 0, &bytes_proc);
  printf ("Write returned %d\n", i);

  printf ("Writing a file mark !\n");
  i = APIWriteFilemarks (TARGET, 1l, 0, 0);
  printf ("Write returned %d\n", i);

  printf ("rewinding the tape...\n");
  i = APIRewind (TARGET, 0);
  printf ("Rewind returned %d\n", i);

  printf("Now counting files ...\n");
  i = 0;
  while (!APISpace(TARGET, 1, 1l))
       ++i;
  printf ("... there are %d files on the tape now\n",i);

  printf ("eraseing the tape...\n");
  i = APIErase (TARGET, 0, 0);
  printf ("erase returned %d\n", i);
#endif
/*
  l = 0l;
  printf ("Now see what kind of errors we get at Overflow\n");
  APIRewind (TARGET, 0);
  while (l < 4080l)
  {
      i = APIRead (TARGET, buf, 32768l, 9, 0, 0, &bytes_proc);
      if (i)
      {
         printf ("reading block # %ld: status: %d\n", l, i);
      }
      else
         printf ("read block # %ld %s:\n\r", l, buf);
      ++l;
  }

  i = APIRewind (TARGET, 0);
  printf ("Rewind returned %d\n", i);
  i = APIErase  (TARGET, 0, 0);
  printf ("Erase returned %d\n", i);
  l = 0l;

  // memset (buf, 0, 10);
  while ((i = APIWrite (TARGET, buf, 32768l, 9, 0, &bytes_proc)))
  {
        ++l;
        printf ("writing block %lu bytes proc: %lu\n", l, bytes_proc);
  }

  printf ("last block was %lu\n", l);
  printf ("error: %lu\n", GetLastError());

*/

/*
  i = APILoad (TARGET, 0, 0, 0, 0);
  printf ("Unload returned %d\n -->", i);
*/

  CloseHandle (hTape);
  free (buf);
}
#endif
