/*
 * $Id: maths.c,v 1.7 2002/02/28 10:49:24 spoel Exp $
 * 
 *                This source code is part of
 * 
 *                 G   R   O   M   A   C   S
 * 
 *          GROningen MAchine for Chemical Simulations
 * 
 *                        VERSION 3.1
 * Copyright (c) 1991-2001, University of Groningen, The Netherlands
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * If you want to redistribute modifications, please consider that
 * scientific software is very special. Version control is crucial -
 * bugs must be traceable. We will be happy to consider code for
 * inclusion in the official distribution, but derived work must not
 * be called official GROMACS. Details are found in the README & COPYING
 * files - if they are missing, get the official version at www.gromacs.org.
 * 
 * To help us fund GROMACS development, we humbly ask that you cite
 * the papers on the package - you can find them in the top README file.
 * 
 * For more info, check our website at http://www.gromacs.org
 * 
 * And Hey:
 * Gyas ROwers Mature At Cryogenic Speed
 */
static char *SRCID_maths_c = "$Id: maths.c,v 1.7 2002/02/28 10:49:24 spoel Exp $";
#include <math.h>
#include <limits.h>
#include "maths.h"

int gmx_nint(real a)
{   
  const real half = .5;
  int   result;
  
  result = (a < 0.) ? ((int)(a - half)) : ((int)(a + half));
  return result;
}

real sign(real x,real y)
{
  if (y < 0)
    return -fabs(x);
  else
    return +fabs(x);
}

/* Double and single precision erf() and erfc() from
 * the GNU C library, for hosts that don't have them.
 */
/*
 * ====================================================
 * Copyright (C) 1993 by Sun Microsystems, Inc. All rights reserved.
 *
 * Developed at SunPro, a Sun Microsystems, Inc. business.
 * Permission to use, copy, modify, and distribute this
 * software is freely granted, provided that this notice
 * is preserved.
 * ====================================================
 */
/* Modified by Naohiko Shimizu/Tokai University, Japan 1997/08/25,
   for performance improvement on pipelined processors.
*/

#if (INT_MAX == 2147483647)
   typedef int erf_int32_t;
   typedef unsigned int erf_u_int32_t;
#elif (LONG_MAX == 2147483647L)
   typedef long erf_int32_t;
   typedef unsigned long erf_u_int32_t;
#elif (SHRT_MAX == 2147483647)
   typedef short erf_int32_t;
   typedef unsigned short erf_u_int32_t;
#else
#  error ERROR: No 32 bit wide integer type found!
#endif


#ifdef DOUBLE

#ifdef __STDC__
static const double
#else
static double
#endif
tiny	    = 1e-300,
half=  5.00000000000000000000e-01, /* 0x3FE00000, 0x00000000 */
one =  1.00000000000000000000e+00, /* 0x3FF00000, 0x00000000 */
two =  2.00000000000000000000e+00, /* 0x40000000, 0x00000000 */
	/* c = (float)0.84506291151 */
erx =  8.45062911510467529297e-01, /* 0x3FEB0AC1, 0x60000000 */
/*
 * Coefficients for approximation to  erf on [0,0.84375]
 */
efx =  1.28379167095512586316e-01, /* 0x3FC06EBA, 0x8214DB69 */
efx8=  1.02703333676410069053e+00, /* 0x3FF06EBA, 0x8214DB69 */
pp[]  =  {1.28379167095512558561e-01, /* 0x3FC06EBA, 0x8214DB68 */
 -3.25042107247001499370e-01, /* 0xBFD4CD7D, 0x691CB913 */
 -2.84817495755985104766e-02, /* 0xBF9D2A51, 0xDBD7194F */
 -5.77027029648944159157e-03, /* 0xBF77A291, 0x236668E4 */
 -2.37630166566501626084e-05}, /* 0xBEF8EAD6, 0x120016AC */
qq[]  =  {0.0, 3.97917223959155352819e-01, /* 0x3FD97779, 0xCDDADC09 */
  6.50222499887672944485e-02, /* 0x3FB0A54C, 0x5536CEBA */
  5.08130628187576562776e-03, /* 0x3F74D022, 0xC4D36B0F */
  1.32494738004321644526e-04, /* 0x3F215DC9, 0x221C1A10 */
 -3.96022827877536812320e-06}, /* 0xBED09C43, 0x42A26120 */
/*
 * Coefficients for approximation to  erf  in [0.84375,1.25]
 */
pa[]  = {-2.36211856075265944077e-03, /* 0xBF6359B8, 0xBEF77538 */
  4.14856118683748331666e-01, /* 0x3FDA8D00, 0xAD92B34D */
 -3.72207876035701323847e-01, /* 0xBFD7D240, 0xFBB8C3F1 */
  3.18346619901161753674e-01, /* 0x3FD45FCA, 0x805120E4 */
 -1.10894694282396677476e-01, /* 0xBFBC6398, 0x3D3E28EC */
  3.54783043256182359371e-02, /* 0x3FA22A36, 0x599795EB */
 -2.16637559486879084300e-03}, /* 0xBF61BF38, 0x0A96073F */
qa[]  =  {0.0, 1.06420880400844228286e-01, /* 0x3FBB3E66, 0x18EEE323 */
  5.40397917702171048937e-01, /* 0x3FE14AF0, 0x92EB6F33 */
  7.18286544141962662868e-02, /* 0x3FB2635C, 0xD99FE9A7 */
  1.26171219808761642112e-01, /* 0x3FC02660, 0xE763351F */
  1.36370839120290507362e-02, /* 0x3F8BEDC2, 0x6B51DD1C */
  1.19844998467991074170e-02}, /* 0x3F888B54, 0x5735151D */
/*
 * Coefficients for approximation to  erfc in [1.25,1/0.35]
 */
ra[]  = {-9.86494403484714822705e-03, /* 0xBF843412, 0x600D6435 */
 -6.93858572707181764372e-01, /* 0xBFE63416, 0xE4BA7360 */
 -1.05586262253232909814e+01, /* 0xC0251E04, 0x41B0E726 */
 -6.23753324503260060396e+01, /* 0xC04F300A, 0xE4CBA38D */
 -1.62396669462573470355e+02, /* 0xC0644CB1, 0x84282266 */
 -1.84605092906711035994e+02, /* 0xC067135C, 0xEBCCABB2 */
 -8.12874355063065934246e+01, /* 0xC0545265, 0x57E4D2F2 */
 -9.81432934416914548592e+00}, /* 0xC023A0EF, 0xC69AC25C */
sa[]  =  {0.0,1.96512716674392571292e+01, /* 0x4033A6B9, 0xBD707687 */
  1.37657754143519042600e+02, /* 0x4061350C, 0x526AE721 */
  4.34565877475229228821e+02, /* 0x407B290D, 0xD58A1A71 */
  6.45387271733267880336e+02, /* 0x40842B19, 0x21EC2868 */
  4.29008140027567833386e+02, /* 0x407AD021, 0x57700314 */
  1.08635005541779435134e+02, /* 0x405B28A3, 0xEE48AE2C */
  6.57024977031928170135e+00, /* 0x401A47EF, 0x8E484A93 */
 -6.04244152148580987438e-02}, /* 0xBFAEEFF2, 0xEE749A62 */
/*
 * Coefficients for approximation to  erfc in [1/.35,28]
 */
rb[]  = {-9.86494292470009928597e-03, /* 0xBF843412, 0x39E86F4A */
 -7.99283237680523006574e-01, /* 0xBFE993BA, 0x70C285DE */
 -1.77579549177547519889e+01, /* 0xC031C209, 0x555F995A */
 -1.60636384855821916062e+02, /* 0xC064145D, 0x43C5ED98 */
 -6.37566443368389627722e+02, /* 0xC083EC88, 0x1375F228 */
 -1.02509513161107724954e+03, /* 0xC0900461, 0x6A2E5992 */
 -4.83519191608651397019e+02}, /* 0xC07E384E, 0x9BDC383F */
sb[]  =  {0.0,3.03380607434824582924e+01, /* 0x403E568B, 0x261D5190 */
  3.25792512996573918826e+02, /* 0x40745CAE, 0x221B9F0A */
  1.53672958608443695994e+03, /* 0x409802EB, 0x189D5118 */
  3.19985821950859553908e+03, /* 0x40A8FFB7, 0x688C246A */
  2.55305040643316442583e+03, /* 0x40A3F219, 0xCEDF3BE6 */
  4.74528541206955367215e+02, /* 0x407DA874, 0xE79FE763 */
 -2.24409524465858183362e+01}; /* 0xC03670E2, 0x42712D62 */


typedef union
{
  double value;
  struct
  {
    erf_u_int32_t msw;
    erf_u_int32_t lsw;
  } parts;
} ieee_double_shape_type;


#define GET_HIGH_WORD(i,d)					\
do {								\
  ieee_double_shape_type gh_u;					\
  gh_u.value = (d);						\
  (i) = gh_u.parts.msw;						\
} while (0)


#define GET_LOW_WORD(i,d)					\
do {								\
  ieee_double_shape_type gl_u;					\
  gl_u.value = (d);						\
  (i) = gl_u.parts.lsw;						\
} while (0)

double gmx_erf(double x)
{
	erf_int32_t hx,ix,i;
	double R,S,P,Q,s,y,z,r;
	GET_HIGH_WORD(hx,x);
	ix = hx&0x7fffffff;
	if(ix>=0x7ff00000) {		/* erf(nan)=nan */
	    i = ((erf_u_int32_t)hx>>31)<<1;
	    return (double)(1-i)+one/x;	/* erf(+-inf)=+-1 */
	}

	if(ix < 0x3feb0000) {		/* |x|<0.84375 */
	    double r1,r2,s1,s2,s3,z2,z4;
	    if(ix < 0x3e300000) { 	/* |x|<2**-28 */
	        if (ix < 0x00800000)
		    return 0.125*(8.0*x+efx8*x);  /*avoid underflow */
		return x + efx*x;
	    }
	    z = x*x;
	    r1 = pp[0]+z*pp[1]; z2=z*z;
	    r2 = pp[2]+z*pp[3]; z4=z2*z2;
	    s1 = one+z*qq[1];
	    s2 = qq[2]+z*qq[3];
	    s3 = qq[4]+z*qq[5];
            r = r1 + z2*r2 + z4*pp[4];
	    s  = s1 + z2*s2 + z4*s3;
	    y = r/s;
	    return x + x*y;
	}
	if(ix < 0x3ff40000) {		/* 0.84375 <= |x| < 1.25 */
	    double s2,s4,s6,P1,P2,P3,P4,Q1,Q2,Q3,Q4;
	    s = fabs(x)-one;
	    P1 = pa[0]+s*pa[1]; s2=s*s;
	    Q1 = one+s*qa[1];   s4=s2*s2;
	    P2 = pa[2]+s*pa[3]; s6=s4*s2;
	    Q2 = qa[2]+s*qa[3];
	    P3 = pa[4]+s*pa[5];
	    Q3 = qa[4]+s*qa[5];
	    P4 = pa[6];
	    Q4 = qa[6];
	    P = P1 + s2*P2 + s4*P3 + s6*P4;
	    Q = Q1 + s2*Q2 + s4*Q3 + s6*Q4;
	    if(hx>=0) return erx + P/Q; else return -erx - P/Q;
	}
	if (ix >= 0x40180000) {		/* inf>|x|>=6 */
	    if(hx>=0) return one-tiny; else return tiny-one;
	}
	x = fabs(x);
 	s = one/(x*x);
	if(ix< 0x4006DB6E) {	/* |x| < 1/0.35 */
	    double R1,R2,R3,R4,S1,S2,S3,S4,s2,s4,s6,s8;
	    R1 = ra[0]+s*ra[1];s2 = s*s;
	    S1 = one+s*sa[1];  s4 = s2*s2;
	    R2 = ra[2]+s*ra[3];s6 = s4*s2;
	    S2 = sa[2]+s*sa[3];s8 = s4*s4;
	    R3 = ra[4]+s*ra[5];
	    S3 = sa[4]+s*sa[5];
	    R4 = ra[6]+s*ra[7];
	    S4 = sa[6]+s*sa[7];
	    R = R1 + s2*R2 + s4*R3 + s6*R4;
	    S = S1 + s2*S2 + s4*S3 + s6*S4 + s8*sa[8];
	} else {	/* |x| >= 1/0.35 */
	    double R1,R2,R3,S1,S2,S3,S4,s2,s4,s6;
	    R1 = rb[0]+s*rb[1];s2 = s*s;
	    S1 = one+s*sb[1];  s4 = s2*s2;
	    R2 = rb[2]+s*rb[3];s6 = s4*s2;
	    S2 = sb[2]+s*sb[3];
	    R3 = rb[4]+s*rb[5];
	    S3 = sb[4]+s*sb[5];
	    S4 = sb[6]+s*sb[7];
	    R = R1 + s2*R2 + s4*R3 + s6*rb[6];
	    S = S1 + s2*S2 + s4*S3 + s6*S4;
	}
	z  = x;
	SET_LOW_WORD(z,0);
	r  =  __ieee754_exp(-z*z-0.5625)*__ieee754_exp((z-x)*(z+x)+R/S);
	if(hx>=0) return one-r/x; else return  r/x-one;
}


double gmx_erfc(double x)
{
	erf_int32_t hx,ix;
	double R,S,P,Q,s,y,z,r;
	GET_HIGH_WORD(hx,x);
	ix = hx&0x7fffffff;
	if(ix>=0x7ff00000) {			/* erfc(nan)=nan */
						/* erfc(+-inf)=0,2 */
	    return (double)(((erf_u_int32_t)hx>>31)<<1)+one/x;
	}

	if(ix < 0x3feb0000) {		/* |x|<0.84375 */
	    double r1,r2,s1,s2,s3,z2,z4;
	    if(ix < 0x3c700000)  	/* |x|<2**-56 */
		return one-x;
	    z = x*x;
	    r1 = pp[0]+z*pp[1]; z2=z*z;
	    r2 = pp[2]+z*pp[3]; z4=z2*z2;
	    s1 = one+z*qq[1];
	    s2 = qq[2]+z*qq[3];
	    s3 = qq[4]+z*qq[5];
            r = r1 + z2*r2 + z4*pp[4];
	    s  = s1 + z2*s2 + z4*s3;
	    y = r/s;
	    if(hx < 0x3fd00000) {  	/* x<1/4 */
		return one-(x+x*y);
	    } else {
		r = x*y;
		r += (x-half);
	        return half - r ;
	    }
	}
	if(ix < 0x3ff40000) {		/* 0.84375 <= |x| < 1.25 */
	    double s2,s4,s6,P1,P2,P3,P4,Q1,Q2,Q3,Q4;
	    s = fabs(x)-one;
	    P1 = pa[0]+s*pa[1]; s2=s*s;
	    Q1 = one+s*qa[1];   s4=s2*s2;
	    P2 = pa[2]+s*pa[3]; s6=s4*s2;
	    Q2 = qa[2]+s*qa[3];
	    P3 = pa[4]+s*pa[5];
	    Q3 = qa[4]+s*qa[5];
	    P4 = pa[6];
	    Q4 = qa[6];
	    P = P1 + s2*P2 + s4*P3 + s6*P4;
	    Q = Q1 + s2*Q2 + s4*Q3 + s6*Q4;
	    if(hx>=0) {
	        z  = one-erx; return z - P/Q;
	    } else {
		z = erx+P/Q; return one+z;
	    }
	}
	if (ix < 0x403c0000) {		/* |x|<28 */
	    x = fabs(x);
 	    s = one/(x*x);
	    if(ix< 0x4006DB6D) {	/* |x| < 1/.35 ~ 2.857143*/
		double R1,R2,R3,R4,S1,S2,S3,S4,s2,s4,s6,s8;
	    R1 = ra[0]+s*ra[1];s2 = s*s;
	    S1 = one+s*sa[1];  s4 = s2*s2;
	    R2 = ra[2]+s*ra[3];s6 = s4*s2;
	    S2 = sa[2]+s*sa[3];s8 = s4*s4;
	    R3 = ra[4]+s*ra[5];
	    S3 = sa[4]+s*sa[5];
	    R4 = ra[6]+s*ra[7];
	    S4 = sa[6]+s*sa[7];
	    R = R1 + s2*R2 + s4*R3 + s6*R4;
	    S = S1 + s2*S2 + s4*S3 + s6*S4 + s8*sa[8];
	    } else {			/* |x| >= 1/.35 ~ 2.857143 */
		double R1,R2,R3,S1,S2,S3,S4,s2,s4,s6;
		if(hx<0&&ix>=0x40180000) return two-tiny;/* x < -6 */
		R1 = rb[0]+s*rb[1];s2 = s*s;
		S1 = one+s*sb[1];  s4 = s2*s2;
		R2 = rb[2]+s*rb[3];s6 = s4*s2;
		S2 = sb[2]+s*sb[3];
		R3 = rb[4]+s*rb[5];
		S3 = sb[4]+s*sb[5];
		S4 = sb[6]+s*sb[7];
		R = R1 + s2*R2 + s4*R3 + s6*rb[6];
		S = S1 + s2*S2 + s4*S3 + s6*S4;
	    }
	    z  = x;
	    SET_LOW_WORD(z,0);
	    r  =  __ieee754_exp(-z*z-0.5625)*
			__ieee754_exp((z-x)*(z+x)+R/S);
	    if(hx>0) return r/x; else return two-r/x;
	} else {
	    if(hx>0) return tiny*tiny; else return two-tiny;
	}
}

#else /* single precision */



#ifdef __STDC__
static const float
#else
static float
#endif
tiny	    = 1e-30,
half=  5.0000000000e-01, /* 0x3F000000 */
one =  1.0000000000e+00, /* 0x3F800000 */
two =  2.0000000000e+00, /* 0x40000000 */
	/* c = (subfloat)0.84506291151 */
erx =  8.4506291151e-01, /* 0x3f58560b */
/*
 * Coefficients for approximation to  erf on [0,0.84375]
 */
efx =  1.2837916613e-01, /* 0x3e0375d4 */
efx8=  1.0270333290e+00, /* 0x3f8375d4 */
pp0  =  1.2837916613e-01, /* 0x3e0375d4 */
pp1  = -3.2504209876e-01, /* 0xbea66beb */
pp2  = -2.8481749818e-02, /* 0xbce9528f */
pp3  = -5.7702702470e-03, /* 0xbbbd1489 */
pp4  = -2.3763017452e-05, /* 0xb7c756b1 */
qq1  =  3.9791721106e-01, /* 0x3ecbbbce */
qq2  =  6.5022252500e-02, /* 0x3d852a63 */
qq3  =  5.0813062117e-03, /* 0x3ba68116 */
qq4  =  1.3249473704e-04, /* 0x390aee49 */
qq5  = -3.9602282413e-06, /* 0xb684e21a */
/*
 * Coefficients for approximation to  erf  in [0.84375,1.25] 
 */
pa0  = -2.3621185683e-03, /* 0xbb1acdc6 */
pa1  =  4.1485610604e-01, /* 0x3ed46805 */
pa2  = -3.7220788002e-01, /* 0xbebe9208 */
pa3  =  3.1834661961e-01, /* 0x3ea2fe54 */
pa4  = -1.1089469492e-01, /* 0xbde31cc2 */
pa5  =  3.5478305072e-02, /* 0x3d1151b3 */
pa6  = -2.1663755178e-03, /* 0xbb0df9c0 */
qa1  =  1.0642088205e-01, /* 0x3dd9f331 */
qa2  =  5.4039794207e-01, /* 0x3f0a5785 */
qa3  =  7.1828655899e-02, /* 0x3d931ae7 */
qa4  =  1.2617121637e-01, /* 0x3e013307 */
qa5  =  1.3637083583e-02, /* 0x3c5f6e13 */
qa6  =  1.1984500103e-02, /* 0x3c445aa3 */
/*
 * Coefficients for approximation to  erfc in [1.25,1/0.35]
 */
ra0  = -9.8649440333e-03, /* 0xbc21a093 */
ra1  = -6.9385856390e-01, /* 0xbf31a0b7 */
ra2  = -1.0558626175e+01, /* 0xc128f022 */
ra3  = -6.2375331879e+01, /* 0xc2798057 */
ra4  = -1.6239666748e+02, /* 0xc322658c */
ra5  = -1.8460508728e+02, /* 0xc3389ae7 */
ra6  = -8.1287437439e+01, /* 0xc2a2932b */
ra7  = -9.8143291473e+00, /* 0xc11d077e */
sa1  =  1.9651271820e+01, /* 0x419d35ce */
sa2  =  1.3765776062e+02, /* 0x4309a863 */
sa3  =  4.3456588745e+02, /* 0x43d9486f */
sa4  =  6.4538726807e+02, /* 0x442158c9 */
sa5  =  4.2900814819e+02, /* 0x43d6810b */
sa6  =  1.0863500214e+02, /* 0x42d9451f */
sa7  =  6.5702495575e+00, /* 0x40d23f7c */
sa8  = -6.0424413532e-02, /* 0xbd777f97 */
/*
 * Coefficients for approximation to  erfc in [1/.35,28]
 */
rb0  = -9.8649431020e-03, /* 0xbc21a092 */
rb1  = -7.9928326607e-01, /* 0xbf4c9dd4 */
rb2  = -1.7757955551e+01, /* 0xc18e104b */
rb3  = -1.6063638306e+02, /* 0xc320a2ea */
rb4  = -6.3756646729e+02, /* 0xc41f6441 */
rb5  = -1.0250950928e+03, /* 0xc480230b */
rb6  = -4.8351919556e+02, /* 0xc3f1c275 */
sb1  =  3.0338060379e+01, /* 0x41f2b459 */
sb2  =  3.2579251099e+02, /* 0x43a2e571 */
sb3  =  1.5367296143e+03, /* 0x44c01759 */
sb4  =  3.1998581543e+03, /* 0x4547fdbb */
sb5  =  2.5530502930e+03, /* 0x451f90ce */
sb6  =  4.7452853394e+02, /* 0x43ed43a7 */
sb7  = -2.2440952301e+01; /* 0xc1b38712 */


typedef union
{
  float value;
  erf_u_int32_t word;
} ieee_float_shape_type;

#define GET_FLOAT_WORD(i,d)					\
do {								\
  ieee_float_shape_type gf_u;					\
  gf_u.value = (d);						\
  (i) = gf_u.word;						\
} while (0)


#define SET_FLOAT_WORD(d,i)					\
do {								\
  ieee_float_shape_type sf_u;					\
  sf_u.word = (i);						\
  (d) = sf_u.value;						\
} while (0)


float gmx_erf(float x)
{
	erf_int32_t hx,ix,i;
	float R,S,P,Q,s,y,z,r;
	GET_FLOAT_WORD(hx,x);
	ix = hx&0x7fffffff;
	if(ix>=0x7f800000) {		/* erf(nan)=nan */
	    i = ((erf_u_int32_t)hx>>31)<<1;
	    return (float)(1-i)+one/x;	/* erf(+-inf)=+-1 */
	}

	if(ix < 0x3f580000) {		/* |x|<0.84375 */
	    if(ix < 0x31800000) { 	/* |x|<2**-28 */
	        if (ix < 0x04000000) 
		    /*avoid underflow */
		    return (float)0.125*((float)8.0*x+efx8*x);
		return x + efx*x;
	    }
	    z = x*x;
	    r = pp0+z*(pp1+z*(pp2+z*(pp3+z*pp4)));
	    s = one+z*(qq1+z*(qq2+z*(qq3+z*(qq4+z*qq5))));
	    y = r/s;
	    return x + x*y;
	}
	if(ix < 0x3fa00000) {		/* 0.84375 <= |x| < 1.25 */
	    s = fabs(x)-one;
	    P = pa0+s*(pa1+s*(pa2+s*(pa3+s*(pa4+s*(pa5+s*pa6)))));
	    Q = one+s*(qa1+s*(qa2+s*(qa3+s*(qa4+s*(qa5+s*qa6)))));
	    if(hx>=0) return erx + P/Q; else return -erx - P/Q;
	}
	if (ix >= 0x40c00000) {		/* inf>|x|>=6 */
	    if(hx>=0) return one-tiny; else return tiny-one;
	}
	x = fabs(x);
 	s = one/(x*x);
	if(ix< 0x4036DB6E) {	/* |x| < 1/0.35 */
	    R=ra0+s*(ra1+s*(ra2+s*(ra3+s*(ra4+s*(
				ra5+s*(ra6+s*ra7))))));
	    S=one+s*(sa1+s*(sa2+s*(sa3+s*(sa4+s*(
				sa5+s*(sa6+s*(sa7+s*sa8)))))));
	} else {	/* |x| >= 1/0.35 */
	    R=rb0+s*(rb1+s*(rb2+s*(rb3+s*(rb4+s*(
				rb5+s*rb6)))));
	    S=one+s*(sb1+s*(sb2+s*(sb3+s*(sb4+s*(
				sb5+s*(sb6+s*sb7))))));
	}
	GET_FLOAT_WORD(ix,x);
	SET_FLOAT_WORD(z,ix&0xfffff000);
	r  =  exp(-z*z-(float)0.5625)*exp((z-x)*(z+x)+R/S);
	if(hx>=0) return one-r/x; else return  r/x-one;
}

float gmx_erfc(float x)
{
	erf_int32_t hx,ix;
	float R,S,P,Q,s,y,z,r;
	GET_FLOAT_WORD(hx,x);
	ix = hx&0x7fffffff;
	if(ix>=0x7f800000) {			/* erfc(nan)=nan */
						/* erfc(+-inf)=0,2 */
	    return (float)(((erf_u_int32_t)hx>>31)<<1)+one/x;
	}

	if(ix < 0x3f580000) {		/* |x|<0.84375 */
	    if(ix < 0x23800000)  	/* |x|<2**-56 */
		return one-x;
	    z = x*x;
	    r = pp0+z*(pp1+z*(pp2+z*(pp3+z*pp4)));
	    s = one+z*(qq1+z*(qq2+z*(qq3+z*(qq4+z*qq5))));
	    y = r/s;
	    if(hx < 0x3e800000) {  	/* x<1/4 */
		return one-(x+x*y);
	    } else {
		r = x*y;
		r += (x-half);
	        return half - r ;
	    }
	}
	if(ix < 0x3fa00000) {		/* 0.84375 <= |x| < 1.25 */
	    s = fabs(x)-one;
	    P = pa0+s*(pa1+s*(pa2+s*(pa3+s*(pa4+s*(pa5+s*pa6)))));
	    Q = one+s*(qa1+s*(qa2+s*(qa3+s*(qa4+s*(qa5+s*qa6)))));
	    if(hx>=0) {
	        z  = one-erx; return z - P/Q; 
	    } else {
		z = erx+P/Q; return one+z;
	    }
	}
	if (ix < 0x41e00000) {		/* |x|<28 */
	    x = fabs(x);
 	    s = one/(x*x);
	    if(ix< 0x4036DB6D) {	/* |x| < 1/.35 ~ 2.857143*/
	        R=ra0+s*(ra1+s*(ra2+s*(ra3+s*(ra4+s*(
				ra5+s*(ra6+s*ra7))))));
	        S=one+s*(sa1+s*(sa2+s*(sa3+s*(sa4+s*(
				sa5+s*(sa6+s*(sa7+s*sa8)))))));
	    } else {			/* |x| >= 1/.35 ~ 2.857143 */
		if(hx<0&&ix>=0x40c00000) return two-tiny;/* x < -6 */
	        R=rb0+s*(rb1+s*(rb2+s*(rb3+s*(rb4+s*(
				rb5+s*rb6)))));
	        S=one+s*(sb1+s*(sb2+s*(sb3+s*(sb4+s*(
				sb5+s*(sb6+s*sb7))))));
	    }
	    GET_FLOAT_WORD(ix,x);
	    SET_FLOAT_WORD(z,ix&0xfffff000);
	    r  =  exp(-z*z-(float)0.5625)*
			exp((z-x)*(z+x)+R/S);
	    if(hx>0) return r/x; else return two-r/x;
	} else {
	    if(hx>0) return tiny*tiny; else return two-tiny;
	}
}

#endif
