! OPTICS_LIB: Optical proecures for for F90
! Compiled/Modified:
!   07/01/06  John Haynes (haynes@atmos.colostate.edu)
!
! m_wat (subroutine)
! m_ice (subroutine)
! mie_int (subroutine)
  
  module optics_lib
  implicit none

  integer, private, parameter :: r8 = selected_real_kind(15)

  contains

! ----------------------------------------------------------------------------
! subroutine M_WAT
! ----------------------------------------------------------------------------
  subroutine m_wat(freq, t, n_r, n_i)
  implicit none
!  
! Purpose:
!   compute complex index of refraction of liquid water
!
! Inputs:
!   [freq]    frequency (GHz)
!   [t]       temperature (C)
!
! Outputs:
!   [n_r]     real part index of refraction
!   [n_i]     imaginary part index of refraction
!
! Reference:
!   Based on the work of Ray (1972)
!
! Coded:
!   03/22/05  John Haynes (haynes@atmos.colostate.edu)
  
! ----- INPUTS -----
  real(8), intent(in) :: freq,t
  
! ----- OUTPUTS -----
  real(8), intent(out) :: n_r, n_i

! ----- INTERNAL -----    
  real(8) ld,es,ei,a,ls,sg,tm1,cos1,sin1
  real(8) e_r,e_i
  real(8) pi
  complex(8) e_comp, sq

  ld = 100._r8*2.99792458E8_r8/(freq*1E9_r8)
  es = 78.54_r8*(1-(4.579E-3_r8*(t-25._r8)+1.19E-5_r8*(t-25._r8)**2 &
       -2.8E-8_r8*(t-25._r8)**3))
  ei = 5.27137_r8+0.021647_r8*t-0.00131198_r8*t**2
  a = -(16.8129_r8/(t+273._r8))+0.0609265_r8
  ls = 0.00033836_r8*exp(2513.98_r8/(t+273._r8))
  sg = 12.5664E8_r8

  tm1 = (ls/ld)**(1-a)
  pi = acos(-1.D0)
  cos1 = cos(0.5_r8*a*pi)
  sin1 = sin(0.5_r8*a*pi)

  e_r = ei + (((es-ei)*(1._r8+tm1*sin1))/(1._r8+2*tm1*sin1+tm1**2))
  e_i = (((es-ei)*tm1*cos1)/(1._r8+2*tm1*sin1+tm1**2)) &
        +((sg*ld)/1.885E11_r8)

  e_comp = cmplx(e_r,e_i,kind=Kind(0d0))
  sq = sqrt(e_comp)
  
  n_r = real(sq)
  n_i = aimag(sq)      
  
  return
  end subroutine m_wat

! ----------------------------------------------------------------------------
! subroutine M_ICE
! ----------------------------------------------------------------------------
  subroutine m_ice(freq,t,n_r,n_i)
  implicit none
!
! Purpose:
!   compute complex index of refraction of ice
!
! Inputs:
!   [freq]    frequency (GHz)
!   [t]       temperature (C)
!
! Outputs:
!   [n_r]     real part index of refraction
!   [n_i]     imaginary part index of refraction
!
! Reference:
!    Fortran 90 port from IDL of REFICE by Stephen G. Warren
!
! Modified:
!   05/31/05  John Haynes (haynes@atmos.colostate.edu)

! ----- INPUTS -----
  real(8), intent(in) :: freq, t
  
! ----- OUTPUTS -----  
  real(8), intent(out) :: n_r,n_i

! Parameters:
  integer(2) :: i,lt1,lt2,nwl,nwlt
  parameter(nwl=468,nwlt=62)

  real(8) :: alam,cutice,pi,t1,t2,tk,wlmax,wlmin, &
            x,x1,x2,y,y1,y2,ylo,yhi

  real(8) :: &
       tabim(nwl),tabimt(nwlt,4),tabre(nwl),tabret(nwlt,4),temref(4), &
       wl(nwl),wlt(nwlt)

! Defines wavelength dependent complex index of refraction for ice.
! Allowable wavelength range extends from 0.045 microns to 8.6 meter
! temperature dependence only considered beyond 167 microns.
! 
! interpolation is done     n_r  vs. log(xlam)
!                           n_r  vs.        t
!                       log(n_i) vs. log(xlam)
!                       log(n_i) vs.        t
!
! Stephen G. Warren - 1983
! Dept. of Atmospheric Sciences
! University of Washington
! Seattle, Wa  98195
!
! Based on
!
!    Warren,S.G.,1984.
!    Optical constants of ice from the ultraviolet to the microwave.
!    Applied Optics,23,1206-1225
!
! Reference temperatures are -1.0,-5.0,-20.0, and -60.0 deg C
 
      data temref/272.16_r8,268.16_r8,253.16_r8,213.16_r8/
 
      data wlmin,wlmax/0.045_r8,8.6e6_r8/
      data cutice/167.0_r8/
 
      data (wl(i),i=1,114)/ &
      0.4430e-01_r8,0.4510e-01_r8,0.4590e-01_r8,0.4680e-01_r8,0.4770e-01_r8,0.4860e-01_r8, &
      0.4960e-01_r8,0.5060e-01_r8,0.5170e-01_r8,0.5280e-01_r8,0.5390e-01_r8,0.5510e-01_r8, &
      0.5640e-01_r8,0.5770e-01_r8,0.5900e-01_r8,0.6050e-01_r8,0.6200e-01_r8,0.6360e-01_r8, &
      0.6530e-01_r8,0.6700e-01_r8,0.6890e-01_r8,0.7080e-01_r8,0.7290e-01_r8,0.7380e-01_r8, &
      0.7510e-01_r8,0.7750e-01_r8,0.8000e-01_r8,0.8270e-01_r8,0.8550e-01_r8,0.8860e-01_r8, &
      0.9180e-01_r8,0.9300e-01_r8,0.9540e-01_r8,0.9920e-01_r8,0.1033e+00_r8,0.1078e+00_r8, &
      0.1100e+00_r8,0.1127e+00_r8,0.1140e+00_r8,0.1181e+00_r8,0.1210e+00_r8,0.1240e+00_r8, &
      0.1272e+00_r8,0.1295e+00_r8,0.1305e+00_r8,0.1319e+00_r8,0.1333e+00_r8,0.1348e+00_r8, &
      0.1362e+00_r8,0.1370e+00_r8,0.1378e+00_r8,0.1387e+00_r8,0.1393e+00_r8,0.1409e+00_r8, &
      0.1425e+00_r8,0.1435e+00_r8,0.1442e+00_r8,0.1450e+00_r8,0.1459e+00_r8,0.1468e+00_r8, &
      0.1476e+00_r8,0.1480e+00_r8,0.1485e+00_r8,0.1494e+00_r8,0.1512e+00_r8,0.1531e+00_r8, &
      0.1540e+00_r8,0.1550e+00_r8,0.1569e+00_r8,0.1580e+00_r8,0.1589e+00_r8,0.1610e+00_r8, &
      0.1625e+00_r8,0.1648e+00_r8,0.1669e+00_r8,0.1692e+00_r8,0.1713e+00_r8,0.1737e+00_r8, &
      0.1757e+00_r8,0.1779e+00_r8,0.1802e+00_r8,0.1809e+00_r8,0.1821e+00_r8,0.1833e+00_r8, &
      0.1843e+00_r8,0.1850e+00_r8,0.1860e+00_r8,0.1870e+00_r8,0.1880e+00_r8,0.1890e+00_r8, &
      0.1900e+00_r8,0.1910e+00_r8,0.1930e+00_r8,0.1950e+00_r8,0.2100e+00_r8,0.2500e+00_r8, &
      0.3000e+00_r8,0.3500e+00_r8,0.4000e+00_r8,0.4100e+00_r8,0.4200e+00_r8,0.4300e+00_r8, &
      0.4400e+00_r8,0.4500e+00_r8,0.4600e+00_r8,0.4700e+00_r8,0.4800e+00_r8,0.4900e+00_r8, &
      0.5000e+00_r8,0.5100e+00_r8,0.5200e+00_r8,0.5300e+00_r8,0.5400e+00_r8,0.5500e+00_r8/
      data (wl(i),i=115,228)/ &
      0.5600e+00_r8,0.5700e+00_r8,0.5800e+00_r8,0.5900e+00_r8,0.6000e+00_r8,0.6100e+00_r8, &
      0.6200e+00_r8,0.6300e+00_r8,0.6400e+00_r8,0.6500e+00_r8,0.6600e+00_r8,0.6700e+00_r8, &
      0.6800e+00_r8,0.6900e+00_r8,0.7000e+00_r8,0.7100e+00_r8,0.7200e+00_r8,0.7300e+00_r8, &
      0.7400e+00_r8,0.7500e+00_r8,0.7600e+00_r8,0.7700e+00_r8,0.7800e+00_r8,0.7900e+00_r8, &
      0.8000e+00_r8,0.8100e+00_r8,0.8200e+00_r8,0.8300e+00_r8,0.8400e+00_r8,0.8500e+00_r8, &
      0.8600e+00_r8,0.8700e+00_r8,0.8800e+00_r8,0.8900e+00_r8,0.9000e+00_r8,0.9100e+00_r8, &
      0.9200e+00_r8,0.9300e+00_r8,0.9400e+00_r8,0.9500e+00_r8,0.9600e+00_r8,0.9700e+00_r8, &
      0.9800e+00_r8,0.9900e+00_r8,0.1000e+01_r8,0.1010e+01_r8,0.1020e+01_r8,0.1030e+01_r8, &
      0.1040e+01_r8,0.1050e+01_r8,0.1060e+01_r8,0.1070e+01_r8,0.1080e+01_r8,0.1090e+01_r8, &
      0.1100e+01_r8,0.1110e+01_r8,0.1120e+01_r8,0.1130e+01_r8,0.1140e+01_r8,0.1150e+01_r8, &
      0.1160e+01_r8,0.1170e+01_r8,0.1180e+01_r8,0.1190e+01_r8,0.1200e+01_r8,0.1210e+01_r8, &
      0.1220e+01_r8,0.1230e+01_r8,0.1240e+01_r8,0.1250e+01_r8,0.1260e+01_r8,0.1270e+01_r8, &
      0.1280e+01_r8,0.1290e+01_r8,0.1300e+01_r8,0.1310e+01_r8,0.1320e+01_r8,0.1330e+01_r8, &
      0.1340e+01_r8,0.1350e+01_r8,0.1360e+01_r8,0.1370e+01_r8,0.1380e+01_r8,0.1390e+01_r8, &
      0.1400e+01_r8,0.1410e+01_r8,0.1420e+01_r8,0.1430e+01_r8,0.1440e+01_r8,0.1449e+01_r8, &
      0.1460e+01_r8,0.1471e+01_r8,0.1481e+01_r8,0.1493e+01_r8,0.1504e+01_r8,0.1515e+01_r8, &
      0.1527e+01_r8,0.1538e+01_r8,0.1563e+01_r8,0.1587e+01_r8,0.1613e+01_r8,0.1650e+01_r8, &
      0.1680e+01_r8,0.1700e+01_r8,0.1730e+01_r8,0.1760e+01_r8,0.1800e+01_r8,0.1830e+01_r8, &
      0.1840e+01_r8,0.1850e+01_r8,0.1855e+01_r8,0.1860e+01_r8,0.1870e+01_r8,0.1890e+01_r8/
      data (wl(i),i=229,342)/ &
      0.1905e+01_r8,0.1923e+01_r8,0.1942e+01_r8,0.1961e+01_r8,0.1980e+01_r8,0.2000e+01_r8, &
      0.2020e+01_r8,0.2041e+01_r8,0.2062e+01_r8,0.2083e+01_r8,0.2105e+01_r8,0.2130e+01_r8, &
      0.2150e+01_r8,0.2170e+01_r8,0.2190e+01_r8,0.2220e+01_r8,0.2240e+01_r8,0.2245e+01_r8, &
      0.2250e+01_r8,0.2260e+01_r8,0.2270e+01_r8,0.2290e+01_r8,0.2310e+01_r8,0.2330e+01_r8, &
      0.2350e+01_r8,0.2370e+01_r8,0.2390e+01_r8,0.2410e+01_r8,0.2430e+01_r8,0.2460e+01_r8, &
      0.2500e+01_r8,0.2520e+01_r8,0.2550e+01_r8,0.2565e+01_r8,0.2580e+01_r8,0.2590e+01_r8, &
      0.2600e+01_r8,0.2620e+01_r8,0.2675e+01_r8,0.2725e+01_r8,0.2778e+01_r8,0.2817e+01_r8, &
      0.2833e+01_r8,0.2849e+01_r8,0.2865e+01_r8,0.2882e+01_r8,0.2899e+01_r8,0.2915e+01_r8, &
      0.2933e+01_r8,0.2950e+01_r8,0.2967e+01_r8,0.2985e+01_r8,0.3003e+01_r8,0.3021e+01_r8, &
      0.3040e+01_r8,0.3058e+01_r8,0.3077e+01_r8,0.3096e+01_r8,0.3115e+01_r8,0.3135e+01_r8, &
      0.3155e+01_r8,0.3175e+01_r8,0.3195e+01_r8,0.3215e+01_r8,0.3236e+01_r8,0.3257e+01_r8, &
      0.3279e+01_r8,0.3300e+01_r8,0.3322e+01_r8,0.3345e+01_r8,0.3367e+01_r8,0.3390e+01_r8, &
      0.3413e+01_r8,0.3436e+01_r8,0.3460e+01_r8,0.3484e+01_r8,0.3509e+01_r8,0.3534e+01_r8, &
      0.3559e+01_r8,0.3624e+01_r8,0.3732e+01_r8,0.3775e+01_r8,0.3847e+01_r8,0.3969e+01_r8, &
      0.4099e+01_r8,0.4239e+01_r8,0.4348e+01_r8,0.4387e+01_r8,0.4444e+01_r8,0.4505e+01_r8, &
      0.4547e+01_r8,0.4560e+01_r8,0.4580e+01_r8,0.4719e+01_r8,0.4904e+01_r8,0.5000e+01_r8, &
      0.5100e+01_r8,0.5200e+01_r8,0.5263e+01_r8,0.5400e+01_r8,0.5556e+01_r8,0.5714e+01_r8, &
      0.5747e+01_r8,0.5780e+01_r8,0.5814e+01_r8,0.5848e+01_r8,0.5882e+01_r8,0.6061e+01_r8, &
      0.6135e+01_r8,0.6250e+01_r8,0.6289e+01_r8,0.6329e+01_r8,0.6369e+01_r8,0.6410e+01_r8/
      data (wl(i),i=343,456)/ &
      0.6452e+01_r8,0.6494e+01_r8,0.6579e+01_r8,0.6667e+01_r8,0.6757e+01_r8,0.6897e+01_r8, &
      0.7042e+01_r8,0.7143e+01_r8,0.7246e+01_r8,0.7353e+01_r8,0.7463e+01_r8,0.7576e+01_r8, &
      0.7692e+01_r8,0.7812e+01_r8,0.7937e+01_r8,0.8065e+01_r8,0.8197e+01_r8,0.8333e+01_r8, &
      0.8475e+01_r8,0.8696e+01_r8,0.8929e+01_r8,0.9091e+01_r8,0.9259e+01_r8,0.9524e+01_r8, &
      0.9804e+01_r8,0.1000e+02_r8,0.1020e+02_r8,0.1031e+02_r8,0.1042e+02_r8,0.1053e+02_r8, &
      0.1064e+02_r8,0.1075e+02_r8,0.1087e+02_r8,0.1100e+02_r8,0.1111e+02_r8,0.1136e+02_r8, &
      0.1163e+02_r8,0.1190e+02_r8,0.1220e+02_r8,0.1250e+02_r8,0.1282e+02_r8,0.1299e+02_r8, &
      0.1316e+02_r8,0.1333e+02_r8,0.1351e+02_r8,0.1370e+02_r8,0.1389e+02_r8,0.1408e+02_r8, &
      0.1429e+02_r8,0.1471e+02_r8,0.1515e+02_r8,0.1538e+02_r8,0.1563e+02_r8,0.1613e+02_r8, &
      0.1639e+02_r8,0.1667e+02_r8,0.1695e+02_r8,0.1724e+02_r8,0.1818e+02_r8,0.1887e+02_r8, &
      0.1923e+02_r8,0.1961e+02_r8,0.2000e+02_r8,0.2041e+02_r8,0.2083e+02_r8,0.2222e+02_r8, &
      0.2260e+02_r8,0.2305e+02_r8,0.2360e+02_r8,0.2460e+02_r8,0.2500e+02_r8,0.2600e+02_r8, &
      0.2857e+02_r8,0.3100e+02_r8,0.3333e+02_r8,0.3448e+02_r8,0.3564e+02_r8,0.3700e+02_r8, &
      0.3824e+02_r8,0.3960e+02_r8,0.4114e+02_r8,0.4276e+02_r8,0.4358e+02_r8,0.4458e+02_r8, &
      0.4550e+02_r8,0.4615e+02_r8,0.4671e+02_r8,0.4736e+02_r8,0.4800e+02_r8,0.4878e+02_r8, &
      0.5003e+02_r8,0.5128e+02_r8,0.5275e+02_r8,0.5350e+02_r8,0.5424e+02_r8,0.5500e+02_r8, &
      0.5574e+02_r8,0.5640e+02_r8,0.5700e+02_r8,0.5746e+02_r8,0.5840e+02_r8,0.5929e+02_r8, &
      0.6000e+02_r8,0.6100e+02_r8,0.6125e+02_r8,0.6250e+02_r8,0.6378e+02_r8,0.6467e+02_r8, &
      0.6558e+02_r8,0.6655e+02_r8,0.6760e+02_r8,0.6900e+02_r8,0.7053e+02_r8,0.7300e+02_r8/
      data (wl(i),i=457,468)/ &
      0.7500e+02_r8,0.7629e+02_r8,0.8000e+02_r8,0.8297e+02_r8,0.8500e+02_r8,0.8680e+02_r8, &
      0.9080e+02_r8,0.9517e+02_r8,0.1000e+03_r8,0.1200e+03_r8,0.1500e+03_r8,0.1670e+03_r8/
      data  wlt/ &
                                       0.1670e+03_r8,0.1778e+03_r8,0.1884e+03_r8, &
      0.1995e+03_r8,0.2113e+03_r8,0.2239e+03_r8,0.2371e+03_r8,0.2512e+03_r8,0.2661e+03_r8, &
      0.2818e+03_r8,0.2985e+03_r8,0.3162e+03_r8,0.3548e+03_r8,0.3981e+03_r8,0.4467e+03_r8, &
      0.5012e+03_r8,0.5623e+03_r8,0.6310e+03_r8,0.7943e+03_r8,0.1000e+04_r8,0.1259e+04_r8, &
      0.2500e+04_r8,0.5000e+04_r8,0.1000e+05_r8,0.2000e+05_r8,0.3200e+05_r8,0.3500e+05_r8, &
      0.4000e+05_r8,0.4500e+05_r8,0.5000e+05_r8,0.6000e+05_r8,0.7000e+05_r8,0.9000e+05_r8, &
      0.1110e+06_r8,0.1200e+06_r8,0.1300e+06_r8,0.1400e+06_r8,0.1500e+06_r8,0.1600e+06_r8, &
      0.1700e+06_r8,0.1800e+06_r8,0.2000e+06_r8,0.2500e+06_r8,0.2900e+06_r8,0.3200e+06_r8, &
      0.3500e+06_r8,0.3800e+06_r8,0.4000e+06_r8,0.4500e+06_r8,0.5000e+06_r8,0.6000e+06_r8, &
      0.6400e+06_r8,0.6800e+06_r8,0.7200e+06_r8,0.7600e+06_r8,0.8000e+06_r8,0.8400e+06_r8, &
      0.9000e+06_r8,0.1000e+07_r8,0.2000e+07_r8,0.5000e+07_r8,0.8600e+07_r8/
      data (tabre(i),i=1,114)/ &
         0.83441_r8,   0.83676_r8,   0.83729_r8,   0.83771_r8,   0.83827_r8,   0.84038_r8, &
         0.84719_r8,   0.85522_r8,   0.86047_r8,   0.86248_r8,   0.86157_r8,   0.86093_r8, &
         0.86419_r8,   0.86916_r8,   0.87764_r8,   0.89296_r8,   0.91041_r8,   0.93089_r8, &
         0.95373_r8,   0.98188_r8,   1.02334_r8,   1.06735_r8,   1.11197_r8,   1.13134_r8, &
         1.15747_r8,   1.20045_r8,   1.23840_r8,   1.27325_r8,   1.32157_r8,   1.38958_r8, &
         1.41644_r8,   1.40906_r8,   1.40063_r8,   1.40169_r8,   1.40934_r8,   1.40221_r8, &
         1.39240_r8,   1.38424_r8,   1.38075_r8,   1.38186_r8,   1.39634_r8,   1.40918_r8, &
         1.40256_r8,   1.38013_r8,   1.36303_r8,   1.34144_r8,   1.32377_r8,   1.30605_r8, &
         1.29054_r8,   1.28890_r8,   1.28931_r8,   1.30190_r8,   1.32025_r8,   1.36302_r8, &
         1.41872_r8,   1.45834_r8,   1.49028_r8,   1.52128_r8,   1.55376_r8,   1.57782_r8, &
         1.59636_r8,   1.60652_r8,   1.61172_r8,   1.61919_r8,   1.62522_r8,   1.63404_r8, &
         1.63689_r8,   1.63833_r8,   1.63720_r8,   1.63233_r8,   1.62222_r8,   1.58269_r8, &
         1.55635_r8,   1.52453_r8,   1.50320_r8,   1.48498_r8,   1.47226_r8,   1.45991_r8, &
         1.45115_r8,   1.44272_r8,   1.43498_r8,   1.43280_r8,   1.42924_r8,   1.42602_r8, &
         1.42323_r8,   1.42143_r8,   1.41897_r8,   1.41660_r8,   1.41434_r8,   1.41216_r8, &
         1.41006_r8,   1.40805_r8,   1.40423_r8,   1.40067_r8,   1.38004_r8,   1.35085_r8, &
         1.33394_r8,   1.32492_r8,   1.31940_r8,   1.31854_r8,   1.31775_r8,   1.31702_r8, &
         1.31633_r8,   1.31569_r8,   1.31509_r8,   1.31452_r8,   1.31399_r8,   1.31349_r8, &
         1.31302_r8,   1.31257_r8,   1.31215_r8,   1.31175_r8,   1.31136_r8,   1.31099_r8/
      data (tabre(i),i=115,228)/ &
         1.31064_r8,   1.31031_r8,   1.30999_r8,   1.30968_r8,   1.30938_r8,   1.30909_r8, &
         1.30882_r8,   1.30855_r8,   1.30829_r8,   1.30804_r8,   1.30780_r8,   1.30756_r8, &
         1.30733_r8,   1.30710_r8,   1.30688_r8,   1.30667_r8,   1.30646_r8,   1.30625_r8, &
         1.30605_r8,   1.30585_r8,   1.30566_r8,   1.30547_r8,   1.30528_r8,   1.30509_r8, &
         1.30491_r8,   1.30473_r8,   1.30455_r8,   1.30437_r8,   1.30419_r8,   1.30402_r8, &
         1.30385_r8,   1.30367_r8,   1.30350_r8,   1.30333_r8,   1.30316_r8,   1.30299_r8, &
         1.30283_r8,   1.30266_r8,   1.30249_r8,   1.30232_r8,   1.30216_r8,   1.30199_r8, &
         1.30182_r8,   1.30166_r8,   1.30149_r8,   1.30132_r8,   1.30116_r8,   1.30099_r8, &
         1.30082_r8,   1.30065_r8,   1.30048_r8,   1.30031_r8,   1.30014_r8,   1.29997_r8, &
         1.29979_r8,   1.29962_r8,   1.29945_r8,   1.29927_r8,   1.29909_r8,   1.29891_r8, &
         1.29873_r8,   1.29855_r8,   1.29837_r8,   1.29818_r8,   1.29800_r8,   1.29781_r8, &
         1.29762_r8,   1.29743_r8,   1.29724_r8,   1.29705_r8,   1.29686_r8,   1.29666_r8, &
         1.29646_r8,   1.29626_r8,   1.29605_r8,   1.29584_r8,   1.29563_r8,   1.29542_r8, &
         1.29521_r8,   1.29499_r8,   1.29476_r8,   1.29453_r8,   1.29430_r8,   1.29406_r8, &
         1.29381_r8,   1.29355_r8,   1.29327_r8,   1.29299_r8,   1.29272_r8,   1.29252_r8, &
         1.29228_r8,   1.29205_r8,   1.29186_r8,   1.29167_r8,   1.29150_r8,   1.29130_r8, &
         1.29106_r8,   1.29083_r8,   1.29025_r8,   1.28962_r8,   1.28891_r8,   1.28784_r8, &
         1.28689_r8,   1.28623_r8,   1.28521_r8,   1.28413_r8,   1.28261_r8,   1.28137_r8, &
         1.28093_r8,   1.28047_r8,   1.28022_r8,   1.27998_r8,   1.27948_r8,   1.27849_r8/
      data (tabre(i),i=229,342)/ &
         1.27774_r8,   1.27691_r8,   1.27610_r8,   1.27535_r8,   1.27471_r8,   1.27404_r8, &
         1.27329_r8,   1.27240_r8,   1.27139_r8,   1.27029_r8,   1.26901_r8,   1.26736_r8, &
         1.26591_r8,   1.26441_r8,   1.26284_r8,   1.26036_r8,   1.25860_r8,   1.25815_r8, &
         1.25768_r8,   1.25675_r8,   1.25579_r8,   1.25383_r8,   1.25179_r8,   1.24967_r8, &
         1.24745_r8,   1.24512_r8,   1.24266_r8,   1.24004_r8,   1.23725_r8,   1.23270_r8, &
         1.22583_r8,   1.22198_r8,   1.21548_r8,   1.21184_r8,   1.20790_r8,   1.20507_r8, &
         1.20209_r8,   1.19566_r8,   1.17411_r8,   1.14734_r8,   1.10766_r8,   1.06739_r8, &
         1.04762_r8,   1.02650_r8,   1.00357_r8,   0.98197_r8,   0.96503_r8,   0.95962_r8, &
         0.97269_r8,   0.99172_r8,   1.00668_r8,   1.02186_r8,   1.04270_r8,   1.07597_r8, &
         1.12954_r8,   1.21267_r8,   1.32509_r8,   1.42599_r8,   1.49656_r8,   1.55095_r8, &
         1.59988_r8,   1.63631_r8,   1.65024_r8,   1.64278_r8,   1.62691_r8,   1.61284_r8, &
         1.59245_r8,   1.57329_r8,   1.55770_r8,   1.54129_r8,   1.52654_r8,   1.51139_r8, &
         1.49725_r8,   1.48453_r8,   1.47209_r8,   1.46125_r8,   1.45132_r8,   1.44215_r8, &
         1.43366_r8,   1.41553_r8,   1.39417_r8,   1.38732_r8,   1.37735_r8,   1.36448_r8, &
         1.35414_r8,   1.34456_r8,   1.33882_r8,   1.33807_r8,   1.33847_r8,   1.34053_r8, &
         1.34287_r8,   1.34418_r8,   1.34634_r8,   1.34422_r8,   1.33453_r8,   1.32897_r8, &
         1.32333_r8,   1.31800_r8,   1.31432_r8,   1.30623_r8,   1.29722_r8,   1.28898_r8, &
         1.28730_r8,   1.28603_r8,   1.28509_r8,   1.28535_r8,   1.28813_r8,   1.30156_r8, &
         1.30901_r8,   1.31720_r8,   1.31893_r8,   1.32039_r8,   1.32201_r8,   1.32239_r8/
      data (tabre(i),i=343,456)/ &
         1.32149_r8,   1.32036_r8,   1.31814_r8,   1.31705_r8,   1.31807_r8,   1.31953_r8, &
         1.31933_r8,   1.31896_r8,   1.31909_r8,   1.31796_r8,   1.31631_r8,   1.31542_r8, &
         1.31540_r8,   1.31552_r8,   1.31455_r8,   1.31193_r8,   1.30677_r8,   1.29934_r8, &
         1.29253_r8,   1.28389_r8,   1.27401_r8,   1.26724_r8,   1.25990_r8,   1.24510_r8, &
         1.22241_r8,   1.19913_r8,   1.17150_r8,   1.15528_r8,   1.13700_r8,   1.11808_r8, &
         1.10134_r8,   1.09083_r8,   1.08734_r8,   1.09254_r8,   1.10654_r8,   1.14779_r8, &
         1.20202_r8,   1.25825_r8,   1.32305_r8,   1.38574_r8,   1.44478_r8,   1.47170_r8, &
         1.49619_r8,   1.51652_r8,   1.53328_r8,   1.54900_r8,   1.56276_r8,   1.57317_r8, &
         1.58028_r8,   1.57918_r8,   1.56672_r8,   1.55869_r8,   1.55081_r8,   1.53807_r8, &
         1.53296_r8,   1.53220_r8,   1.53340_r8,   1.53289_r8,   1.51705_r8,   1.50097_r8, &
         1.49681_r8,   1.49928_r8,   1.50153_r8,   1.49856_r8,   1.49053_r8,   1.46070_r8, &
         1.45182_r8,   1.44223_r8,   1.43158_r8,   1.41385_r8,   1.40676_r8,   1.38955_r8, &
         1.34894_r8,   1.31039_r8,   1.26420_r8,   1.23656_r8,   1.21663_r8,   1.20233_r8, &
         1.19640_r8,   1.19969_r8,   1.20860_r8,   1.22173_r8,   1.24166_r8,   1.28175_r8, &
         1.32784_r8,   1.38657_r8,   1.46486_r8,   1.55323_r8,   1.60379_r8,   1.61877_r8, &
         1.62963_r8,   1.65712_r8,   1.69810_r8,   1.72065_r8,   1.74865_r8,   1.76736_r8, &
         1.76476_r8,   1.75011_r8,   1.72327_r8,   1.68490_r8,   1.62398_r8,   1.59596_r8, &
         1.58514_r8,   1.59917_r8,   1.61405_r8,   1.66625_r8,   1.70663_r8,   1.73713_r8, &
         1.76860_r8,   1.80343_r8,   1.83296_r8,   1.85682_r8,   1.87411_r8,   1.89110_r8/
      data (tabre(i),i=457,468)/ &
         1.89918_r8,   1.90432_r8,   1.90329_r8,   1.88744_r8,   1.87499_r8,   1.86702_r8, &
         1.85361_r8,   1.84250_r8,   1.83225_r8,   1.81914_r8,   1.82268_r8,   1.82961_r8/
      data (tabret(i,1),i=1,nwlt)/ &
                                          1.82961_r8,   1.83258_r8,   1.83149_r8, &
         1.82748_r8,   1.82224_r8,   1.81718_r8,   1.81204_r8,   1.80704_r8,   1.80250_r8, &
         1.79834_r8,   1.79482_r8,   1.79214_r8,   1.78843_r8,   1.78601_r8,   1.78434_r8, &
         1.78322_r8,   1.78248_r8,   1.78201_r8,   1.78170_r8,   1.78160_r8,   1.78190_r8, &
         1.78300_r8,   1.78430_r8,   1.78520_r8,   1.78620_r8,   1.78660_r8,   1.78680_r8, &
         1.78690_r8,   1.78700_r8,   1.78700_r8,   1.78710_r8,   1.78710_r8,   1.78720_r8, &
         1.78720_r8,   1.78720_r8,   1.78720_r8,   1.78720_r8,   1.78720_r8,   1.78720_r8, &
         1.78720_r8,   1.78720_r8,   1.78720_r8,   1.78720_r8,   1.78720_r8,   1.78720_r8, &
         1.78720_r8,   1.78720_r8,   1.78720_r8,   1.78720_r8,   1.78720_r8,   1.78720_r8, &
         1.78720_r8,   1.78720_r8,   1.78720_r8,   1.78720_r8,   1.78720_r8,   1.78720_r8, &
         1.78720_r8,   1.78720_r8,   1.78720_r8,   1.78720_r8,   1.78800_r8/
      data (tabret(i,2),i=1,nwlt)/ &
                               1.82961_r8,   1.83258_r8,   1.83149_r8,   1.82748_r8, &
         1.82224_r8,   1.81718_r8,   1.81204_r8,   1.80704_r8,   1.80250_r8,   1.79834_r8, &
         1.79482_r8,   1.79214_r8,   1.78843_r8,   1.78601_r8,   1.78434_r8,   1.78322_r8, &
         1.78248_r8,   1.78201_r8,   1.78170_r8,   1.78160_r8,   1.78190_r8,   1.78300_r8, &
         1.78430_r8,   1.78520_r8,   1.78610_r8,   1.78630_r8,   1.78640_r8,   1.78650_r8, &
         1.78650_r8,   1.78650_r8,   1.78650_r8,   1.78650_r8,   1.78650_r8,   1.78650_r8, &
         1.78650_r8,   1.78650_r8,   1.78650_r8,   1.78650_r8,   1.78650_r8,   1.78650_r8, &
         1.78650_r8,   1.78650_r8,   1.78650_r8,   1.78650_r8,   1.78650_r8,   1.78650_r8, &
         1.78650_r8,   1.78650_r8,   1.78650_r8,   1.78650_r8,   1.78650_r8,   1.78650_r8, &
         1.78650_r8,   1.78650_r8,   1.78650_r8,   1.78650_r8,   1.78650_r8,   1.78650_r8, &
         1.78650_r8,   1.78650_r8,   1.78650_r8,   1.78720_r8/
      data(tabret(i,3),i=1,nwlt)/ &
                    1.82961_r8,   1.83258_r8,   1.83149_r8,   1.82748_r8,   1.82224_r8, &
         1.81718_r8,   1.81204_r8,   1.80704_r8,   1.80250_r8,   1.79834_r8,   1.79482_r8, &
         1.79214_r8,   1.78843_r8,   1.78601_r8,   1.78434_r8,   1.78322_r8,   1.78248_r8, &
         1.78201_r8,   1.78160_r8,   1.78140_r8,   1.78160_r8,   1.78220_r8,   1.78310_r8, &
         1.78380_r8,   1.78390_r8,   1.78400_r8,   1.78400_r8,   1.78400_r8,   1.78400_r8, &
         1.78400_r8,   1.78390_r8,   1.78380_r8,   1.78370_r8,   1.78370_r8,   1.78370_r8, &
         1.78370_r8,   1.78370_r8,   1.78370_r8,   1.78370_r8,   1.78370_r8,   1.78370_r8, &
         1.78370_r8,   1.78370_r8,   1.78370_r8,   1.78370_r8,   1.78370_r8,   1.78370_r8, &
         1.78370_r8,   1.78370_r8,   1.78370_r8,   1.78370_r8,   1.78370_r8,   1.78370_r8, &
         1.78370_r8,   1.78370_r8,   1.78370_r8,   1.78370_r8,   1.78370_r8,   1.78370_r8, &
         1.78370_r8,   1.78400_r8,   1.78450_r8/
      data (tabret(i,4),i=1,nwlt)/ &
         1.82961_r8,   1.83258_r8,   1.83149_r8,   1.82748_r8,   1.82224_r8,   1.81718_r8, &
         1.81204_r8,   1.80704_r8,   1.80250_r8,   1.79834_r8,   1.79482_r8,   1.79214_r8, &
         1.78843_r8,   1.78601_r8,   1.78434_r8,   1.78322_r8,   1.78248_r8,   1.78201_r8, &
         1.78150_r8,   1.78070_r8,   1.78010_r8,   1.77890_r8,   1.77790_r8,   1.77730_r8, &
         1.77720_r8,   1.77720_r8,   1.77720_r8,   1.77720_r8,   1.77720_r8,   1.77720_r8, &
         1.77720_r8,   1.77720_r8,   1.77720_r8,   1.77720_r8,   1.77720_r8,   1.77720_r8, &
         1.77720_r8,   1.77720_r8,   1.77720_r8,   1.77720_r8,   1.77720_r8,   1.77720_r8, &
         1.77720_r8,   1.77720_r8,   1.77720_r8,   1.77720_r8,   1.77720_r8,   1.77720_r8, &
         1.77720_r8,   1.77720_r8,   1.77720_r8,   1.77720_r8,   1.77720_r8,   1.77720_r8, &
         1.77720_r8,   1.77720_r8,   1.77720_r8,   1.77720_r8,   1.77720_r8,   1.77720_r8, &
         1.77720_r8,   1.77800_r8/
      data(tabim(i),i=1,114)/ &
      0.1640e+00_r8,0.1730e+00_r8,0.1830e+00_r8,0.1950e+00_r8,0.2080e+00_r8,0.2230e+00_r8, &
      0.2400e+00_r8,0.2500e+00_r8,0.2590e+00_r8,0.2680e+00_r8,0.2790e+00_r8,0.2970e+00_r8, &
      0.3190e+00_r8,0.3400e+00_r8,0.3660e+00_r8,0.3920e+00_r8,0.4160e+00_r8,0.4400e+00_r8, &
      0.4640e+00_r8,0.4920e+00_r8,0.5170e+00_r8,0.5280e+00_r8,0.5330e+00_r8,0.5340e+00_r8, &
      0.5310e+00_r8,0.5240e+00_r8,0.5100e+00_r8,0.5000e+00_r8,0.4990e+00_r8,0.4680e+00_r8, &
      0.3800e+00_r8,0.3600e+00_r8,0.3390e+00_r8,0.3180e+00_r8,0.2910e+00_r8,0.2510e+00_r8, &
      0.2440e+00_r8,0.2390e+00_r8,0.2390e+00_r8,0.2440e+00_r8,0.2470e+00_r8,0.2240e+00_r8, &
      0.1950e+00_r8,0.1740e+00_r8,0.1720e+00_r8,0.1800e+00_r8,0.1940e+00_r8,0.2130e+00_r8, &
      0.2430e+00_r8,0.2710e+00_r8,0.2890e+00_r8,0.3340e+00_r8,0.3440e+00_r8,0.3820e+00_r8, &
      0.4010e+00_r8,0.4065e+00_r8,0.4050e+00_r8,0.3890e+00_r8,0.3770e+00_r8,0.3450e+00_r8, &
      0.3320e+00_r8,0.3150e+00_r8,0.2980e+00_r8,0.2740e+00_r8,0.2280e+00_r8,0.1980e+00_r8, &
      0.1720e+00_r8,0.1560e+00_r8,0.1100e+00_r8,0.8300e-01_r8,0.5800e-01_r8,0.2200e-01_r8, &
      0.1000e-01_r8,0.3000e-02_r8,0.1000e-02_r8,0.3000e-03_r8,0.1000e-03_r8,0.3000e-04_r8, &
      0.1000e-04_r8,0.3000e-05_r8,0.1000e-05_r8,0.7000e-06_r8,0.4000e-06_r8,0.2000e-06_r8, &
      0.1000e-06_r8,0.6377e-07_r8,0.3750e-07_r8,0.2800e-07_r8,0.2400e-07_r8,0.2200e-07_r8, &
      0.1900e-07_r8,0.1750e-07_r8,0.1640e-07_r8,0.1590e-07_r8,0.1325e-07_r8,0.8623e-08_r8, &
      0.5504e-08_r8,0.3765e-08_r8,0.2710e-08_r8,0.2510e-08_r8,0.2260e-08_r8,0.2080e-08_r8, &
      0.1910e-08_r8,0.1540e-08_r8,0.1530e-08_r8,0.1550e-08_r8,0.1640e-08_r8,0.1780e-08_r8, &
      0.1910e-08_r8,0.2140e-08_r8,0.2260e-08_r8,0.2540e-08_r8,0.2930e-08_r8,0.3110e-08_r8/
      data(tabim(i),i=115,228)/ &
      0.3290e-08_r8,0.3520e-08_r8,0.4040e-08_r8,0.4880e-08_r8,0.5730e-08_r8,0.6890e-08_r8, &
      0.8580e-08_r8,0.1040e-07_r8,0.1220e-07_r8,0.1430e-07_r8,0.1660e-07_r8,0.1890e-07_r8, &
      0.2090e-07_r8,0.2400e-07_r8,0.2900e-07_r8,0.3440e-07_r8,0.4030e-07_r8,0.4300e-07_r8, &
      0.4920e-07_r8,0.5870e-07_r8,0.7080e-07_r8,0.8580e-07_r8,0.1020e-06_r8,0.1180e-06_r8, &
      0.1340e-06_r8,0.1400e-06_r8,0.1430e-06_r8,0.1450e-06_r8,0.1510e-06_r8,0.1830e-06_r8, &
      0.2150e-06_r8,0.2650e-06_r8,0.3350e-06_r8,0.3920e-06_r8,0.4200e-06_r8,0.4440e-06_r8, &
      0.4740e-06_r8,0.5110e-06_r8,0.5530e-06_r8,0.6020e-06_r8,0.7550e-06_r8,0.9260e-06_r8, &
      0.1120e-05_r8,0.1330e-05_r8,0.1620e-05_r8,0.2000e-05_r8,0.2250e-05_r8,0.2330e-05_r8, &
      0.2330e-05_r8,0.2170e-05_r8,0.1960e-05_r8,0.1810e-05_r8,0.1740e-05_r8,0.1730e-05_r8, &
      0.1700e-05_r8,0.1760e-05_r8,0.1820e-05_r8,0.2040e-05_r8,0.2250e-05_r8,0.2290e-05_r8, &
      0.3040e-05_r8,0.3840e-05_r8,0.4770e-05_r8,0.5760e-05_r8,0.6710e-05_r8,0.8660e-05_r8, &
      0.1020e-04_r8,0.1130e-04_r8,0.1220e-04_r8,0.1290e-04_r8,0.1320e-04_r8,0.1350e-04_r8, &
      0.1330e-04_r8,0.1320e-04_r8,0.1320e-04_r8,0.1310e-04_r8,0.1320e-04_r8,0.1320e-04_r8, &
      0.1340e-04_r8,0.1390e-04_r8,0.1420e-04_r8,0.1480e-04_r8,0.1580e-04_r8,0.1740e-04_r8, &
      0.1980e-04_r8,0.2500e-04_r8,0.5400e-04_r8,0.1040e-03_r8,0.2030e-03_r8,0.2708e-03_r8, &
      0.3511e-03_r8,0.4299e-03_r8,0.5181e-03_r8,0.5855e-03_r8,0.5899e-03_r8,0.5635e-03_r8, &
      0.5480e-03_r8,0.5266e-03_r8,0.4394e-03_r8,0.3701e-03_r8,0.3372e-03_r8,0.2410e-03_r8, &
      0.1890e-03_r8,0.1660e-03_r8,0.1450e-03_r8,0.1280e-03_r8,0.1030e-03_r8,0.8600e-04_r8, &
      0.8220e-04_r8,0.8030e-04_r8,0.8500e-04_r8,0.9900e-04_r8,0.1500e-03_r8,0.2950e-03_r8/
      data(tabim(i),i=229,342)/ &
      0.4687e-03_r8,0.7615e-03_r8,0.1010e-02_r8,0.1313e-02_r8,0.1539e-02_r8,0.1588e-02_r8, &
      0.1540e-02_r8,0.1412e-02_r8,0.1244e-02_r8,0.1068e-02_r8,0.8414e-03_r8,0.5650e-03_r8, &
      0.4320e-03_r8,0.3500e-03_r8,0.2870e-03_r8,0.2210e-03_r8,0.2030e-03_r8,0.2010e-03_r8, &
      0.2030e-03_r8,0.2140e-03_r8,0.2320e-03_r8,0.2890e-03_r8,0.3810e-03_r8,0.4620e-03_r8, &
      0.5480e-03_r8,0.6180e-03_r8,0.6800e-03_r8,0.7300e-03_r8,0.7820e-03_r8,0.8480e-03_r8, &
      0.9250e-03_r8,0.9200e-03_r8,0.8920e-03_r8,0.8700e-03_r8,0.8900e-03_r8,0.9300e-03_r8, &
      0.1010e-02_r8,0.1350e-02_r8,0.3420e-02_r8,0.7920e-02_r8,0.2000e-01_r8,0.3800e-01_r8, &
      0.5200e-01_r8,0.6800e-01_r8,0.9230e-01_r8,0.1270e+00_r8,0.1690e+00_r8,0.2210e+00_r8, &
      0.2760e+00_r8,0.3120e+00_r8,0.3470e+00_r8,0.3880e+00_r8,0.4380e+00_r8,0.4930e+00_r8, &
      0.5540e+00_r8,0.6120e+00_r8,0.6250e+00_r8,0.5930e+00_r8,0.5390e+00_r8,0.4910e+00_r8, &
      0.4380e+00_r8,0.3720e+00_r8,0.3000e+00_r8,0.2380e+00_r8,0.1930e+00_r8,0.1580e+00_r8, &
      0.1210e+00_r8,0.1030e+00_r8,0.8360e-01_r8,0.6680e-01_r8,0.5400e-01_r8,0.4220e-01_r8, &
      0.3420e-01_r8,0.2740e-01_r8,0.2200e-01_r8,0.1860e-01_r8,0.1520e-01_r8,0.1260e-01_r8, &
      0.1060e-01_r8,0.8020e-02_r8,0.6850e-02_r8,0.6600e-02_r8,0.6960e-02_r8,0.9160e-02_r8, &
      0.1110e-01_r8,0.1450e-01_r8,0.2000e-01_r8,0.2300e-01_r8,0.2600e-01_r8,0.2900e-01_r8, &
      0.2930e-01_r8,0.3000e-01_r8,0.2850e-01_r8,0.1730e-01_r8,0.1290e-01_r8,0.1200e-01_r8, &
      0.1250e-01_r8,0.1340e-01_r8,0.1400e-01_r8,0.1750e-01_r8,0.2400e-01_r8,0.3500e-01_r8, &
      0.3800e-01_r8,0.4200e-01_r8,0.4600e-01_r8,0.5200e-01_r8,0.5700e-01_r8,0.6900e-01_r8, &
      0.7000e-01_r8,0.6700e-01_r8,0.6500e-01_r8,0.6400e-01_r8,0.6200e-01_r8,0.5900e-01_r8/
      data(tabim(i),i=343,456)/ &
      0.5700e-01_r8,0.5600e-01_r8,0.5500e-01_r8,0.5700e-01_r8,0.5800e-01_r8,0.5700e-01_r8, &
      0.5500e-01_r8,0.5500e-01_r8,0.5400e-01_r8,0.5200e-01_r8,0.5200e-01_r8,0.5200e-01_r8, &
      0.5200e-01_r8,0.5000e-01_r8,0.4700e-01_r8,0.4300e-01_r8,0.3900e-01_r8,0.3700e-01_r8, &
      0.3900e-01_r8,0.4000e-01_r8,0.4200e-01_r8,0.4400e-01_r8,0.4500e-01_r8,0.4600e-01_r8, &
      0.4700e-01_r8,0.5100e-01_r8,0.6500e-01_r8,0.7500e-01_r8,0.8800e-01_r8,0.1080e+00_r8, &
      0.1340e+00_r8,0.1680e+00_r8,0.2040e+00_r8,0.2480e+00_r8,0.2800e+00_r8,0.3410e+00_r8, &
      0.3790e+00_r8,0.4090e+00_r8,0.4220e+00_r8,0.4220e+00_r8,0.4030e+00_r8,0.3890e+00_r8, &
      0.3740e+00_r8,0.3540e+00_r8,0.3350e+00_r8,0.3150e+00_r8,0.2940e+00_r8,0.2710e+00_r8, &
      0.2460e+00_r8,0.1980e+00_r8,0.1640e+00_r8,0.1520e+00_r8,0.1420e+00_r8,0.1280e+00_r8, &
      0.1250e+00_r8,0.1230e+00_r8,0.1160e+00_r8,0.1070e+00_r8,0.7900e-01_r8,0.7200e-01_r8, &
      0.7600e-01_r8,0.7500e-01_r8,0.6700e-01_r8,0.5500e-01_r8,0.4500e-01_r8,0.2900e-01_r8, &
      0.2750e-01_r8,0.2700e-01_r8,0.2730e-01_r8,0.2890e-01_r8,0.3000e-01_r8,0.3400e-01_r8, &
      0.5300e-01_r8,0.7550e-01_r8,0.1060e+00_r8,0.1350e+00_r8,0.1761e+00_r8,0.2229e+00_r8, &
      0.2746e+00_r8,0.3280e+00_r8,0.3906e+00_r8,0.4642e+00_r8,0.5247e+00_r8,0.5731e+00_r8, &
      0.6362e+00_r8,0.6839e+00_r8,0.7091e+00_r8,0.6790e+00_r8,0.6250e+00_r8,0.5654e+00_r8, &
      0.5433e+00_r8,0.5292e+00_r8,0.5070e+00_r8,0.4883e+00_r8,0.4707e+00_r8,0.4203e+00_r8, &
      0.3771e+00_r8,0.3376e+00_r8,0.3056e+00_r8,0.2835e+00_r8,0.3170e+00_r8,0.3517e+00_r8, &
      0.3902e+00_r8,0.4509e+00_r8,0.4671e+00_r8,0.4779e+00_r8,0.4890e+00_r8,0.4899e+00_r8, &
      0.4873e+00_r8,0.4766e+00_r8,0.4508e+00_r8,0.4193e+00_r8,0.3880e+00_r8,0.3433e+00_r8/
      data(tabim(i),i=457,468)/ &
      0.3118e+00_r8,0.2935e+00_r8,0.2350e+00_r8,0.1981e+00_r8,0.1865e+00_r8,0.1771e+00_r8, &
      0.1620e+00_r8,0.1490e+00_r8,0.1390e+00_r8,0.1200e+00_r8,0.9620e-01_r8,0.8300e-01_r8/
      data(tabimt(i,1),i=1,nwlt)/ &
                                       0.8300e-01_r8,0.6900e-01_r8,0.5700e-01_r8, &
      0.4560e-01_r8,0.3790e-01_r8,0.3140e-01_r8,0.2620e-01_r8,0.2240e-01_r8,0.1960e-01_r8, &
      0.1760e-01_r8,0.1665e-01_r8,0.1620e-01_r8,0.1550e-01_r8,0.1470e-01_r8,0.1390e-01_r8, &
      0.1320e-01_r8,0.1250e-01_r8,0.1180e-01_r8,0.1060e-01_r8,0.9540e-02_r8,0.8560e-02_r8, &
      0.6210e-02_r8,0.4490e-02_r8,0.3240e-02_r8,0.2340e-02_r8,0.1880e-02_r8,0.1740e-02_r8, &
      0.1500e-02_r8,0.1320e-02_r8,0.1160e-02_r8,0.8800e-03_r8,0.6950e-03_r8,0.4640e-03_r8, &
      0.3400e-03_r8,0.3110e-03_r8,0.2940e-03_r8,0.2790e-03_r8,0.2700e-03_r8,0.2640e-03_r8, &
      0.2580e-03_r8,0.2520e-03_r8,0.2490e-03_r8,0.2540e-03_r8,0.2640e-03_r8,0.2740e-03_r8, &
      0.2890e-03_r8,0.3050e-03_r8,0.3150e-03_r8,0.3460e-03_r8,0.3820e-03_r8,0.4620e-03_r8, &
      0.5000e-03_r8,0.5500e-03_r8,0.5950e-03_r8,0.6470e-03_r8,0.6920e-03_r8,0.7420e-03_r8, &
      0.8200e-03_r8,0.9700e-03_r8,0.1950e-02_r8,0.5780e-02_r8,0.9700e-02_r8/
      data(tabimt(i,2),i=1,nwlt)/ &
                            0.8300e-01_r8,0.6900e-01_r8,0.5700e-01_r8,0.4560e-01_r8, &
      0.3790e-01_r8,0.3140e-01_r8,0.2620e-01_r8,0.2240e-01_r8,0.1960e-01_r8,0.1760e-01_r8, &
      0.1665e-01_r8,0.1600e-01_r8,0.1500e-01_r8,0.1400e-01_r8,0.1310e-01_r8,0.1230e-01_r8, &
      0.1150e-01_r8,0.1080e-01_r8,0.9460e-02_r8,0.8290e-02_r8,0.7270e-02_r8,0.4910e-02_r8, &
      0.3300e-02_r8,0.2220e-02_r8,0.1490e-02_r8,0.1140e-02_r8,0.1060e-02_r8,0.9480e-03_r8, &
      0.8500e-03_r8,0.7660e-03_r8,0.6300e-03_r8,0.5200e-03_r8,0.3840e-03_r8,0.2960e-03_r8, &
      0.2700e-03_r8,0.2520e-03_r8,0.2440e-03_r8,0.2360e-03_r8,0.2300e-03_r8,0.2280e-03_r8, &
      0.2250e-03_r8,0.2200e-03_r8,0.2160e-03_r8,0.2170e-03_r8,0.2200e-03_r8,0.2250e-03_r8, &
      0.2320e-03_r8,0.2390e-03_r8,0.2600e-03_r8,0.2860e-03_r8,0.3560e-03_r8,0.3830e-03_r8, &
      0.4150e-03_r8,0.4450e-03_r8,0.4760e-03_r8,0.5080e-03_r8,0.5400e-03_r8,0.5860e-03_r8, &
      0.6780e-03_r8,0.1280e-02_r8,0.3550e-02_r8,0.5600e-02_r8/
      data(tabimt(i,3),i=1,nwlt)/ &
                 0.8300e-01_r8,0.6900e-01_r8,0.5700e-01_r8,0.4560e-01_r8,0.3790e-01_r8, &
      0.3140e-01_r8,0.2620e-01_r8,0.2190e-01_r8,0.1880e-01_r8,0.1660e-01_r8,0.1540e-01_r8, &
      0.1470e-01_r8,0.1350e-01_r8,0.1250e-01_r8,0.1150e-01_r8,0.1060e-01_r8,0.9770e-02_r8, &
      0.9010e-02_r8,0.7660e-02_r8,0.6520e-02_r8,0.5540e-02_r8,0.3420e-02_r8,0.2100e-02_r8, &
      0.1290e-02_r8,0.7930e-03_r8,0.5700e-03_r8,0.5350e-03_r8,0.4820e-03_r8,0.4380e-03_r8, &
      0.4080e-03_r8,0.3500e-03_r8,0.3200e-03_r8,0.2550e-03_r8,0.2120e-03_r8,0.2000e-03_r8, &
      0.1860e-03_r8,0.1750e-03_r8,0.1660e-03_r8,0.1560e-03_r8,0.1490e-03_r8,0.1440e-03_r8, &
      0.1350e-03_r8,0.1210e-03_r8,0.1160e-03_r8,0.1160e-03_r8,0.1170e-03_r8,0.1200e-03_r8, &
      0.1230e-03_r8,0.1320e-03_r8,0.1440e-03_r8,0.1680e-03_r8,0.1800e-03_r8,0.1900e-03_r8, &
      0.2090e-03_r8,0.2160e-03_r8,0.2290e-03_r8,0.2400e-03_r8,0.2600e-03_r8,0.2920e-03_r8, &
      0.6100e-03_r8,0.1020e-02_r8,0.1810e-02_r8/
      data(tabimt(i,4),i=1,nwlt)/ &
      0.8300e-01_r8,0.6900e-01_r8,0.5700e-01_r8,0.4450e-01_r8,0.3550e-01_r8,0.2910e-01_r8, &
      0.2440e-01_r8,0.1970e-01_r8,0.1670e-01_r8,0.1400e-01_r8,0.1235e-01_r8,0.1080e-01_r8, &
      0.8900e-02_r8,0.7340e-02_r8,0.6400e-02_r8,0.5600e-02_r8,0.5000e-02_r8,0.4520e-02_r8, &
      0.3680e-02_r8,0.2990e-02_r8,0.2490e-02_r8,0.1550e-02_r8,0.9610e-03_r8,0.5950e-03_r8, &
      0.3690e-03_r8,0.2670e-03_r8,0.2510e-03_r8,0.2290e-03_r8,0.2110e-03_r8,0.1960e-03_r8, &
      0.1730e-03_r8,0.1550e-03_r8,0.1310e-03_r8,0.1130e-03_r8,0.1060e-03_r8,0.9900e-04_r8, &
      0.9300e-04_r8,0.8730e-04_r8,0.8300e-04_r8,0.7870e-04_r8,0.7500e-04_r8,0.6830e-04_r8, &
      0.5600e-04_r8,0.4960e-04_r8,0.4550e-04_r8,0.4210e-04_r8,0.3910e-04_r8,0.3760e-04_r8, &
      0.3400e-04_r8,0.3100e-04_r8,0.2640e-04_r8,0.2510e-04_r8,0.2430e-04_r8,0.2390e-04_r8, &
      0.2370e-04_r8,0.2380e-04_r8,0.2400e-04_r8,0.2460e-04_r8,0.2660e-04_r8,0.4450e-04_r8, &
      0.8700e-04_r8,0.1320e-03_r8/
 
  pi = acos(-1.0_r8)
  n_r=0.0_r8
  n_i=0.0_r8

! // convert frequency to wavelength (um)
  alam=3E5_r8/freq
  if((alam < wlmin) .or. (alam > wlmax)) then
    print *, 'm_ice: wavelength out of bounds'
    stop
  endif

! // convert temperature to K
  tk = t + 273.16_r8

  if (alam < cutice) then

!   // region from 0.045 microns to 167.0 microns - no temperature depend
    do i=2,nwl
      if(alam < wl(i)) continue
    enddo
    x1=log(wl(i-1))
    x2=log(wl(i))
    y1=tabre(i-1)
    y2=tabre(i)
    x=log(alam)
    y=((x-x1)*(y2-y1)/(x2-x1))+y1
    n_r=y
    y1=log(abs(tabim(i-1)))
    y2=log(abs(tabim(i)))
    y=((x-x1)*(y2-y1)/(x2-x1))+y1
    n_i=exp(y)

  else

!   // region from 167.0 microns to 8.6 meters - temperature dependence
    if(tk > temref(1)) tk=temref(1)
    if(tk < temref(4)) tk=temref(4)
    do 11 i=2,4
      if(tk.ge.temref(i)) go to 12
    11 continue
    12 lt1=i
    lt2=i-1
    do 13 i=2,nwlt
      if(alam.le.wlt(i)) go to 14
    13 continue
    14 x1=log(wlt(i-1))
    x2=log(wlt(i))
    y1=tabret(i-1,lt1)
    y2=tabret(i,lt1)
    x=log(alam)
    ylo=((x-x1)*(y2-y1)/(x2-x1))+y1
    y1=tabret(i-1,lt2)
    y2=tabret(i,lt2)
    yhi=((x-x1)*(y2-y1)/(x2-x1))+y1
    t1=temref(lt1)
    t2=temref(lt2)
    y=((tk-t1)*(yhi-ylo)/(t2-t1))+ylo
    n_r=y
    y1=log(abs(tabimt(i-1,lt1)))
    y2=log(abs(tabimt(i,lt1)))
    ylo=((x-x1)*(y2-y1)/(x2-x1))+y1
    y1=log(abs(tabimt(i-1,lt2)))
    y2=log(abs(tabimt(i,lt2)))
    yhi=((x-x1)*(y2-y1)/(x2-x1))+y1
    y=((tk-t1)*(yhi-ylo)/(t2-t1))+ylo
    n_i=exp(y)

  endif

  end subroutine m_ice

! ----------------------------------------------------------------------------
! subroutine MIEINT
! ----------------------------------------------------------------------------
!
!     General purpose Mie scattering routine for single particles
!     Author: R Grainger 1990
!     History:
!     G Thomas, March 2005: Added calculation of Phase function and
!     code to ensure correct calculation of backscatter coeficient
!     Options/Extend_Source
!
      Subroutine MieInt(Dx, SCm, Inp, Dqv, Dqxt, Dqsc, Dbsc, Dg, Xs1, Xs2, DPh, Error)

      Integer(2)  Imaxx
      Parameter (Imaxx = 12000)
      Real(4)     RIMax          ! largest real part of refractive index
      Parameter (RIMax = 2.5_r8)
      Real(4)     IRIMax         ! largest imaginary part of refractive index
      Parameter (IRIMax = -2)
      Integer(2)  Itermax
      Parameter (Itermax = 12000 * 2.5_r8)
                                ! must be large enough to cope with the
                                ! largest possible nmx = x * abs(scm) + 15
                                ! or nmx =  Dx + 4.05*Dx**(1./3.) + 2.0
      Integer(2)  Imaxnp
      Parameter (Imaxnp = 10000)  ! Change this as required
!     INPUT
      Real (8)     Dx
      Complex(8)  SCm
      Integer(4)  Inp
      Real (8)     Dqv(Inp)
!     OUTPUT
      Complex(8)  Xs1(InP)
      Complex(8)  Xs2(InP)
      Real (8)     Dqxt
      Real (8)     Dqsc
      Real (8)     Dg
      Real (8)     Dbsc
      Real (8)     DPh(InP)
      Integer(4)  Error
!     LOCAL
      Integer(2)  I
      Integer(2)  NStop
      Integer(2)  NmX
      Integer(4)  N    ! N*N > 32767 ie N > 181
      Integer(4)  Inp2
      Real (8)     Chi,Chi0,Chi1
      Real (8)     APsi,APsi0,APsi1
      Real (8)     Pi0(Imaxnp)
      Real (8)     Pi1(Imaxnp)
      Real (8)     Taun(Imaxnp)
      Real (8)     Psi,Psi0,Psi1
      Complex(4)  Ir
      Complex(8) Cm
      Complex(8) A,ANM1,APB
      Complex(8) B,BNM1,AMB
      Complex(8) D(Itermax)
      Complex(8) Sp(Imaxnp)
      Complex(8) Sm(Imaxnp)
      Complex(8) Xi,Xi0,Xi1
      Complex(8) Y
!     ACCELERATOR VARIABLES
      Integer(2)  Tnp1
      Integer(2)  Tnm1
      Real (8)     Dn
      Real (8)     Rnx
      Real (8)     S(Imaxnp)
      Real (8)     T(Imaxnp)
      Real (8)     Turbo
      Real (8)     A2
      Complex(8) A1
      
      If ((Dx.Gt.Imaxx) .Or. (InP.Gt.ImaxNP)) Then
        Error = 1
        Return
      EndIf
      Cm = SCm
      Ir = 1 / Cm
      Y =  Dx * Cm
      If (Dx.Lt.0.02_r8) Then
         NStop = 2
      Else
         If (Dx.Le.8.0_r8) Then
            NStop = Dx + 4.00_r8*Dx**(1._r8/3._r8) + 2.0_r8
         Else
            If (Dx.Lt. 4200.0_r8) Then
               NStop = Dx + 4.05_r8*Dx**(1._r8/3._r8) + 2.0_r8
            Else
               NStop = Dx + 4.00_r8*Dx**(1._r8/3._r8) + 2.0_r8
            End If
         End If
      End If
      NmX = Max(Real(NStop),Real(Abs(Y))) + 15._r8
      If (Nmx .gt. Itermax) then
          Error = 1
          Return
      End If
      Inp2 = Inp+1
      D(NmX) = cmplx(0,0,kind=Kind(0d0))
      Do N = Nmx-1,1,-1
         A1 = (N+1) / Y
         D(N) = A1 - 1/(A1+D(N+1))
      End Do
      Do I =1,Inp2
         Sm(I) = cmplx(0,0,kind=Kind(0d0))
         Sp(I) = cmplx(0,0,kind=Kind(0d0))
         Pi0(I) = 0
         Pi1(I) = 1
      End Do
      Psi0 = Cos(Dx)
      Psi1 = Sin(Dx)
      Chi0 =-Sin(Dx)
      Chi1 = Cos(Dx)
      APsi0 = Psi0
      APsi1 = Psi1
      Xi0 = cmplx(APsi0,Chi0,kind=Kind(0d0))
      Xi1 = cmplx(APsi1,Chi1,kind=Kind(0d0))
      Dg = 0
      Dqsc = 0
      Dqxt = 0
      Tnp1 = 1
      Do N = 1,Nstop
         DN = N
         Tnp1 = Tnp1 + 2
         Tnm1 = Tnp1 - 2
         A2 = Tnp1 / (DN*(DN+1D0))
         Turbo = (DN+1D0) / DN
         Rnx = DN/Dx
         Psi = Dble(Tnm1)*Psi1/Dx - Psi0
         APsi = Psi
         Chi = Tnm1*Chi1/Dx       - Chi0
         Xi = cmplx(APsi,Chi,kind=Kind(0d0))
         A = ((D(N)*Ir+Rnx)*APsi-APsi1) / ((D(N)*Ir+Rnx)*  Xi-  Xi1)
         B = ((D(N)*Cm+Rnx)*APsi-APsi1) / ((D(N)*Cm+Rnx)*  Xi-  Xi1)
         Dqxt = Tnp1 *      Dble(A + B)          + Dqxt
         Dqsc = Tnp1 * (A*Conjg(A) + B*Conjg(B)) + Dqsc
         If (N.Gt.1) then
         Dg = Dg + (dN*dN - 1) * Dble(ANM1*Conjg(A) + BNM1 * Conjg(B)) / dN + TNM1 * Dble(ANM1*Conjg(BNM1)) / (dN*dN - dN)
         End If
         Anm1 = A
         Bnm1 = B
         APB = A2 * (A + B)
         AMB = A2 * (A - B)
         Do I = 1,Inp2
            If (I.GT.Inp) Then
               S(I) = -Pi1(I)
            Else
               S(I) = Dqv(I) * Pi1(I)
            End If
            T(I) = S(I) - Pi0(I)
            Taun(I) = N*T(I) - Pi0(I)
            Sp(I) = APB * (Pi1(I) + Taun(I)) + Sp(I)
            Sm(I) = AMB * (Pi1(I) - Taun(I)) + Sm(I)
            Pi0(I) = Pi1(I)
            Pi1(I) = S(I) + T(I)*Turbo
         End Do
         Psi0 = Psi1
         Psi1 = Psi
         Apsi1 = Psi1
         Chi0 = Chi1
         Chi1 = Chi
         Xi1 = cmplx(APsi1,Chi1,kind=Kind(0d0))
      End Do
      If (Dg .GT.0) Dg = 2 * Dg / Dqsc
      Dqsc =  2 * Dqsc / Dx**2
      Dqxt =  2 * Dqxt / Dx**2
      Do I = 1,Inp
         Xs1(I) = (Sp(I)+Sm(I)) / 2
         Xs2(I) = (Sp(I)-Sm(I)) / 2
         Dph(I) = 2 * Dble(Xs1(I)*Conjg(Xs1(I)) + Xs2(I)*Conjg(Xs2(I))) / (Dx**2 * Dqsc)
      End Do
      Dbsc = 4 * Abs(( (Sp(Inp2)+Sm(Inp2))/2 )**2) / Dx**2
      Error = 0
      Return
      End subroutine MieInt

  end module optics_lib
