/*******************************************************************************
 * bbox.h
 *
 * This module contains all defines, typedefs, and prototypes for BBOX.CPP.
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/backend/bounding/bbox.h $
 * $Revision: #2 $
 * $Change: 5047 $
 * $DateTime: 2010/06/30 07:58:31 $
 * $Author: thorsten $
 *******************************************************************************/

/* NOTE: FRAME.H contains other bound stuff. */

#ifndef BBOX_H
#define BBOX_H

#include <map>

namespace pov
{

/*****************************************************************************
* Global preprocessor defines
******************************************************************************/

/* Generate additional bbox statistics. */

#define BBOX_EXTRA_STATS 1


/*****************************************************************************
* Global typedefs
******************************************************************************/

typedef int VECTORI[3];

class Rayinfo
{
	public:
		BBOX_VECT slab_num;
		BBOX_VECT slab_den;
		VECTORI nonzero;
		VECTORI positive;

		explicit Rayinfo(const Ray& ray)
		{
			DBL t;

			slab_num[X] = ray.Origin[X];
			slab_num[Y] = ray.Origin[Y];
			slab_num[Z] = ray.Origin[Z];

			if((nonzero[X] = ((t = ray.Direction[X]) != 0.0)) != 0)
			{
				slab_den[X] = 1.0 / t;
				positive[X] = (ray.Direction[X] > 0.0);
			}

			if((nonzero[Y] = ((t = ray.Direction[Y]) != 0.0)) != 0)
			{
				slab_den[Y] = 1.0 / t;
				positive[Y] = (ray.Direction[Y] > 0.0);
			}

			if((nonzero[Z] = ((t = ray.Direction[Z]) != 0.0)) != 0)
			{
				slab_den[Z] = 1.0 / t;
				positive[Z] = (ray.Direction[Z] > 0.0);
			}
		}
};

/*****************************************************************************
* Global functions
******************************************************************************/

struct PriorityQueue
{
	struct Qelem
	{
		DBL depth;
		BBOX_TREE *node;
	};

	unsigned QSize;
	unsigned Max_QSize;
	Qelem *Queue;

	PriorityQueue();
	~PriorityQueue();
};

void Build_BBox_Tree(BBOX_TREE **Root, size_t numOfFiniteObjects, BBOX_TREE **&Finite, size_t numOfInfiniteObjects, BBOX_TREE **Infinite, size_t& maxfinitecount);
void Build_Bounding_Slabs(BBOX_TREE **Root, vector<ObjectPtr>& objects, unsigned int& numberOfFiniteObjects, unsigned int& numberOfInfiniteObjects, unsigned int& numberOfLightSources);

void Recompute_BBox(BBOX *bbox, const TRANSFORM *trans);
void Recompute_Inverse_BBox(BBOX *bbox, const TRANSFORM *trans);
bool Intersect_BBox_Tree(PriorityQueue& pqueue, const BBOX_TREE *Root, Ray& ray, Intersection *Best_Intersection, TraceThreadData *Thread);
bool Intersect_BBox_Tree(PriorityQueue& pqueue, const BBOX_TREE *Root, Ray& ray, Intersection *Best_Intersection, const RayObjectCondition& precondition, const RayObjectCondition& postcondition, TraceThreadData *Thread);
void Check_And_Enqueue(PriorityQueue& Queue, const BBOX_TREE *Node, const BBOX *BBox, Rayinfo *rayinfo, TraceThreadData *Thread);
void Priority_Queue_Delete(PriorityQueue& Queue, DBL *key, const BBOX_TREE **Node);
void Destroy_BBox_Tree(BBOX_TREE *Node);


/*****************************************************************************
* Inline functions
******************************************************************************/

// Calculate the volume of a bounding box. [DB 8/94]
inline void BOUNDS_VOLUME(DBL& a, const BBOX& b)
{
	a = b.Lengths[X] * b.Lengths[Y] * b.Lengths[Z];
}

}

#endif
