/*******************************************************************************
 * interior.cpp
 *
 * This module contains all functions for interior stuff.
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/backend/interior/interior.cpp $
 * $Revision: #2 $
 * $Change: 5047 $
 * $DateTime: 2010/06/30 07:58:31 $
 * $Author: thorsten $
 *******************************************************************************/

// frame.h must always be the first POV file included (pulls in platform config)
#include "backend/frame.h"
#include "backend/interior/interior.h"
#include "backend/texture/texture.h"

// this must be the last file included
#include "base/povdebug.h"

namespace pov
{

/* How many subrays to trace for dispersive media */
#define DEFAULT_DISP_NELEMS  7

Interior::Interior()
{
	References = 1;

	IOR = 0.0;
	Old_Refract = 1.0;

	Dispersion  = 1.0;
	Disp_NElems = DEFAULT_DISP_NELEMS;

	Caustics = 0.0;

	Fade_Distance = 0.0;
	Fade_Power    = 0.0;

	hollow = false;
}

Interior::Interior(const Interior& source)
{
	References = 1;
	Disp_NElems = source.Disp_NElems;
	Dispersion = source.Dispersion;
	Old_Refract = source.Old_Refract;
	Fade_Distance = source.Fade_Distance;
	Fade_Power = source.Fade_Power;
	Fade_Colour = source.Fade_Colour;
	media = source.media;
	hollow = source.hollow;
	IOR = source.IOR;
	Caustics = source.Caustics;
}

Interior::~Interior()
{
}

void Interior::Transform(const TRANSFORM *trans)
{
	for(vector<Media>::iterator i(media.begin());i != media.end(); i++)
		i->Transform(trans);
}

void Interior::PostProcess()
{
	for(vector<Media>::iterator i(media.begin());i != media.end(); i++)
		i->PostProcess();
}

/*****************************************************************************
*
* FUNCTION
*
*   Destroy_Interior
*
* INPUT
*
*   Interior - interior to destroy
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Dieter Bayer
*
* DESCRIPTION
*
*   Destroy an interior.
*
* CHANGES
*
*   Dec 1994 : Creation.
*
******************************************************************************/

void Destroy_Interior(Interior *interior)
{
	if((interior != NULL) && (--(interior->References) == 0))
		delete interior;
}

/*****************************************************************************
*
* FUNCTION
*
*   Copy_Interior_Pointer
*
* INPUT
*
*   Old - interior to copy
*
* OUTPUT
*
* RETURNS
*
*   INTERIOR * - new interior
*
* AUTHOR
*
*   Dieter Bayer
*
* DESCRIPTION
*
*   Copy an interior by increasing number of references.
*
* CHANGES
*
*   Dec 1994 : Creation.
*
******************************************************************************/

Interior *Copy_Interior_Pointer(Interior *Old)
{
  if (Old != NULL)
  {
    Old->References++;
  }

  return(Old);
}

MATERIAL *Create_Material()
{
  MATERIAL *New;

  New = (MATERIAL *)POV_MALLOC(sizeof(MATERIAL), "material");
  
  New->Texture  = NULL;
  New->Interior_Texture  = NULL;
  New->interior = NULL;

  return(New);
}

MATERIAL *Copy_Material(MATERIAL *Old)
{
  MATERIAL *New;

  if (Old != NULL)
  {
    New = Create_Material();

    *New = *Old;

    New->Texture  = Copy_Textures(Old->Texture);
    New->Interior_Texture  = Copy_Textures(Old->Interior_Texture);
    if (Old->interior != NULL)
      New->interior = new Interior(*(Old->interior));

    return(New);
  }
  else
  {
    return(NULL);
  }
}

void Destroy_Material(MATERIAL *Material)
{
  if (Material != NULL) 
  {
    Destroy_Textures(Material->Texture);
    Destroy_Textures(Material->Interior_Texture);
    Destroy_Interior(Material->interior);

    POV_FREE(Material);
  }
}

}
