/*******************************************************************************
 * camera.cpp
 *
 * This module implements methods for managing the viewpoint.
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/backend/scene/camera.cpp $
 * $Revision: #2 $
 * $Change: 5047 $
 * $DateTime: 2010/06/30 07:58:31 $
 * $Author: thorsten $
 *******************************************************************************/

// frame.h must always be the first POV file included (pulls in platform config)
#include "backend/frame.h"
#include "backend/scene/camera.h"
#include "backend/texture/normal.h"
#include "backend/math/vector.h"
#include "backend/math/matrices.h"

// this must be the last file included
#include "base/povdebug.h"

namespace pov
{

/*****************************************************************************
*
* FUNCTION
*
*   Translate_Camera
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void Camera::Translate(const VECTOR Vector)
{
	VAddEq(Location, Vector);
}



/*****************************************************************************
*
* FUNCTION
*
*   Rotate_Camera
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void Camera::Rotate(const VECTOR Vector)
{
	TRANSFORM Trans;

	Compute_Rotation_Transform(&Trans, Vector);
	Transform(&Trans);
}



/*****************************************************************************
*
* FUNCTION
*
*   Scale_Camera
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void Camera::Scale(const VECTOR Vector)
{
	TRANSFORM Trans;

	Compute_Scaling_Transform(&Trans, Vector);
	Transform(&Trans);
}



/*****************************************************************************
*
* FUNCTION
*
*   Transform_Camera
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void Camera::Transform(const TRANSFORM *Trans)
{
	MTransPoint(Location, Location, Trans);
	MTransDirection(Direction, Direction, Trans);
	MTransDirection(Up, Up, Trans);
	MTransDirection(Right, Right, Trans);
}



/*****************************************************************************
*
* METHOD
*
*   Camera::Init
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void Camera::Init()
{
	Make_Vector(Location,    0.0,  0.0, 0.0);
	Make_Vector(Direction,   0.0,  0.0, 1.0);
	Make_Vector(Up,          0.0,  1.0, 0.0);
	Make_Vector(Right,       1.33, 0.0, 0.0);
	Make_Vector(Sky,         0.0,  1.0, 0.0);
	Make_Vector(Look_At,     0.0,  0.0, 1.0);
	Make_Vector(Focal_Point, 0.0,  0.0, 1.0);

	/* Init focal blur stuff (not used by default). */
	Blur_Samples   = 0;
	Confidence     = 0.9;
	Variance       = 1.0 / 10000.0;
	Aperture       = 0.0;
	Focal_Distance = -1.0;

	/* Set default camera type and viewing angle. [DB 7/94] */
	Type = PERSPECTIVE_CAMERA;
	Angle = 90.0;

	/* Default view angle for spherical camera. [MH 6/99] */
	H_Angle = 360;
	V_Angle = 180;

	/* Do not perturb primary rays by default. [DB 7/94] */
	Tnormal = NULL;

	Trans = Create_Transform();
}

/*****************************************************************************
*
* FUNCTION
*
*   Create_Camera
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

Camera::Camera()
{
	Init();
}



/*****************************************************************************
*
* FUNCTION
*
*   Copy_Camera
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

Camera& Camera::operator=(const Camera& src)
{
	Assign_Vector(Location, src.Location);
	Assign_Vector(Direction, src.Direction);
	Assign_Vector(Up, src.Up);
	Assign_Vector(Right, src.Right);
	Assign_Vector(Sky, src.Sky);
	Assign_Vector(Look_At, src.Look_At);
	Assign_Vector(Focal_Point, src.Focal_Point);

	Focal_Distance = src.Focal_Distance;
	Aperture = src.Aperture;
	Blur_Samples = src.Blur_Samples;
	Confidence = src.Confidence;
	Variance = src.Variance;
	Type = src.Type;
	Angle = src.Angle;
	H_Angle = src.H_Angle;
	V_Angle = src.V_Angle;

	if (Tnormal != NULL)
		Destroy_Tnormal(Tnormal);
	Tnormal = src.Tnormal ? Copy_Tnormal(src.Tnormal) : NULL;
	if (Trans != NULL)
		Destroy_Transform(Trans);
	Trans = src.Trans ? Copy_Transform(src.Trans) : NULL;

	return *this;
}

Camera::Camera(const Camera& src)
{
	Tnormal = NULL;
	Trans = NULL;
	operator=(src);
}

/*****************************************************************************
*
* FUNCTION
*
*   Destroy_Camera
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Dieter Bayer
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

Camera::~Camera()
{
	Destroy_Tnormal(Tnormal);
	Destroy_Transform(Trans);
}

}
