/*******************************************************************************
 * task.cpp
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/backend/support/task.cpp $
 * $Revision: #3 $
 * $Change: 5050 $
 * $DateTime: 2010/06/30 12:23:03 $
 * $Author: thorsten $
 *******************************************************************************/

#include <cassert>
#include <stdexcept>

#include <boost/bind.hpp>

// frame.h must always be the first POV file included (pulls in platform config)
#include "backend/frame.h"

#include "base/types.h"
#include "base/timer.h"

#include "backend/support/task.h"

// this must be the last file included
#include "base/povdebug.h"

namespace pov
{

using namespace pov_base;

Task::Task(TaskData *td, boost::function1<void, Exception&> f) :
	taskData(td),
	fatalErrorHandler(f),
	stopRequested(false),
	paused(false),
	done(false),
	failed(kNoError),
	timer(NULL),
	realTime(-1),
	cpuTime(-1),
	taskThread(NULL),
	povmsContext(NULL)
{
	if (td == NULL)
		throw POV_EXCEPTION_STRING("Internal error: TaskData is NULL in Task constructor");
	td->task = this;
}

Task::~Task()
{
	Stop();

	// NOTE: Freeing taskData is the responsiblity of the task creator!
	taskData = NULL;
}

int Task::FailureCode(int defval)
{
	if(failed == kNoError)
		return defval;
	else
		return failed;
}

POV_LONG Task::ConsumedRealTime() const
{
	return realTime;
}

POV_LONG Task::ConsumedCPUTime() const
{
	return cpuTime;
}

void Task::Start()
{
	if((done == false) && (taskThread == NULL))
	{
		taskThread = new Thread(boost::bind(&Task::TaskThread, this), POV_DEFAULT_THREAD_STACK_SIZE); // TODO - make stack size definable
		taskThread->SetThreadDataPtr(this);
	}
}

void Task::RequestStop()
{
	stopRequested = true;
}

void Task::Stop()
{
	stopRequested = true;

	if(taskThread != NULL)
	{
		taskThread->Join();
		delete taskThread;
		taskThread = NULL;
	}
}

void Task::Pause()
{
	paused = true;
}

void Task::Resume()
{
	paused = false;
}

POV_LONG Task::ElapsedRealTime() const
{
	assert(timer != NULL);
	return timer->ElapsedRealTime();
}

POV_LONG Task::ElapsedCPUTime() const
{
	assert(timer != NULL);
	return timer->ElapsedCPUTime();
}

void Task::TaskThread()
{
	int result;

	if((result = POVMS_OpenContext(&povmsContext)) != kNoErr)
	{
		failed = result;
		timer = NULL;
		done = true;
		return;
	}

	POV_SYS_THREAD_STARTUP

	Timer tasktime(true); // only keep CPU time of this thread (if supported)

	timer = &tasktime;

	try
	{
		Run();
	}
	catch(StopThreadException&)
	{
		try
		{
			Stopped();
		}
		catch(pov_base::Exception& e)
		{
			FatalErrorHandler(e);
			failed = e.code(kUncategorizedError);
		}
		catch(std::exception& e)
		{
			FatalErrorHandler(POV_EXCEPTION_STRING(e.what()));
			failed = kUncategorizedError;
		}
		catch(...)
		{
			FatalErrorHandler(POV_EXCEPTION_STRING("An unknown error occured stopping a task!"));
			failed = kUncategorizedError;
		}
	}
	catch(bad_alloc&)
	{
		// can't raise a new exception object here as the alloc will probably fail
#if 0
		FatalErrorHandler(POV_EXCEPTION_STRING("Out of memory! There is not enough memory available for\n"
		                                       "POV-Ray to complete a task. Usually this suggests the scene\n"
		                                       "you are trying to render is too complex for your hardware\n"
		                                       "to handle. Refer to the platform-specific documentation for\n"
		                                       "details, it might contain hints how to work around this.\n"));
#endif
		failed = kOutOfMemoryErr;
	}
	catch(pov_base::Exception& e)
	{
		FatalErrorHandler(e);
		failed = e.code(kUncategorizedError);
	}
	catch(std::exception& e)
	{
		FatalErrorHandler(POV_EXCEPTION_STRING(e.what()));
		failed = kUncategorizedError;
	}
	catch(...)
	{
		FatalErrorHandler(POV_EXCEPTION_STRING("An unknown error occured running a task!"));
		failed = kUncategorizedError;
	}

	realTime = tasktime.ElapsedRealTime();
	if(tasktime.HasValidCPUTime() == true)
		cpuTime = tasktime.ElapsedCPUTime();
	else
		cpuTime = -1;

	try
	{
		Finish();
	}
	catch(pov_base::Exception& e)
	{
		FatalErrorHandler(e);
	}
	catch(std::exception& e)
	{
		FatalErrorHandler(POV_EXCEPTION_STRING(e.what()));
	}
	catch(...)
	{
		FatalErrorHandler(POV_EXCEPTION_STRING("An unknown error occured finishing a task!"));
	}

	timer = NULL;
	done = true;

	POV_SYS_THREAD_CLEANUP

	(void)POVMS_CloseContext(povmsContext);
}

}
