/*******************************************************************************
 * taskqueue.cpp
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/backend/support/taskqueue.cpp $
 * $Revision: #2 $
 * $Change: 5047 $
 * $DateTime: 2010/06/30 07:58:31 $
 * $Author: thorsten $
 *******************************************************************************/

#include <boost/bind.hpp>

// frame.h must always be the first POV file included (pulls in platform config)
#include "backend/frame.h"
#include "backend/support/task.h"
#include "backend/support/taskqueue.h"

// this must be the last file included
#include "base/povdebug.h"

namespace pov
{

TaskQueue::TaskQueue() : failed(kNoError)
{
}

TaskQueue::~TaskQueue()
{
	Stop();
}

void TaskQueue::Stop()
{
	RecursiveMutex::ScopedLock lock(queueMutex);

	// we pass through this list twice; the first time through only sets the cancel
	// flag, and the second time through waits for the threads to exit. if we only
	// the the one pass (and wait), the cancel flag for later threads would remain
	// unset whilst we wait for earlier threads to reach a point where they check
	// the status of the flag, which tends to cause the shutdown to take longer
	// (particularly with a large number of threads vs a small number of CPU's).
	for(list<TaskEntry>::iterator i(activeTasks.begin()); i != activeTasks.end(); i++)
		i->GetTask()->RequestStop();
	for(list<TaskEntry>::iterator i(activeTasks.begin()); i != activeTasks.end(); i++)
		i->GetTask()->Stop();

	activeTasks.clear();
	while(queuedTasks.empty() == false)
		queuedTasks.pop();
}

void TaskQueue::Pause()
{
	RecursiveMutex::ScopedLock lock(queueMutex);

	for(list<TaskEntry>::iterator i(activeTasks.begin()); i != activeTasks.end(); i++)
		i->GetTask()->Pause();
}

void TaskQueue::Resume()
{
	RecursiveMutex::ScopedLock lock(queueMutex);

	for(list<TaskEntry>::iterator i(activeTasks.begin()); i != activeTasks.end(); i++)
		i->GetTask()->Resume();
}

bool TaskQueue::IsPaused()
{
	RecursiveMutex::ScopedLock lock(queueMutex);

	bool paused = false;

	for(list<TaskEntry>::iterator i(activeTasks.begin()); i != activeTasks.end(); i++)
		paused = paused || i->GetTask()->IsPaused();

	return paused;
}

bool TaskQueue::IsRunning()
{
	RecursiveMutex::ScopedLock lock(queueMutex);

	bool running = !queuedTasks.empty();

	for(list<TaskEntry>::iterator i(activeTasks.begin()); i != activeTasks.end(); i++)
		running = running || i->GetTask()->IsRunning();

	return running;
}

bool TaskQueue::IsDone()
{
	RecursiveMutex::ScopedLock lock(queueMutex);

	bool done = queuedTasks.empty();

	for(list<TaskEntry>::iterator i(activeTasks.begin()); i != activeTasks.end(); i++)
		done = done && i->GetTask()->IsDone();

	return done;
}

bool TaskQueue::Failed()
{
	RecursiveMutex::ScopedLock lock(queueMutex);

	return (failed != kNoError);
}

int TaskQueue::FailureCode(int defval)
{
	RecursiveMutex::ScopedLock lock(queueMutex);

	if(failed == kNoError)
		return defval;
	else
		return failed;
}

Task::TaskData *TaskQueue::AppendTask(Task *task)
{
	RecursiveMutex::ScopedLock lock(queueMutex);

	failed = false;

	queuedTasks.push(TaskEntry(shared_ptr<Task>(task)));

	return task->GetDataPtr();
}

void TaskQueue::AppendSync()
{
	RecursiveMutex::ScopedLock lock(queueMutex);

	queuedTasks.push(TaskEntry::kSync);
}

void TaskQueue::AppendMessage(POVMS_Message& msg)
{
	RecursiveMutex::ScopedLock lock(queueMutex);

	queuedTasks.push(TaskEntry(msg));
}

void TaskQueue::AppendFunction(const boost::function1<void, TaskQueue&>& fn)
{
	RecursiveMutex::ScopedLock lock(queueMutex);

	queuedTasks.push(TaskEntry(fn));
}

bool TaskQueue::Process()
{
	RecursiveMutex::ScopedLock lock(queueMutex);

	for(list<TaskEntry>::iterator i(activeTasks.begin()); i != activeTasks.end();)
	{
		if(failed == kNoError)
			failed = i->GetTask()->FailureCode();

		if(i->GetTask()->IsDone() == true)
		{
			list<TaskEntry>::iterator e(i);
			i++;
			activeTasks.erase(e);
		}
		else
			i++;
	}

	if(failed != kNoError)
	{
		Stop();
		return false;
	}

	if(queuedTasks.empty() == false)
	{
		switch(queuedTasks.front().GetEntryType())
		{
			case TaskEntry::kTask:
			{
				activeTasks.push_back(queuedTasks.front());
				queuedTasks.front().GetTask()->Start();
				queuedTasks.pop();
				break;
			}
			case TaskEntry::kSync:
			{
				if(activeTasks.empty() == true)
					queuedTasks.pop();
				else
					return false;
				break;
			}
			case TaskEntry::kMessage:
			{
				try { POVMS_SendMessage(queuedTasks.front().GetMessage()); } catch(pov_base::Exception&) { }
				queuedTasks.pop();
				break;
			}
			case TaskEntry::kFunction:
			{
				try { queuedTasks.front().GetFunction()(*this); } catch(pov_base::Exception&) { }
				queuedTasks.pop();
				break;
			}
		}
	}

	return (queuedTasks.empty() == false);
}

}
