/*******************************************************************************
 * image.h
 *
 * This file contains ... TODO ...
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/base/image/image.h $
 * $Revision: #3 $
 * $Change: 5050 $
 * $DateTime: 2010/06/30 12:23:03 $
 * $Author: thorsten $
 *******************************************************************************/

#ifndef POVRAY_BASE_IMAGE_H
#define POVRAY_BASE_IMAGE_H

#include <vector>
#include <list>

#include <boost/weak_ptr.hpp>

#include "base/configbase.h"
#include "base/fileinputoutput.h"
#include "base/pov_err.h"
#include "base/types.h"
#include "base/thread.h"

namespace pov_base
{

class GammaCurve;

/**
 *  Class holding a shared reference to a gamma curve.
 *  @note   This is a boost shared_ptr; the assignment operator (=) may not work as expected,
 *          as it is in fact implemented as a swap operation.
 *          To clone a reference, use "GammaCurvePtr myNewPtr(myOldPtr);".
 */
typedef boost::shared_ptr<GammaCurve> GammaCurvePtr;

/**
 *  Abstract class representing an encoding gamma curve (or, more generally, transfer function).
 *  In this generic form, the gamma curve may be arbitrarily complex.
 *  @note   To conserve memory, derived classes should prevent duplicates from being instantiated.
 *          This base class provides a caching mechanism to help accomplish this.
 */
class GammaCurve
{
	public:
		/**
		 *  Encoding function.
		 *  This function is to be implemented by subclasses to define the encoding rules of that particular gamma curve.
		 *  @note           Input values 0.0 and 1.0 should be mapped to output values 0.0 and 1.0, respectively.
		 *                  Input values that cannot be mapped should be clipped to the nearest valid value.
		 *  @param[in]  x   Input value.
		 *  @return         Output value.
		 */
		virtual float Encode(float x) const = 0;
		
		/**
		 *  Decoding function.
		 *  This function is to be implemented by subclasses to define the decoding rules of that particular gamma curve.
		 *  @note           Input values 0.0 and 1.0 should be mapped to output values 0.0 and 1.0, respectively.
		 *                  Input values that cannot be mapped should be clipped to the nearest valid value.
		 *  @param[in]  x   Input value.
		 *  @return         Output value.
		 */
		virtual float Decode(float x) const = 0;
		
		/**
		 *  Approximated power-law gamma
		 *  This function is to be implemented by subclasses to return an average overall gamma to be used as a
		 *  gamma-law approximation of the particular gamma curve.
		 */
		virtual float ApproximateEncodingGamma() const = 0;
		
		/**
		 *  Retrieves a lookup table for faster decoding.
		 *  This feature is intended to be used for low-dynamic-range, 8/16 bit depth, non-linearly encoded images
		 *  to be kept in encoded format during render to reduce the memory footprint without unnecessarily degrading
		 *  performance.
		 *  @note           The lookup table pointer is only valid during the lifetime of the GammaCurve object.
		 *                  Any entity holding a pointer to the lookup table must therefore also maintain a strong
		 *                  pointer to the GammaCurve object.
		 *  @param[in]  max The maximum encoded value; must be either 255 for 8-bit depth, or 65535 for 16-bit depth.
		 *  @return         Pointer to a table of pre-computed Decode() results for values from 0.0 to 1.0
		 *                  in increments of 1.0/max.
		 */
		float* GetLookupTable(unsigned int max);
		
		/**
		 *  Convenience function to test whether a gamma curve pointer refers to a neutral curve.
		 *  @param[in]  p   The gamma curve pointer to test.
		 *  @return         @c true if the gamma curve pointer is empty or the gamma curve is neutral, @c false otherwise.
		 */
		static bool IsNeutral(const GammaCurvePtr& p) { return (!p || p->IsNeutral()); }
		
		/**
		 *  Convenience function to apply encoding according to a given gamma curve pointer.
		 *  @note           If an empty gamma curve pointer is passed, neutral encoding is applied.
		 *  @param[in]  p   The gamma curve pointer to use for encoding.
		 *  @param[in]  x   The value to encode, typically in the range from 0.0 to 1.0.
		 *  @return         The encoded value, typically in the range from 0.0 to 1.0.
		 */
		static float Encode(const GammaCurvePtr& p, float x) { if (IsNeutral(p)) return x; else return p->Encode(x); }
		
		/**
		 *  Convenience function to apply encoding according to a given gamma curve pointer.
		 *  @note           If an empty gamma curve pointer is passed, neutral encoding is applied.
		 *  @param[in]  p   The gamma curve pointer to use for encoding.
		 *  @param[in]  c   The colour to encode, typically in the range from 0.0 to 1.0.
		 *  @return         The encoded colour, typically in the range from 0.0 to 1.0.
		 */
		static Colour Encode(const GammaCurvePtr& p, const Colour& c)
		{
			if (IsNeutral(p))
				return c;
			else
				return Colour(p->Encode(c.red()), p->Encode(c.green()), p->Encode(c.blue()), c.filter(), c.transm());
		}
		
		/**
		 *  Applies decoding according to a given gamma curve pointer.
		 *  @note           If an empty gamma curve pointer is passed, neutral decoding is applied.
		 *  @param[in]  p   The gamma curve pointer to use for decoding.
		 *  @param[in]  x   The value to decode, typically in the range from 0.0 to 1.0.
		 *  @return         The decoded value, typically in the range from 0.0 to 1.0.
		 */
		static float Decode(const GammaCurvePtr& p, float x) { if (IsNeutral(p)) return x; else return p->Decode(x); }
	protected:
		/**
		 *  Cached lookup table for 8-bit lookup.
		 *  This member variable caches the pointer returned by a first call to @c GetLookupTable(255) to avoid creating
		 *  multiple copies of the table.
		 */
		float *lookupTable8;

		/**
		 *  Cached lookup table for 16-bit lookup.
		 *  This member variable caches the pointer returned by a first call to @c GetLookupTable(65535) to avoid creating
		 *  multiple copies of the table.
		 */
		float *lookupTable16;
		
		/**
		 *  Mutex to guard access to @c lookupTable8 and @c lookupTable16.
		 */
		Mutex lutMutex;
		
		/**
		 *  Constructor.
		 */
		GammaCurve() : lookupTable8(NULL), lookupTable16(NULL) {}
		
		/**
		 *  Destructor.
		 */
		~GammaCurve() { if (lookupTable8) delete[] lookupTable8; if (lookupTable16) delete[] lookupTable16; }
		
		/**
		 *  Function to test whether two gamma curves match.
		 *  This function is to be implemented by subclasses to define how to test for matching gamma curves.
		 *  @param[in]  p   Pointer to the gamma curve to compare with.
		 *  @return         @c true if the gamma curve will produce the same result as this instance, @c false otherwise.
		 */
		virtual bool Matches(const GammaCurvePtr& p) const { return false; }

		/**
		 *  Function to test whether the gamma curve is neutral.
		 *  This function is to be implemented by subclasses to define how to test for neutral gamma curves.
		 *  @return         @c true if this gamma curve is neutral (i.e. maps any value to itself), @c false otherwise.
		 */
		virtual bool IsNeutral() const { return false; }

		/**
		 *  Cache of all gamma curves currently in use.
		 *  This static member variable caches pointers of gamma curve instances currently in use, forming the basis
		 *  of the @c GetMatching() mechanism to avoid duplicate instances.
		 */
		static list<boost::weak_ptr<GammaCurve> > cache;

		/// Mutex to guard access to GammaCurve cache.
		static Mutex cacheMutex;

		/**
		 *  Function to manage the gamma curve cache.
		 *  This static function will look up a gamma curve from the cache to match the supplied one, or encache the
		 *  supplied one if no match has been found.
		 *  @note           Derived classes allowing for multiple instances can pass any newly created instance
		 *                  through this function to make sure no duplicate gamma curve instances are ever in use.
		 *                  For this purpose, no references to the instance supplied shall be retained; instead,
		 *                  only the instance returned by this function shall be kept.
		 *  @param      p   Pointer to the gamma curve to look up or encache.
		 *  @return         A matching encached gamma curve (possibly, but not necessarily, the instance supplied).
		 */
		static GammaCurvePtr GetMatching(GammaCurvePtr p);
};

/**
 *  Class representing a neutral gamma curve.
 */
class NeutralGammaCurve : public GammaCurve
{
	public:
		static GammaCurvePtr Get() { if (!instance) instance.reset(new NeutralGammaCurve()); return instance; }
		float Encode(float x) const;
		float Decode(float x) const;
		float ApproximateEncodingGamma() const;
	private:
		static GammaCurvePtr instance;
		bool Matches(const GammaCurvePtr&) const;
		bool IsNeutral() const;
		NeutralGammaCurve();
};

/**
 *  Class representing the IEC 61966-2-1 sRGB transfer function.
 *  @note   While the sRGB transfer functionn can be approximated with a classic power-law curve
 *          having a constant gamma of 1/2.2, the two are not identical. This class represents
 *          the exact function as specified in IEC 61966-2-1.
 */
class SRGBGammaCurve : public GammaCurve
{
	public:
		static GammaCurvePtr Get() { if (!instance) instance.reset(new SRGBGammaCurve()); return instance; }
		float Encode(float x) const;
		float Decode(float x) const;
		float ApproximateEncodingGamma() const;
	private:
		static GammaCurvePtr instance;
		SRGBGammaCurve();
};

/**
 *  Class representing the ITU-R BT.709 transfer function.
 *  @note   This class does @e not account for the "black digital count" and "white digital count" being defined
 *          as 16/255 and 235/255, respectively.
 */
class ITURBT709GammaCurve : public GammaCurve
{
	public:
		static GammaCurvePtr Get() { if (!instance) instance.reset(new ITURBT709GammaCurve()); return instance; }
		float Encode(float x) const;
		float Decode(float x) const;
		float ApproximateEncodingGamma() const;
	private:
		static GammaCurvePtr instance;
		ITURBT709GammaCurve();
};

/**
 *  Class representing the Rec1361 transfer function.
 *  This transfer function is a wide-gamut extension to that specified in ITU-R BT.709.
 */
class Rec1361GammaCurve : public GammaCurve
{
	public:
		static GammaCurvePtr Get() { if (!instance) instance.reset(new Rec1361GammaCurve()); return instance; }
		float Encode(float x) const;
		float Decode(float x) const;
		float ApproximateEncodingGamma() const;
	private:
		static GammaCurvePtr instance;
		Rec1361GammaCurve();
};

/**
 *  Class representing a classic constant-gamma (power-law) gamma encoding curve.
 */
class PowerLawGammaCurve : public GammaCurve
{
	public:
		static GammaCurvePtr GetByEncodingGamma(float gamma);
		static GammaCurvePtr GetByDecodingGamma(float gamma);
		float Encode(float x) const;
		float Decode(float x) const;
		float ApproximateEncodingGamma() const;
	protected:
		float encGamma;
		PowerLawGammaCurve(float encGamma);
		bool Matches(const GammaCurvePtr&) const;
		bool IsNeutral() const;
		static bool IsNeutral(float gamma);
};

/**
 *  Class representing a scaled-encoding variant of another gamma curves.
 */
class ScaledGammaCurve : public GammaCurve
{
	public:
		static GammaCurvePtr GetByEncoding(GammaCurvePtr, float encodingFactor);
		static GammaCurvePtr GetByDecoding(float decodingFactor, GammaCurvePtr);
		float Encode(float x) const;
		float Decode(float x) const;
		float ApproximateEncodingGamma() const;
	protected:
		GammaCurvePtr baseGamma;
		float encFactor;
		ScaledGammaCurve(GammaCurvePtr, float);
		bool Matches(const GammaCurvePtr&) const;
		bool IsNeutral() const;
		static bool IsNeutral(float factor);
};

/**
 *  Linear encoding function.
 *  This function maps a floating-point value in the range 0..1 linearly to in an integer value in the range 0..max.
 *  @note   Floating-point values outside the range 0..1 are clipped, mapping them to 0 or max, respectively.
 *  @param  x       value to encode
 *  @param  max     encoded value representing 1.0
 */
inline unsigned int IntEncode(float x, unsigned int max) { return (unsigned int)floor( clip(x, 0.0f, 1.0f) * float(max) + 0.5f ); }

/**
 *  Linear decoding function.
 *  This function maps an integer value in the range 0..max linearly to a floating-point value in the range 0..1.
 *  @param  x       value to decode
 *  @param  max     encoded value representing 1.0
 */
inline float IntDecode(unsigned int x, unsigned int max) { return float(x) / float(max); }

/**
 *  Generic encoding function.
 *  This function maps a floating-point value in the range 0..1 to in an integer value in the range 0..max.
 *  @note   Floating-point values outside the range 0..1 (after applying the transfer function) are clipped,
 *          mapping them to 0 or max, respectively.
 *  @param  g       transfer function (gamma curve) to use
 *  @param  x       value to encode
 *  @param  max     encoded value representing 1.0
 */
inline unsigned int IntEncode(const GammaCurvePtr& g, float x, unsigned int max) { return IntEncode(GammaCurve::Encode(g,x),max); }

/**
 *  Generic decoding function.
 *  This function maps an integer value in the range 0..max to a floating-point value in the range 0..1
 *  using a specific transfer function (gamma curve).
 *  @param  g       transfer function (gamma curve) to use
 *  @param  x       value to decode
 *  @param  max     encoded value representing 1.0
 */
inline float IntDecode(const GammaCurvePtr& g, unsigned int x, unsigned int max) { return GammaCurve::Decode(g,IntDecode(x,max)); }


/**
 *  Generic image data container.
 *
 *  @note   Except for access functions having a @c premul parameter as well as those named
 *          @c GetEncodedSomethingValue or GetEncodedSomethingValue, all other access functions are unaware of
 *          premultiplied vs. non-premultiplied alpha issues, and will access the data in whatever format
 *          it is stored (as far as alpha handling goes).
 *
 *  @note   When backed by a gamma-encoded data container, unsigned int access methods are presumed
 *          to read/write raw encoded data, while float access methods will read/write logical
 *          linear values.
 */
class Image
{
	public:
		struct RGBMapEntry
		{
			float red;
			float green;
			float blue;

			RGBMapEntry() : red(0.0f), green(0.0f), blue(0.0f) { }
			RGBMapEntry(float r, float g, float b) : red(r), green(g), blue(b) { }
		};

		struct RGBAMapEntry
		{
			float red;
			float green;
			float blue;
			float alpha;

			RGBAMapEntry() : red(0.0f), green(0.0f), blue(0.0f), alpha(0.0f) { }
			RGBAMapEntry(float r, float g, float b, float a) : red(r), green(g), blue(b), alpha(a) { }
		};

		struct RGBFTMapEntry
		{
			float red;
			float green;
			float blue;
			float filter;
			float transm;

			RGBFTMapEntry() : red(0.0f), green(0.0f), blue(0.0f), filter(0.0f), transm(0.0f) { }
			RGBFTMapEntry(float r, float g, float b, float f, float t) : red(r), green(g), blue(b), filter(f), transm(t) { }
		};

		enum ColourMapType
		{
			NoColourMap,
			RGBColourMap,
			RGBAColourMap,
			RGBFTColourMap
		};

		enum ImageDataType
		{
			/// Value used to indicate that image decoder is free to pick the most fitting type.
			Undefined,
			/// Palette-based image with 2 palette entries.
			Bit_Map,
			/// Palette-based image with up to 256 palette entries.
			Colour_Map,
			/// Single-channel (grayscale) image using 8-bit linear encoding.
			Gray_Int8,
			/// Single-channel (grayscale) image using 16-bit linear encoding.
			Gray_Int16,
			/// Dual-channel (grayscale and alpha) image using 8-bit linear encoding.
			GrayA_Int8,
			/// Dual-channel (grayscale and alpha) image using 16-bit linear encoding.
			GrayA_Int16,
			/// 3-channel (colour) image using 8-bit linear encoding.
			RGB_Int8,
			/// 3-channel (colour) image using 16-bit linear encoding.
			RGB_Int16,
			/// 4-channel (colour and alpha) image using 8-bit linear encoding.
			RGBA_Int8,
			/// 4-channel (colour and alpha) image using 16-bit linear encoding.
			RGBA_Int16,
			/// 5-channel (colour, filter and transmit) image using single-precision floating-point encoding.
			RGBFT_Float,
			/// 3-channel (colour) image using 8-bit gamma encoding.
			RGB_Gamma8,
			/// 3-channel (colour) image using 16-bit gamma encoding.
			RGB_Gamma16,
			/// 4-channel (colour and alpha) image using 8-bit gamma colour encoding and 8-bit linear alpha encoding.
			RGBA_Gamma8,
			/// 4-channel (colour and alpha) image using 16-bit gamma colour encoding and 16-bit linear alpha encoding.
			RGBA_Gamma16,
			/// Single-channel (grayscale) image using 8-bit gamma encoding.
			Gray_Gamma8,
			/// Single-channel (grayscale) image using 16-bit gamma encoding.
			Gray_Gamma16,
			/// Dual-channel (grayscale and alpha) image using 8-bit gamma greyscale encoding and 8-bit linear alpha encoding.
			GrayA_Gamma8,
			/// Dual-channel (grayscale and alpha) image using 16-bit gamma greyscale encoding and 16-bit linear alpha encoding.
			GrayA_Gamma16
		};

		enum ImageFileType
		{
			GIF,
			POT,
			SYS,
			IFF,
			TGA,
			PGM,
			PPM,
			PNG,
			JPEG,
			TIFF,
			BMP,
			EXR,
			HDR
		};

		struct ReadOptions
		{
			ImageDataType itype;
			GammaCurvePtr defaultGamma; // the gamma curve to use by default for converting to working colour space (NULL or a neutral curve to use raw encoded values)
			bool gammaOverride;         // whether to apply defaultGamma even if the file indicates a different gamma
			bool gammacorrect;          // whether to do any gamma correction at all; if false, raw encoded values are used
			mutable vector<string> warnings;

			ReadOptions() : itype(Undefined), gammaOverride(false), gammacorrect(false) { }
		};

		struct WriteOptions
		{
			unsigned char bpcc; // bits per colour component
			bool alphachannel;
			bool grayscale;
			unsigned char compress;
//			float gamma;
			GammaCurvePtr encodingGamma;    // the gamma curve to use for encoding if the file format leaves any choice (NULL to use file format recommendation)
			unsigned int offset_x;
			unsigned int offset_y;

			WriteOptions() : bpcc(8), alphachannel(false), grayscale(false), compress(0) /*, gamma(1.0f) */, offset_x(0), offset_y(0) { }
		};

		virtual ~Image() { }

		static Image *Create(unsigned int w, unsigned int h, ImageDataType t, bool f = false);
		static Image *Create(unsigned int w, unsigned int h, ImageDataType t, const vector<RGBMapEntry>& m, bool f = false);
		static Image *Create(unsigned int w, unsigned int h, ImageDataType t, const vector<RGBAMapEntry>& m, bool f = false);
		static Image *Create(unsigned int w, unsigned int h, ImageDataType t, const vector<RGBFTMapEntry>& m, bool f = false);

		// ftype = use this image type, if "Undefined" use best match
		static Image *Read(ImageFileType ftype, IStream *file, const ReadOptions& options = ReadOptions());

		// bitperpixel = use this number of bits per pixel or closest supported match, if "0" use best match
		// compress = if "0" use no compression, other values use fomat specific compression (TBD)
		static void Write(ImageFileType ftype, OStream *file, const Image *image, const WriteOptions& options = WriteOptions());

		unsigned int GetWidth() const { return width; }
		unsigned int GetHeight() const { return height; }
		ImageDataType GetImageDataType() const { return type; }

		/// Returns true if image is fully opaque.
		virtual bool IsOpaque() const = 0;

		virtual bool IsGrayscale() const = 0;
		virtual bool IsColour() const = 0;
		virtual bool IsFloat() const = 0;
		virtual bool IsInt() const = 0;

		/// Returns true if backed by a palette-based container.
        virtual bool IsIndexed() const = 0;

		/// Returns true if backed by a gamma-encoded data container.
		virtual bool IsGammaEncoded() const = 0;

        /// Returns true if container features a genuine alpha channel.
		virtual bool HasAlphaChannel() const = 0;

		/// Returns true if container features genuine filter & transmit channels.
        virtual bool HasFilterTransmit() const = 0;

		/// Returns true if container features any way of storing transparency information.
        virtual bool HasTransparency() const { return (HasAlphaChannel() || HasFilterTransmit()); }

        /// Returns the maximum value supported by int access methods or for palette indices (1, 255 or 65535).
		virtual unsigned int GetMaxIntValue() const = 0;

        /// Specifies whether container holds color data premultiplied with alpha
        void SetPremultiplied(bool b) { premultiplied = b; } // TODO - mechanism not fully functional yet for image reading

        /// Returns true if container holds data premultiplied with alpha
        bool IsPremultiplied() const { return premultiplied; }

		/**
         *  Requests the image container to perform deferred decoding of integer values.
		 *  In order for the request to be honored, the requested value range must match the container's native
		 *  bit depth, and the container must have a neutral encoding gamma curve set at present.
		 *  @note           If the request is honored, this will also affect subsequent unsigned int read accesses.
		 *  @param[in,out]  gamma   Gamma encoding curve of the encoded material. Set to empty by the function
		 *                          if the request is accepted.
		 *  @param[in]      max     Maximum encoded value. In order for the request to be honored, this must match
		 *                          the image container's native bit depth.
		 *  @return                 true it the request is accepted, false otherwise.
         */
		virtual bool TryDeferDecoding(GammaCurvePtr& gamma, unsigned int max) = 0;

		void GetRGBIndexedValue(unsigned char index, float& red, float& green, float& blue) const;
		void GetRGBAIndexedValue(unsigned char index, float& red, float& green, float& blue, float& alpha) const;
		void GetRGBFTIndexedValue(unsigned char index, float& red, float& green, float& blue, float& filter, float& transm) const;

		void SetRGBIndexedValue(unsigned char index, float red, float green, float blue);
		void SetRGBAIndexedValue(unsigned char index, float red, float green, float blue, float alpha);
		void SetRGBFTIndexedValue(unsigned char index, float red, float green, float blue, float filter, float transm);
		void SetRGBFTIndexedValue(unsigned char index, const Colour& colour);

		virtual bool GetBitValue(unsigned int x, unsigned int y) const = 0;
		virtual float GetGrayValue(unsigned int x, unsigned int y) const = 0;
		virtual void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const = 0;
		virtual void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const = 0;
		virtual void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const = 0;
		virtual void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const = 0;
		virtual unsigned char GetIndexedValue(unsigned int x, unsigned int y);

		virtual void SetBitValue(unsigned int x, unsigned int y, bool bit) = 0;
		virtual void SetGrayValue(unsigned int x, unsigned int y, float gray) = 0;
		virtual void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray) = 0;
		virtual void SetGrayAValue(unsigned int x, unsigned int y, float gray, float alpha) = 0;
		virtual void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int alpha) = 0;
		virtual void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue) = 0;
		virtual void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue) = 0;
		virtual void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float alpha) = 0;
		virtual void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha) = 0;
		virtual void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float filter, float transm) = 0;
		virtual void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col) = 0;
		virtual void SetIndexedValue(unsigned int x, unsigned int y, unsigned char index);

		// convenience functions for image file decoding
		void SetEncodedGrayValue(unsigned int x, unsigned int y, const GammaCurvePtr&, unsigned int max, unsigned int gray);
		void SetEncodedGrayAValue(unsigned int x, unsigned int y, const GammaCurvePtr&, unsigned int max, unsigned int gray, unsigned int alpha, bool premul = false);
		void SetEncodedRGBValue(unsigned int x, unsigned int y, const GammaCurvePtr&, unsigned int max, unsigned int red, unsigned int green, unsigned int blue);
		void SetEncodedRGBAValue(unsigned int x, unsigned int y, const GammaCurvePtr&, unsigned int max, unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha, bool premul = false);
		void SetEncodedGrayValue(unsigned int x, unsigned int y, const GammaCurvePtr&, float gray);
		void SetEncodedGrayAValue(unsigned int x, unsigned int y, const GammaCurvePtr&, float gray, float alpha, bool premul = false);
        void SetEncodedRGBValue(unsigned int x, unsigned int y, const GammaCurvePtr&, float red, float green, float blue);
        void SetEncodedRGBAValue(unsigned int x, unsigned int y, const GammaCurvePtr&, float red, float green, float blue, float alpha, bool premul = false);

		// convenience functions for image file encoding
		unsigned int GetEncodedGrayValue(unsigned int x, unsigned int y, const GammaCurvePtr&, unsigned int max) const;
		void GetEncodedGrayAValue(unsigned int x, unsigned int y, const GammaCurvePtr&, unsigned int max, unsigned int& gray, unsigned int& alpha, bool premul = false) const;
		void GetEncodedRGBValue(unsigned int x, unsigned int y, const GammaCurvePtr&, unsigned int max, unsigned int& red, unsigned int& green, unsigned int& blue) const;
		void GetEncodedRGBAValue(unsigned int x, unsigned int y, const GammaCurvePtr&, unsigned int max, unsigned int& red, unsigned int& green, unsigned int& blue, unsigned int& alpha, bool premul = false) const;
		float GetEncodedGrayValue(unsigned int x, unsigned int y, const GammaCurvePtr&) const;
		void GetEncodedGrayAValue(unsigned int x, unsigned int y, const GammaCurvePtr&, float& gray, float& alpha, bool premul = false) const;
        void GetEncodedRGBValue(unsigned int x, unsigned int y, const GammaCurvePtr&, float& red, float& green, float& blue) const;
        void GetEncodedRGBAValue(unsigned int x, unsigned int y, const GammaCurvePtr&, float& red, float& green, float& blue, float& alpha, bool premul = false) const;

		// convenience functions for image evaluation
		void GetRGBValue(unsigned int x, unsigned int y, RGBColour& colour, bool premul = false) const;
		void GetRGBFTValue(unsigned int x, unsigned int y, Colour& colour, bool premul = false) const;

		virtual void FillBitValue(bool bit) = 0;
		virtual void FillGrayValue(float gray) = 0;
		virtual void FillGrayValue(unsigned int gray) = 0;
		virtual void FillGrayAValue(float gray, float alpha) = 0;
		virtual void FillGrayAValue(unsigned int gray, unsigned int alpha) = 0;
		virtual void FillRGBValue(float red, float green, float blue) = 0;
		virtual void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue) = 0;
		virtual void FillRGBAValue(float red, float green, float blue, float alpha) = 0;
		virtual void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha) = 0;
		virtual void FillRGBFTValue(float red, float green, float blue, float filter, float transm) = 0;
		virtual void FillIndexedValue(unsigned char index);

		unsigned int GetColourMapSize() const;

		void GetColourMap(vector<RGBMapEntry>& m) const;
		void GetColourMap(vector<RGBAMapEntry>& m) const;
		void GetColourMap(vector<RGBFTMapEntry>& m) const;

		void SetColourMap(const vector<RGBMapEntry>& m);
		void SetColourMap(const vector<RGBAMapEntry>& m);
		void SetColourMap(const vector<RGBFTMapEntry>& m);
/*
		void CopyTo(unsigned int x, unsigned int y, const Image& srcimage)
		{
			// TODO
		}
		void CopyToScaled(unsigned int x, unsigned int y, unsigned int w, unsigned int h, const Image& srcimage, bool smooth = false)
		{
			// TODO
		}*/
	protected:
		struct MapEntry
		{
			float red;
			float green;
			float blue;
			float filter; // alpha = filter
			float transm;

			MapEntry() : red(0.0f), green(0.0f), blue(0.0f), filter(0.0f), transm(0.0f) { }
			MapEntry(float r, float g, float b, float f, float t) : red(r), green(g), blue(b), filter(f), transm(t) { }
			MapEntry(const RGBMapEntry& e) : red(e.red), green(e.green), blue(e.blue), filter(0.0f), transm(0.0f) { }
			MapEntry(const RGBAMapEntry& e) : red(e.red), green(e.green), blue(e.blue), filter(e.alpha), transm(0.0f) { }
			MapEntry(const RGBFTMapEntry& e) : red(e.red), green(e.green), blue(e.blue), filter(e.filter), transm(e.transm) { }
		};

		vector<MapEntry> colormap;
		ColourMapType colormaptype;
		unsigned int width;
		unsigned int height;
		ImageDataType type;
        bool premultiplied;

		Image(unsigned int w, unsigned int h, ImageDataType t) :
			width(w), height(h), type(t), colormaptype(NoColourMap), premultiplied(false) { }

		Image(unsigned int w, unsigned int h, ImageDataType t, const vector<RGBMapEntry>& m) :
			width(w), height(h), type(t), colormaptype(RGBColourMap), premultiplied(false) { colormap.resize(max(m.size(), sizeof(unsigned char) * 256)); colormap.assign(m.begin(), m.end()); }

		Image(unsigned int w, unsigned int h, ImageDataType t, const vector<RGBAMapEntry>& m) :
			width(w), height(h), type(t), colormaptype(RGBAColourMap), premultiplied(false) { colormap.resize(max(m.size(), sizeof(unsigned char) * 256)); colormap.assign(m.begin(), m.end()); }

		Image(unsigned int w, unsigned int h, ImageDataType t, const vector<RGBFTMapEntry>& m) :
			width(w), height(h), type(t), colormaptype(RGBFTColourMap), premultiplied(false) { colormap.resize(max(m.size(), sizeof(unsigned char) * 256)); colormap.assign(m.begin(), m.end()); }

		float RGB2Gray(float red, float green, float blue) const
		{
			return (red * 0.297f + green * 0.589f + blue * 0.114f);
		}
	private:
		/// not available
		Image();
		/// not available
		Image(const Image&);
		/// not available
		Image& operator=(Image&);
};

}

#endif // POVRAY_BASE_IMAGE_H
