/*******************************************************************************
 * thread.h
 *
 * This module contains all defines, typedefs, and prototypes for the
 * C++ interface of thread.cpp.
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/base/textstream.h $
 * $Revision: #1 $
 * $Change: 5014 $
 * $DateTime: 2010/06/13 03:51:51 $
 * $Author: thorsten $
 *******************************************************************************/

#ifndef POVRAY_BASE_THREAD_H
#define POVRAY_BASE_THREAD_H

#include "base/configbase.h"
#include "base/types.h"

#include <set>

#include <boost/function.hpp>

// define either one of these in platform config file
// #define POV_USE_WINDOWS_THREADS 1
// #define POV_USE_POSIX_THREADS 1

#if(POV_USE_POSIX_THREADS == 1)
	#undef POV_USE_WINDOWS_THREADS
	#define POV_USE_WINDOWS_THREADS 0
#elif(POV_USE_WINDOWS_THREADS == 1)
	#undef POV_USE_POSIX_THREADS
	#define POV_USE_POSIX_THREADS 0
#else
	#error Threading model for POV-Ray is not defined!
#endif

// include the magic system headers needed to support threads
#if(POV_USE_POSIX_THREADS == 1)
	#include <pthread.h>
#elif(POV_USE_WINDOWS_THREADS == 1)
	#include <WinDef.h> // Use this to avoid dragging all of Windows.h into POV-Ray code and avoid the mess that follows! [trf]
#endif

// sufficient thread stack size for all management threads
#define POV_MINIMUM_THREAD_STACK_SIZE (sizeof(void*) * 1024 * 16) // 64/128 KB for 32/64 bit pointer architectures respectively
// sufficient thread stack size for all ray-tracing, parsing and other processing related threads
#define POV_DEFAULT_THREAD_STACK_SIZE (sizeof(void*) * 1024 * 1024) // 4/8 MB for 32/64 bit pointer architectures respectively

namespace pov_base
{

#if(POV_USE_POSIX_THREADS == 1)
	#define POV_PLATFORM_THREAD_RUN_FUNCTION_DECL void *
	#define POV_PLATFORM_THREAD_RUN_FUNCTION_ARGS void *threadptr

	struct PlatformThreadHandle { pthread_t thread; pthread_attr_t attr; };
	typedef pthread_key_t PlatformThreadLocalStorageHandle;
	typedef pthread_mutex_t PlatformMutexHandle;
	struct PlatformRecursiveMutexHandle { pthread_mutex_t mutex; pthread_mutexattr_t attr; };
#elif(POV_USE_WINDOWS_THREADS == 1)
	#define POV_PLATFORM_THREAD_RUN_FUNCTION_DECL DWORD WINAPI
	#define POV_PLATFORM_THREAD_RUN_FUNCTION_ARGS void *threadptr

	typedef HANDLE PlatformThreadHandle;
	typedef DWORD PlatformThreadLocalStorageHandle;
	struct PlatformMutexHandle { LPCRITICAL_SECTION handle; volatile bool entered; };
	typedef PlatformRecursiveMutexHandle LPCRITICAL_SECTION handle;
#endif

class Mutex;

class Mutex
{
	public:
		class ScopedLock
		{
			public:
				ScopedLock(Mutex& m) :
					mutex(m),
					locked(true)
				{
					mutex.Lock();
				}

				ScopedLock(Mutex& m, bool lock) :
					mutex(m),
					locked(lock)
				{
					if(lock == true)
						mutex.Lock();
				}
		
				~ScopedLock()
				{
					locked = false;
					mutex.Unlock();
				}
		
				inline void Lock()
				{
					mutex.Lock();
					locked = true;
				}
			
				inline bool TryLock()
				{
					locked = mutex.TryLock();
					return locked;
				}
		
				inline void Unlock()
				{
					locked = false;
					mutex.Unlock();
				}

				inline bool IsLocked() const
				{
					return locked;
				}
			private:
				Mutex& mutex;
				bool locked;
		};

		Mutex();
		~Mutex();

		void Lock();
		bool TryLock();
		void Unlock();
	private:
		PlatformMutexHandle platformMutexHandle;
};

class RecursiveMutex;

class RecursiveMutex
{
	public:
		class ScopedLock
		{
			public:
				ScopedLock(RecursiveMutex& m) :
					mutex(m),
					locked(true)
				{
					mutex.Lock();
				}

				ScopedLock(RecursiveMutex& m, bool lock) :
					mutex(m),
					locked(lock)
				{
					if(lock == true)
						mutex.Lock();
				}
			
				~ScopedLock()
				{
					locked = false;
					mutex.Unlock();
				}
		
				inline void Lock()
				{
					mutex.Lock();
					locked = true;
				}
			
				inline bool TryLock()
				{
					locked = mutex.TryLock();
					return locked;
				}
		
				inline void Unlock()
				{
					locked = false;
					mutex.Unlock();
				}

				inline bool IsLocked() const
				{
					return locked;
				}
			private:
				RecursiveMutex& mutex;
				bool locked;
		};

		RecursiveMutex();
		~RecursiveMutex();

		void Lock();
		bool TryLock();
		void Unlock();
	private:
		PlatformRecursiveMutexHandle platformMutexHandle;
};

class Thread
{
	public:
		typedef void *ThreadHandle;
		typedef std::vector<ThreadHandle> ThreadList;

		Thread(boost::function0<void> fn, size_t stack = POV_DEFAULT_THREAD_STACK_SIZE);
		~Thread();

		void Join();

		bool operator==(const Thread& other) const;
		bool operator!=(const Thread& other) const;

		bool operator<(const Thread& other) const;

		void *GetThreadDataPtr() const;
		void SetThreadDataPtr(void *ptr);

		static void Yield();
		static void Sleep(unsigned int msec);

		 // WARNING: Do not call the static methods below before entering main!
		static Thread& GetCurrentThread();
		static ThreadList GetAllThreads();
		static Thread *AdoptThread();
		static void ModifyThread(ThreadHandle th, boost::function1<void, PlatformThreadHandle *> fn);
	private:
		class TLSHelper
		{
			public:
				TLSHelper();
				~TLSHelper();

				PlatformThreadLocalStorageHandle handle;
		};

		typedef std::set<Thread *> Threads;

		static Mutex threadMutex;
		static Threads threads;
		static TLSHelper platformThreadLocalStorageHandle;

		PlatformThreadHandle platformThreadHandle;
		boost::function0<void> threadFunction;
		void *threadDataPtr;

		volatile bool joinable;
		
		Thread(); // unavailable
		Thread(const Thread&); // unavailable
		Thread& operator=(const Thread&); // unavailable

		Thread(PlatformThreadHandle& pth);

		void Run();

		static POV_PLATFORM_THREAD_RUN_FUNCTION_DECL PlatformRunFunction(POV_PLATFORM_THREAD_RUN_FUNCTION_ARGS);
};

}

#endif
