/****************************************************************************
 *               compareimage.cpp
 *
 * This program implements PPM image comparison.
 *
 * from Persistence of Vision(tm) Ray Tracer version 3.6.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2004 Persistence of Vision Raytracer Pty. Ltd.
 *---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 *---------------------------------------------------------------------------
 * This program is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 *****************************************************************************/

namespace std { }

#include <exception>
#include <bitset>
#include <cstring>
#include <cstdio>
#include <cstdlib>
#include <cstdarg>
#include <climits>
#include <cctype>
#include <cmath>

#include "fileinputoutput.h"

USING_POV_BASE_NAMESPACE

using namespace std;

#define POV_MALLOC(size,msg) pov_malloc ((size), __FILE__, __LINE__, (msg))

const int IS16BITIMAGE = 0x00020000;

typedef struct Image8_Line_Struct IMAGE8_LINE;
typedef struct Image16_Line_Struct IMAGE16_LINE;

struct Image8_Line_Struct
{
  unsigned char *red, *green, *blue, *transm;
};

struct Image16_Line_Struct
{
  unsigned short *red, *green, *blue, *transm;
};

typedef struct Image_Struct IMAGE;

struct Image_Struct
{
  int References; /* Keeps track of number of pointers to this structure */
  int Map_Type;
  int File_Type;
  int Image_Type; /* What this image is being used for */
  int iwidth, iheight;
  union
  {
    IMAGE8_LINE *rgb8_lines;
    IMAGE16_LINE *rgb16_lines;
    unsigned short **gray16_lines;
    unsigned char **map_lines;
  } data;
};

double Compute_MSE(IMAGE *img1, IMAGE *img2);
double Compute_PSNR(double mse);
void Read_PPM_Image(IMAGE *Image, char *name);
int Read_ASCII_File_Number(IStream *filep);
void Get_Pixel(IMAGE *Image, unsigned int ixcoor, unsigned int iycoor, unsigned int& red, unsigned int& green, unsigned int& blue);
int Error(const char *format,...);
void *pov_malloc(size_t size, const char *file, int line, const char *msg);

int main(int argc, char **argv)
{
	try
	{
		if(argc != 5)
		{
			puts("Usage:   compareimages <masterimage> <testimage> <mse> <psnr>");
			puts("Returns:  0 in case of a match (the result is valid)\n"
			     "         -1 in case of no match (the result is invalid)\n"
			     "         -2 in case of an error\n");
			return -2;
		}

		IMAGE masterImage;
		IMAGE testImage;

		int expect_mse = (int)atoi(argv[3]);
		int expect_psnr = (int)atoi(argv[4]);

		fprintf(stdout, "Expecting MSE below %d\n", expect_mse);
		fprintf(stdout, "Expecting PSNR over %d\n", expect_psnr);

		fprintf(stdout, "Reading image '%s'\n", argv[1]);
		Read_PPM_Image(&masterImage, argv[1]);

		fprintf(stdout, "Reading image '%s'\n", argv[2]);
		Read_PPM_Image(&testImage, argv[2]);

		fprintf(stdout, "Computing MSE...\n", argv[2]);
		double mse = Compute_MSE(&masterImage, &testImage);

		if(mse == HUGE_VAL)
		{
			fprintf(stdout, "The MSE is very huge and there is no point to continue!\n");
			fprintf(stdout, "Result: The images do not match!\n");
			return -1;
		}
		else if(mse < 10e-5)
		{
			fprintf(stdout, "The MSE is zero and thus the images are identical!\n");
			fprintf(stdout, "Result: The the images match!\n");
			return 0;
		}

		fprintf(stdout, "Computing PSNR...\n", argv[2]);
		double psnr = Compute_PSNR(mse);

		fprintf(stdout, "MSE is %13.5f\n", mse);
		fprintf(stdout, "PSNR is %12.5f\n", psnr);

		if((psnr > double(expect_psnr)) && (mse < double(expect_mse)))
		{
			fprintf(stdout, "Result: The the images match!\n");
			return 0;
		}
		else
		{
			fprintf(stdout, "Result: The images do not match!\n");
			return -1;
		}
	}
	catch(exception& e)
	{
		Error("Caught C++ exception '%s'.", e.what());
	}
	catch(...)
	{
		Error("Caught unknown C++ exception!");
	}

	// unreachable
	return -2;
}

double Compute_MSE(IMAGE *img1, IMAGE *img2)
{
	if((img1->iwidth != img2->iwidth) || (img1->iheight != img2->iheight))
		Error("Image size does not match.");	

	if(img1->Image_Type != img2->Image_Type)
		Error("Image color depth does not match.");	

	unsigned int red1, green1, blue1, red2, green2, blue2;
	double mse_red = 0.0, mse_green = 0.0, mse_blue = 0.0;

	// MSE = sum(x = 0, x < WIDTH, sum(y = 0, y < HEIGHT, pow((IMAGE1(x, y) - IMAGE2(x, y)), 2))) / (WIDTH * HEIGHT)

	for(unsigned int y = 0; y < img1->iheight; y++)
	{
		for(unsigned int x = 0; x < img1->iwidth; x++)
		{
			// get pixels with 16 bit precision
			Get_Pixel(img1, x, y, red1, green1, blue1);
			Get_Pixel(img2, x, y, red2, green2, blue2);

			red1 = red1 - red2;
			green1 = green1 - green2;
			blue1 = blue1 - blue2;

			mse_red += double(red1 * red1);
			mse_green += double(green1 * green1);
			mse_blue += double(blue1 * blue1);

			// limit the maximum error
			if((mse_red > HUGE_VAL) || (mse_green > HUGE_VAL) || (mse_blue > HUGE_VAL))
				return HUGE_VAL;
		}
	}

	mse_red = mse_red / double(img1->iheight * img1->iwidth);
	mse_green = mse_green / double(img1->iheight * img1->iwidth);
	mse_blue = mse_blue / double(img1->iheight * img1->iwidth);

	// limit the maximum error
	if((mse_red > HUGE_VAL) || (mse_green > HUGE_VAL) || (mse_blue > HUGE_VAL))
		return HUGE_VAL;

	// weight red to 0.297, green to 0.589 and blue to 0.114
	return ((mse_red * 0.297) + (mse_green * 0.589) + (mse_blue * 0.114));
}

double Compute_PSNR(double mse)
{
	// 65535 = 2**PRECISIONBITS - 1
	// PSNR =20 * log10(65535 / sqrt(MSE))

	return 20.0 * log10(65535.0 / sqrt(mse));
}

void Read_PPM_Image(IMAGE *Image, char *name)
{
  IStream *filep;
  unsigned char header[2];
  char line[1024];
  char *ptr;
  int nbr;

  int width, height;
  unsigned int depth;

  IMAGE8_LINE *line_data;
  IMAGE16_LINE *line_16_data;
  int data_hi, data_lo;
  int x, i;

  // --- Start by trying to open the file --- 
  filep = new IStream(0);
  if(filep->open(name) == 0)
    Error ("Cannot open PPM image %s.", name);

  // --- Read Header --- 
  if (!filep->read((char *)header, 2))
    Error ("Cannot read header of PPM image %s.", name);

  if(header[0] != 'P') Error ("File is not in PPM format.");

  if((header[1] != '3') && (header[1] != '6'))
    Error ("File is not in PPM format (type %d).", header[1]);

  do
  {
    filep->getline (line, 1024);
    line[1023] = '\0';
    if ((ptr = strchr(line, '#')) != NULL) *ptr = '\0';  // remove comment 
  }
  while (line[0]=='\0');  // read until line without comment from beginning 

  // --- First: two numbers: with and height --- 
  if (sscanf(line,"%d %d",&width, &height) != 2)
    Error ("Cannot read width and height from PPM image.");

  if (width <= 0 || height <= 0)
    Error ("Invalid width or height read from PPM image.");

  do
  {
    filep->getline (line, 1024) ;
    line[1023] = '\0';
    if ((ptr = strchr(line, '#')) != NULL) *ptr = '\0';  // remove comment 
  }
  while (line[0]=='\0');  // read until line without comment from beginning 

  // --- Second: one number: color depth --- 
  if (sscanf(line,"%d",&depth) != 1)
    Error ("Cannot read color depth from PPM image.");

  if ((depth > 65535) || (depth < 1))
    Error ("Unsupported number of colors (%d) in PPM image.", depth);

  Image->Image_Type = 0;
  Image->iwidth = width;
  Image->iheight = height;

  if (depth < 256)
  {
    Image->data.rgb8_lines = (IMAGE8_LINE *)POV_MALLOC(height * sizeof(IMAGE8_LINE), "PPM image");

    for (i = 0; i < height; i++)
    {
      line_data = &Image->data.rgb8_lines[i];

      line_data->red    = (unsigned char *)POV_MALLOC(width, "PPM image line");
      line_data->green  = (unsigned char *)POV_MALLOC(width, "PPM image line");
      line_data->blue   = (unsigned char *)POV_MALLOC(width, "PPM image line");
      line_data->transm = (unsigned char *)NULL;

      if (header[1] == '3') // --- ASCII PPM file (type 3) --- 
      {
        for (x = 0; x < width; x++)
        {
          nbr = Read_ASCII_File_Number(filep);
          if (nbr >= 0) line_data->red[x] = (nbr*255)/depth;
          else Error ("Cannot read image data from PPM image.");
          nbr = Read_ASCII_File_Number(filep);
          if (nbr >= 0) line_data->green[x] = (nbr*255)/depth;
          else Error ("Cannot read image data from PPM image.");
          nbr = Read_ASCII_File_Number(filep);
          if (nbr >= 0) line_data->blue[x] = (nbr*255)/depth;
          else Error ("Cannot read image data from PPM image.");
        }
      }
      else                  // --- binary PPM file (type 6) --- 
      {
        for (x = 0; x < width; x++)
        {
          if ((nbr = filep->Read_Byte ()) == EOF)
            Error("Cannot read data from PPM image.");

          line_data->red[x] = (nbr*255)/depth;

          if ((nbr = filep->Read_Byte ()) == EOF)
            Error("Cannot read data from PPM image.");

          line_data->green[x] = (nbr*255)/depth;

          if ((nbr = filep->Read_Byte ()) == EOF)
            Error("Cannot read data from PPM image.");

          line_data->blue[x] = (nbr*255)/depth;
        }
      }
    }
  }
  else // --- 16 bit PPM (binary or ASCII) --- 
  {
    Image->Image_Type |= IS16BITIMAGE;

    Image->data.rgb16_lines = (IMAGE16_LINE *)POV_MALLOC(height * sizeof(IMAGE16_LINE), "PPM image");

    for (i = 0; i < height; i++)
    {
      line_16_data = &Image->data.rgb16_lines[i];

      line_16_data->red    = (unsigned short *)POV_MALLOC(width * sizeof(unsigned short), "PPM image line");
      line_16_data->green  = (unsigned short *)POV_MALLOC(width * sizeof(unsigned short), "PPM image line");
      line_16_data->blue   = (unsigned short *)POV_MALLOC(width * sizeof(unsigned short), "PPM image line");
      line_16_data->transm = (unsigned short *)NULL;

      if (header[1] == '3') // --- ASCII PPM file (type 3) --- 
      {
        for (x = 0; x < width; x++)
        {
          nbr = Read_ASCII_File_Number(filep);
          if (nbr >= 0) line_16_data->red[x] = (nbr*65535)/depth;
          else Error ("Cannot read image data from PPM image.");

          nbr = Read_ASCII_File_Number(filep);
          if (nbr >= 0) line_16_data->green[x] = (nbr*65535)/depth;
          else Error ("Cannot read image data from PPM image.");

          nbr = Read_ASCII_File_Number(filep);
          if (nbr >= 0) line_16_data->blue[x] = (nbr*65535)/depth;
          else Error ("Cannot read image data from PPM image.");
        }
      }
      else                  // --- binary PPM file (type 6) --- 
      {
        for (x = 0; x < width; x++)
        {
          if ((data_hi = filep->Read_Byte ()) == EOF)
            Error ("Cannot read data from PPM image.");
          if ((data_lo = filep->Read_Byte ()) == EOF)
            Error ("Cannot read data from PPM image.");
          line_16_data->red[x] = (256*data_hi + data_lo)*65535/depth;

          if ((data_hi = filep->Read_Byte ()) == EOF)
            Error ("Cannot read data from PPM image.");
          if ((data_lo = filep->Read_Byte ()) == EOF)
            Error ("Cannot read data from PPM image.");
          line_16_data->green[x] = (256*data_hi + data_lo)*65535/depth;

          if ((data_hi = filep->Read_Byte ()) == EOF)
            Error ("Cannot read data from PPM image.");
          if ((data_lo = filep->Read_Byte ()) == EOF)
            Error ("Cannot read data from PPM image.");
          line_16_data->blue[x] = (256*data_hi + data_lo)*65535/depth;
        }
      }
    }
  }

  // Close the image file 

  delete filep;
}

int Read_ASCII_File_Number(IStream *filep)
{
  int value;
  int pos = 0;
  char buffer[50];

  do
  {
    value = filep->Read_Byte();
  }
  while ( isspace(value) );

  if ( isdigit(value) ) buffer[pos] = (char)value;
  else return -1;

  while ( !isspace(value = filep->Read_Byte()) && (pos<48))
  {
    if ( isdigit(value) )
    {
      pos++;
      buffer[pos] = (char)value;
    }
    else return -1;
  }

  buffer[pos+1] = '\0';

  value = atoi(buffer);

  return value;
}

void Get_Pixel(IMAGE *Image, unsigned int ixcoor, unsigned int iycoor, unsigned int& red, unsigned int& green, unsigned int& blue)
{
	IMAGE16_LINE *line16 = NULL;
	IMAGE8_LINE *line8 = NULL;

	if((Image->Image_Type & IS16BITIMAGE) == IS16BITIMAGE)
	{
		line16 = &Image->data.rgb16_lines[iycoor];
		red = (unsigned int)line16->red[ixcoor];
		green = (unsigned int)line16->green[ixcoor];
		blue = (unsigned int)line16->blue[ixcoor];
	}
	else
	{
		line8 = &Image->data.rgb8_lines[iycoor];
		red = ((unsigned int)line8->red[ixcoor]) * 256;
		green = ((unsigned int)line8->green[ixcoor]) * 256;
		blue = ((unsigned int)line8->blue[ixcoor]) * 256;
	}
}

int Error(const char *format,...)
{
	va_list marker;
	char localvsbuffer[4096];

	sprintf(localvsbuffer, "Error: ");

	va_start(marker, format);
	vsnprintf(localvsbuffer + strlen(localvsbuffer), 4095 - strlen(localvsbuffer), format, marker);
	va_end(marker);

	fprintf(stderr, "%s\n\n", localvsbuffer);

	exit(-2);
}

void *pov_malloc(size_t size, const char *file, int line, const char *msg)
{
	void *block;

	if(size == 0)
		Error("Attempt to malloc zero size block (File: %s Line: %d).", file, line);
	block = malloc(size);

	if(block == NULL)
		Error("Out of memory.  Cannot allocate %ld bytes for %s (File: %s Line: %d).", (long)size, msg, file, line);

	return block;
}
