
      module mo_heatnirco2

      use shr_kind_mod, only : r8 => shr_kind_r8
      use cam_logfile,  only : iulog

      implicit none

      private
      public :: heatnirco2
      public :: heatnirco2_init

      integer,parameter :: ndpara = 62
      integer,parameter :: ncolgr = 10

      real(r8)  :: xspara(ndpara)
      real(r8)  :: zppara(ndpara) = (/ &
        -17.00_r8,  -16.75_r8,  -16.50_r8,  -16.25_r8,  -16.00_r8, &
        -15.75_r8,  -15.50_r8,  -15.25_r8,  -15.00_r8,  -14.75_r8, &
        -14.50_r8,  -14.25_r8,  -14.00_r8,  -13.75_r8,  -13.50_r8, &
        -13.25_r8,  -13.00_r8,  -12.75_r8,  -12.50_r8,  -12.25_r8, &
        -12.00_r8,  -11.75_r8,  -11.50_r8,  -11.25_r8,  -11.00_r8, &
        -10.75_r8,  -10.50_r8,  -10.25_r8,  -10.00_r8,   -9.75_r8, &
         -9.50_r8,   -9.25_r8,   -9.00_r8,   -8.75_r8,   -8.50_r8, &
         -8.25_r8,   -8.00_r8,   -7.75_r8,   -7.50_r8,   -7.25_r8, &
         -7.00_r8,   -6.75_r8,   -6.50_r8,   -6.25_r8,   -6.00_r8, &
         -5.75_r8,   -5.50_r8,   -5.25_r8,   -5.00_r8,   -4.75_r8, &
         -4.50_r8,   -4.25_r8,   -4.00_r8,   -3.75_r8,   -3.50_r8, &
         -3.25_r8,   -3.00_r8,   -2.75_r8,   -2.50_r8,   -2.25_r8, &
         -2.00_r8,   -1.75_r8 /)
      real(r8)  :: co2stand(ndpara)
      real(r8)  :: colmpara(ndpara,ncolgr)
      real(r8)  :: corrnormpara(ndpara,ncolgr)

      contains

      subroutine heatnirco2( co2, scco2, pmid, htng, kbot, zarg, sza )

      use abortutils, only : endrun

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer,intent(in)     :: kbot                    ! vertical dimension
      real(r8), intent(in)   :: sza
      real(r8), intent(in)   :: zarg(kbot)              ! midpoint geopot (m)
      real(r8), intent(in)   :: co2(kbot)               ! co2 concentration (mol/mol)
      real(r8), intent(in)   :: scco2(kbot)             ! co2 slant column (molec/cm^2)
      real(r8), intent(in)   :: pmid(kbot)              ! model midpoint pressure (Pa)
      real(r8), intent(out)  :: htng(kbot)              ! co2 near ir heating (K/day)

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      real(r8), parameter :: smallvalue = 1.0e-20_r8
      real(r8), parameter :: pa2hPa     = 1.e-2_r8

      integer  :: icolm, icolmp1
      integer  :: k, kk, kndx
      real(r8) :: reldcolm
      real(r8) :: delp
      real(r8) :: pinterp
      real(r8) :: colzpint
      real(r8) :: co2std
      real(r8) :: colparai(ncolgr)
      real(r8) :: corrnorai(ncolgr)

      htng(:kbot) = smallvalue
#ifdef NIR_DEBUG
      write(iulog,*) ' '
      write(iulog,*) '==============================================='
      write(iulog,*) 'heatnirco2: kbot,sza = ',kbot,sza
      write(iulog,*) 'heatnirco2: co2'
      write(iulog,'(1p,5g15.7)') co2(:)
      write(iulog,*) 'heatnirco2: scco2'
      write(iulog,'(1p,5g15.7)') scco2(:)
      write(iulog,*) 'heatnirco2: pmid'
      write(iulog,'(1p,5g15.7)') pmid(:)
      write(iulog,*) 'heatnirco2: zarg'
      write(iulog,'(1p,5g15.7)') zarg(:)
      write(iulog,*) 'heatnirco2: xspara'
      write(iulog,'(1p,5g15.7)') xspara(:)
#endif
!-----------------------------------------------------------------------
! 	... vertical and column interpolation
!-----------------------------------------------------------------------
level_loop : &
      do k = 1,kbot
!-----------------------------------------------------------------------
! 	... first setup pressure interpolation
!-----------------------------------------------------------------------
         pinterp  = pmid(k) * pa2hPa
         colzpint = log( scco2(k) )
         if( pinterp <= xspara(ndpara) ) then
            colparai(:)  = colmpara(ndpara,:)
            corrnorai(:) = corrnormpara(ndpara,:)
            co2std       = co2stand(ndpara)
         else if( pinterp > xspara(1) ) then
            colparai(:)  = colmpara(1,:)
            corrnorai(:) = corrnormpara(1,:)
            co2std       = co2stand(1)
         else
            do kk = ndpara-1,1,-1
               if( pinterp <= xspara(kk) ) then
                  kndx = kk + 1
                  delp         = (pinterp - xspara(kndx))/(xspara(kk) - xspara(kndx))
                  colparai(:)  = colmpara(kndx,:) + delp*(colmpara(kk,:) - colmpara(kndx,:))
                  corrnorai(:) = corrnormpara(kndx,:) &
                                 + delp*(corrnormpara(kk,:) - corrnormpara(kndx,:))
                  co2std       = co2stand(kndx) + delp*(co2stand(kk) - co2stand(kndx))
                  exit
               end if
            end do
         end if

!-----------------------------------------------------------------------
! Linear interpolation over column density for given altitude point
!-----------------------------------------------------------------------
         if( colzpint < colparai(1) ) then
            htng(k) = corrnorai(1)
         else if( colzpint >= colparai(ncolgr) ) then
            htng(k) = corrnorai(ncolgr)
         else
loop1:      do icolm = 1,ncolgr-1
               icolmp1 = icolm + 1
               if( colzpint >= colparai(icolm) .and. &
                   colzpint <  colparai(icolmp1) ) then
                  reldcolm = (colzpint - colparai(icolm)) &
                             /(colparai(icolmp1) - colparai(icolm))
                  htng(k) = corrnorai(icolm) &
                          + (corrnorai(icolmp1) - corrnorai(icolm))*reldcolm
                  exit loop1
               end if
            end do loop1
         end if
#ifdef NIR_DEBUG
         write(iulog,*) 'heatnirco2: k, kk, kndx, icolm, icolmp1 = ',k,kk,kndx,icolm,icolmp1
         write(iulog,*) 'heatnirco2: colparai'
         write(iulog,'(1p,5g15.7)') colparai(:)
         write(iulog,*) 'heatnirco2: corrnorai'
         write(iulog,'(1p,5g15.7)') corrnorai(:)
         write(iulog,*) 'heatnirco2: pinterp,colzpint,delp,reldcolm,co2std,htng'
         write(iulog,'(1p,6g15.7)') pinterp,colzpint,delp,reldcolm,co2std,htng(k)
#endif
         
!-----------------------------------------------------------------------
! From normalized value to the one corresponding to the given vmrco2
!-----------------------------------------------------------------------
         htng(k) = htng(k) * co2(k)/co2std
      end do level_loop

      end subroutine heatnirco2

      subroutine heatnirco2_init
!-----------------------------------------------------------------------
! Called once per run from init (init.F) to define module data.
!-----------------------------------------------------------------------

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      integer :: k
      
      do k = 1,ndpara
         xspara(k) = 5.e-7_r8 * exp( -zppara(k) )
      end do

      co2stand = (/ &
        3.6000e-04_r8,  3.6000e-04_r8,  3.6000e-04_r8,  3.6000e-04_r8,  3.6000e-04_r8, &
        3.6000e-04_r8,  3.6000e-04_r8,  3.6000e-04_r8,  3.6000e-04_r8,  3.6000e-04_r8, &
        3.6000e-04_r8,  3.6000e-04_r8,  3.6000e-04_r8,  3.6000e-04_r8,  3.6000e-04_r8, &
        3.6000e-04_r8,  3.6000e-04_r8,  3.6000e-04_r8,  3.6000e-04_r8,  3.6000e-04_r8, &
        3.6000e-04_r8,  3.6000e-04_r8,  3.6000e-04_r8,  3.6000e-04_r8,  3.6000e-04_r8, &
        3.6000e-04_r8,  3.6000e-04_r8,  3.6000e-04_r8,  3.6000e-04_r8,  3.6000e-04_r8, &
        3.6000e-04_r8,  3.6000e-04_r8,  3.5867e-04_r8,  3.5534e-04_r8,  3.5134e-04_r8, &
        3.4401e-04_r8,  3.3235e-04_r8,  3.1668e-04_r8,  2.9902e-04_r8,  2.8102e-04_r8, &
        2.6235e-04_r8,  2.4335e-04_r8,  2.2435e-04_r8,  2.0535e-04_r8,  1.8635e-04_r8, &
        1.6735e-04_r8,  1.4835e-04_r8,  1.3002e-04_r8,  1.1202e-04_r8,  9.5350e-05_r8, &
        8.2013e-05_r8,  7.1344e-05_r8,  6.2010e-05_r8,  5.3675e-05_r8,  4.6341e-05_r8, &
        3.9341e-05_r8,  3.2341e-05_r8,  2.6006e-05_r8,  2.0672e-05_r8,  1.5672e-05_r8, &
        1.1338e-05_r8,  8.0032e-06_r8 /)

      colmpara(1,:) = (/ &
        4.5112e+01_r8,  4.5647e+01_r8,  4.6183e+01_r8,  4.6719e+01_r8,  4.7254e+01_r8, &
        4.7789e+01_r8,  4.8325e+01_r8,  4.8861e+01_r8,  4.9396e+01_r8,  4.9931e+01_r8 /)
      colmpara(2,:) = (/ &
        4.4862e+01_r8,  4.5398e+01_r8,  4.5934e+01_r8,  4.6469e+01_r8,  4.7004e+01_r8, &
        4.7540e+01_r8,  4.8076e+01_r8,  4.8611e+01_r8,  4.9147e+01_r8,  4.9682e+01_r8 /)
      colmpara(3,:) = (/ &
        4.4613e+01_r8,  4.5148e+01_r8,  4.5684e+01_r8,  4.6220e+01_r8,  4.6755e+01_r8, &
        4.7290e+01_r8,  4.7826e+01_r8,  4.8362e+01_r8,  4.8897e+01_r8,  4.9432e+01_r8 /)
      colmpara(4,:) = (/ &
        4.4363e+01_r8,  4.4899e+01_r8,  4.5435e+01_r8,  4.5970e+01_r8,  4.6506e+01_r8, &
        4.7041e+01_r8,  4.7577e+01_r8,  4.8112e+01_r8,  4.8647e+01_r8,  4.9183e+01_r8 /)
      colmpara(5,:) = (/ &
        4.4114e+01_r8,  4.4650e+01_r8,  4.5185e+01_r8,  4.5721e+01_r8,  4.6256e+01_r8, &
        4.6791e+01_r8,  4.7327e+01_r8,  4.7863e+01_r8,  4.8398e+01_r8,  4.8933e+01_r8 /)
      colmpara(6,:) = (/ &
        4.3865e+01_r8,  4.4401e+01_r8,  4.4936e+01_r8,  4.5471e+01_r8,  4.6007e+01_r8, &
        4.6542e+01_r8,  4.7078e+01_r8,  4.7613e+01_r8,  4.8149e+01_r8,  4.8684e+01_r8 /)
      colmpara(7,:) = (/ &
        4.3615e+01_r8,  4.4151e+01_r8,  4.4686e+01_r8,  4.5222e+01_r8,  4.5757e+01_r8, &
        4.6293e+01_r8,  4.6828e+01_r8,  4.7364e+01_r8,  4.7899e+01_r8,  4.8435e+01_r8 /)
      colmpara(8,:) = (/ &
        4.3366e+01_r8,  4.3901e+01_r8,  4.4437e+01_r8,  4.4972e+01_r8,  4.5508e+01_r8, &
        4.6043e+01_r8,  4.6579e+01_r8,  4.7114e+01_r8,  4.7650e+01_r8,  4.8185e+01_r8 /)
      colmpara(9,:) = (/ &
        4.3116e+01_r8,  4.3652e+01_r8,  4.4188e+01_r8,  4.4723e+01_r8,  4.5258e+01_r8, &
        4.5794e+01_r8,  4.6329e+01_r8,  4.6865e+01_r8,  4.7400e+01_r8,  4.7936e+01_r8 /)
      colmpara(10,:) = (/ &
        4.2867e+01_r8,  4.3402e+01_r8,  4.3938e+01_r8,  4.4473e+01_r8,  4.5009e+01_r8, &
        4.5544e+01_r8,  4.6080e+01_r8,  4.6616e+01_r8,  4.7151e+01_r8,  4.7686e+01_r8 /)
      colmpara(11,:) = (/ &
        4.2617e+01_r8,  4.3153e+01_r8,  4.3688e+01_r8,  4.4224e+01_r8,  4.4759e+01_r8, &
        4.5295e+01_r8,  4.5830e+01_r8,  4.6366e+01_r8,  4.6901e+01_r8,  4.7437e+01_r8 /)
      colmpara(12,:) = (/ &
        4.2368e+01_r8,  4.2903e+01_r8,  4.3439e+01_r8,  4.3974e+01_r8,  4.4510e+01_r8, &
        4.5045e+01_r8,  4.5581e+01_r8,  4.6116e+01_r8,  4.6651e+01_r8,  4.7187e+01_r8 /)
      colmpara(13,:) = (/ &
        4.2118e+01_r8,  4.2654e+01_r8,  4.3189e+01_r8,  4.3725e+01_r8,  4.4260e+01_r8, &
        4.4796e+01_r8,  4.5331e+01_r8,  4.5867e+01_r8,  4.6402e+01_r8,  4.6938e+01_r8 /)
      colmpara(14,:) = (/ &
        4.1869e+01_r8,  4.2404e+01_r8,  4.2940e+01_r8,  4.3475e+01_r8,  4.4011e+01_r8, &
        4.4546e+01_r8,  4.5082e+01_r8,  4.5617e+01_r8,  4.6153e+01_r8,  4.6688e+01_r8 /)
      colmpara(15,:) = (/ &
        4.1619e+01_r8,  4.2155e+01_r8,  4.2690e+01_r8,  4.3225e+01_r8,  4.3761e+01_r8, &
        4.4296e+01_r8,  4.4832e+01_r8,  4.5367e+01_r8,  4.5903e+01_r8,  4.6438e+01_r8 /)
      colmpara(16,:) = (/ &
        4.1369e+01_r8,  4.1905e+01_r8,  4.2440e+01_r8,  4.2976e+01_r8,  4.3511e+01_r8, &
        4.4046e+01_r8,  4.4582e+01_r8,  4.5118e+01_r8,  4.5653e+01_r8,  4.6188e+01_r8 /)
      colmpara(17,:) = (/ &
        4.1119e+01_r8,  4.1655e+01_r8,  4.2190e+01_r8,  4.2726e+01_r8,  4.3261e+01_r8, &
        4.3797e+01_r8,  4.4332e+01_r8,  4.4868e+01_r8,  4.5403e+01_r8,  4.5938e+01_r8 /)
      colmpara(18,:) = (/ &
        4.0869e+01_r8,  4.1405e+01_r8,  4.1940e+01_r8,  4.2476e+01_r8,  4.3011e+01_r8, &
        4.3547e+01_r8,  4.4082e+01_r8,  4.4618e+01_r8,  4.5153e+01_r8,  4.5688e+01_r8 /)
      colmpara(19,:) = (/ &
        4.0619e+01_r8,  4.1154e+01_r8,  4.1690e+01_r8,  4.2226e+01_r8,  4.2761e+01_r8, &
        4.3296e+01_r8,  4.3832e+01_r8,  4.4367e+01_r8,  4.4903e+01_r8,  4.5438e+01_r8 /)
      colmpara(20,:) = (/ &
        4.0368e+01_r8,  4.0904e+01_r8,  4.1439e+01_r8,  4.1975e+01_r8,  4.2510e+01_r8, &
        4.3046e+01_r8,  4.3581e+01_r8,  4.4117e+01_r8,  4.4652e+01_r8,  4.5188e+01_r8 /)
      colmpara(21,:) = (/ &
        4.0117e+01_r8,  4.0653e+01_r8,  4.1189e+01_r8,  4.1724e+01_r8,  4.2259e+01_r8, &
        4.2795e+01_r8,  4.3330e+01_r8,  4.3866e+01_r8,  4.4402e+01_r8,  4.4937e+01_r8 /)
      colmpara(22,:) = (/ &
        3.9866e+01_r8,  4.0402e+01_r8,  4.0937e+01_r8,  4.1473e+01_r8,  4.2008e+01_r8, &
        4.2544e+01_r8,  4.3080e+01_r8,  4.3615e+01_r8,  4.4150e+01_r8,  4.4686e+01_r8 /)
      colmpara(23,:) = (/ &
        3.9615e+01_r8,  4.0150e+01_r8,  4.0686e+01_r8,  4.1221e+01_r8,  4.1757e+01_r8, &
        4.2292e+01_r8,  4.2828e+01_r8,  4.3363e+01_r8,  4.3899e+01_r8,  4.4434e+01_r8 /)
      colmpara(24,:) = (/ &
        3.9363e+01_r8,  3.9898e+01_r8,  4.0434e+01_r8,  4.0969e+01_r8,  4.1505e+01_r8, &
        4.2040e+01_r8,  4.2576e+01_r8,  4.3111e+01_r8,  4.3647e+01_r8,  4.4182e+01_r8 /)
      colmpara(25,:) = (/ &
        3.9110e+01_r8,  3.9645e+01_r8,  4.0181e+01_r8,  4.0716e+01_r8,  4.1252e+01_r8, &
        4.1787e+01_r8,  4.2323e+01_r8,  4.2858e+01_r8,  4.3394e+01_r8,  4.3929e+01_r8 /)
      colmpara(26,:) = (/ &
        3.8856e+01_r8,  3.9391e+01_r8,  3.9927e+01_r8,  4.0462e+01_r8,  4.0998e+01_r8, &
        4.1533e+01_r8,  4.2069e+01_r8,  4.2604e+01_r8,  4.3140e+01_r8,  4.3675e+01_r8 /)
      colmpara(27,:) = (/ &
        3.8601e+01_r8,  3.9136e+01_r8,  3.9672e+01_r8,  4.0207e+01_r8,  4.0743e+01_r8, &
        4.1278e+01_r8,  4.1814e+01_r8,  4.2349e+01_r8,  4.2885e+01_r8,  4.3420e+01_r8 /)
      colmpara(28,:) = (/ &
        3.8344e+01_r8,  3.8879e+01_r8,  3.9415e+01_r8,  3.9950e+01_r8,  4.0486e+01_r8, &
        4.1021e+01_r8,  4.1557e+01_r8,  4.2092e+01_r8,  4.2628e+01_r8,  4.3163e+01_r8 /)
      colmpara(29,:) = (/ &
        3.8085e+01_r8,  3.8620e+01_r8,  3.9156e+01_r8,  3.9692e+01_r8,  4.0227e+01_r8, &
        4.0762e+01_r8,  4.1298e+01_r8,  4.1833e+01_r8,  4.2369e+01_r8,  4.2904e+01_r8 /)
      colmpara(30,:) = (/ &
        3.7823e+01_r8,  3.8359e+01_r8,  3.8894e+01_r8,  3.9430e+01_r8,  3.9965e+01_r8, &
        4.0501e+01_r8,  4.1036e+01_r8,  4.1572e+01_r8,  4.2107e+01_r8,  4.2642e+01_r8 /)
      colmpara(31,:) = (/ &
        3.7558e+01_r8,  3.8093e+01_r8,  3.8628e+01_r8,  3.9164e+01_r8,  3.9699e+01_r8, &
        4.0235e+01_r8,  4.0771e+01_r8,  4.1306e+01_r8,  4.1842e+01_r8,  4.2377e+01_r8 /)
      colmpara(32,:) = (/ &
        3.7287e+01_r8,  3.7823e+01_r8,  3.8358e+01_r8,  3.8894e+01_r8,  3.9429e+01_r8, &
        3.9965e+01_r8,  4.0500e+01_r8,  4.1036e+01_r8,  4.1571e+01_r8,  4.2107e+01_r8 /)
      colmpara(33,:) = (/ &
        3.7011e+01_r8,  3.7547e+01_r8,  3.8082e+01_r8,  3.8617e+01_r8,  3.9153e+01_r8, &
        3.9689e+01_r8,  4.0224e+01_r8,  4.0759e+01_r8,  4.1295e+01_r8,  4.1830e+01_r8 /)
      colmpara(34,:) = (/ &
        3.6728e+01_r8,  3.7263e+01_r8,  3.7799e+01_r8,  3.8334e+01_r8,  3.8870e+01_r8, &
        3.9405e+01_r8,  3.9941e+01_r8,  4.0476e+01_r8,  4.1012e+01_r8,  4.1547e+01_r8 /)
      colmpara(35,:) = (/ &
        3.6437e+01_r8,  3.6972e+01_r8,  3.7508e+01_r8,  3.8043e+01_r8,  3.8578e+01_r8, &
        3.9114e+01_r8,  3.9650e+01_r8,  4.0185e+01_r8,  4.0720e+01_r8,  4.1256e+01_r8 /)
      colmpara(36,:) = (/ &
        3.6136e+01_r8,  3.6672e+01_r8,  3.7207e+01_r8,  3.7743e+01_r8,  3.8278e+01_r8, &
        3.8814e+01_r8,  3.9349e+01_r8,  3.9885e+01_r8,  4.0420e+01_r8,  4.0956e+01_r8 /)
      colmpara(37,:) = (/ &
        3.5827e+01_r8,  3.6363e+01_r8,  3.6898e+01_r8,  3.7434e+01_r8,  3.7969e+01_r8, &
        3.8505e+01_r8,  3.9040e+01_r8,  3.9575e+01_r8,  4.0111e+01_r8,  4.0647e+01_r8 /)
      colmpara(38,:) = (/ &
        3.5511e+01_r8,  3.6046e+01_r8,  3.6582e+01_r8,  3.7117e+01_r8,  3.7653e+01_r8, &
        3.8188e+01_r8,  3.8724e+01_r8,  3.9259e+01_r8,  3.9795e+01_r8,  4.0331e+01_r8 /)
      colmpara(39,:) = (/ &
        3.5189e+01_r8,  3.5724e+01_r8,  3.6259e+01_r8,  3.6795e+01_r8,  3.7331e+01_r8, &
        3.7866e+01_r8,  3.8402e+01_r8,  3.8937e+01_r8,  3.9472e+01_r8,  4.0008e+01_r8 /)
      colmpara(40,:) = (/ &
        3.4860e+01_r8,  3.5395e+01_r8,  3.5931e+01_r8,  3.6466e+01_r8,  3.7002e+01_r8, &
        3.7538e+01_r8,  3.8073e+01_r8,  3.8608e+01_r8,  3.9144e+01_r8,  3.9679e+01_r8 /)
      colmpara(41,:) = (/ &
        3.4525e+01_r8,  3.5061e+01_r8,  3.5596e+01_r8,  3.6132e+01_r8,  3.6667e+01_r8, &
        3.7203e+01_r8,  3.7738e+01_r8,  3.8274e+01_r8,  3.8809e+01_r8,  3.9345e+01_r8 /)
      colmpara(42,:) = (/ &
        3.4184e+01_r8,  3.4719e+01_r8,  3.5254e+01_r8,  3.5790e+01_r8,  3.6326e+01_r8, &
        3.6861e+01_r8,  3.7396e+01_r8,  3.7932e+01_r8,  3.8467e+01_r8,  3.9003e+01_r8 /)
      colmpara(43,:) = (/ &
        3.3835e+01_r8,  3.4370e+01_r8,  3.4906e+01_r8,  3.5441e+01_r8,  3.5977e+01_r8, &
        3.6512e+01_r8,  3.7048e+01_r8,  3.7583e+01_r8,  3.8119e+01_r8,  3.8654e+01_r8 /)
      colmpara(44,:) = (/ &
        3.3478e+01_r8,  3.4014e+01_r8,  3.4549e+01_r8,  3.5085e+01_r8,  3.5620e+01_r8, &
        3.6155e+01_r8,  3.6691e+01_r8,  3.7227e+01_r8,  3.7762e+01_r8,  3.8297e+01_r8 /)
      colmpara(45,:) = (/ &
        3.3112e+01_r8,  3.3648e+01_r8,  3.4183e+01_r8,  3.4719e+01_r8,  3.5254e+01_r8, &
        3.5790e+01_r8,  3.6325e+01_r8,  3.6861e+01_r8,  3.7396e+01_r8,  3.7932e+01_r8 /)
      colmpara(46,:) = (/ &
        3.2738e+01_r8,  3.3273e+01_r8,  3.3808e+01_r8,  3.4344e+01_r8,  3.4880e+01_r8, &
        3.5415e+01_r8,  3.5950e+01_r8,  3.6486e+01_r8,  3.7021e+01_r8,  3.7557e+01_r8 /)
      colmpara(47,:) = (/ &
        3.2354e+01_r8,  3.2889e+01_r8,  3.3425e+01_r8,  3.3960e+01_r8,  3.4496e+01_r8, &
        3.5031e+01_r8,  3.5567e+01_r8,  3.6102e+01_r8,  3.6638e+01_r8,  3.7173e+01_r8 /)
      colmpara(48,:) = (/ &
        3.1963e+01_r8,  3.2498e+01_r8,  3.3034e+01_r8,  3.3569e+01_r8,  3.4105e+01_r8, &
        3.4640e+01_r8,  3.5176e+01_r8,  3.5711e+01_r8,  3.6247e+01_r8,  3.6782e+01_r8 /)
      colmpara(49,:) = (/ &
        3.1567e+01_r8,  3.2103e+01_r8,  3.2638e+01_r8,  3.3174e+01_r8,  3.3709e+01_r8, &
        3.4244e+01_r8,  3.4780e+01_r8,  3.5316e+01_r8,  3.5851e+01_r8,  3.6386e+01_r8 /)
      colmpara(50,:) = (/ &
        3.1171e+01_r8,  3.1707e+01_r8,  3.2242e+01_r8,  3.2778e+01_r8,  3.3314e+01_r8, &
        3.3849e+01_r8,  3.4384e+01_r8,  3.4920e+01_r8,  3.5456e+01_r8,  3.5991e+01_r8 /)
      colmpara(51,:) = (/ &
        3.0777e+01_r8,  3.1313e+01_r8,  3.1848e+01_r8,  3.2384e+01_r8,  3.2919e+01_r8, &
        3.3455e+01_r8,  3.3990e+01_r8,  3.4526e+01_r8,  3.5062e+01_r8,  3.5597e+01_r8 /)
      colmpara(52,:) = (/ &
        3.0379e+01_r8,  3.0914e+01_r8,  3.1450e+01_r8,  3.1985e+01_r8,  3.2521e+01_r8, &
        3.3056e+01_r8,  3.3592e+01_r8,  3.4127e+01_r8,  3.4663e+01_r8,  3.5198e+01_r8 /)
      colmpara(53,:) = (/ &
        2.9970e+01_r8,  3.0506e+01_r8,  3.1041e+01_r8,  3.1577e+01_r8,  3.2112e+01_r8, &
        3.2648e+01_r8,  3.3183e+01_r8,  3.3719e+01_r8,  3.4254e+01_r8,  3.4790e+01_r8 /)
      colmpara(54,:) = (/ &
        2.9547e+01_r8,  3.0083e+01_r8,  3.0618e+01_r8,  3.1154e+01_r8,  3.1689e+01_r8, &
        3.2225e+01_r8,  3.2760e+01_r8,  3.3296e+01_r8,  3.3831e+01_r8,  3.4367e+01_r8 /)
      colmpara(55,:) = (/ &
        2.9104e+01_r8,  2.9640e+01_r8,  3.0175e+01_r8,  3.0711e+01_r8,  3.1246e+01_r8, &
        3.1782e+01_r8,  3.2317e+01_r8,  3.2853e+01_r8,  3.3388e+01_r8,  3.3924e+01_r8 /)
      colmpara(56,:) = (/ &
        2.8636e+01_r8,  2.9171e+01_r8,  2.9707e+01_r8,  3.0242e+01_r8,  3.0778e+01_r8, &
        3.1313e+01_r8,  3.1849e+01_r8,  3.2384e+01_r8,  3.2920e+01_r8,  3.3455e+01_r8 /)
      colmpara(57,:) = (/ &
        2.8140e+01_r8,  2.8675e+01_r8,  2.9211e+01_r8,  2.9746e+01_r8,  3.0282e+01_r8, &
        3.0817e+01_r8,  3.1353e+01_r8,  3.1888e+01_r8,  3.2424e+01_r8,  3.2960e+01_r8 /)
      colmpara(58,:) = (/ &
        2.7615e+01_r8,  2.8151e+01_r8,  2.8686e+01_r8,  2.9221e+01_r8,  2.9757e+01_r8, &
        3.0292e+01_r8,  3.0828e+01_r8,  3.1364e+01_r8,  3.1899e+01_r8,  3.2435e+01_r8 /)
      colmpara(59,:) = (/ &
        2.7048e+01_r8,  2.7583e+01_r8,  2.8119e+01_r8,  2.8654e+01_r8,  2.9190e+01_r8, &
        2.9725e+01_r8,  3.0261e+01_r8,  3.0796e+01_r8,  3.1332e+01_r8,  3.1867e+01_r8 /)
      colmpara(60,:) = (/ &
        2.6417e+01_r8,  2.6952e+01_r8,  2.7488e+01_r8,  2.8023e+01_r8,  2.8558e+01_r8, &
        2.9094e+01_r8,  2.9630e+01_r8,  3.0165e+01_r8,  3.0701e+01_r8,  3.1236e+01_r8 /)
      colmpara(61,:) = (/ &
        2.5690e+01_r8,  2.6226e+01_r8,  2.6761e+01_r8,  2.7297e+01_r8,  2.7832e+01_r8, &
        2.8368e+01_r8,  2.8903e+01_r8,  2.9439e+01_r8,  2.9974e+01_r8,  3.0510e+01_r8 /)
      colmpara(62,:) = (/ &
        2.4753e+01_r8,  2.5288e+01_r8,  2.5824e+01_r8,  2.6359e+01_r8,  2.6895e+01_r8, &
        2.7430e+01_r8,  2.7966e+01_r8,  2.8501e+01_r8,  2.9036e+01_r8,  2.9572e+01_r8 /)

      corrnormpara(1,:) = (/ &
        1.0127e+00_r8,  7.6998e-01_r8,  5.8224e-01_r8,  4.2806e-01_r8,  2.9084e-01_r8, &
        1.9402e-01_r8,  1.3023e-01_r8,  9.1149e-02_r8,  6.4704e-02_r8,  4.7688e-02_r8/)
      corrnormpara(2,:) = (/ &
        1.0187e+00_r8,  7.7869e-01_r8,  5.9453e-01_r8,  4.4852e-01_r8,  3.1375e-01_r8, &
        2.1065e-01_r8,  1.3653e-01_r8,  9.1144e-02_r8,  6.1994e-02_r8,  4.4571e-02_r8/)
      corrnormpara(3,:) = (/ &
        1.0270e+00_r8,  7.8203e-01_r8,  5.9534e-01_r8,  4.5240e-01_r8,  3.2403e-01_r8, &
        2.2302e-01_r8,  1.4233e-01_r8,  8.9070e-02_r8,  5.6653e-02_r8,  3.8611e-02_r8/)
      corrnormpara(4,:) = (/ &
        1.0373e+00_r8,  7.8500e-01_r8,  5.9297e-01_r8,  4.4837e-01_r8,  3.2384e-01_r8, &
        2.2687e-01_r8,  1.4450e-01_r8,  8.4366e-02_r8,  4.9511e-02_r8,  3.1025e-02_r8/)
      corrnormpara(5,:) = (/ &
        1.0490e+00_r8,  7.8929e-01_r8,  5.9130e-01_r8,  4.4231e-01_r8,  3.1728e-01_r8, &
        2.2191e-01_r8,  1.4024e-01_r8,  7.6380e-02_r8,  4.1139e-02_r8,  2.2905e-02_r8/)
      corrnormpara(6,:) = (/ &
        1.0617e+00_r8,  7.9456e-01_r8,  5.9057e-01_r8,  4.3581e-01_r8,  3.0571e-01_r8, &
        2.0872e-01_r8,  1.2728e-01_r8,  6.2907e-02_r8,  3.0269e-02_r8,  1.4243e-02_r8/)
      corrnormpara(7,:) = (/ &
        1.0773e+00_r8,  8.0212e-01_r8,  5.9180e-01_r8,  4.2987e-01_r8,  2.8925e-01_r8, &
        1.8822e-01_r8,  1.0372e-01_r8,  3.4590e-02_r8,  1.2018e-02_r8,  3.8835e-03_r8/)
      corrnormpara(8,:) = (/ &
        1.1003e+00_r8,  8.1612e-01_r8,  5.9836e-01_r8,  4.2760e-01_r8,  2.8986e-01_r8, &
        1.8413e-01_r8,  9.7462e-02_r8,  4.0988e-02_r8,  7.2898e-03_r8, -9.4405e-03_r8/)
      corrnormpara(9,:) = (/ &
        1.1409e+00_r8,  8.4433e-01_r8,  6.1714e-01_r8,  4.3559e-01_r8,  2.9400e-01_r8, &
        1.8117e-01_r8,  8.9445e-02_r8,  3.9187e-02_r8, -2.7993e-03_r8, -2.3482e-02_r8/)
      corrnormpara(10,:) = (/ &
        1.2102e+00_r8,  8.9617e-01_r8,  6.5632e-01_r8,  4.6226e-01_r8,  3.0804e-01_r8, &
        1.8559e-01_r8,  8.5822e-02_r8,  3.2302e-02_r8, -1.1882e-02_r8, -3.1824e-02_r8/)
      corrnormpara(11,:) = (/ &
        1.3155e+00_r8,  9.7740e-01_r8,  7.2026e-01_r8,  5.1267e-01_r8,  3.4425e-01_r8, &
        2.1050e-01_r8,  1.0137e-01_r8,  4.2290e-02_r8, -6.6371e-03_r8, -2.9570e-02_r8/)
      corrnormpara(12,:) = (/ &
        1.4568e+00_r8,  1.0872e+00_r8,  8.0860e-01_r8,  5.8668e-01_r8,  3.7564e-01_r8, &
        2.2742e-01_r8,  1.1522e-01_r8,  6.0656e-02_r8,  1.5509e-02_r8, -1.3943e-02_r8/)
      corrnormpara(13,:) = (/ &
        1.6291e+00_r8,  1.2209e+00_r8,  9.1448e-01_r8,  6.7723e-01_r8,  4.4810e-01_r8, &
        2.8628e-01_r8,  1.6087e-01_r8,  8.2129e-02_r8,  3.7453e-02_r8,  1.2558e-02_r8/)
      corrnormpara(14,:) = (/ &
        1.8231e+00_r8,  1.3678e+00_r8,  1.0284e+00_r8,  7.7316e-01_r8,  5.4154e-01_r8, &
        3.6841e-01_r8,  2.3079e-01_r8,  1.3201e-01_r8,  7.5752e-02_r8,  4.5007e-02_r8/)
      corrnormpara(15,:) = (/ &
        2.0245e+00_r8,  1.5178e+00_r8,  1.1409e+00_r8,  8.6451e-01_r8,  6.2741e-01_r8, &
        4.4552e-01_r8,  2.9960e-01_r8,  1.9064e-01_r8,  1.2103e-01_r8,  7.7739e-02_r8/)
      corrnormpara(16,:) = (/ &
        2.2205e+00_r8,  1.6645e+00_r8,  1.2499e+00_r8,  9.4698e-01_r8,  7.0103e-01_r8, &
        5.1247e-01_r8,  3.5917e-01_r8,  2.4008e-01_r8,  1.5973e-01_r8,  1.0640e-01_r8/)
      corrnormpara(17,:) = (/ &
        2.4095e+00_r8,  1.8035e+00_r8,  1.3532e+00_r8,  1.0211e+00_r8,  7.6117e-01_r8, &
        5.6591e-01_r8,  4.0665e-01_r8,  2.7886e-01_r8,  1.9017e-01_r8,  1.2938e-01_r8/)
      corrnormpara(18,:) = (/ &
        2.5945e+00_r8,  1.9315e+00_r8,  1.4449e+00_r8,  1.0869e+00_r8,  8.0855e-01_r8, &
        6.0410e-01_r8,  4.3994e-01_r8,  3.0564e-01_r8,  2.1118e-01_r8,  1.4559e-01_r8/)
      corrnormpara(19,:) = (/ &
        2.7711e+00_r8,  2.0436e+00_r8,  1.5179e+00_r8,  1.1386e+00_r8,  8.4343e-01_r8, &
        6.2755e-01_r8,  4.5825e-01_r8,  3.2012e-01_r8,  2.2249e-01_r8,  1.5473e-01_r8/)
      corrnormpara(20,:) = (/ &
        2.9222e+00_r8,  2.1322e+00_r8,  1.5680e+00_r8,  1.1720e+00_r8,  8.6482e-01_r8, &
        6.3853e-01_r8,  4.6350e-01_r8,  3.2373e-01_r8,  2.2527e-01_r8,  1.5753e-01_r8/)
      corrnormpara(21,:) = (/ &
        3.0229e+00_r8,  2.1936e+00_r8,  1.5980e+00_r8,  1.1857e+00_r8,  8.7186e-01_r8, &
        6.3904e-01_r8,  4.5877e-01_r8,  3.1914e-01_r8,  2.2164e-01_r8,  1.5567e-01_r8/)
      corrnormpara(22,:) = (/ &
        3.0587e+00_r8,  2.2270e+00_r8,  1.6153e+00_r8,  1.1860e+00_r8,  8.6554e-01_r8, &
        6.3031e-01_r8,  4.4655e-01_r8,  3.0841e-01_r8,  2.1341e-01_r8,  1.5074e-01_r8/)
      corrnormpara(23,:) = (/ &
        3.0304e+00_r8,  2.2330e+00_r8,  1.6267e+00_r8,  1.1810e+00_r8,  8.4969e-01_r8, &
        6.1375e-01_r8,  4.2865e-01_r8,  2.9308e-01_r8,  2.0168e-01_r8,  1.4361e-01_r8/)
      corrnormpara(24,:) = (/ &
        2.9488e+00_r8,  2.2160e+00_r8,  1.6360e+00_r8,  1.1770e+00_r8,  8.2992e-01_r8, &
        5.9196e-01_r8,  4.0652e-01_r8,  2.7399e-01_r8,  1.8695e-01_r8,  1.3468e-01_r8/)
      corrnormpara(25,:) = (/ &
        2.8341e+00_r8,  2.2280e+00_r8,  1.6729e+00_r8,  1.1866e+00_r8,  8.3395e-01_r8, &
        5.9046e-01_r8,  4.0163e-01_r8,  2.7945e-01_r8,  1.8343e-01_r8,  1.2405e-01_r8/)
      corrnormpara(26,:) = (/ &
        2.6932e+00_r8,  2.1887e+00_r8,  1.6860e+00_r8,  1.2020e+00_r8,  8.3694e-01_r8, &
        5.8129e-01_r8,  3.8726e-01_r8,  2.7069e-01_r8,  1.7119e-01_r8,  1.1151e-01_r8/)
      corrnormpara(27,:) = (/ &
        2.5405e+00_r8,  2.0861e+00_r8,  1.6620e+00_r8,  1.2163e+00_r8,  8.0881e-01_r8, &
        5.3543e-01_r8,  3.3890e-01_r8,  2.1588e-01_r8,  1.3605e-01_r8,  9.7129e-02_r8/)
      corrnormpara(28,:) = (/ &
        2.3765e+00_r8,  2.0097e+00_r8,  1.6517e+00_r8,  1.2423e+00_r8,  8.4083e-01_r8, &
        5.4485e-01_r8,  3.3131e-01_r8,  2.1040e-01_r8,  1.2250e-01_r8,  8.1111e-02_r8/)
      corrnormpara(29,:) = (/ &
        2.2055e+00_r8,  1.9224e+00_r8,  1.6294e+00_r8,  1.2636e+00_r8,  8.7801e-01_r8, &
        5.6100e-01_r8,  3.2565e-01_r8,  2.0542e-01_r8,  1.0852e-01_r8,  6.4218e-02_r8/)
      corrnormpara(30,:) = (/ &
        2.0096e+00_r8,  1.7961e+00_r8,  1.5707e+00_r8,  1.2597e+00_r8,  8.8893e-01_r8, &
        5.5944e-01_r8,  3.0422e-01_r8,  1.8113e-01_r8,  8.4620e-02_r8,  4.6343e-02_r8/)
      corrnormpara(31,:) = (/ &
        1.8388e+00_r8,  1.6814e+00_r8,  1.5111e+00_r8,  1.2497e+00_r8,  8.9772e-01_r8, &
        5.6143e-01_r8,  2.8445e-01_r8,  1.5706e-01_r8,  6.0365e-02_r8,  2.9833e-02_r8/)
      corrnormpara(32,:) = (/ &
        1.6396e+00_r8,  1.5305e+00_r8,  1.4078e+00_r8,  1.1951e+00_r8,  8.7151e-01_r8, &
        5.4223e-01_r8,  2.4998e-01_r8,  1.2041e-01_r8,  2.5746e-02_r8,  7.6820e-03_r8/)
      corrnormpara(33,:) = (/ &
        1.5550e+00_r8,  1.4667e+00_r8,  1.3663e+00_r8,  1.1810e+00_r8,  8.7444e-01_r8, &
        5.4784e-01_r8,  2.3483e-01_r8,  9.4771e-02_r8, -5.0855e-03_r8, -1.1990e-02_r8/)
      corrnormpara(34,:) = (/ &
        1.5314e+00_r8,  1.4440e+00_r8,  1.3477e+00_r8,  1.1757e+00_r8,  8.8300e-01_r8, &
        5.6149e-01_r8,  2.2941e-01_r8,  7.1049e-02_r8, -4.1493e-02_r8, -3.9332e-02_r8/)
      corrnormpara(35,:) = (/ &
        1.7270e+00_r8,  1.6064e+00_r8,  1.4824e+00_r8,  1.2918e+00_r8,  9.9076e-01_r8, &
        6.5768e-01_r8,  2.9160e-01_r8,  9.1000e-02_r8, -5.7683e-02_r8, -6.3116e-02_r8/)
      corrnormpara(36,:) = (/ &
        2.0397e+00_r8,  1.8621e+00_r8,  1.6875e+00_r8,  1.4588e+00_r8,  1.1422e+00_r8, &
        7.9506e-01_r8,  3.9605e-01_r8,  1.3928e-01_r8, -6.2587e-02_r8, -9.5685e-02_r8/)
      corrnormpara(37,:) = (/ &
        2.2668e+00_r8,  2.0382e+00_r8,  1.8186e+00_r8,  1.5593e+00_r8,  1.2376e+00_r8, &
        8.9109e-01_r8,  4.7986e-01_r8,  1.7537e-01_r8, -7.7256e-02_r8, -1.5153e-01_r8/)
      corrnormpara(38,:) = (/ &
        2.3460e+00_r8,  2.0887e+00_r8,  1.8443e+00_r8,  1.5690e+00_r8,  1.2543e+00_r8, &
        9.1963e-01_r8,  5.1685e-01_r8,  1.8720e-01_r8, -9.8736e-02_r8, -2.1873e-01_r8/)
      corrnormpara(39,:) = (/ &
        2.3330e+00_r8,  2.0637e+00_r8,  1.8110e+00_r8,  1.5281e+00_r8,  1.2244e+00_r8, &
        9.0429e-01_r8,  5.2074e-01_r8,  1.9076e-01_r8, -1.0517e-01_r8, -2.6267e-01_r8/)
      corrnormpara(40,:) = (/ &
        2.2348e+00_r8,  1.9681e+00_r8,  1.7201e+00_r8,  1.4351e+00_r8,  1.1484e+00_r8, &
        8.4754e-01_r8,  4.9071e-01_r8,  1.8348e-01_r8, -9.6347e-02_r8, -2.6481e-01_r8/)
      corrnormpara(41,:) = (/ &
        2.1025e+00_r8,  1.8445e+00_r8,  1.6021e+00_r8,  1.3071e+00_r8,  1.0438e+00_r8, &
        7.6659e-01_r8,  4.4062e-01_r8,  1.6942e-01_r8, -7.5640e-02_r8, -2.2199e-01_r8/)
      corrnormpara(42,:) = (/ &
        1.9475e+00_r8,  1.7005e+00_r8,  1.4648e+00_r8,  1.1508e+00_r8,  9.1109e-01_r8, &
        6.6102e-01_r8,  3.6639e-01_r8,  1.3488e-01_r8, -6.7300e-02_r8, -1.6573e-01_r8/)
      corrnormpara(43,:) = (/ &
        1.7615e+00_r8,  1.5319e+00_r8,  1.3075e+00_r8,  9.7985e-01_r8,  7.5181e-01_r8, &
        5.1793e-01_r8,  2.4758e-01_r8,  5.6814e-02_r8, -1.0011e-01_r8, -1.3994e-01_r8/)
      corrnormpara(44,:) = (/ &
        1.5496e+00_r8,  1.3439e+00_r8,  1.1379e+00_r8,  8.1610e-01_r8,  5.7226e-01_r8, &
        3.1847e-01_r8,  4.8379e-02_r8, -9.4852e-02_r8, -1.9907e-01_r8, -1.7544e-01_r8/)
      corrnormpara(45,:) = (/ &
        1.3282e+00_r8,  1.1519e+00_r8,  9.7038e-01_r8,  6.7744e-01_r8,  3.8117e-01_r8, &
        4.9884e-02_r8, -2.7429e-01_r8, -3.5478e-01_r8, -3.8500e-01_r8, -2.7837e-01_r8/)
      corrnormpara(46,:) = (/ &
        1.1036e+00_r8,  9.5974e-01_r8,  8.0834e-01_r8,  5.6022e-01_r8,  1.8634e-01_r8, &
       -2.7016e-01_r8, -7.2208e-01_r8, -7.3811e-01_r8, -6.7652e-01_r8, -4.4553e-01_r8/)
      corrnormpara(47,:) = (/ &
        8.8569e-01_r8,  7.7516e-01_r8,  6.5602e-01_r8,  4.5814e-01_r8,  2.0128e-02_r8, &
       -5.5226e-01_r8, -1.1772e+00_r8, -1.1949e+00_r8, -1.0850e+00_r8, -6.9480e-01_r8/)
      corrnormpara(48,:) = (/ &
        6.9026e-01_r8,  6.0924e-01_r8,  5.1984e-01_r8,  3.6822e-01_r8, -7.9705e-02_r8, &
       -6.9601e-01_r8, -1.4791e+00_r8, -1.6532e+00_r8, -1.6410e+00_r8, -1.1362e+00_r8/)
      corrnormpara(49,:) = (/ &
        5.1458e-01_r8,  4.5782e-01_r8,  3.9370e-01_r8,  2.8169e-01_r8, -1.1179e-01_r8, &
       -6.7907e-01_r8, -1.5384e+00_r8, -2.0210e+00_r8, -2.3049e+00_r8, -1.9254e+00_r8/)
      corrnormpara(50,:) = (/ &
        3.6681e-01_r8,  3.2819e-01_r8,  2.8391e-01_r8,  2.0345e-01_r8, -1.0147e-01_r8, &
       -5.6092e-01_r8, -1.3819e+00_r8, -2.1793e+00_r8, -2.8408e+00_r8, -2.9029e+00_r8/)
      corrnormpara(51,:) = (/ &
        2.4572e-01_r8,  2.1994e-01_r8,  1.8976e-01_r8,  1.3317e-01_r8, -8.4483e-02_r8, &
       -4.2521e-01_r8, -1.1176e+00_r8, -2.0715e+00_r8, -2.9853e+00_r8, -3.5512e+00_r8/)
      corrnormpara(52,:) = (/ &
        1.3765e-01_r8,  1.2118e-01_r8,  1.0134e-01_r8,  6.3601e-02_r8, -8.1307e-02_r8, &
       -3.1424e-01_r8, -8.3137e-01_r8, -1.7228e+00_r8, -2.6429e+00_r8, -3.4631e+00_r8/)
      corrnormpara(53,:) = (/ &
        3.9753e-02_r8,  2.9513e-02_r8,  1.7194e-02_r8, -6.4057e-03_r8, -9.6451e-02_r8, &
       -2.4263e-01_r8, -5.8604e-01_r8, -1.2695e+00_r8, -2.0044e+00_r8, -2.7729e+00_r8/)
      corrnormpara(54,:) = (/ &
       -4.9228e-02_r8, -6.7635e-02_r8, -8.4465e-02_r8, -1.0228e-01_r8, -1.5492e-01_r8, &
       -2.4047e-01_r8, -4.4807e-01_r8, -9.0507e-01_r8, -1.4082e+00_r8, -1.9908e+00_r8/)
      corrnormpara(55,:) = (/ &
       -1.2942e-01_r8, -1.5302e-01_r8, -1.7275e-01_r8, -1.8669e-01_r8, -2.1598e-01_r8, &
       -2.6364e-01_r8, -3.8122e-01_r8, -6.5929e-01_r8, -9.6948e-01_r8, -1.3556e+00_r8/)
      corrnormpara(56,:) = (/ &
       -1.9101e-01_r8, -2.0825e-01_r8, -2.2230e-01_r8, -2.3063e-01_r8, -2.4588e-01_r8, &
       -2.7113e-01_r8, -3.3375e-01_r8, -4.8833e-01_r8, -6.6192e-01_r8, -8.8584e-01_r8/)
      corrnormpara(57,:) = (/ &
       -2.2531e-01_r8, -2.3179e-01_r8, -2.3703e-01_r8, -2.4020e-01_r8, -2.4747e-01_r8, &
       -2.5995e-01_r8, -2.9111e-01_r8, -3.6874e-01_r8, -4.5632e-01_r8, -5.6825e-01_r8/)
      corrnormpara(58,:) = (/ &
       -2.3044e-01_r8, -2.3034e-01_r8, -2.3035e-01_r8, -2.3068e-01_r8, -2.3375e-01_r8, &
       -2.3952e-01_r8, -2.5404e-01_r8, -2.8977e-01_r8, -3.3031e-01_r8, -3.7962e-01_r8/)
      corrnormpara(59,:) = (/ &
       -2.1246e-01_r8, -2.1209e-01_r8, -2.1172e-01_r8, -2.1162e-01_r8, -2.1269e-01_r8, &
       -2.1506e-01_r8, -2.2143e-01_r8, -2.3725e-01_r8, -2.5530e-01_r8, -2.7715e-01_r8/)
      corrnormpara(60,:) = (/ &
       -1.7345e-01_r8, -1.7352e-01_r8, -1.7358e-01_r8, -1.7338e-01_r8, -1.7352e-01_r8, &
       -1.7428e-01_r8, -1.7682e-01_r8, -1.8326e-01_r8, -1.9064e-01_r8, -1.9988e-01_r8/)
      corrnormpara(61,:) = (/ &
       -1.2718e-01_r8, -1.2718e-01_r8, -1.2722e-01_r8, -1.2698e-01_r8, -1.2675e-01_r8, &
       -1.2695e-01_r8, -1.2775e-01_r8, -1.2992e-01_r8, -1.3249e-01_r8, -1.3566e-01_r8/)
      corrnormpara(62,:) = (/ &
       -8.6627e-02_r8, -8.6640e-02_r8, -8.6627e-02_r8, -8.6447e-02_r8, -8.6173e-02_r8, &
       -8.6140e-02_r8, -8.6281e-02_r8, -8.6855e-02_r8, -8.7563e-02_r8, -8.8431e-02_r8/)

      end subroutine heatnirco2_init

      end module mo_heatnirco2
