/*******************************************************************************
 * boundingtask.cpp
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/backend/bounding/boundingtask.cpp $
 * $Revision: #2 $
 * $Change: 5047 $
 * $DateTime: 2010/06/30 07:58:31 $
 * $Author: thorsten $
 *******************************************************************************/

#include <set>

#include <boost/bind.hpp>

// frame.h must always be the first POV file included (pulls in platform config)
#include "backend/frame.h"
#include "backend/math/vector.h"
#include "backend/math/matrices.h"
#include "backend/scene/objects.h"
#include "backend/support/bsptree.h"
#include "backend/bounding/boundingtask.h"
#include "backend/shape/cones.h"
#include "backend/texture/texture.h"
#include "backend/texture/pigment.h"

// this must be the last file included
#include "base/povdebug.h"

namespace pov
{

class SceneObjects : public BSPTree::Objects
{
	public:
		vector<ObjectPtr> infinite;
		vector<ObjectPtr> finite;
		unsigned int numLights;

		SceneObjects(vector<ObjectPtr>& objects)
		{
			numLights = 0;
			for(vector<ObjectPtr>::iterator i(objects.begin()); i != objects.end(); i++)
			{
				if(Test_Flag((*i), INFINITE_FLAG))
				{
					infinite.push_back(*i);
					if (((*i)->Type & LIGHT_SOURCE_OBJECT) != 0)
						numLights++;
				}
				else
					finite.push_back(*i);
			}
		}

		virtual ~SceneObjects()
		{
			// nothing to do
		}

		virtual unsigned int size() const
		{
			return finite.size();
		}

		virtual float GetMin(unsigned int axis, unsigned int i) const
		{
			return finite[i]->BBox.lowerleft[axis];
		}

		virtual float GetMax(unsigned int axis, unsigned int i) const
		{
			return (finite[i]->BBox.lowerleft[axis] + finite[i]->BBox.length[axis]);
		}
};

class BSPProgress : public BSPTree::Progress
{
	public:
		BSPProgress(RenderBackend::SceneId sid, POVMSAddress addr) :
			sceneId(sid),
			frontendAddress(addr),
			lastProgressTime(0)
		{
		}

		virtual void operator()(unsigned int nodes) const
		{
			if((timer.ElapsedRealTime() - lastProgressTime) > 1000) // update progress at most every second
			{
				POVMS_Object obj(kPOVObjectClass_BoundingProgress);
				obj.SetLong(kPOVAttrib_RealTime, timer.ElapsedRealTime());
				obj.SetLong(kPOVAttrib_CurrentNodeCount, nodes);
				RenderBackend::SendSceneOutput(sceneId, frontendAddress, kPOVMsgIdent_Progress, obj);

				Task::CurrentTaskCooperate();

				lastProgressTime = timer.ElapsedRealTime();
			}
		}
	private:
		RenderBackend::SceneId sceneId;
		POVMSAddress frontendAddress;
		Timer timer;
		mutable POV_LONG lastProgressTime;

		BSPProgress();
};

BoundingTask::BoundingTask(shared_ptr<SceneData> sd, unsigned int bt) :
	Task(new SceneThreadData(sd), boost::bind(&BoundingTask::SendFatalError, this, _1)),
	sceneData(sd),
	boundingThreshold(bt)
{
}

BoundingTask::~BoundingTask()
{
}

void BoundingTask::AppendObject(ObjectPtr p)
{
	sceneData->objects.push_back(p);
}

void BoundingTask::Run()
{
	if((sceneData->objects.size() < boundingThreshold) || (sceneData->boundingMethod == 0))
	{
		SceneObjects objects(sceneData->objects);
		sceneData->boundingMethod = 0;
		sceneData->numberOfFiniteObjects = objects.finite.size();
		sceneData->numberOfInfiniteObjects = objects.infinite.size() - objects.numLights;
		return;
	}

	switch(sceneData->boundingMethod)
	{
		case 2:
		{
			// new BSP tree code
			SceneObjects objects(sceneData->objects);
			BSPProgress progress(sceneData->sceneId, sceneData->frontendAddress);

			sceneData->objects.clear();
			sceneData->objects.insert(sceneData->objects.end(), objects.finite.begin(), objects.finite.end());
			sceneData->objects.insert(sceneData->objects.end(), objects.infinite.begin(), objects.infinite.end());
			sceneData->numberOfFiniteObjects = objects.finite.size();
			sceneData->numberOfInfiniteObjects = objects.infinite.size() - objects.numLights;
			sceneData->tree = new BSPTree(sceneData->bspMaxDepth, sceneData->bspObjectIsectCost, sceneData->bspBaseAccessCost, sceneData->bspChildAccessCost, sceneData->bspMissChance);
			sceneData->tree->build(progress, objects,
			                       sceneData->nodes, sceneData->splitNodes, sceneData->objectNodes, sceneData->emptyNodes,
			                       sceneData->maxObjects, sceneData->averageObjects, sceneData->maxDepth, sceneData->averageDepth,
			                       sceneData->aborts, sceneData->averageAborts, sceneData->averageAbortObjects, sceneData->inputFile);
			break;
		}
		case 1:
		{
			// old bounding box code
			unsigned int numberOfLightSources;

			Build_Bounding_Slabs(&(sceneData->boundingSlabs), sceneData->objects, sceneData->numberOfFiniteObjects,
				                   sceneData->numberOfInfiniteObjects, numberOfLightSources);
			break;
		}
	}
}

void BoundingTask::Stopped()
{
}

void BoundingTask::Finish()
{
	GetSceneDataPtr()->timeType = SceneThreadData::kBoundingTime;
	GetSceneDataPtr()->realTime = ConsumedRealTime();
	GetSceneDataPtr()->cpuTime = ConsumedCPUTime();
}

void BoundingTask::SendFatalError(Exception& e)
{
	// if the front-end has been told about this exception already, we don't tell it again
	if (e.frontendnotified(true))
		return;

	POVMS_Message msg(kPOVObjectClass_ControlData, kPOVMsgClass_SceneOutput, kPOVMsgIdent_Error);

	msg.SetString(kPOVAttrib_EnglishText, e.what());
	msg.SetInt(kPOVAttrib_Error, 0);
	msg.SetInt(kPOVAttrib_SceneId, sceneData->sceneId);
	msg.SetSourceAddress(sceneData->backendAddress);
	msg.SetDestinationAddress(sceneData->frontendAddress);

	POVMS_SendMessage(msg);
}

}
