/*******************************************************************************
 * spectral.h
 *
 * This module contains all defines, typedefs, and prototypes for spectral.cpp.
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/smp/source/backend/colour/spectral.h $
 * $Revision: #7 $
 * $Change: 4715 $
 * $DateTime: 2009/02/21 10:00:39 $
 * $Author: chrisc $
 *******************************************************************************/

#ifndef SPECTRAL_H
#define SPECTRAL_H

namespace pov
{

#define SPECTRAL_VIOLET         380.0   // extreme violet
#define SPECTRAL_RED            730.0   // extreme red
#define SPECTRAL_BANDWIDTH      (SPECTRAL_RED-SPECTRAL_VIOLET)
#define SPECTRAL_CENTER         ((SPECTRAL_VIOLET + SPECTRAL_RED)/2)    // TODO - maybe should define this as yellow

/// Class representing a spectral band.
class SpectralBand
{
	public:
        /// Default Constructor.
        SpectralBand():
            wavelength  ( (SPECTRAL_VIOLET + SPECTRAL_RED)/2 ), // deliberately NOT using SPECTRAL_CENTER
            bandwidth   ( SPECTRAL_BANDWIDTH )
        {}

        /// Construct by abstract band index (ranging from 0 to N-1).
        SpectralBand(unsigned int band, unsigned int bands):
            wavelength  ( SPECTRAL_VIOLET + SPECTRAL_BANDWIDTH*( ((float)band+0.5)/(float)bands ) ),
            bandwidth   ( SPECTRAL_BANDWIDTH/(float)bands )
        {}

        /// Construct by spectral band and sub-band index (ranging from 1 to N).
        SpectralBand(const SpectralBand& super, unsigned int subBand, unsigned int subBands):
            wavelength  ( super.wavelength + super.bandwidth*( ((float)subBand+0.5)/(float)subBands - 0.5 ) ),
            bandwidth   ( super.bandwidth/(float)subBands )
        {}

        /// Construct by physical parameters.
        SpectralBand(float wl, float bw):
            wavelength(wl),
            bandwidth(bw)
        {}

        /// Get peak wavelength.
        float GetWavelength() const
        {
            return wavelength;
        }

        /// Get corrected IOR
        double GetDispersionIOR(double nominalIOR, double nominalDispersion) const
        {
            return nominalIOR * pow(nominalDispersion, -(wavelength-SPECTRAL_CENTER)/SPECTRAL_BANDWIDTH );
        }

        /// Get hue
        RGBColour GetHue() const
        {
            return (GetHueIntegral(wavelength+bandwidth/2) - GetHueIntegral(wavelength-bandwidth/2)) * (SPECTRAL_BANDWIDTH/bandwidth);
        }

    private:
        /// Peak wavelength.
        float wavelength;
        /// Nominal bandwidth.
        float bandwidth;

        static RGBColour GetHueIntegral(double wavelength);
};

}

#endif // SPECTRAL_H
