/*******************************************************************************
 * renderbackend.cpp
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/backend/control/renderbackend.cpp $
 * $Revision: #2 $
 * $Change: 5047 $
 * $DateTime: 2010/06/30 07:58:31 $
 * $Author: thorsten $
 *******************************************************************************/

// frame.h must always be the first POV file included (pulls in platform config)
#include "backend/frame.h"
#include "base/povms.h"
#include "base/povmscpp.h"
#include "base/povmsgid.h"
#include "base/pov_err.h"
#include "backend/povray.h"
#include "backend/control/renderbackend.h"
#include "backend/scene/scene.h"
#include "backend/scene/view.h"

// this must be the last file included
#include "base/povdebug.h"

namespace pov
{

using namespace pov_base;

POVMSContext RenderBackend::context = NULL;

RenderBackend::RenderBackend(POVMSContext ctx,  bool (*val)(POVMSAddress)) :
	POVMS_MessageReceiver(ctx),
	validateFrontendAddress(val),
	scenecounter(0),
	viewcounter(0)
{
	context = ctx;

	InstallFront(kPOVMsgClass_BackendControl, kPOVMsgIdent_CreateScene, this, &RenderBackend::CreateScene);
	InstallFront(kPOVMsgClass_BackendControl, kPOVMsgIdent_CloseScene, this, &RenderBackend::CloseScene);

	InstallFront(kPOVMsgClass_SceneControl, kPOVMsgIdent_CreateView, this, &RenderBackend::CreateView);
	InstallFront(kPOVMsgClass_SceneControl, kPOVMsgIdent_CloseView, this, &RenderBackend::CloseView);

	InstallFront(kPOVMsgClass_SceneControl, kPOVMsgIdent_StartParser, this, &RenderBackend::StartParser);
	InstallFront(kPOVMsgClass_SceneControl, kPOVMsgIdent_StopParser, this, &RenderBackend::StopParser);
	InstallFront(kPOVMsgClass_SceneControl, kPOVMsgIdent_PauseParser, this, &RenderBackend::PauseParser);
	InstallFront(kPOVMsgClass_SceneControl, kPOVMsgIdent_ResumeParser, this, &RenderBackend::ResumeParser);

	InstallFront(kPOVMsgClass_ViewControl, kPOVMsgIdent_StartRender, this, &RenderBackend::StartRender);
	InstallFront(kPOVMsgClass_ViewControl, kPOVMsgIdent_StopRender, this, &RenderBackend::StopRender);
	InstallFront(kPOVMsgClass_ViewControl, kPOVMsgIdent_PauseRender, this, &RenderBackend::PauseRender);
	InstallFront(kPOVMsgClass_ViewControl, kPOVMsgIdent_ResumeRender, this, &RenderBackend::ResumeRender);

	InstallFront(kPOVMsgClass_FileAccess, kPOVMsgIdent_ReadFile, this, &RenderBackend::ReadFile);
}

RenderBackend::~RenderBackend()
{
}

void RenderBackend::SendSceneOutput(SceneId sid, POVMSAddress addr, POVMSType ident, POVMS_Object& obj)
{
	POVMS_Message msg(obj, kPOVMsgClass_SceneOutput, ident);

	msg.SetInt(kPOVAttrib_SceneId, sid);
	msg.SetDestinationAddress(addr);

	POVMS_SendMessage(RenderBackend::context, msg, NULL, kPOVMSSendMode_NoReply); // POVMS context provide for source address access only!
}

void RenderBackend::SendViewOutput(ViewId vid, POVMSAddress addr, POVMSType ident, POVMS_Object& obj)
{
	POVMS_Message msg(obj, kPOVMsgClass_ViewOutput, ident);

	msg.SetInt(kPOVAttrib_ViewId, vid);
	msg.SetDestinationAddress(addr);

	POVMS_SendMessage(RenderBackend::context, msg, NULL, kPOVMSSendMode_NoReply); // POVMS context provide for source address access only!
}

void RenderBackend::SendFindFile(POVMSContext ctx, SceneId sid, POVMSAddress addr, const vector<POVMSUCS2String>& filenames, POVMSUCS2String& filename)
{
	POVMS_Message msg(kPOVObjectClass_FileData, kPOVMsgClass_FileAccess, kPOVMsgIdent_FindFile);
	POVMS_Message result(kPOVObjectClass_FileData, kPOVMsgClass_FileAccess, kPOVMsgIdent_FindFile);
	POVMS_List files;

	for(vector<POVMSUCS2String>::const_iterator i(filenames.begin()); i != filenames.end(); i++)
	{
		POVMS_Attribute attr(i->c_str());
		files.Append(attr);
	}

	msg.Set(kPOVAttrib_ReadFile, files);

	msg.SetInt(kPOVAttrib_SceneId, sid);
	msg.SetDestinationAddress(addr);

	POVMS_SendMessage(ctx, msg, &result, kPOVMSSendMode_WaitReply);

	filename = result.TryGetUCS2String(kPOVAttrib_ReadFile, "");
}

void RenderBackend::SendReadFile(POVMSContext ctx, SceneId sid, POVMSAddress addr, const POVMSUCS2String& filename, POVMSUCS2String& localfile, POVMSUCS2String& fileurl)
{
	POVMS_Message msg(kPOVObjectClass_FileData, kPOVMsgClass_FileAccess, kPOVMsgIdent_ReadFile);
	POVMS_Message result(kPOVObjectClass_FileData, kPOVMsgClass_FileAccess, kPOVMsgIdent_ReadFile);

	msg.SetUCS2String(kPOVAttrib_ReadFile, filename.c_str());

	msg.SetInt(kPOVAttrib_SceneId, sid);
	msg.SetDestinationAddress(addr);

	POVMS_SendMessage(ctx, msg, &result, kPOVMSSendMode_WaitReply);

	localfile = result.TryGetUCS2String(kPOVAttrib_LocalFile, "");
	fileurl = result.TryGetUCS2String(kPOVAttrib_FileURL, "");
}

void RenderBackend::SendCreatedFile(POVMSContext ctx, SceneId sid, POVMSAddress addr, const POVMSUCS2String& filename)
{
	POVMS_Message msg(kPOVObjectClass_FileData, kPOVMsgClass_FileAccess, kPOVMsgIdent_CreatedFile);

	msg.SetUCS2String(kPOVAttrib_CreatedFile, filename.c_str());

	msg.SetInt(kPOVAttrib_SceneId, sid);
	msg.SetDestinationAddress(addr);

	POVMS_SendMessage(ctx, msg, NULL, kPOVMSSendMode_NoReply);
}

void RenderBackend::SendSuccessResult(POVMSAddress addr)
{
	POVMS_Message result(kPOVObjectClass_ResultData, kPOVMsgClass_BackendControl, kPOVMsgIdent_Done);

	result.SetDestinationAddress(addr);

	POVMS_SendMessage(RenderBackend::context, result, NULL, kPOVMSSendMode_NoReply); // POVMS context provide for source address access only!
}

void RenderBackend::SendFailedResult(int error, POVMSAddress addr)
{
	POVMS_Message result(kPOVObjectClass_ResultData, kPOVMsgClass_BackendControl, kPOVMsgIdent_Failed);

	result.SetInt(kPOVAttrib_ErrorNumber, error);
	result.SetString(kPOVAttrib_EnglishText, pov_base::Exception::lookup_code(error));
	result.SetDestinationAddress(addr);

	POVMS_SendMessage(RenderBackend::context, result, NULL, kPOVMSSendMode_NoReply); // POVMS context provide for source address access only!
}

void RenderBackend::SendFailedResult(const pov_base::Exception& e, POVMSAddress addr)
{
	POVMS_Message result(kPOVObjectClass_ResultData, kPOVMsgClass_BackendControl, kPOVMsgIdent_Failed);

	result.SetInt(kPOVAttrib_ErrorNumber, e.code(kCannotHandleRequestErr));
	// pov_base::Exception(...) does a code->string lookup if a string isn't supplied
	result.SetString(kPOVAttrib_EnglishText, e.what());
	result.SetDestinationAddress(addr);

	POVMS_SendMessage(RenderBackend::context, result, NULL, kPOVMSSendMode_NoReply); // POVMS context provide for source address access only!
}

void RenderBackend::SendFailedResult(const char *str, POVMSAddress addr)
{
	POVMS_Message result(kPOVObjectClass_ResultData, kPOVMsgClass_BackendControl, kPOVMsgIdent_Failed);

	result.SetString(kPOVAttrib_EnglishText, str);
	result.SetDestinationAddress(addr);

	POVMS_SendMessage(RenderBackend::context, result, NULL, kPOVMSSendMode_NoReply); // POVMS context provide for source address access only!
}

void RenderBackend::SendSceneSuccessResult(SceneId sid, POVMSAddress addr)
{
	POVMS_Message result(kPOVObjectClass_ResultData, kPOVMsgClass_SceneOutput, kPOVMsgIdent_Done);

	result.SetInt(kPOVAttrib_SceneId, sid);
	result.SetDestinationAddress(addr);

	POVMS_SendMessage(RenderBackend::context, result, NULL, kPOVMSSendMode_NoReply); // POVMS context provide for source address access only!
}

void RenderBackend::SendSceneFailedResult(SceneId sid, int error, POVMSAddress addr)
{
	POVMS_Message result(kPOVObjectClass_ResultData, kPOVMsgClass_SceneOutput, kPOVMsgIdent_Failed);

	result.SetInt(kPOVAttrib_SceneId, sid);
	result.SetInt(kPOVAttrib_ErrorNumber, error);
	result.SetString(kPOVAttrib_EnglishText, pov_base::Exception::lookup_code(error));
	result.SetDestinationAddress(addr);

	POVMS_SendMessage(RenderBackend::context, result, NULL, kPOVMSSendMode_NoReply); // POVMS context provide for source address access only!
}

void RenderBackend::SendSceneFailedResult(SceneId sid, const pov_base::Exception& e, POVMSAddress addr)
{
	POVMS_Message result(kPOVObjectClass_ResultData, kPOVMsgClass_SceneOutput, kPOVMsgIdent_Failed);

	result.SetInt(kPOVAttrib_SceneId, sid);
	result.SetInt(kPOVAttrib_ErrorNumber, e.code(kCannotHandleRequestErr));
	// pov_base::Exception(...) does a code->string lookup if a string isn't supplied
	result.SetString(kPOVAttrib_EnglishText, e.what());
	result.SetDestinationAddress(addr);

	POVMS_SendMessage(RenderBackend::context, result, NULL, kPOVMSSendMode_NoReply); // POVMS context provide for source address access only!
}

void RenderBackend::SendSceneFailedResult(SceneId sid, const char *str, POVMSAddress addr)
{
	POVMS_Message result(kPOVObjectClass_ResultData, kPOVMsgClass_SceneOutput, kPOVMsgIdent_Failed);

	result.SetInt(kPOVAttrib_SceneId, sid);
	result.SetString(kPOVAttrib_EnglishText, str);
	result.SetDestinationAddress(addr);

	POVMS_SendMessage(RenderBackend::context, result, NULL, kPOVMSSendMode_NoReply); // POVMS context provide for source address access only!
}

void RenderBackend::SendViewSuccessResult(ViewId vid, POVMSAddress addr)
{
	POVMS_Message result(kPOVObjectClass_ResultData, kPOVMsgClass_ViewOutput, kPOVMsgIdent_Done);

	result.SetInt(kPOVAttrib_ViewId, vid);
	result.SetDestinationAddress(addr);

	POVMS_SendMessage(RenderBackend::context, result, NULL, kPOVMSSendMode_NoReply); // POVMS context provide for source address access only!
}

void RenderBackend::SendViewFailedResult(ViewId vid, int error, POVMSAddress addr)
{
	POVMS_Message result(kPOVObjectClass_ResultData, kPOVMsgClass_ViewOutput, kPOVMsgIdent_Failed);

	result.SetInt(kPOVAttrib_ViewId, vid);
	result.SetInt(kPOVAttrib_ErrorNumber, error);
	result.SetString(kPOVAttrib_EnglishText, pov_base::Exception::lookup_code(error));
	result.SetDestinationAddress(addr);

	POVMS_SendMessage(RenderBackend::context, result, NULL, kPOVMSSendMode_NoReply); // POVMS context provide for source address access only!
}

void RenderBackend::SendViewFailedResult(ViewId vid, const pov_base::Exception& e, POVMSAddress addr)
{
	POVMS_Message result(kPOVObjectClass_ResultData, kPOVMsgClass_ViewOutput, kPOVMsgIdent_Failed);

	result.SetInt(kPOVAttrib_ViewId, vid);
	result.SetInt(kPOVAttrib_ErrorNumber, e.code(kCannotHandleRequestErr));
	// pov_base::Exception(...) does a code->string lookup if a string isn't supplied
	result.SetString(kPOVAttrib_EnglishText, e.what());
	result.SetDestinationAddress(addr);

	POVMS_SendMessage(RenderBackend::context, result, NULL, kPOVMSSendMode_NoReply); // POVMS context provide for source address access only!
}

void RenderBackend::SendViewFailedResult(ViewId vid, const char *str, POVMSAddress addr)
{
	POVMS_Message result(kPOVObjectClass_ResultData, kPOVMsgClass_ViewOutput, kPOVMsgIdent_Failed);

	result.SetInt(kPOVAttrib_ViewId, vid);
	result.SetString(kPOVAttrib_EnglishText, str);
	result.SetDestinationAddress(addr);

	POVMS_SendMessage(RenderBackend::context, result, NULL, kPOVMSSendMode_NoReply); // POVMS context provide for source address access only!
}

void RenderBackend::CreateScene(POVMS_Message& msg, POVMS_Message& result, int)
{
	if(ValidateFrontendAddress(msg.GetSourceAddress(), result) == false)
		return;

	try
	{
		POVMSAddress backendAddress = POVMSInvalidAddress;
		int err = POVMS_GetContextAddress(context, &backendAddress);

		if(err != kNoErr)
			throw POV_EXCEPTION_CODE (err);

		shared_ptr<Scene> scene(new Scene(backendAddress, msg.GetSourceAddress(), scenecounter + 1));

		scenecounter++;

		POVMS_Message newresult(result, result.GetClass(), kPOVMsgIdent_Done);
		result = newresult;
		result.SetInt(kPOVAttrib_SceneId, scenecounter);

		scenes[scenecounter] = scene;
		try
		{
			scene2views[scenecounter] = set<ViewId>();
		}
		catch(pov_base::Exception&)
		{
			scenes.erase(scenecounter);
			throw;
		}
	}
	catch(pov_base::Exception& e)
	{
		MakeFailedResult(e, result);
	}
}

void RenderBackend::CloseScene(POVMS_Message& msg, POVMS_Message& result, int)
{
	if(ValidateFrontendAddress(msg.GetSourceAddress(), result) == false)
		return;

	try
	{
		SceneId sid = msg.GetInt(kPOVAttrib_SceneId);

		map<SceneId, shared_ptr<Scene> >::iterator i(scenes.find(sid));

		if(i == scenes.end())
			throw POV_EXCEPTION_CODE(kInvalidIdentifierErr);

		if(scene2views[sid].size() > 0)
			throw POV_EXCEPTION_CODE(kNotNowErr);

		try { scenes.erase(sid); } catch(pov_base::Exception&) { }
		try { scene2views.erase(sid); } catch(pov_base::Exception&) { }

		MakeDoneResult(result);
	}
	catch(pov_base::Exception& e)
	{
		MakeFailedResult(e, result);
	}
}

void RenderBackend::CreateView(POVMS_Message& msg, POVMS_Message& result, int)
{
	if(ValidateFrontendAddress(msg.GetSourceAddress(), result) == false)
		return;

	try
	{
		SceneId sid = msg.GetInt(kPOVAttrib_SceneId);

		map<SceneId, shared_ptr<Scene> >::iterator i(scenes.find(sid));

		if(i == scenes.end())
			throw POV_EXCEPTION_CODE(kInvalidIdentifierErr);

		shared_ptr<View> view(i->second->NewView(msg.TryGetInt(kPOVAttrib_Width, 160), msg.TryGetInt(kPOVAttrib_Height, 120), viewcounter + 1));

		viewcounter++;

		POVMS_Message newresult(result, result.GetClass(), kPOVMsgIdent_Done);
		result = newresult;
		result.SetInt(kPOVAttrib_ViewId, viewcounter);

		views[viewcounter] = view;
		try { view2scene[viewcounter] = sid; } catch(pov_base::Exception&) { views.erase(viewcounter); throw; }
		try { scene2views[sid].insert(viewcounter); } catch(pov_base::Exception&) { views.erase(viewcounter); view2scene.erase(viewcounter); throw; }
	}
	catch(pov_base::Exception& e)
	{
		MakeFailedResult(e, result);
	}
}

void RenderBackend::CloseView(POVMS_Message& msg, POVMS_Message& result, int)
{
	if(ValidateFrontendAddress(msg.GetSourceAddress(), result) == false)
		return;

	try
	{
		ViewId vid = msg.GetInt(kPOVAttrib_ViewId);

		map<ViewId, shared_ptr<View> >::iterator i(views.find(vid));

		if(i == views.end())
			throw POV_EXCEPTION_CODE(kInvalidIdentifierErr);

		try { views.erase(vid); } catch(pov_base::Exception&) { }
		try { scene2views[view2scene[vid]].erase(vid); } catch(pov_base::Exception&) { }
		try { view2scene.erase(vid); } catch(pov_base::Exception&) { }

		MakeDoneResult(result);
	}
	catch(pov_base::Exception& e)
	{
		MakeFailedResult(e, result);
	}
}

void RenderBackend::StartParser(POVMS_Message& msg, POVMS_Message&, int)
{
	try
	{
		SceneId sid = msg.GetInt(kPOVAttrib_SceneId);

		try
		{
			if(validateFrontendAddress(msg.GetSourceAddress()) == false)
				throw POV_EXCEPTION_CODE(kAuthorisationErr);

			map<SceneId, shared_ptr<Scene> >::iterator i(scenes.find(sid));

			if(i == scenes.end())
				throw POV_EXCEPTION_CODE(kInvalidIdentifierErr);

			i->second->StartParser(msg);
		}
		catch(pov_base::Exception& e)
		{
			SendSceneFailedResult(sid, e, msg.GetSourceAddress());
		}
	}
	catch(pov_base::Exception& e)
	{
		SendFailedResult(e, msg.GetSourceAddress());
	}

}

void RenderBackend::StopParser(POVMS_Message& msg, POVMS_Message& result, int)
{
	if(ValidateFrontendAddress(msg.GetSourceAddress(), result) == false)
		return;

	try
	{
		SceneId sid = msg.GetInt(kPOVAttrib_SceneId);

		map<SceneId, shared_ptr<Scene> >::iterator i(scenes.find(sid));

		if(i == scenes.end())
			throw POV_EXCEPTION_CODE(kInvalidIdentifierErr);

		if((i->second->IsParsing() == false) && (i->second->IsPaused() == false))
			throw POV_EXCEPTION_CODE(kNotNowErr);

		i->second->StopParser();

		MakeDoneResult(result);
	}
	catch(pov_base::Exception& e)
	{
		MakeFailedResult(e, result);
	}
}

void RenderBackend::PauseParser(POVMS_Message& msg, POVMS_Message& result, int)
{
	if(ValidateFrontendAddress(msg.GetSourceAddress(), result) == false)
		return;

	try
	{
		SceneId sid = msg.GetInt(kPOVAttrib_SceneId);

		map<SceneId, shared_ptr<Scene> >::iterator i(scenes.find(sid));

		if(i == scenes.end())
			throw POV_EXCEPTION_CODE(kInvalidIdentifierErr);

		if((i->second->IsParsing() == false) && (i->second->IsPaused() == false))
			throw POV_EXCEPTION_CODE(kNotNowErr);

		i->second->PauseParser();

		MakeDoneResult(result);
	}
	catch(pov_base::Exception& e)
	{
		MakeFailedResult(e, result);
	}
}

void RenderBackend::ResumeParser(POVMS_Message& msg, POVMS_Message& result, int)
{
	if(ValidateFrontendAddress(msg.GetSourceAddress(), result) == false)
		return;

	try
	{
		SceneId sid = msg.GetInt(kPOVAttrib_SceneId);

		map<SceneId, shared_ptr<Scene> >::iterator i(scenes.find(sid));

		if(i == scenes.end())
			throw POV_EXCEPTION_CODE(kInvalidIdentifierErr);

		if((i->second->IsParsing() == false) && (i->second->IsPaused() == false))
			throw POV_EXCEPTION_CODE(kNotNowErr);

		i->second->ResumeParser();

		MakeDoneResult(result);
	}
	catch(pov_base::Exception& e)
	{
		MakeFailedResult(e, result);
	}
}

void RenderBackend::StartRender(POVMS_Message& msg, POVMS_Message&, int)
{
	try
	{
		ViewId vid = msg.GetInt(kPOVAttrib_ViewId);

		try
		{
			if(validateFrontendAddress(msg.GetSourceAddress()) == false)
				throw POV_EXCEPTION_CODE(kAuthorisationErr);

			map<ViewId, shared_ptr<View> >::iterator i(views.find(vid));

			if(i == views.end())
				throw POV_EXCEPTION_CODE(kInvalidIdentifierErr);

			if(i->second->IsRendering() == true)
				throw POV_EXCEPTION_CODE(kNotNowErr);

			i->second->StartRender(msg);
		}
		catch(pov_base::Exception& e)
		{
			SendViewFailedResult(vid, e, msg.GetSourceAddress());
		}
	}
	catch(pov_base::Exception& e)
	{
		SendFailedResult(e, msg.GetSourceAddress());
	}
}

void RenderBackend::StopRender(POVMS_Message& msg, POVMS_Message& result, int)
{
	if(ValidateFrontendAddress(msg.GetSourceAddress(), result) == false)
		return;

	try
	{
		ViewId vid = msg.GetInt(kPOVAttrib_ViewId);

		map<ViewId, shared_ptr<View> >::iterator i(views.find(vid));

		if(i == views.end())
			throw POV_EXCEPTION_CODE(kInvalidIdentifierErr);

		if((i->second->IsRendering() == false) && (i->second->IsPaused() == false))
			throw POV_EXCEPTION_CODE(kNotNowErr);

		i->second->StopRender();

		MakeDoneResult(result);
	}
	catch(pov_base::Exception& e)
	{
		MakeFailedResult(e, result);
	}
}

void RenderBackend::PauseRender(POVMS_Message& msg, POVMS_Message& result, int)
{
	if(ValidateFrontendAddress(msg.GetSourceAddress(), result) == false)
		return;

	try
	{
		ViewId vid = msg.GetInt(kPOVAttrib_ViewId);

		map<ViewId, shared_ptr<View> >::iterator i(views.find(vid));

		if(i == views.end())
			throw POV_EXCEPTION_CODE(kInvalidIdentifierErr);

		if((i->second->IsRendering() == false) && (i->second->IsPaused() == false))
			throw POV_EXCEPTION_CODE(kNotNowErr);

		i->second->PauseRender();

		MakeDoneResult(result);
	}
	catch(pov_base::Exception& e)
	{
		MakeFailedResult(e, result);
	}
}

void RenderBackend::ResumeRender(POVMS_Message& msg, POVMS_Message& result, int)
{
	if(ValidateFrontendAddress(msg.GetSourceAddress(), result) == false)
		return;

	try
	{
		ViewId vid = msg.GetInt(kPOVAttrib_ViewId);

		map<ViewId, shared_ptr<View> >::iterator i(views.find(vid));

		if(i == views.end())
			throw POV_EXCEPTION_CODE(kInvalidIdentifierErr);

		if((i->second->IsRendering() == false) && (i->second->IsPaused() == false))
			throw POV_EXCEPTION_CODE(kNotNowErr);

		i->second->ResumeRender();

		MakeDoneResult(result);
	}
	catch(pov_base::Exception& e)
	{
		MakeFailedResult(e, result);
	}
}

void RenderBackend::ReadFile(POVMS_Message& msg, POVMS_Message& result, int)
{
	if(ValidateFrontendAddress(msg.GetSourceAddress(), result) == false)
		return;

	MakeFailedResult(kCannotHandleRequestErr, result);
}

bool RenderBackend::ValidateFrontendAddress(POVMSAddress addr, POVMS_Message& result)
{
	if(validateFrontendAddress(addr) == false)
	{
		MakeFailedResult(kAuthorisationErr, result);

		return false;
	}
	else
		return true;
}

void RenderBackend::MakeFailedResult(int error, POVMS_Message& result)
{
	POVMS_Message newmsg(result, result.GetClass(), kPOVMsgIdent_Failed);
	result = newmsg;
	result.SetInt(kPOVAttrib_ErrorNumber, error);
	result.SetString(kPOVAttrib_EnglishText, pov_base::Exception::lookup_code(error));
}

void RenderBackend::MakeFailedResult(const pov_base::Exception& e, POVMS_Message& result)
{
	POVMS_Message newmsg(result, result.GetClass(), kPOVMsgIdent_Failed);
	result = newmsg;
	result.SetInt(kPOVAttrib_ErrorNumber, e.code(kCannotHandleRequestErr));
	// pov_base::Exception(...) does a code->string lookup if a string isn't supplied
	result.SetString(kPOVAttrib_EnglishText, e.what());
}

void RenderBackend::MakeFailedResult(const char *str, POVMS_Message& result)
{
	POVMS_Message newmsg(result, result.GetClass(), kPOVMsgIdent_Failed);
	result = newmsg;
	result.SetString(kPOVAttrib_EnglishText, str);
}

void RenderBackend::MakeDoneResult(POVMS_Message& result)
{
	POVMS_Message newmsg(result, result.GetClass(), kPOVMsgIdent_Done);
	result = newmsg;
}

}

/*

const int gStreamTypeUtilDataCount = 6;
const POVMSType gStreamTypeUtilData[gStreamTypeUtilDataCount] =
{
	kPOVAttrib_DebugFile,
	kPOVAttrib_FatalFile,
	kPOVAttrib_RenderFile,
	kPOVAttrib_StatisticsFile,
	kPOVAttrib_WarningFile,
	kPOVAttrib_AllFile
};

const char *gStreamDefaultFile[gStreamTypeUtilDataCount] =
{
	"debug.out",
	"fatal.out",
	"render.out",
	"stats.out",
	"warning.out",
	"alltext.out"
};

POVMSObject gStartedStreamMessageData;
POVMSObject *gStartedStreamMessage = NULL;

const int Quality_Values[12]=
{
  QUALITY_0, QUALITY_1, QUALITY_2, QUALITY_3, QUALITY_4,
  QUALITY_5, QUALITY_6, QUALITY_7, QUALITY_8, QUALITY_9
};

bool Matches(const char *v1, const char *v2);
bool IsTrue(const char *value);
bool IsFalse(const char *value);
int SetCommandOption(POVMSObjectPtr msg, POVMSType key, SHELLDATA *data);

bool Matches(const char *v1, const char *v2) // TODO
{
	int i = 0;
	int ans = 1;

	while((ans) && (v1[i] != 0) && (v2[i] != 0))
	{
		ans = ans && (int)(v1[i] == tolower(v2[i]));
		i++;
	}

	return (ans != 0);
}

bool IsTrue(const char *value) // TODO
{
	return (Matches("on",value)  || Matches("true",value) || 
			  Matches("yes",value) || Matches("1",value));
}

bool IsFalse(const char *value) // TODO
{
	return (Matches("off",value)  || Matches("false",value) || 
			  Matches("no",value)	|| Matches("0",value));
}

int SetCommandOption(POVMSObjectPtr msg, POVMSType key, SHELLDATA *data) // TODO
{
	POVMSObject obj;
	int len = POV_MAX_CMD_LENGTH;
	int err;

	err = POVMSObject_Get(msg, &obj, key);
	if(err == 0)
	{
		err = POVMSUtil_GetString(&obj, kPOVAttrib_CommandString, data->Command, &len);
		if(err == 0)
		{
			int ret = 0;

			err = POVMSUtil_GetInt(&obj, kPOVAttrib_ReturnAction, &ret);
			if(err == 0)
			{
		 		data->Inverse = (ret < 0);
				if(ret < 0)
					ret = -ret;
				switch(ret)
				{
					case 'I':
					case 'i':
						data->Ret = IGNORE_RET;
						break;
					case 'Q':
					case 'q':
						data->Ret = QUIT_RET;
						break;
					case 'U':
					case 'u':
						data->Ret = USER_RET;
						break;
					case 'F':
					case 'f':
						data->Ret = FATAL_RET;
						break;
					case 'S':
					case 's':
						data->Ret = SKIP_ONCE_RET;
						break;
					case 'A':
					case 'a':
						data->Ret = ALL_SKIP_RET;
						break;
				}
			}
		}
		(void)POVMSObject_Delete(&obj);
	}

	return err;
}

void RenderBackend::RenderOptions(POVMSObjectPtr msg, POVMSObjectPtr result, int)
{
	POVMSAttribute attr;
	POVMSInt i;
	POVMSFloat f;
	POVMSBool b;
	int l = 0;

	// TODO FIXME HACK [trf]
	scene = new Scene();

	if(gStartedStreamMessage == NULL)
	{
		if(POVMSObject_New(&gStartedStreamMessageData, kPOVMSType_WildCard) == kNoErr)
			gStartedStreamMessage = &gStartedStreamMessageData;
	}

	l = sizeof(unsigned long);
	if(POVMSObject_Get(msg, &attr, kPOVAttrib_PreviewRefCon) == kNoErr)
	{
		(void)POVMSAttr_Get(&attr, kPOVMSType_WildCard, (void *)(&opts.Preview_RefCon), &l);
		(void)POVMSAttr_Delete(&attr);
	}
	if(POVMSUtil_GetInt(msg, kPOVAttrib_WarningLevel, &i) == 0)
		opts.Warning_Level = i;
	if(POVMSUtil_GetInt(msg, kPOVAttrib_Height, &i) == 0)
		scene->sceneData->iniHeight = i;
	if(POVMSUtil_GetInt(msg, kPOVAttrib_Width, &i) == 0)
		scene->sceneData->iniWidth = i;
	if(POVMSUtil_GetFloat(msg, kPOVAttrib_StartColumn, &f) == 0)
	{
		if(f >= 0.0 && f < 1.0)
		{
			opts.First_Column = -1;
			opts.First_Column_Percent = f;
		}
		else
			opts.First_Column = (int)f;
	}
	if(POVMSUtil_GetFloat(msg, kPOVAttrib_EndColumn, &f) == 0)
	{
		if((f >= 0.0 && f < 1.0) || ((f >= 0.0 && f <= 1.0) && (opts.First_Column < 1)))
		{
			opts.Last_Column = -1;
			opts.Last_Column_Percent = f;
		}
		else
			opts.Last_Column = (int)f;
	}
	if(POVMSUtil_GetFloat(msg, kPOVAttrib_StartRow, &f) == 0)
	{
		if(f >= 0.0 && f < 1.0)
		{
			opts.First_Line = -1;
			opts.First_Line_Percent = f;
		}
		else
			opts.First_Line = (int)f;
	}
	if(POVMSUtil_GetFloat(msg, kPOVAttrib_EndRow, &f) == 0)
	{
		if((f >= 0.0 && f < 1.0) || ((f >= 0.0 && f <= 1.0) && (opts.First_Line < 1)))
		{
			opts.Last_Line = -1;
			opts.Last_Line_Percent = f;
		}
		else
			opts.Last_Line = (int)f;
	}
	if(POVMSUtil_GetBool(msg, kPOVAttrib_TestAbort, &b) == 0)
	{
		if(b == true)
			opts.Options |= EXITENABLE;
		else
			opts.Options &= ~EXITENABLE;
	}
	if(POVMSUtil_GetInt(msg, kPOVAttrib_TestAbortCount, &i) == 0)
		opts.Abort_Test_Counter = i;
	if(POVMSUtil_GetBool(msg, kPOVAttrib_ContinueTrace, &b) == 0)
	{
		if(b == true)
			opts.Options |= CONTINUE_TRACE;
		else
			opts.Options &= ~CONTINUE_TRACE;
	}
	l = FILE_NAME_LENGTH;
	(void)POVMSUtil_GetString(msg, kPOVAttrib_CreateIni, opts.Ini_Output_File_Name, &l);
	if(POVMSUtil_GetFloat(msg, kPOVAttrib_Clock, &f) == 0)
		opts.FrameSeq.Clock_Value = f;
	if(POVMSUtil_GetInt(msg, kPOVAttrib_InitialFrame, &i) == 0)
		opts.FrameSeq.InitialFrame = i;
	if(POVMSUtil_GetInt(msg, kPOVAttrib_FinalFrame, &i) == 0)
		opts.FrameSeq.FinalFrame = i;
	if(POVMSUtil_GetFloat(msg, kPOVAttrib_InitialClock, &f) == 0)
		opts.FrameSeq.InitialClock = f;
	if(POVMSUtil_GetFloat(msg, kPOVAttrib_FinalClock, &f) == 0)
		opts.FrameSeq.FinalClock = f;
	if(POVMSUtil_GetFloat(msg, kPOVAttrib_SubsetStartFrame, &f) == 0)
	{
		if(f > 0.0 && f < 1.0)
			opts.FrameSeq.SubsetStartPercent = f;
		else
			opts.FrameSeq.SubsetStartFrame = (int)f;
	}
	if(POVMSUtil_GetFloat(msg, kPOVAttrib_SubsetEndFrame, &f) == 0)
	{
		if(f > 0.0 && f < 1.0)
			opts.FrameSeq.SubsetEndPercent = f;
		else
			opts.FrameSeq.SubsetEndFrame = (int)f;
	}
	if(POVMSUtil_GetBool(msg, kPOVAttrib_CyclicAnimation, &b) == 0)
	{
		if(b == true)
			opts.Options |= CYCLIC_ANIMATION;
		else
			opts.Options &= ~CYCLIC_ANIMATION;
	}
	if(POVMSUtil_GetBool(msg, kPOVAttrib_FieldRender, &b) == 0)
		opts.FrameSeq.Field_Render_Flag = b;
	if(POVMSUtil_GetBool(msg, kPOVAttrib_OddField, &b) == 0)
		opts.FrameSeq.Odd_Field_Flag = b;
	if(POVMSUtil_GetBool(msg, kPOVAttrib_PauseWhenDone, &b) == 0)
	{
		if(b == true)
			opts.Options |= PROMPTEXIT;
		else
			opts.Options &= ~PROMPTEXIT;
	}
	if(POVMSUtil_GetBool(msg, kPOVAttrib_Verbose, &b) == 0)
	{
		if(b == true)
			opts.Options |= VERBOSE;
		else
			opts.Options &= ~VERBOSE;
	}
	if(POVMSUtil_GetBool(msg, kPOVAttrib_DrawVistas, &b) == 0)
	{
		if(b == true)
			opts.Options |= USE_VISTA_DRAW;
		else
			opts.Options &= ~USE_VISTA_DRAW;
	}
	if(POVMSUtil_GetBool(msg, kPOVAttrib_Display, &b) == 0)
	{
		if(b == true)
			opts.Options |= DISPLAY;
		else
			opts.Options &= ~DISPLAY;
	}
	if(POVMSUtil_GetInt(msg, kPOVAttrib_VideoMode, &i) == 0)
		opts.DisplayFormat = (char)toupper(i);
	if(POVMSUtil_GetInt(msg, kPOVAttrib_Palette, &i) == 0)
		opts.PaletteOption = (char)toupper(i);
	if(POVMSUtil_GetFloat(msg, kPOVAttrib_DisplayGamma, &f) == 0)
	{
		if(f > 0.0)
			opts.DisplayGamma = f;
	}
	if(POVMSUtil_GetInt(msg, kPOVAttrib_PreviewStartSize, &i) == 0)
		opts.PreviewGridSize_Start = i;
	if(POVMSUtil_GetInt(msg, kPOVAttrib_PreviewEndSize, &i) == 0)
		opts.PreviewGridSize_End = i;
	if(POVMSUtil_GetBool(msg, kPOVAttrib_OutputToFile, &b) == 0)
	{
		if(b == true)
			opts.Options |= DISKWRITE;
		else
			opts.Options &= ~DISKWRITE;
	}
	if(POVMSUtil_GetInt(msg, kPOVAttrib_OutputFileType, &i) == 0)
		opts.OutputFormat = (char)tolower(i);
	if(POVMSUtil_GetInt(msg, kPOVAttrib_Compression, &i) == 0)
	{
		if(opts.OutputFormat == 'j')
		{
			opts.OutputQuality = i;
			opts.OutputQuality = max(0, opts.OutputQuality);
			opts.OutputQuality = min(100, opts.OutputQuality);
		}
	}
	if(POVMSUtil_GetBool(msg, kPOVAttrib_OutputAlpha, &b) == 0)
	{
		if(b == true)
			opts.Options |= OUTPUT_ALPHA;
		else
			opts.Options &= ~OUTPUT_ALPHA;
	}
	if(POVMSUtil_GetInt(msg, kPOVAttrib_BitsPerColor, &i) == 0)
	{
		if(opts.OutputFormat != 'j')
		{
			opts.OutputQuality = i;
			opts.OutputQuality = max(5, opts.OutputQuality);
			opts.OutputQuality = min(16, opts.OutputQuality);
		}
	}
	l = FILE_NAME_LENGTH;
	if(POVMSUtil_GetString(msg, kPOVAttrib_OutputFile, opts.Output_File_Name, &l) == 0)
	{
		if(!strcmp(opts.Output_File_Name, "-") || !strcmp(opts.Output_File_Name, "stdout"))
		{
			strcpy(opts.Output_File_Name, "stdout");
			opts.Options |= TO_STDOUT;
		}
	}
	l = FILE_NAME_LENGTH;
	opts.Ini_Output_File_Name[0] = '\0'; // FIXME
	(void)POVMSUtil_GetString(msg, kPOVAttrib_CreateIni, opts.Ini_Output_File_Name, &l);

#if PRECISION_TIMER_AVAILABLE
	if(POVMSUtil_GetBool(msg, kPOVAttrib_CreateHistogram, &b) == 0)
		opts.histogram_on = b;
	if(POVMSUtil_GetInt(msg, kPOVAttrib_HistogramFileType, &i) == 0)
	{
		char *def_ext = NULL;

		switch(i)
		{
			case 'C':
			case 'c':
				opts.histogram_type = CSV;
	 		 opts.histogram_file_type = NO_FILE; // CSV has special handling in histogram output, so this is correct [trf]
				def_ext = ".csv";
				break;
			case 'S':
			case 's':
				opts.histogram_type = SYS ;
	 		 opts.histogram_file_type = SYS_FILE;
				def_ext = SYS_DEF_EXT;
				break ;
			case 'P' :
			case 'p' :
				opts.histogram_type = PPM;
	 		 opts.histogram_file_type = PPM_FILE;
				def_ext = ".ppm";
				break;
			case 'T':
			case 't':
				opts.histogram_type = TARGA;
	 		 opts.histogram_file_type = TGA_FILE;
				def_ext = ".tga";
				break;
			case 'N':
			case 'n':
				opts.histogram_type = PNG;
	 		 opts.histogram_file_type = PNG_FILE;
				def_ext = ".png";
				break;
			default:
				opts.histogram_type = TARGA;
	 		 opts.histogram_file_type = TGA_FILE;
				Warning(0, "Unknown histogram output type '%c'.", (char)i);
				break ;
		}

		// Process the histogram file name now, if it hasn't
		// yet been specified, and in case it isn't set later.
		if (opts.histogram_on && (opts.Histogram_File_Name[0] == '\0') && (def_ext != NULL))
			sprintf(opts.Histogram_File_Name, "histgram%s", def_ext);
	}
	l = FILE_NAME_LENGTH;
	if(POVMSUtil_GetString(msg, kPOVAttrib_HistogramFile, opts.Histogram_File_Name, &l) == 0)
	{
		if(opts.histogram_on && opts.Histogram_File_Name[0] == '\0')
		{
			char *def_ext = NULL;

			switch(opts.histogram_type)
			{
				case CSV:	 def_ext = ".csv"; break;
				case TARGA:  def_ext = ".tga"; break;
				case PNG:	 def_ext = ".png"; break;
				case PPM:	 def_ext = ".ppm"; break;
				case SYS:	 def_ext = SYS_DEF_EXT; break;
				case NONE:	 def_ext = ""; break;	// To quiet warnings
			}
			sprintf(opts.Histogram_File_Name, "histgram%s", def_ext);
		}
	}
	if(POVMSUtil_GetInt(msg, kPOVAttrib_HistogramGridSizeX, &i) == 0)
		opts.histogram_x = i;
	if(POVMSUtil_GetInt(msg, kPOVAttrib_HistogramGridSizeY, &i) == 0)
		opts.histogram_y = i;

#endif // PRECISION_TIMER_AVAILABLE
	(void)SetCommandOption(msg, kPOVAttrib_PreSceneCommand, &opts.Shellouts[PRE_SCENE_SHL]);
	(void)SetCommandOption(msg, kPOVAttrib_PreFrameCommand, &opts.Shellouts[PRE_FRAME_SHL]);
	(void)SetCommandOption(msg, kPOVAttrib_PostSceneCommand, &opts.Shellouts[POST_SCENE_SHL]);
	(void)SetCommandOption(msg, kPOVAttrib_PostFrameCommand, &opts.Shellouts[POST_FRAME_SHL]);
	(void)SetCommandOption(msg, kPOVAttrib_UserAbortCommand, &opts.Shellouts[USER_ABORT_SHL]);
	(void)SetCommandOption(msg, kPOVAttrib_FatalErrorCommand, &opts.Shellouts[FATAL_SHL]);
	l = 0;
	if(POVMSUtil_GetStringLength(msg, kPOVAttrib_InputFile, &l) == 0)
	{
		scene->inputFile.resize(l, ' ');
		if(POVMSUtil_GetString(msg, kPOVAttrib_InputFile, (char *)scene->inputFile.c_str(), &l) == 0) // HACK (char *) [trf]
		{
			if(!strcmp(opts.Input_File_Name, "-") || !strcmp(opts.Input_File_Name, "stdin"))
			{
				scene->inputFile = string("stdin");
				opts.Options |= FROM_STDIN;
			}
		}
	}
	if(POVMSObject_Get(msg, &attr, kPOVAttrib_LibraryPath) == 0)
	{
		int cnt = 0;

		if(POVMSAttrList_Count(&attr, &cnt) == 0)
		{
			POVMSAttribute item;
			int ii,iii;
			bool rem = false;

			for(ii = 1; ii <= cnt; ii++)
			{
				if(POVMSAttrList_GetNth(&attr, ii, &item) == 0)
				{
					l = 0;
					if(POVMSAttr_Size(&item, &l) == 0)
					{
						if(l > 0)
						{
							if(opts.Library_Path_Index >= MAX_LIBRARIES)
								Error ("Too many library directories specified.");
							opts.Library_Paths[opts.Library_Path_Index] = (char *)POV_MALLOC(l, "library paths");
							if(POVMSAttr_Get(&item, kPOVMSType_CString, opts.Library_Paths[opts.Library_Path_Index], &l) == 0)
								rem = false;
							else
								rem = true;

							// remove path again if the same one already exists
							for(iii = 0; iii < opts.Library_Path_Index - 1; iii++)
							{
								if(strcmp(opts.Library_Paths[iii], opts.Library_Paths[opts.Library_Path_Index]) == 0)
								{
									rem = true;
									break;
								}
							}

							if(rem == true)
							{
								POV_FREE(opts.Library_Paths[opts.Library_Path_Index]);
								opts.Library_Paths[opts.Library_Path_Index] = NULL;
							}
							else
								opts.Library_Path_Index++;
						}
					}
					(void)POVMSAttr_Delete(&item);
				}
			}
		}
		(void)POVMSAttr_Delete(&attr);
	}
	if(POVMSUtil_GetFloat(msg, kPOVAttrib_Version, &f) == 0)
		opts.Language_Version = (int)(f * 100 + 0.5);

	(void)POVMSObject_Delete(gStartedStreamMessage);
	(void)POVMSObject_New(gStartedStreamMessage, kPOVMSType_WildCard);
	if(POVMSUtil_GetBool(msg, kPOVAttrib_AllConsole, &b) == 0)
	{
		(void)POVMSUtil_SetBool(gStartedStreamMessage, kPOVAttrib_AllConsole, b);
		(void)POVMSUtil_SetBool(gStartedStreamMessage, kPOVAttrib_DebugConsole, b);
		(void)POVMSUtil_SetBool(gStartedStreamMessage, kPOVAttrib_FatalConsole, b);
		(void)POVMSUtil_SetBool(gStartedStreamMessage, kPOVAttrib_RenderConsole, b);
		(void)POVMSUtil_SetBool(gStartedStreamMessage, kPOVAttrib_StatisticsConsole, b);
		(void)POVMSUtil_SetBool(gStartedStreamMessage, kPOVAttrib_WarningConsole, b);
	}
	if(POVMSUtil_GetBool(msg, kPOVAttrib_DebugConsole, &b) == 0)
		(void)POVMSUtil_SetBool(gStartedStreamMessage, kPOVAttrib_DebugConsole, b);
	if(POVMSUtil_GetBool(msg, kPOVAttrib_FatalConsole, &b) == 0)
		(void)POVMSUtil_SetBool(gStartedStreamMessage, kPOVAttrib_FatalConsole, b);
	if(POVMSUtil_GetBool(msg, kPOVAttrib_RenderConsole, &b) == 0)
		(void)POVMSUtil_SetBool(gStartedStreamMessage, kPOVAttrib_RenderConsole, b);
	if(POVMSUtil_GetBool(msg, kPOVAttrib_StatisticsConsole, &b) == 0)
		(void)POVMSUtil_SetBool(gStartedStreamMessage, kPOVAttrib_StatisticsConsole, b);
	if(POVMSUtil_GetBool(msg, kPOVAttrib_WarningConsole, &b) == 0)
	  (void)POVMSUtil_SetBool(gStartedStreamMessage, kPOVAttrib_WarningConsole, b);
	for(i = 0; i < gStreamTypeUtilDataCount; i++)
	{
		l = 0;
		if(POVMSUtil_GetStringLength(msg, gStreamTypeUtilData[i], &l) == kNoErr)
		{
			 char *str = (char *)POV_MALLOC(l, "stream name");

			 if(POVMSUtil_GetString(msg, gStreamTypeUtilData[i], str, &l) == kNoErr)
			 {
				 if(l > 1)
				 {
					 if(IsTrue(str) == true)
						 (void)POVMSUtil_SetString(gStartedStreamMessage, gStreamTypeUtilData[i], gStreamDefaultFile[i]);
					 else if(IsFalse(str) == false)
						 (void)POVMSUtil_SetString(gStartedStreamMessage, gStreamTypeUtilData[i], str);
				 }
			 }

			 POV_FREE(str);
		}
	}

	if(POVMSUtil_GetInt(msg, kPOVAttrib_Quality, &i) == 0)
	{
		opts.Quality = i;
		// Emit a warning about the "radiosity" quality levels for
		// now.	We can get rid of this some time in the future.
		if ((opts.Quality == 10) || (opts.Quality == 11))
		{
			Warning(0, "Quality settings 10 and 11 are no longer valid.");
			opts.Quality = 9;
		}
		else if ((opts.Quality < 0) || (opts.Quality > 9))
			 Error("Illegal Quality setting.");
		opts.Quality_Flags = Quality_Values[opts.Quality];
	}
	if(POVMSUtil_GetBool(msg, kPOVAttrib_Bounding, &b) == 0)
		opts.Use_Slabs = b;
	if(POVMSUtil_GetInt(msg, kPOVAttrib_BoundingThreshold, &i) == 0)
	{
		if(opts.BBox_Threshold < 1)
			Warning(0, "Too small bounding threshold adjusted to its minimum of one.");
		opts.BBox_Threshold = max(1, i);
	}
	if(POVMSUtil_GetBool(msg, kPOVAttrib_LightBuffer, &b) == 0)
	{
		if(b == true)
			opts.Options |= USE_LIGHT_BUFFER;
		else
			opts.Options &= ~USE_LIGHT_BUFFER;
	}
	if(POVMSUtil_GetBool(msg, kPOVAttrib_VistaBuffer, &b) == 0)
	{
		if(b == true)
			opts.Options |= USE_VISTA_BUFFER;
		else
			opts.Options &= ~USE_VISTA_BUFFER;
	}
	if(POVMSUtil_GetBool(msg, kPOVAttrib_Radiosity, &b) == 0)
	{
		Warning(0, "Radiosity commandline/INI switch is not needed in POV-Ray 3.5.\n"
					  "Add a radiosity{}-block to your scene to turn on radiosity.");
	}
	if(POVMSUtil_GetBool(msg, kPOVAttrib_RemoveBounds, &b) == 0)
	{
		if(b == true)
			opts.Options |= REMOVE_BOUNDS;
		else
			opts.Options &= ~REMOVE_BOUNDS;
	}
	if(POVMSUtil_GetBool(msg, kPOVAttrib_SplitUnions, &b) == 0)
	{
		if(b == true)
			opts.Options |= SPLIT_UNION;
		else
			opts.Options &= ~SPLIT_UNION;
	}
	if(POVMSUtil_GetBool(msg, kPOVAttrib_Antialias, &b) == 0)
	{
		if(b == true)
			opts.Options |= ANTIALIAS;
		else
			opts.Options &= ~ANTIALIAS;
	}
	if(POVMSUtil_GetInt(msg, kPOVAttrib_SamplingMethod, &i) == 0)
		opts.Tracing_Method = i;
	if(POVMSUtil_GetFloat(msg, kPOVAttrib_AntialiasThreshold, &f) == 0)
		opts.Antialias_Threshold = f;
	if(POVMSUtil_GetInt(msg, kPOVAttrib_AntialiasDepth, &i) == 0)
	{
		opts.AntialiasDepth = i;
		if(opts.AntialiasDepth < 1)
			opts.AntialiasDepth = 1;
		if(opts.AntialiasDepth > 9)
			opts.AntialiasDepth = 9;
	}
	if(POVMSUtil_GetBool(msg, kPOVAttrib_Jitter, &b) == 0)
	{
		if(b == true)
			opts.Options |= JITTER;
		else
			opts.Options &= ~JITTER;
	}
	if(POVMSUtil_GetFloat(msg, kPOVAttrib_JitterAmount, &f) == 0)
	{
		opts.JitterScale = f;
		if(opts.JitterScale <= 0.0)
			opts.Options &= ~JITTER;
	}
	if(POVMSObject_Exist(msg, kPOVAttrib_IncludeHeader) == 0)
	{
		l = FILE_NAME_LENGTH;
		opts.Header_File_Name[0] = '\0';
		(void)POVMSUtil_GetString(msg, kPOVAttrib_IncludeHeader, opts.Header_File_Name, &l);
	}
//	(void)POVMSObject_Get(msg, &opts.Declared_Variables, kPOVAttrib_Declare);

//	if(result != NULL)
//		(void)BuildRenderOptions(result);
}

void RenderBackend::RenderAll(POVMSObjectPtr msg, POVMSObjectPtr, int)
{
	scene->Parse(msg);
	static shared_ptr<View> view(scene->NewView());
	view->StartRender(msg);
}

void RenderBackend::RenderArea(POVMSObjectPtr msg, POVMSObjectPtr, int)
{
	POVMSInt l,r,t,b;
	int err;

	err = POVMSUtil_GetInt(msg, kPOVAttrib_Left, &l);
	if(err == 0)
		err = POVMSUtil_GetInt(msg, kPOVAttrib_Right, &r);
	if(err == 0)
		err = POVMSUtil_GetInt(msg, kPOVAttrib_Top, &t);
	if(err == 0)
		err = POVMSUtil_GetInt(msg, kPOVAttrib_Bottom, &b);
	if(err == 0)
	{
	// TODO FIXME
//		opts.First_Column = l;
//		opts.Last_Column = r;
//		opts.First_Line = t;
//		opts.Last_Line = b;

	}
}

void RenderBackend::RenderStop(POVMSObjectPtr, POVMSObjectPtr, int)
{
	Stop_Flag = true;
}
*/
