/*******************************************************************************
 * radiositytask.cpp
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/backend/render/radiositytask.cpp $
 * $Revision: #3 $
 * $Change: 5050 $
 * $DateTime: 2010/06/30 12:23:03 $
 * $Author: thorsten $
 *******************************************************************************/

#include <boost/bind.hpp>

// frame.h must always be the first POV file included (pulls in platform config)
#include "backend/frame.h"

#include "base/types.h"
#include "base/timer.h"

#include "backend/render/radiositytask.h"
#include "backend/scene/threaddata.h"
#include "backend/scene/scene.h"
#include "backend/scene/view.h"

// this must be the last file included
#include "base/povdebug.h"

namespace pov
{

using namespace pov_base;

RadiosityTask::RadiosityTask(ViewData *vd, DBL ptsz, unsigned int pts, unsigned int nt) :
	RenderTask(vd),
	trace(vd, GetViewDataPtr(), vd->GetSceneData()->parsedMaxTraceLevel, vd->GetSceneData()->parsedAdcBailout,
	      vd->GetQualityFeatureFlags(), cooperate, media, radiosity, !vd->GetSceneData()->radiositySettings.vainPretrace),
	pretraceStartSize(vd->GetSceneData()->radiositySettings.pretraceStart),
	pretraceEndSize(vd->GetSceneData()->radiositySettings.pretraceEnd),
	cooperate(*this),
	media(GetViewDataPtr(), &trace, &photonGatherer),
	radiosity(vd->GetSceneData(), GetViewDataPtr(),
	          vd->GetSceneData()->radiositySettings, vd->GetRadiosityCache(), cooperate, pts, Vector3d(vd->GetCamera().Location)),
	photonGatherer(&vd->GetSceneData()->surfacePhotonMap, vd->GetSceneData()->photonSettings),
	pretraceStep(pts),
	pretraceSize(ptsz),
	nominalThreads(nt)
{
}

RadiosityTask::~RadiosityTask()
{
}

// TODO: we probably want a radiosity task control thread rather that having each thread knowing
// its thread number and using this to work out its own pixel list. ideally this would be pre-
// computed once for all threads and passed to them, with the control thread co-ordinating the
// resulting data and sending stats etc. or something along those lines.
void RadiosityTask::Run()
{
	//RandomIntSequence rands(0, 2, 2047);
	//RandomIntSequence::Generator randgen(&rands);
	RandomDoubleSequence rands(-1.0, 1.0, 2047);
	RandomDoubleSequence::Generator randgen(&rands);

	DBL width = GetViewData()->GetWidth();
	DBL height = GetViewData()->GetHeight();

	POVRect rect;
	vector<Vector2d> pixelpositions;
	vector<Colour> pixelcolors;
	unsigned int serial;

	while(GetViewData()->GetNextRectangle(rect, serial) == true)
	{
		radiosity.BeforeTile(serial % nominalThreads);
		randgen.SetSeed(pretraceStep * 17 + serial * 13); // make sure our jitter is different (but reproducible) for each pass and tile

		unsigned int px = (rect.GetWidth()  + pretraceSize - 1) / pretraceSize;
		unsigned int py = (rect.GetHeight() + pretraceSize - 1) / pretraceSize;

		double startX   = ceil((DBL(rect.left)  - 0.5) / pretraceSize) * pretraceSize;
		double startY   = ceil((DBL(rect.top)   - 0.5) / pretraceSize) * pretraceSize;
		double endX     = DBL(rect.right)       + 0.5;
		double endY     = DBL(rect.bottom)      + 0.5;

		// make sure we start with the pixel buffer cleared
		pixelpositions.clear();
		pixelpositions.reserve(px * py);
		pixelcolors.clear();
		pixelcolors.reserve(px * py);

		double jitter = min(1.0, pretraceSize / 2.0);
		double offset = (pretraceSize - 1.0) / 2.0;
		for(DBL y = startY; y < endY; y += pretraceSize)
		{
			for(DBL x = startX; x < endX; x += pretraceSize)
			{
				Colour col;

				trace(x + offset + jitter * randgen(), y + offset + jitter * randgen(), width, height, col);
				GetViewDataPtr()->Stats()[Number_Of_Pixels]++;

				pixelpositions.push_back(Vector2d(x, y));
				pixelcolors.push_back(col);

				Cooperate();
			}
		}

		radiosity.AfterTile();

		GetViewDataPtr()->AfterTile();
		if(pixelpositions.size() > 0)
			GetViewData()->CompletedRectangle(rect, serial, pixelpositions, pixelcolors, int(ceil(pretraceSize)), false);
		else
			GetViewData()->CompletedRectangle(rect, serial);
	}
}

void RadiosityTask::Stopped()
{
	// nothing to do for now [trf]
}

void RadiosityTask::Finish()
{
	GetViewDataPtr()->timeType = SceneThreadData::kRadiosityTime;
	GetViewDataPtr()->realTime = ConsumedRealTime();
	GetViewDataPtr()->cpuTime = ConsumedCPUTime();
}

}
