/*******************************************************************************
 * ray.cpp
 *
 * This module implements the code pertaining to rays.
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/backend/render/ray.cpp $
 * $Revision: #2 $
 * $Change: 5047 $
 * $DateTime: 2010/06/30 07:58:31 $
 * $Author: thorsten $
 *******************************************************************************/

// frame.h must always be the first POV file included (pulls in platform config)
#include "backend/frame.h"
#include "backend/math/vector.h"
#include "povray.h"
#include "backend/interior/interior.h"
#include "backend/render/ray.h"
#include "backend/texture/texture.h"

// this must be the last file included
#include "base/povdebug.h"

namespace pov
{

Ray::Ray(RayType rt, bool shadowTest, bool photon, bool radiosity, bool monochromatic, bool pretrace)
{
	Make_Vector(Origin, 0.0, 0.0, 0.0);
	Make_Vector(Direction, 0.0, 0.0, 0.0);

	SetFlags(rt, shadowTest, photon, radiosity, monochromatic, pretrace);
	hollowRay = true;
	ClearInteriors();
}

Ray::Ray(const VECTOR ov, const VECTOR dv, RayType rt, bool shadowTest, bool photon, bool radiosity, bool monochromatic, bool pretrace)
{
	Assign_Vector(Origin, ov);
	Assign_Vector(Direction, dv);

	SetFlags(rt, shadowTest, photon, radiosity, monochromatic, pretrace);
	hollowRay = true;
	ClearInteriors();
}

Ray::~Ray()
{
}

void Ray::AppendInterior(Interior *i)
{
	interiors.push_back(i);
	hollowRay = hollowRay && i->hollow;
}

void Ray::AppendInteriors(RayInteriorVector& ii)
{
//	interiors.reserve(interiors.size() + ii.size()); // TODO FIXME - remove FixedSimpleVector from POV-Ray 3.7

	for(RayInteriorVector::iterator it(ii.begin()); it != ii.end(); it++)
	{
		interiors.push_back(*it);
		hollowRay = hollowRay && (*it)->hollow;
	}
}

bool Ray::RemoveInterior(const Interior *i)
{
	bool checkhollow = false;
	bool found = false;

	for(RayInteriorVector::iterator it(interiors.begin()); it != interiors.end(); it++)
	{
		if(*it == i)
		{
			checkhollow = ((*it)->hollow == false);
			interiors.erase(it);
			found = true;
			break;
		}
	}

	if(checkhollow == true)
	{
		hollowRay = true;

		for(RayInteriorVector::iterator it(interiors.begin()); it != interiors.end(); it++)
			hollowRay = hollowRay && (*it)->hollow;
	}

	return found;
}

bool Ray::IsInterior(const Interior *i) const
{
	for(RayInteriorVector::const_iterator it(interiors.begin()); it != interiors.end(); it++)
	{
		if(*it == i)
			return true;
	}

	return false;
}

void Ray::SetSpectralBand(const SpectralBand& sb)
{
	spectralBand = sb;
	monochromaticRay = true;
}

const SpectralBand& Ray::GetSpectralBand() const
{
	return spectralBand;
}

void Ray::SetFlags(RayType rt, bool shadowTest, bool photon, bool radiosity, bool monochromatic, bool pretrace)
{
	primaryRay = (rt == PrimaryRay);
	reflectionRay = (rt == ReflectionRay);
	refractionRay = (rt == RefractionRay);
	shadowTestRay = shadowTest;
	photonRay = photon;
	radiosityRay = radiosity;
	monochromaticRay = monochromatic;
	pretraceRay = pretrace;
}

void Ray::SetFlags(RayType rt, const Ray& other)
{
	primaryRay = (rt == PrimaryRay);
	reflectionRay = (rt == ReflectionRay);
	refractionRay = (rt == RefractionRay);
	shadowTestRay = other.IsShadowTestRay();
	photonRay = other.IsPhotonRay();
	radiosityRay = other.IsRadiosityRay();
	monochromaticRay = other.IsMonochromaticRay();
	pretraceRay = other.IsPretraceRay();
}

}
