/*******************************************************************************
 * tracepixel.cpp
 *
 * This module implements the TracePixel class, which mostly pertains to
 * setting up the camera, the initial ray for each pixel rendered, and
 * calling TraceRay() for said pixels. It also contains focal blur code,
 * as this is part of camera and ray setup.
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/backend/render/tracepixel.cpp $
 * $Revision: #2 $
 * $Change: 5047 $
 * $DateTime: 2010/06/30 07:58:31 $
 * $Author: thorsten $
 *******************************************************************************/

#include <vector>

#include <boost/scoped_array.hpp>

// frame.h must always be the first POV file included (pulls in platform config)
#include "backend/frame.h"
#include "backend/colour/colour.h"
#include "backend/math/vector.h"
#include "backend/math/chi2.h"
#include "backend/math/matrices.h"
#include "backend/render/trace.h"
#include "backend/render/tracepixel.h"
#include "backend/support/jitter.h"
#include "backend/texture/normal.h"

// this must be the last file included
#include "base/povdebug.h"

namespace pov
{

#ifdef DYNAMIC_HASHTABLE
extern unsigned short *hashTable; // GLOBAL VARIABLE
#else
extern ALIGN16 unsigned short hashTable[]; // GLOBAL VARIABLE
#endif

const int Grid1Size    = 4;
const int HexGrid2Size = 7;
const int HexGrid3Size = 19;
const int HexGrid4Size = 37;

// Grid size (n x n) used while jittering focal blur sub-pixel position.
const int SUB_PIXEL_GRID_SIZE = 16;

const TracePixel::VEC2 Grid1[Grid1Size] =
{
	{-0.25,  0.25},
	{ 0.25,  0.25},
	{-0.25, -0.25},
	{ 0.25, -0.25}
};

const DBL HexJitter2 = 0.144338;
const int HexGrid2Samples[2] = { 7, 0 };
const TracePixel::VEC2 HexGrid2[HexGrid2Size] =
{
	{-0.288675,  0.000000},
	{ 0.000000,  0.000000},
	{ 0.288675,  0.000000},
	{-0.144338,  0.250000},
	{-0.144338, -0.250000},
	{ 0.144338,  0.250000},
	{ 0.144338, -0.250000}
};

const DBL HexJitter3 = 0.096225;
const int HexGrid3Samples[4] = { 7, 6, 6, 0 };
const TracePixel::VEC2 HexGrid3[HexGrid3Size] =
{
	{-0.192450,  0.333333},
	{-0.192450, -0.333333},
	{ 0.192450,  0.333333},
	{ 0.192450, -0.333333},
	{ 0.384900,  0.000000},
	{-0.384900,  0.000000},
	{ 0.000000,  0.000000},

	{ 0.000000,  0.333333},
	{ 0.000000, -0.333333},
	{-0.288675,  0.166667},
	{-0.288675, -0.166667},
	{ 0.288675,  0.166667},
	{ 0.288675, -0.166667},

	{-0.096225,  0.166667},
	{-0.096225, -0.166667},
	{ 0.096225,  0.166667},
	{ 0.096225, -0.166667},
	{-0.192450,  0.000000},
	{ 0.192450,  0.000000}
};

const DBL HexJitter4 = 0.0721688;
const int HexGrid4Samples[9] = { 7, 6, 6, 4, 4, 4, 4, 2, 0 };
const TracePixel::VEC2 HexGrid4[HexGrid4Size] =
{
	{ 0.000000,  0.000000},
	{-0.216506,  0.375000},
	{ 0.216506, -0.375000},
	{-0.216506, -0.375000},
	{ 0.216506,  0.375000},
	{-0.433013,  0.000000},
	{ 0.433013,  0.000000},

	{-0.144338,  0.250000},
	{ 0.144338, -0.250000},
	{-0.144338, -0.250000},
	{ 0.144338,  0.250000},
	{-0.288675,  0.000000},
	{ 0.288675,  0.000000},

	{-0.072169,  0.125000},
	{ 0.072169, -0.125000},
	{-0.072169, -0.125000},
	{ 0.072169,  0.125000},
	{-0.144338,  0.000000},
	{ 0.144338,  0.000000},

	{-0.360844,  0.125000},
	{-0.360844, -0.125000},
	{ 0.360844,  0.125000},
	{ 0.360844, -0.125000},

	{-0.288675,  0.250000},
	{-0.288675, -0.250000},
	{ 0.288675,  0.250000},
	{ 0.288675, -0.250000},

	{-0.072169,  0.375000},
	{-0.072169, -0.375000},
	{ 0.072169,  0.375000},
	{ 0.072169, -0.375000},

	{-0.216506,  0.125000},
	{-0.216506, -0.125000},
	{ 0.216506,  0.125000},
	{ 0.216506, -0.125000},

	{ 0.000000,  0.250000},
	{ 0.000000, -0.250000},
};

inline int PseudoRandom(int v)
{
	return int(hashTable[int(v & 0x0fff)]);
}

TracePixel::TracePixel(ViewData *vd, TraceThreadData *td, unsigned int mtl, DBL adcb, unsigned int qf,
                       CooperateFunctor& cf, MediaFunctor& mf, RadiosityFunctor& af, bool pt) :
                       Trace(vd->GetSceneData(), td, qf, cf, mf, af),
                       sceneData(vd->GetSceneData()),
                       threadData(td),
                       focalBlurData(NULL),
                       maxTraceLevel(mtl),
                       adcBailout(adcb),
                       pretrace(pt)
{
	SetupCamera(vd->GetCamera());
}

TracePixel::~TracePixel()
{
	if(focalBlurData != NULL)
		delete focalBlurData;
}

void TracePixel::SetupCamera(const Camera& cam)
{
	bool normalise = false;
	camera = cam;
	useFocalBlur = false;
	precomputeContainingInteriors = true;
	cameraDirection = Vector3d(camera.Direction);
	cameraRight = Vector3d(camera.Right);
	cameraUp =  Vector3d(camera.Up);
	cameraLocation =  Vector3d(camera.Location);
	aspectRatio = 4.0 / 3.0;
	VLength(cameraLengthRight, *cameraRight);
	VLength(cameraLengthUp, *cameraUp);

	switch(camera.Type)
	{
		case CYL_1_CAMERA:
		case CYL_3_CAMERA:
			aspectRatio = cameraLengthUp;
			normalise = true;
			break;
		case CYL_2_CAMERA:
		case CYL_4_CAMERA:
			aspectRatio = cameraLengthRight;
			normalise = true;
			break;
		case ULTRA_WIDE_ANGLE_CAMERA:
			aspectRatio = cameraLengthUp / cameraLengthRight;
			normalise = true;
			break;
		case OMNIMAX_CAMERA:
		case FISHEYE_CAMERA:
			aspectRatio = cameraLengthRight / cameraLengthUp;
			normalise = true;
			break;
		default:
			aspectRatio = cameraLengthRight / cameraLengthUp;
			break;
	}

	if(normalise == true)
	{
		cameraRight.normalize();
		cameraUp.normalize();
		cameraDirection.normalize();
	}

	if(focalBlurData != NULL)
	{
		delete focalBlurData;
		focalBlurData = NULL;
	}

	// TODO: there is little point in calculating the grid separately for each thread.
	// since all threads in a given render must have identical grids, we should calculate
	// it once, and then duplicate the calculated data when starting up the threads.
	// (Possibly we could store it in the view).
	useFocalBlur = ((camera.Aperture != 0.0) && (camera.Blur_Samples > 0));
	if(useFocalBlur == true)
		focalBlurData = new FocalBlurData(camera);
}

void TracePixel::operator()(DBL x, DBL y, DBL width, DBL height, Colour& colour)
{
	if(useFocalBlur == false)
	{
		Ray ray;

		if(CreateCameraRay(ray, x, y, width, height, 0) == true)
		{
			colour.clear();
			Trace::TraceTicket ticket(maxTraceLevel, adcBailout, sceneData->outputAlpha && (sceneData->languageVersion < 370));
			TraceRay(ray, colour, 1.0, ticket, false);
		}
		else
			colour.set(0.0, 0.0, 0.0, 0.0, 1.0);
	}
	else
		TraceRayWithFocalBlur(colour, x, y, width, height);
}

bool TracePixel::CreateCameraRay(Ray& ray, DBL x, DBL y, DBL width, DBL height, size_t ray_number)
{
	DBL x0 = 0.0, y0 = 0.0;
	DBL cx, sx, cy, sy, ty, rad, phi;
	VECTOR V1;
	TRANSFORM Trans;

	// Move to center of pixel
	x += 0.5;
	y -= 0.5;

	// Set ray flags
	ray.SetFlags(Ray::PrimaryRay, false, false, false, false, pretrace);

	// Create primary ray according to the camera used. 
	Assign_Vector(ray.Origin, *cameraLocation);

	switch(camera.Type)
	{
		// Perspective projection (Pinhole camera; POV standard).
		case PERSPECTIVE_CAMERA:
			// Convert the x coordinate to be a DBL from -0.5 to 0.5. 
			x0 = x / width - 0.5;

			// Convert the y coordinate to be a DBL from -0.5 to 0.5. 
			y0 = ((height - 1) - y) / height - 0.5;

			// Create primary ray. 
			VLinComb3(ray.Direction, 1.0, *cameraDirection, x0, *cameraRight, y0, *cameraUp);

			// Do focal blurring (by Dan Farmer). 
			if(useFocalBlur)
			{
				JitterCameraRay(ray, x, y, ray_number);
				InitRayContainerState(ray, true);
			}
			else
				InitRayContainerState(ray);
			break;

		// Orthographic projection.
		case ORTHOGRAPHIC_CAMERA:
			// Convert the x coordinate to be a DBL from -0.5 to 0.5. 
			x0 = x / width - 0.5;

			// Convert the y coordinate to be a DBL from -0.5 to 0.5. 
			y0 = ((height - 1) - y) / height - 0.5;

			// Create primary ray. 
			Assign_Vector(ray.Direction, *cameraDirection);

			VLinComb3(ray.Origin, 1.0, *cameraLocation, x0, *cameraRight, y0, *cameraUp);

			if(useFocalBlur)
				JitterCameraRay(ray, x, y, ray_number);

			InitRayContainerState(ray, true);
			break;

		// Fisheye camera.
		case FISHEYE_CAMERA:
			// Convert the x coordinate to be a DBL from -1.0 to 1.0. 
			x0 = 2.0 * x / width - 1.0;

			// Convert the y coordinate to be a DBL from -1.0 to 1.0. 
			y0 = 2.0 * ((height - 1) - y) / height - 1.0;

			// This code would do what Warp wants 
			x0 *= cameraLengthRight;
			y0 *= cameraLengthUp;

			rad = sqrt(x0 * x0 + y0 * y0);

			// If the pixel lies outside the unit circle no ray is traced. 

			if(rad > 1.0)
				return false;

			if(rad == 0.0)
				phi = 0.0;
			else if(x0 < 0.0)
				phi = M_PI - asin(y0 / rad);
			else
				phi = asin(y0 / rad);

			// Get spherical coordinates. 
			x0 = phi;

			// Set vertical angle to half viewing angle. 
			y0 = rad * camera.Angle * M_PI_360;

			// Create primary ray. 
			cx = cos(x0);  sx = sin(x0);
			cy = cos(y0);  sy = sin(y0);

			VLinComb3(ray.Direction, cx * sy, *cameraRight, sx * sy, *cameraUp, cy, *cameraDirection);

			if(useFocalBlur)
				JitterCameraRay(ray, x, y, ray_number);

			InitRayContainerState(ray);
			break;

		// Omnimax camera.
		case OMNIMAX_CAMERA:
			// Convert the x coordinate to be a DBL from -1.0 to 1.0. 
			x0 = 2.0 * x / width - 1.0;

			// Convert the y coordinate to be a DBL from -1.0 to 1.0. 
			y0 = 2.0 * ((height - 1) - y) / height - 1.0;

			// Get polar coordinates. 
			if(aspectRatio > 1.0)
			{
				if(aspectRatio > 1.283458)
				{
					x0 *= aspectRatio/1.283458;
					y0 = (y0-1.0)/1.283458 + 1.0;
				}
				else
					y0 = (y0-1.0)/aspectRatio + 1.0;
			}
			else
				y0 /= aspectRatio;

			rad = sqrt(x0 * x0 + y0 * y0);

			// If the pixel lies outside the unit circle no ray is traced. 

			if(rad > 1.0)
				return false;

			if(rad == 0.0)
				phi = 0.0;
			else if (x0 < 0.0)
				phi = M_PI - asin(y0 / rad);
			else
				phi = asin(y0 / rad);

			// Get spherical coordinates. 
			x0 = phi;

			y0 = 1.411269 * rad - 0.09439 * rad * rad * rad + 0.25674 * rad * rad * rad * rad * rad;

			cx = cos(x0);  sx = sin(x0);
			cy = cos(y0);  sy = sin(y0);

			// We can't see below 45 degrees under the projection axis. 
			if (sx * sy < tan(135.0 * M_PI_180) * cy)
				return false;

			VLinComb3(ray.Direction, cx * sy, *cameraRight, sx * sy, *cameraUp, cy, *cameraDirection);

			if(useFocalBlur)
				JitterCameraRay(ray, x, y, ray_number);

			InitRayContainerState(ray);
			break;

		// Panoramic camera from Graphic Gems III.
		case PANORAMIC_CAMERA:
			// Convert the x coordinate to be a DBL from 0.0 to 1.0. 
			x0 = x / width;

			// Convert the y coordinate to be a DBL from -1.0 to 1.0. 
			y0 = 2.0 * ((height - 1) - y) / height - 1.0;

			// Get cylindrical coordinates. 
			x0 = (1.0 - x0) * M_PI;
			y0 = M_PI_2 * y0;

			cx = cos(x0);
			sx = sin(x0);

			if(fabs(M_PI_2 - fabs(y0)) < EPSILON)
			{
				if (y0 > 0.0)
					ty = BOUND_HUGE;
				else
					ty = - BOUND_HUGE;
			}
			else
				ty = tan(y0);

			// Create primary ray. 
			VLinComb3(ray.Direction, cx, *cameraRight, ty, *cameraUp, sx, *cameraDirection);

			if(useFocalBlur)
				JitterCameraRay(ray, x, y, ray_number);

			InitRayContainerState(ray);
			break;

		// Ultra wide angle camera written by Dan Farmer.
		case ULTRA_WIDE_ANGLE_CAMERA:
			// Convert the x coordinate to be a DBL from -0.5 to 0.5. 
			x0 = x / width - 0.5;

			// Convert the y coordinate to be a DBL from -0.5 to 0.5. 
			y0 = ((height - 1) - y) / height - 0.5;

			// Create primary ray. 
			x0 *= camera.Angle * M_PI_180; // NK 1998 - changed to M_PI_180 
			// 1999 July 10 Bugfix - as per suggestion of Gerald K. Dobiasovsky
			// added aspectRatio
			y0 *= camera.Angle * aspectRatio * M_PI_180; // NK 1998 - changed to M_PI_180 

			cx = cos(x0);  sx = sin(x0);
			cy = cos(y0);  sy = sin(y0);

			VLinComb3(ray.Direction, sx, *cameraRight, sy, *cameraUp, cx * cy, *cameraDirection);

			if(useFocalBlur)
				JitterCameraRay(ray, x, y, ray_number);

			InitRayContainerState(ray);
			break;

		// Cylinder camera 1. Axis in "up" direction
		case CYL_1_CAMERA:
			// Convert the x coordinate to be a DBL from -0.5 to 0.5. 
			x0 = x / width - 0.5;

			// Convert the y coordinate to be a DBL from -0.5 to 0.5. 
			y0 = ((height - 1) - y) / height - 0.5;

			// Create primary ray. 
			x0 *= camera.Angle * M_PI_180;
			y0 *= aspectRatio;

			cx = cos(x0);
			sx = sin(x0);

			VLinComb3(ray.Direction, sx, *cameraRight, y0, *cameraUp, cx, *cameraDirection);

			if(useFocalBlur)
				JitterCameraRay(ray, x, y, ray_number);

			InitRayContainerState(ray);
			break;

		// Cylinder camera 2. Axis in "right" direction
		case CYL_2_CAMERA:
			// Convert the x coordinate to be a DBL from -0.5 to 0.5. 
			x0 = x / width - 0.5;

			// Convert the y coordinate to be a DBL from -0.5 to 0.5. 
			y0 = ((height - 1) - y) / height - 0.5;

			y0 *= camera.Angle * M_PI_180;
			x0 *= aspectRatio;

			cy = cos(y0);
			sy = sin(y0);

			VLinComb3(ray.Direction, x0, *cameraRight, sy, *cameraUp, cy, *cameraDirection);

			if(useFocalBlur)
				JitterCameraRay(ray, x, y, ray_number);

			InitRayContainerState(ray);
			break;

		// Cylinder camera 3. Axis in "up" direction, orthogonal in "right"
		case CYL_3_CAMERA:
			// Convert the x coordinate to be a DBL from -0.5 to 0.5. 
			x0 = x / width - 0.5;

			// Convert the y coordinate to be a DBL from -0.5 to 0.5. 
			y0 = ((height - 1) - y) / height - 0.5;

			// Create primary ray. 
			x0 *= camera.Angle * M_PI_180;
			y0 *= aspectRatio;

			cx = cos(x0);
			sx = sin(x0);

			VLinComb2(ray.Direction, sx, *cameraRight, cx, *cameraDirection);

			VLinComb2(ray.Origin, 1.0, *cameraLocation, y0, *cameraUp);

			if(useFocalBlur)
				JitterCameraRay(ray, x, y, ray_number);

			InitRayContainerState(ray, true);
			break;

		// Cylinder camera 4. Axis in "right" direction, orthogonal in "up"
		case CYL_4_CAMERA:
			// Convert the x coordinate to be a DBL from -0.5 to 0.5. 
			x0 = x / width - 0.5;

			// Convert the y coordinate to be a DBL from -0.5 to 0.5. 
			y0 = ((height - 1) - y) / height - 0.5;

			// Create primary ray. 
			y0 *= camera.Angle * M_PI_180;
			x0 *= aspectRatio;

			cy = cos(y0);
			sy = sin(y0);

			VLinComb2(ray.Direction, sy, *cameraUp, cy, *cameraDirection);

			VLinComb2(ray.Origin, 1.0, *cameraLocation, x0, *cameraRight);

			if(useFocalBlur)
				JitterCameraRay(ray, x, y, ray_number);

			InitRayContainerState(ray, true);
			break;

		// spherical camera: x is horizontal, y vertical, V_Angle - vertical FOV, H_Angle - horizontal FOV
		case SPHERICAL_CAMERA:
			// Convert the x coordinate to be a DBL from -0.5 to 0.5. 
			x0 = x / width - 0.5;

			// Convert the y coordinate to be a DBL from -0.5 to 0.5. 
			y0 = ((height - 1) - y) / height - 0.5;

			// get angle in radians 
			y0 *= (camera.V_Angle / 360) * TWO_M_PI;
			x0 *= (camera.H_Angle / 360) * TWO_M_PI;

			// find latitude for y in 3D space 
			Compute_Axis_Rotation_Transform(&Trans, *cameraRight, -y0);
			MTransPoint (V1, *cameraDirection, &Trans);

			// Now take V1 and find longitude based on x 
			Compute_Axis_Rotation_Transform(&Trans, *cameraUp, x0);

			// Create primary ray. 
			MTransPoint(ray.Direction, V1, &Trans);

			if(useFocalBlur)
				JitterCameraRay(ray, x, y, ray_number);

			InitRayContainerState(ray);
			break;
		default:
			throw POV_EXCEPTION_STRING("Unknown camera type in create_ray().");
	}

	if(camera.Tnormal != NULL)
	{
		VNormalize(ray.Direction, ray.Direction);
		Make_Vector(V1, x0, y0, 0.0);
		Perturb_Normal(ray.Direction, camera.Tnormal, V1, NULL, NULL, threadData);
	}

	VNormalize(ray.Direction, ray.Direction);

	return true;
}

void TracePixel::InitRayContainerState(Ray& ray, bool compute)
{
	if((compute == true) || (precomputeContainingInteriors == true)) // TODO - check this logic, in particular that of compute!
	{
		precomputeContainingInteriors = false;
		containingInteriors.clear();

		if(sceneData->boundingMethod == 2)
		{
			HasInteriorPointObjectCondition precond;
			ContainingInteriorsPointObjectCondition postcond(containingInteriors);
			BSPInsideCondFunctor ifn(Vector3d(ray.Origin), sceneData->objects, threadData, precond, postcond);

			mailbox.clear();
			(*sceneData->tree)(Vector3d(ray.Origin), ifn, mailbox);

			// test infinite objects
			for(vector<ObjectPtr>::iterator object = sceneData->objects.begin() + sceneData->numberOfFiniteObjects; object != sceneData->objects.end(); object++)
				if(((*object)->interior != NULL) && Inside_BBox(ray.Origin, (*object)->BBox) && (*object)->Inside(ray.Origin, threadData))
					containingInteriors.push_back((*object)->interior);
		}
		else if((sceneData->boundingMethod == 0) || (sceneData->boundingSlabs == NULL))
		{
			for(vector<ObjectPtr>::iterator object = sceneData->objects.begin(); object != sceneData->objects.end(); object++)
				if(((*object)->interior != NULL) && Inside_BBox(ray.Origin, (*object)->BBox) && (*object)->Inside(ray.Origin, threadData))
					containingInteriors.push_back((*object)->interior);
		}
		else
		{
			InitRayContainerStateTree(ray, sceneData->boundingSlabs);
		}
	}

	ray.AppendInteriors(containingInteriors);
}

/*****************************************************************************
*
* METHOD
*
*   InitRayContainerStateTree
*
* AUTHOR
*
*   Dieter Bayer
*
* DESCRIPTION
*
*   Step down the bounding box hierarchy and test for all node wether
*   the ray's origin is inside or not. If it's inside a node descend
*   further. If a leaf is reached and the ray's origin is inside the
*   leaf object insert the objects data into the ray's containing lists.
*
* CHANGES
*
*   Mar 1996 : Creation.
*
******************************************************************************/

void TracePixel::InitRayContainerStateTree(Ray& ray, BBOX_TREE *node)
{
	/* Check current node. */
	if(!Inside_BBox(ray.Origin, node->BBox))
		return;
	if(node->Entries == 0)
	{
		/* This is a leaf so test contained object. */
		ObjectPtr object = ObjectPtr(node->Node);
		if((object->interior != NULL) && object->Inside(ray.Origin, threadData))
			containingInteriors.push_back(object->interior);
	}
	else
	{
		/* This is a node containing leaves to be checked. */
		for(int i = 0; i < node->Entries; i++)
			InitRayContainerStateTree(ray, node->Node[i]);
	}
}

void TracePixel::TraceRayWithFocalBlur(Colour& colour, DBL x, DBL y, DBL width, DBL height)
{
	int nr;     // Number of current samples. 
	int level;  // Index into number of samples list. 
	int max_s;  // Number of samples to take before next confidence test. 
	int dxi, dyi;
	int i;
	DBL dx, dy, n, randx, randy;
	Colour C, V1, S1, S2;
	int seed = int(x * 313.0 + 11.0) + int(y * 311.0 + 17.0);
	Ray ray;

	colour.clear();
	V1.clear();
	S1.clear();
	S2.clear();

	nr = 0;
	level = 0;

	do
	{
		// Trace number of rays given by the list Current_Number_Of_Samples[]. 
		max_s = 4;

		if(focalBlurData->Current_Number_Of_Samples != NULL)
		{
			if(focalBlurData->Current_Number_Of_Samples[level] > 0)
			{
				max_s = focalBlurData->Current_Number_Of_Samples[level];
				level++;
			}
		}

		for(i = 0; (i < max_s) && (nr < camera.Blur_Samples); i++)
		{
			// Choose sub-pixel location. 
			dxi = PseudoRandom(seed + nr) % SUB_PIXEL_GRID_SIZE;
			dyi = PseudoRandom(seed + nr + 1) % SUB_PIXEL_GRID_SIZE;

			dx = (DBL)(2 * dxi + 1) / (DBL)(2 * SUB_PIXEL_GRID_SIZE) - 0.5;
			dy = (DBL)(2 * dyi + 1) / (DBL)(2 * SUB_PIXEL_GRID_SIZE) - 0.5;

			Jitter2d(dx, dy, randx, randy);

			// Add jitter to sub-pixel location. 
			dx += (randx - 0.5) / (DBL)(SUB_PIXEL_GRID_SIZE);
			dy += (randy - 0.5) / (DBL)(SUB_PIXEL_GRID_SIZE);

			// remove interiors accumulated from previous iteration (if any)
			ray.ClearInteriors();

			// Create and trace ray.
			if(CreateCameraRay(ray, x + dx, y + dy, width, height, nr))
			{
				// Increase_Counter(stats[Number_Of_Samples]);

				C.clear();
				Trace::TraceTicket ticket(maxTraceLevel, adcBailout, sceneData->outputAlpha && (sceneData->languageVersion < 370));
				TraceRay(ray, C, 1.0, ticket, false);

				colour += C;
			}
			else
				C = Colour(0.0, 0.0, 0.0, 0.0, 1.0);

			// Add color to color sum. 

			S1[pRED]    += C[pRED];
			S1[pGREEN]  += C[pGREEN];
			S1[pBLUE]   += C[pBLUE];
			S1[pTRANSM] += C[pTRANSM];

			// Add color to squared color sum. 

			S2[pRED]    += Sqr(C[pRED]);
			S2[pGREEN]  += Sqr(C[pGREEN]);
			S2[pBLUE]   += Sqr(C[pBLUE]);
			S2[pTRANSM] += Sqr(C[pTRANSM]);

			nr++;
		}

		// Get variance of samples. 

		n = (DBL)nr;

		V1[pRED]    = (S2[pRED]    / n - Sqr(S1[pRED]    / n)) / n;
		V1[pGREEN]  = (S2[pGREEN]  / n - Sqr(S1[pGREEN]  / n)) / n;
		V1[pBLUE]   = (S2[pBLUE]   / n - Sqr(S1[pBLUE]   / n)) / n;
		V1[pTRANSM] = (S2[pTRANSM] / n - Sqr(S1[pTRANSM] / n)) / n;

		// Exit if samples are likely too be good enough. 

		if((V1[pRED]  < focalBlurData->Sample_Threshold[nr - 1]) && (V1[pGREEN]  < focalBlurData->Sample_Threshold[nr - 1]) &&
		   (V1[pBLUE] < focalBlurData->Sample_Threshold[nr - 1]) && (V1[pTRANSM] < focalBlurData->Sample_Threshold[nr - 1]))
			break;
	}
	while(nr < camera.Blur_Samples);

	colour /= (DBL)nr;
}

void TracePixel::JitterCameraRay(Ray& ray, DBL x, DBL y, size_t ray_number)
{
	DBL xjit, yjit, xlen, ylen, r;
	VECTOR temp_xperp, temp_yperp, deflection;

	r = camera.Aperture * 0.5;

	Jitter2d(x, y, xjit, yjit);
	xjit = focalBlurData->Max_Jitter * (((xjit + 0.5) * 2.0) - 1.0);
	yjit = focalBlurData->Max_Jitter * (((yjit + 0.5) * 2.0) - 1.0);

	xlen = r * (focalBlurData->Sample_Grid[ray_number].x + xjit);
	ylen = r * (focalBlurData->Sample_Grid[ray_number].y + yjit);

	// Deflect the position of the eye by the size of the aperture, and in
	// a direction perpendicular to the current direction of view.

	VScale(temp_xperp, focalBlurData->XPerp, xlen);
	VScale(temp_yperp, focalBlurData->YPerp, ylen);

	VSub(deflection, temp_xperp, temp_yperp);

	VAdd(ray.Origin, camera.Location, deflection);

	// Deflect the direction of the ray in the opposite direction we deflected
	// the eye position.  This makes sure that we are looking at the same place
	// when the distance from the eye is equal to "Focal_Distance".

	VScale(ray.Direction, ray.Direction, focalBlurData->Focal_Distance);
	VSub(ray.Direction, ray.Direction, deflection);

	VNormalize(ray.Direction, ray.Direction);
}

TracePixel::FocalBlurData::FocalBlurData(const Camera& camera)
{
	// Create list of thresholds for confidence test.
	Sample_Threshold = new DBL[camera.Blur_Samples];
	if(camera.Blur_Samples > 1)
	{
		DBL T1 = camera.Variance / chdtri((DBL)(camera.Blur_Samples-1), camera.Confidence);
		for(int i = 0; i < camera.Blur_Samples; i++)
			Sample_Threshold[i] = T1 * chdtri((DBL)(i + 1), camera.Confidence);
	}
	else
		Sample_Threshold[0] = 0.0;

	// Create list of sample positions. 
	Sample_Grid = new VEC2[camera.Blur_Samples];

	// Choose sample list and the best standard grid to use.

	// Default is 4x4 standard grid. 
	const VEC2 *Standard_Sample_Grid = &Grid1[0];
	int Standard_Sample_Grid_Size = 4;
	Current_Number_Of_Samples = NULL;

	// Check for 37 samples hexgrid. 
	if(camera.Blur_Samples >= HexGrid4Size)
	{
		Standard_Sample_Grid = &HexGrid4[0];
		Standard_Sample_Grid_Size = HexGrid4Size;
		Current_Number_Of_Samples = &HexGrid4Samples[0];
	}
	// Check for 19 samples hexgrid. 
	else if(camera.Blur_Samples >= HexGrid3Size)
	{
		Standard_Sample_Grid = &HexGrid3[0];
		Standard_Sample_Grid_Size = HexGrid3Size;
		Current_Number_Of_Samples = &HexGrid3Samples[0];
	}
	// Check for 7 samples hexgrid. 
	else if(camera.Blur_Samples >= HexGrid2Size)
	{
		Standard_Sample_Grid = &HexGrid2[0];
		Standard_Sample_Grid_Size = HexGrid2Size;
		Current_Number_Of_Samples = &HexGrid2Samples[0];
	}

	// Get max. jitter. 
	switch(camera.Blur_Samples)
	{
		case HexGrid2Size:
			Max_Jitter = HexJitter2;
			break;
		case HexGrid3Size:
			Max_Jitter = HexJitter3;
			break;
		case HexGrid4Size:
			Max_Jitter = HexJitter4;
			break;
		default:
			Max_Jitter = 1.0 / (2.0 * sqrt((DBL)camera.Blur_Samples));
			break;
	}

	// Copy standard grid to sample grid. 
	for(int i = 0; i < min(Standard_Sample_Grid_Size, camera.Blur_Samples); i++)
		Sample_Grid[i] = Standard_Sample_Grid[i];

	// Choose remaining samples from a uniform grid to get "best" coverage. 
	if(camera.Blur_Samples > Standard_Sample_Grid_Size)
	{
		// Get sub-pixel grid size (I want it to be odd). 
		int Grid_Size = (int)sqrt((DBL)camera.Blur_Samples) + 1;

		if((Grid_Size & 1) == 0)
			Grid_Size++;

		// Allocate temporary grid.
		boost::scoped_array<char> Grid_Data (new char [Grid_Size * Grid_Size]);
		char *p = Grid_Data.get();
		memset(p, 0, Grid_Size * Grid_Size);
		vector<char *> Grid(Grid_Size);
		for(int i = 0; i < Grid_Size; i++, p += Grid_Size)
			Grid[i] = p;

		// Mark sub-pixels already covered. 
		for(int i = 0; i < Standard_Sample_Grid_Size; i++)
		{
			int xi = (int)((Sample_Grid[i].x + 0.5) * (DBL)Grid_Size);
			int yi = (int)((Sample_Grid[i].y + 0.5) * (DBL)Grid_Size);
			Grid[yi][xi]++;
		}

		RandomIntSequence rands(0, Grid_Size - 1, camera.Blur_Samples * 2);
		RandomIntSequence::Generator randgen(&rands);

		// Distribute remaining samples. 
		for(int i = Standard_Sample_Grid_Size; i < camera.Blur_Samples; i++)
		{
			int xi = randgen(); // NOTE: Grid size limited in random number generator!
			int yi = randgen(); // NOTE: Grid size limited in random number generator!

			// just find a close unset one if the random one did not find one right away
			while(Grid[yi][xi])
			{
				xi = ((xi + 1) % Grid_Size);
				int ycount = Grid_Size ;
				while(Grid[yi][xi] && ycount--)
					yi = ((yi + 1) % Grid_Size);
			}

			Sample_Grid[i].x = (DBL)(2 * xi + 1) / (DBL)(2 * Grid_Size) - 0.5;
			Sample_Grid[i].y = (DBL)(2 * yi + 1) / (DBL)(2 * Grid_Size) - 0.5;

			Grid[yi][xi]++;
		}
	}

	// Calculate vectors perpendicular to the current ray
	// We're making a "+" (crosshair) on the film plane.

	// XPerp = vector perpendicular to y/z plane 
	VCross(XPerp, camera.Up, camera.Direction);
	VNormalize(XPerp, XPerp);

	// YPerp = vector perpendicular to x/z plane 
	VCross(YPerp, camera.Direction, XPerp);
	VNormalize(YPerp, YPerp);

	// Get adjusted distance to focal plane. 
	DBL len;
	VLength(len, camera.Direction);
	Focal_Distance = camera.Focal_Distance / len;
}

TracePixel::FocalBlurData::~FocalBlurData()
{
	delete[] Sample_Grid;
}

}
