/*******************************************************************************
 * octree.h
 *
 * Oct-tree routine prototypes.  Use by Radiosity calculation routies.
 *
 * Implemented by and (c) 1994 Jim McElhiney, mcelhiney@acm.org or cserve 71201,1326
 * All standard POV distribution rights granted.  All other rights reserved.
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/backend/support/octree.h $
 * $Revision: #2 $
 * $Change: 5047 $
 * $DateTime: 2010/06/30 07:58:31 $
 * $Author: thorsten $
 *******************************************************************************/

#ifndef OCTREE_H
#define OCTREE_H

namespace pov
{
using namespace pov_base;

/*****************************************************************************
* Global preprocessor defines
******************************************************************************/

#define OT_BIAS 10000000.

#define MAX3(a,b,c) ( ((a)>(b)) ? max((a),(c)) : max((b),(c)) )


/*****************************************************************************
* Global typedefs
******************************************************************************/

typedef unsigned short OT_TILE;
#define OT_TILE_MAX USHRT_MAX

typedef unsigned char OT_PASS;
#define OT_PASS_FIRST 1 // 0 is reserved
#define OT_PASS_FINAL UCHAR_MAX
#define OT_PASS_MAX   (OT_PASS_FINAL-2) // OT_PASS_FINAL-1 is reserved

typedef unsigned char OT_DEPTH;
#define OT_DEPTH_MAX UCHAR_MAX

typedef struct ot_block_struct OT_BLOCK;
typedef struct ot_id_struct OT_ID;
typedef struct ot_node_struct OT_NODE;
typedef struct ot_read_param_struct OT_READ_PARAM;
typedef struct ot_read_info_struct OT_READ_INFO;

// Each node in the oct-tree has a (possibly null) linked list of these data blocks off it.
struct ot_block_struct
{
	// TODO for memory efficiency we could probably use single-precision data types for the vector stuff
	OT_BLOCK          *next;      // next block in the same node
	Vector3d          Point;
	Vector3d          S_Normal;
	Vector3d          To_Nearest_Surface;
	RGBColour         dx, dy, dz; // gradients, not colors, but used only to manipulate colors [trf]
	RGBColour         Illuminance;
	SNGL              Harmonic_Mean_Distance;
	SNGL              Nearest_Distance;
	OT_TILE           TileId;     // tile in which this sample was taken
	OT_PASS           Pass;       // pass during which this sample was taken (OT_PASS_FINAL for final render)
	OT_DEPTH          Bounce_Depth;
};

// This is the information necessary to name an oct-tree node.
struct ot_id_struct
{
	int x, y, z;
	int Size;
};

// These are the structures that make up the oct-tree itself, known as nodes
struct ot_node_struct
{
	OT_ID    Id;
	OT_BLOCK *Values;
	OT_NODE  *Kids[8];
};

// These are informations the octree reader needs to know
struct ot_read_param_struct
{
	DBL       RealErrorBound;
};

// These are informations the octree reader generates
struct ot_read_info_struct
{
	RGBColour Gather_Total;
	long      Gather_Total_Count;
	DBL       Brightness;
	bool      FirstRadiosityPass;
};

/*****************************************************************************
* Global functions
******************************************************************************/

void ot_ins (OT_NODE **root, OT_BLOCK *new_block, OT_ID *new_id);
bool ot_dist_traverse (OT_NODE *subtree, const Vector3d& point, int bounce_depth, bool (*func)(OT_BLOCK *block, void *handle1), void *handle2);
void ot_index_sphere (const Vector3d& point, DBL radius, OT_ID *id);
void ot_index_box (const Vector3d& min_point, const Vector3d& max_point, OT_ID *id);
bool ot_save_tree (OT_NODE *root, OStream *fd);
bool ot_write_block (OT_BLOCK *bl, void * handle);
bool ot_free_tree (OT_NODE **root_ptr);
bool ot_read_file (OT_NODE **root, IStream * fd, const OT_READ_PARAM* param, OT_READ_INFO* info);
void ot_newroot (OT_NODE **root_ptr);
void ot_parent (OT_ID *dad, OT_ID *kid);

}

#endif
