/*******************************************************************************
 * randomsequences.cpp
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/backend/support/randomsequences.cpp $
 * $Revision: #2 $
 * $Change: 5047 $
 * $DateTime: 2010/06/30 07:58:31 $
 * $Author: thorsten $
 *******************************************************************************/

#include <cassert>
#include <stdexcept>

#include <boost/random/mersenne_twister.hpp>
#include <boost/random/uniform_int.hpp>
#include <boost/random/uniform_real.hpp>
#include <boost/random/uniform_on_sphere.hpp>
#include <boost/random/variate_generator.hpp>

// frame.h must always be the first POV file included (pulls in platform config)
#include "backend/frame.h"
#include "backend/support/randomsequences.h"

// this must be the last file included
#include "base/povdebug.h"

namespace pov
{

using namespace pov_base;

using boost::uniform_int;
using boost::uniform_real;
using boost::variate_generator;
using boost::mt19937;

vector<int> RandomInts(int minval, int maxval, size_t count)
{
	mt19937 generator;
	uniform_int<int> distribution(minval, maxval);
	variate_generator<mt19937, uniform_int<int> > sequence(generator, distribution);
	vector<int> rands(count);

	for(size_t i = 0; i < count; i++)
		rands[i] = sequence();

	return rands;
}

vector<double> RandomDoubles(double minval, double maxval, size_t count)
{
	mt19937 generator;
	uniform_real<double> distribution(minval, maxval);
	variate_generator<mt19937, uniform_real<double> > sequence(generator, distribution);
	vector<double> rands(count);

	for(size_t i = 0; i < count; i++)
		rands[i] = sequence();

	return rands;
}

//vector<Vector3d> RandomSphere(size_t count)
//{
// Use boost::uniform_on_sphere! [trf]
//}

RandomIntSequence::RandomIntSequence(int minval, int maxval, size_t count) :
	values(RandomInts(minval, maxval, count))
{
}

RandomIntSequence::Generator::Generator(RandomIntSequence *seq, size_t seedindex) :
	sequence(seq),
	index(seedindex)
{
}

int RandomIntSequence::operator()(size_t seedindex)
{
	seedindex = seedindex % values.size();
	return values[seedindex];
}

int RandomIntSequence::Generator::operator()()
{
	index = (index + 1) % sequence->values.size();
	return (*sequence)(index);
}

int RandomIntSequence::Generator::operator()(size_t seedindex)
{
	return (*sequence)(seedindex);
}

size_t RandomIntSequence::Generator::GetSeed()
{
	return index;
}

void RandomIntSequence::Generator::SetSeed(size_t seedindex)
{
	index = seedindex % sequence->values.size();
}

RandomDoubleSequence::RandomDoubleSequence(double minval, double maxval, size_t count) :
	values(RandomDoubles(minval, maxval, count))
{
}

RandomDoubleSequence::Generator::Generator(RandomDoubleSequence *seq, size_t seedindex) :
	sequence(seq),
	index(seedindex)
{
}

double RandomDoubleSequence::operator()(size_t seedindex)
{
	seedindex = seedindex % values.size();
	return values[seedindex];
}

double RandomDoubleSequence::Generator::operator()()
{
	index = (index + 1) % sequence->values.size();
	return (*sequence)(index);
}

double RandomDoubleSequence::Generator::operator()(size_t seedindex)
{
	return (*sequence)(seedindex);
}

size_t RandomDoubleSequence::Generator::GetSeed()
{
	return index;
}

void RandomDoubleSequence::Generator::SetSeed(size_t seedindex)
{
	index = seedindex % sequence->values.size();
}

}
