/*******************************************************************************
 * statistics.cpp
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/backend/support/statistics.cpp $
 * $Revision: #2 $
 * $Change: 5047 $
 * $DateTime: 2010/06/30 07:58:31 $
 * $Author: thorsten $
 *******************************************************************************/

#include <vector>
#include <cassert>

// frame.h must always be the first POV file included (pulls in platform config)
#include "backend/frame.h"
#include "base/types.h"
#include "backend/support/statistics.h"

// this must be the last file included
#include "base/povdebug.h"

namespace pov
{

const unsigned int kMaxReadAttempts = 10;

template <typename T>
bool Counter<T>::SafeRead(unsigned int maxattempts, T *result) const
{
	// Use pointers to the value because even though it is declared
	// as volatile, we cannot trust all compilers enough to not
	// optimize the accesses!
	const volatile T *p1 = &value;
	const volatile T *p2 = &value;

	do
	{
		T val1 = *p1;
		T val2 = *p2;

		if (val2 == val1)
		{
			*result = val2;
			return true;
		}
	} while (maxattempts--);

	return false;
}

template <typename T, int numElem>
void StatisticsBase<T, numElem>::operator+=(const StatisticsBase<T, numElem>& other)
{
	for (int i = 0; i < numElem; i++)
	{
		T temp;
		other.counters[i].SafeRead(kMaxReadAttempts, &temp);
		counters[i] += temp;
	}
}

template <typename T, int numElem>
StatisticsBase<T, numElem> StatisticsBase<T, numElem>::operator+(const StatisticsBase<T, numElem>& other)
{
	StatisticsBase<T, numElem> result(*this);

	result += other;
	return result;
}

template <typename T, int numElem>
void StatisticsBase<T, numElem>::clear()
{
	for (int i = 0; i < numElem; i++)
		counters[i] = 0; // assumes this is a valid operation for type T
}

template class StatisticsBase<POV_ULONG, MaxIntStat>;
template class StatisticsBase<double, MaxFPStat>;

}
