/*******************************************************************************
 * task.h
 *
 * This file contains ... TODO ...
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/backend/support/task.h $
 * $Revision: #3 $
 * $Change: 5050 $
 * $DateTime: 2010/06/30 12:23:03 $
 * $Author: thorsten $
 *******************************************************************************/

#ifndef POVRAY_BACKEND_TASK_H
#define POVRAY_BACKEND_TASK_H

#include <vector>
#include <queue>

#include <boost/function.hpp>

#include "backend/frame.h"

#include "base/timer.h"
#include "base/pov_err.h"
#include "base/thread.h"

namespace pov
{

using namespace pov_base;

class Task;

class Task
{
		struct StopThreadException { };
	public:
		class TaskData
		{
				friend class Task;
			public:
				TaskData() { }
				virtual ~TaskData() { }
			private:
				Task *task;
		};

		Task(TaskData *td, const boost::function1<void, Exception&> f);
		virtual ~Task();

		inline bool IsPaused() { return !done && paused; }
		inline bool IsRunning() { return !done && !paused && !stopRequested && (taskThread != NULL); }
		inline bool IsDone() { return done; }
		inline bool Failed() { return done && (failed != kNoError); }

		int FailureCode(int defval = kNoError);

		POV_LONG ConsumedRealTime() const;
		POV_LONG ConsumedCPUTime() const;

		void Start();
		void RequestStop();
		void Stop();
		void Pause();
		void Resume();

		inline void Cooperate()
		{
			if(stopRequested == true)
				throw StopThreadException();
			else if(paused == true)
			{
				while(paused == true)
				{
					Thread::Yield();
					if(stopRequested == true)
						throw StopThreadException();
				}
			}
		}

		inline static void CurrentTaskCooperate()
		{
			void *ptr = Thread::GetCurrentThread().GetThreadDataPtr();

			if(ptr != NULL)
				reinterpret_cast<Task *>(ptr)->Cooperate();
		}

		inline TaskData *GetDataPtr() { return taskData; }

		inline POVMSContext GetPOVMSContext() { return povmsContext; }
	protected:
		virtual void Run() = 0;
		virtual void Stopped() = 0;
		virtual void Finish() = 0;

		POV_LONG ElapsedRealTime() const;
		POV_LONG ElapsedCPUTime() const;
	private:
        /// task data pointer
		TaskData *taskData;
		/// task fatal error handler
		boost::function1<void, Exception&> fatalErrorHandler;
		/// stop request flag
		volatile bool stopRequested;
		/// paused flag
		volatile bool paused;
		/// done flag
		volatile bool done;
		/// failed code
		volatile int failed;
		// pointer to timer or NULL
		Timer *timer;
		// real time spend in task
		POV_LONG realTime;
		// CPU time spend in task
		POV_LONG cpuTime;
		/// task thread
		Thread *taskThread;
		/// POVMS message receiving context
		POVMSContext povmsContext;

		inline void FatalErrorHandler(const Exception& e)
		{
			Exception pe(e);
			fatalErrorHandler(pe);
		}

		inline void FatalErrorHandler(Exception& e) { fatalErrorHandler(e); }

		/// not available
		Task();

		/// not available
		Task(const Task&);

		/// not available
		Task& operator=(const Task&);

		void TaskThread();

		static void TaskDataCleanup(Task::TaskData *) { }
};

}

#endif // POVRAY_BACKEND_TASK_H
