/*******************************************************************************
 * image.cpp
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/base/image/image.cpp $
 * $Revision: #3 $
 * $Change: 5050 $
 * $DateTime: 2010/06/30 12:23:03 $
 * $Author: thorsten $
 *******************************************************************************/

#include <vector>
#include <algorithm>
#include <cassert>

// configbase.h must always be the first POV file included within base *.cpp files
#include "base/configbase.h"

#include "base/image/image.h"
#include "base/image/targa.h"
#include "base/image/iff.h"
#include "base/image/pgm.h"
#include "base/image/ppm.h"
#include "base/image/bmp.h"

#include "base/platformbase.h"

#ifdef USE_SYSPROTO
#include "syspovprotobase.h"
#endif

#define CHECK_BOUNDS(x,y) assert(((x) < width) && ((y) < height))

#define ALPHA_OPAQUE                (1.0f)
#define ALPHA_OPAQUE_INT(MAX)       (MAX)

#define FT_OPAQUE                   (0.0f)
#define FT_OPAQUE_INT(MAX)          (0)

#define IS_NONZERO_RGB(r,g,b)       ((r)*(g)*(b) != 0.0f) // TODO FIXME - [CLi] this tests whether *all* channels are nonzero - is this desired?
#define IS_NONZERO_RGB_INT(r,g,b)   ((r)*(g)*(b) != 0)    // TODO FIXME - [CLi] this tests whether *all* channels are nonzero - is this desired?

// this must be the last file included
#include "base/povdebug.h"

namespace pov_base
{

// definitions of static GammaCurve member variables to satisfy the linker
list<boost::weak_ptr<GammaCurve> > GammaCurve::cache;
Mutex GammaCurve::cacheMutex;

// definitions of static GammaCurve-derivatives' member variables to satisfy the linker
GammaCurvePtr NeutralGammaCurve::instance;
GammaCurvePtr SRGBGammaCurve::instance;
GammaCurvePtr ITURBT709GammaCurve::instance;
GammaCurvePtr Rec1361GammaCurve::instance;

float* GammaCurve::GetLookupTable(unsigned int max)
{
	assert(max == 255 || max == 65535); // shouldn't happen, but it won't hurt to check in debug versions

    // Get a reference to the lookup table pointer we're dealing with, so we don't need to duplicate all the remaining code.
    float*& lookupTable = (max == 255 ? lookupTable8 : lookupTable16);

    // Make sure we're not racing any other thread that might currently be busy creating the LUT.
    Mutex::ScopedLock lock(lutMutex);

    // Create the LUT if it doesn't exist yet.
	if (!lookupTable)
	{
        float* tempTable = new float[max+1];
		for (unsigned int i = 0; i <= max; i ++)
			tempTable[i] = Decode(IntDecode(i, max));

        // hook up the table only as soon as it is completed, so that querying the table does not need to
		// care about thread-safety.
        lookupTable = tempTable;
	}

	return lookupTable;
}

GammaCurvePtr GammaCurve::GetMatching(GammaCurvePtr newInstance)
{
	GammaCurvePtr oldInstance;
	bool cached = false;

    // If new instance is a neutral gamma curve, return /the/ instance of /the/ neutral gamma curve instead.
	if (newInstance->IsNeutral())
		return NeutralGammaCurve::Get();

    // See if we have a matching gamma curve in our chache already

    // make sure the cache doesn't get tampered with while we're working on it
    Mutex::ScopedLock lock(cacheMutex);

    // Check if we already have created a matching gamma curve object; if so, return that object instead.
	// Also, make sure we get the new object stored (as we're using weak pointers, we may have stale entries;
	// it also won't hurt if we store the new instance, even if we decide to discard it)
	for(list<boost::weak_ptr<GammaCurve> >::iterator i(cache.begin()); i != cache.end(); i++)
	{
		oldInstance = (*i).lock();
		if (!oldInstance)
		{
            // Found a stale entry in the cache where we could store the new instance, in case we don't find any match.
            // As the cache uses weak pointers, we can just as well store the new instance now right away,
            // and leave it up to the weak pointer mechanism to clean up in case we find an existing instance.
			if (!cached)
				(*i) = newInstance;
			cached = true;
		}
		else if (oldInstance->Matches(newInstance))
		{
            // Found a matching curve in the cache, so use that instead, and (as far as we're concerned)
            // just forget that the new instance ever existed (allowing the shared_ptr mechanism to garbage-collect it)
			return oldInstance;
		}
	}

    // No matching gamma curve in the cache yet

    // Store the new entry in the cache if we haven't done so already.
    if (!cached)
		cache.push_back(newInstance);

    return newInstance;
}

NeutralGammaCurve::NeutralGammaCurve() {}
float NeutralGammaCurve::Encode(float x) const { return x; }
float NeutralGammaCurve::Decode(float x) const { return x; }
float NeutralGammaCurve::ApproximateEncodingGamma() const { return 1.0f; }
bool NeutralGammaCurve::Matches(const GammaCurvePtr& p) const { return GammaCurve::IsNeutral(p); }
bool NeutralGammaCurve::IsNeutral() const { return true; }

SRGBGammaCurve::SRGBGammaCurve() {}
float SRGBGammaCurve::Encode(float x) const
{
	// (the threshold of 0.00304 occasionally found on the net was from an older draft)
	if (x <= 0.0031308f) return x * 12.92f;
	else                 return 1.055f * pow(x, 1.0f/2.4f) - 0.055f;
}
float SRGBGammaCurve::Decode(float x) const
{
	// (the threshold of 0.03928 occasionally found on the net was from an older draft)
	if (x < 0.04045f) return x / 12.92f;
	else              return pow((x + 0.055f) / 1.055f, 2.4f);
}
float SRGBGammaCurve::ApproximateEncodingGamma() const { return 1.0f/2.2f; }

ITURBT709GammaCurve::ITURBT709GammaCurve() {}
float ITURBT709GammaCurve::Encode(float x) const
{
	if (x < 0.018f) return x * 4.5f;
	else            return 1.099f * pow(x, 0.45f) - 0.099f;
}
float ITURBT709GammaCurve::Decode(float x) const
{
	if (x < 0.081f) return x / 4.5f;
	else            return pow((x + 0.099f) / 1.099f, 1.0f/0.45f);
}
float ITURBT709GammaCurve::ApproximateEncodingGamma() const { return 1.0f/1.9f; } // very rough approximation

Rec1361GammaCurve::Rec1361GammaCurve() {}
float Rec1361GammaCurve::Encode(float x) const
{
	if      (x < -0.0045f) return (1.099f * pow(-4*x, 0.45f) - 0.099f) / 4;
	else if (x <  0.018f)  return x * 4.5f;
	else                   return 1.099f * pow(x,0.45f) - 0.099f;
}
float Rec1361GammaCurve::Decode(float x) const
{
	if      (x < -0.02025f) return pow((4*x + 0.099f) / 1.099f, 1.0f/0.45f) / -4;
	else if (x <  0.081f)   return x / 4.5f;
	else                    return pow((x + 0.099f) / 1.099f, 1.0f/0.45f);
}
float Rec1361GammaCurve::ApproximateEncodingGamma() const { return 1.0f/1.9f; } // very rough approximation of the x>0 section

PowerLawGammaCurve::PowerLawGammaCurve(float gamma) : encGamma(gamma) {}
GammaCurvePtr PowerLawGammaCurve::GetByEncodingGamma(float gamma) { return GetMatching(GammaCurvePtr(new PowerLawGammaCurve(gamma))); }
GammaCurvePtr PowerLawGammaCurve::GetByDecodingGamma(float gamma) { return GetMatching(GammaCurvePtr(new PowerLawGammaCurve(1.0f/gamma))); }
float PowerLawGammaCurve::Encode(float x) const { return pow(max(x,0.0f), encGamma); }
float PowerLawGammaCurve::Decode(float x) const { return pow(max(x,0.0f), 1.0f/encGamma); }
float PowerLawGammaCurve::ApproximateEncodingGamma() const { return encGamma; }
bool PowerLawGammaCurve::Matches(const GammaCurvePtr& p) const
{
	PowerLawGammaCurve* other = dynamic_cast<PowerLawGammaCurve*>(p.get());
	if (!other) return false;
	return IsNeutral(this->encGamma / other->encGamma);
}
bool PowerLawGammaCurve::IsNeutral() const { return IsNeutral(this->encGamma); }
bool PowerLawGammaCurve::IsNeutral(float gamma) { return fabs(1.0 - gamma) <= 0.01; }

ScaledGammaCurve::ScaledGammaCurve(GammaCurvePtr gamma, float factor) : baseGamma(gamma), encFactor(factor)
{
	ScaledGammaCurve* other = dynamic_cast<ScaledGammaCurve*>(baseGamma.get());
	if (other) // if base gamma curve is a scaled one as well, compute a combined scaling factor instead of nesting
	{
		baseGamma = other->baseGamma;
		encFactor *= other->encFactor;
	}
}
GammaCurvePtr ScaledGammaCurve::GetByEncoding(GammaCurvePtr gamma, float factor) { return GetMatching(GammaCurvePtr(new ScaledGammaCurve(gamma, factor))); }
GammaCurvePtr ScaledGammaCurve::GetByDecoding(float factor, GammaCurvePtr gamma) { return GetMatching(GammaCurvePtr(new ScaledGammaCurve(gamma, 1.0f/factor))); }
float ScaledGammaCurve::Encode(float x) const { if (baseGamma) return baseGamma->Encode(x) * encFactor; else return x * encFactor; }
float ScaledGammaCurve::Decode(float x) const { if (baseGamma) return baseGamma->Decode(x / encFactor); else return x / encFactor; }
float ScaledGammaCurve::ApproximateEncodingGamma() const { if (baseGamma) return baseGamma->ApproximateEncodingGamma(); else return 1.0; }
bool ScaledGammaCurve::Matches(const GammaCurvePtr& p) const
{
	ScaledGammaCurve* other = dynamic_cast<ScaledGammaCurve*>(p.get());
	if (!other) return false;
	return (this->baseGamma == other->baseGamma) && IsNeutral(this->encFactor / other->encFactor);
}
bool ScaledGammaCurve::IsNeutral() const { return (!this->baseGamma) && IsNeutral(this->encFactor); }
bool ScaledGammaCurve::IsNeutral(float scale) { return fabs(1.0 - scale) <= 1e-6; }


template<class Allocator = allocator<bool> >
class BitMapImage : public Image
{
	public:
		BitMapImage(unsigned int w, unsigned int h) :
			Image(w, h, Bit_Map) { pixels.resize(w * h); FillBitValue(false); }
		BitMapImage(unsigned int w, unsigned int h, const vector<RGBMapEntry>& m) :
			Image(w, h, Bit_Map, m) { pixels.resize(w * h); FillBitValue(false); }
		BitMapImage(unsigned int w, unsigned int h, const vector<RGBAMapEntry>& m) :
			Image(w, h, Bit_Map, m) { pixels.resize(w * h); FillBitValue(false); }
		BitMapImage(unsigned int w, unsigned int h, const vector<RGBFTMapEntry>& m) :
			Image(w, h, Bit_Map, m) { pixels.resize(w * h); FillBitValue(false); }
		~BitMapImage() { }

		bool IsOpaque() const
		{
			return true;
		}
		bool IsGrayscale() const
		{
			return true;
		}
		bool IsColour() const
		{
			return false;
		}
		bool IsFloat() const
		{
			return false;
		}
		bool IsInt() const
		{
			return true;
		}
		bool IsIndexed() const
		{
			return false;
		}
		bool IsGammaEncoded() const
		{
			return false;
		}
		bool HasAlphaChannel() const
		{
			return false;
		}
		bool HasFilterTransmit() const
		{
			return false;
		}
		unsigned int GetMaxIntValue() const
		{
			return 1;
		}
		bool TryDeferDecoding(GammaCurvePtr&, unsigned int)
		{
			return false;
		}

		bool GetBitValue(unsigned int x, unsigned int y) const
		{
			CHECK_BOUNDS(x, y);
			return pixels[x + y * width];
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			CHECK_BOUNDS(x, y);
			if(pixels[x + y * width] == true)
				return 1.0f;
			else
				return 0.0f;
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			gray = GetGrayValue(x, y);
			alpha = ALPHA_OPAQUE;
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			red = green = blue = GetGrayValue(x, y);
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			red = green = blue = GetGrayValue(x, y);
			alpha = ALPHA_OPAQUE;
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			red = green = blue = GetGrayValue(x, y);
			filter = transm = FT_OPAQUE;
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = bit;
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = (gray != 0.0f);
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = (gray != 0);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float)
		{
            // TODO FIXME - [CLi] This ignores opacity information; other bit-based code doesn't.
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = (gray != 0.0f);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int)
		{
            // TODO FIXME - [CLi] This ignores opacity information; other bit-based code doesn't.
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = (gray != 0);
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = IS_NONZERO_RGB(red, green, blue);
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = IS_NONZERO_RGB_INT(red, green, blue);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float)
		{
            // TODO FIXME - [CLi] This ignores opacity information; other bit-based code doesn't.
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int)
		{
            // TODO FIXME - [CLi] This ignores opacity information; other bit-based code doesn't.
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float, float)
		{
            // TODO FIXME - [CLi] This ignores opacity information; other bit-based code doesn't.
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
            // TODO FIXME - [CLi] This ignores opacity information; other bit-based code doesn't.
			SetRGBValue(x, y, col.red(), col.green(), col.blue());
		}

		void FillBitValue(bool bit)
		{
			fill(pixels.begin(), pixels.end(), bit);
		}
		void FillGrayValue(float gray)
		{
			FillBitValue(gray != 0.0f);
		}
		void FillGrayValue(unsigned int gray)
		{
			FillBitValue(gray != 0);
		}
		void FillGrayAValue(float gray, float)
		{
			FillBitValue(gray != 0.0f);
		}
		void FillGrayAValue(unsigned int gray, unsigned int)
		{
			FillBitValue(gray != 0);
		}
		void FillRGBValue(float red, float green, float blue)
		{
			FillBitValue(IS_NONZERO_RGB(red, green, blue));
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
			FillBitValue(IS_NONZERO_RGB_INT(red, green, blue));
		}
		void FillRGBAValue(float red, float green, float blue, float)
		{
            // TODO FIXME - [CLi] This ignores opacity information; other bit-based code doesn't.
			FillRGBValue(red, green, blue);
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int)
		{
            // TODO FIXME - [CLi] This ignores opacity information; other bit-based code doesn't.
			FillRGBValue(red, green, blue);
		}
		void FillRGBFTValue(float red, float green, float blue, float, float)
		{
            // TODO FIXME - [CLi] This ignores opacity information; other bit-based code doesn't.
			FillRGBValue(red, green, blue);
		}
	private:
		vector<bool, Allocator> pixels;
};

typedef BitMapImage<> MemoryBitMapImage;

template<class Allocator = allocator<unsigned char> >
class ColourMapImage : public Image
{
	public:
		ColourMapImage(unsigned int w, unsigned int h, const vector<RGBMapEntry>& m) :
			Image(w, h, Colour_Map, m) { pixels.resize(w * h); FillBitValue(false); }
		ColourMapImage(unsigned int w, unsigned int h, const vector<RGBAMapEntry>& m) :
			Image(w, h, Colour_Map, m) { pixels.resize(w * h); FillBitValue(false); }
		ColourMapImage(unsigned int w, unsigned int h, const vector<RGBFTMapEntry>& m) :
			Image(w, h, Colour_Map, m) { pixels.resize(w * h); FillBitValue(false); }
		~ColourMapImage() { }

		bool IsOpaque() const
		{
			if((colormaptype != RGBAColourMap) && (colormaptype != RGBFTColourMap))
				return true;

			bool transp = false;

			for(size_t i = 0; i < colormap.size(); i++)
			{
				if(colormaptype == RGBAColourMap)
					transp = (colormap[i].filter < 1.0); // with RGBAColourMap, .filter is actually alpha
				else if(colormaptype == RGBFTColourMap)
					transp = (colormap[i].transm > 0.0);
				if(transp == true)
					break;
			}

			if(transp == false)
				return true;

			if(colormaptype == RGBAColourMap)
			{
				for(typename vector<unsigned char, Allocator>::const_iterator i(pixels.begin()); i != pixels.end(); i++)
				{
					if(colormap[*i].filter < 1.0) // with RGBAColourMap, .filter is actually alpha
						return false;
				}
			}
			else
			{
				for(typename vector<unsigned char, Allocator>::const_iterator i(pixels.begin()); i != pixels.end(); i++)
				{
					if(colormap[*i].transm > 0.0)
						return false;
				}
			}

			return true;
		}
		bool IsGrayscale() const
		{
			return false;
		}
		bool IsColour() const
		{
			return true;
		}
		bool IsFloat() const
		{
			return false;
		}
		bool IsInt() const
		{
			return true;
		}
		bool IsIndexed() const
		{
			return true;
		}
		bool IsGammaEncoded() const
		{
			return false;
		}
		bool HasAlphaChannel() const
		{
			return (colormaptype == RGBAColourMap);
		}
		bool HasFilterTransmit() const
		{
			return (colormaptype == RGBFTColourMap);
		}
		unsigned int GetMaxIntValue() const
		{
			return 255;
		}
		bool TryDeferDecoding(GammaCurvePtr&, unsigned int)
		{
			return false;
		}

		bool GetBitValue(unsigned int x, unsigned int y) const
		{
			float red, green, blue, filter, transm, alpha;
			switch(colormaptype)
			{
				case RGBColourMap:
					GetRGBValue(x, y, red, green, blue);
					return IS_NONZERO_RGB(red, green, blue);
				case RGBAColourMap:
                    // TODO FIXME - [CLi] This takes into account opacity information; other bit-based code doesn't.
					GetRGBAValue(x, y, red, green, blue, alpha);
					return IS_NONZERO_RGB(red, green, blue) && (alpha == ALPHA_OPAQUE);
				case RGBFTColourMap:
                    // TODO FIXME - [CLi] This takes into account opacity information; other bit-based code doesn't.
					GetRGBFTValue(x, y, red, green, blue, filter, transm);
					return IS_NONZERO_RGB(red, green, blue) && (filter == FT_OPAQUE) && (transm == FT_OPAQUE);
				default:
					return false;
			}
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			float red, green, blue, filter, transm, alpha;
			switch(colormaptype)
			{
				case RGBColourMap:
					GetRGBValue(x, y, red, green, blue);
					return RGB2Gray(red, green, blue);
				case RGBAColourMap:
					GetRGBAValue(x, y, red, green, blue, alpha);
					return RGB2Gray(red, green, blue);
				case RGBFTColourMap:
					GetRGBFTValue(x, y, red, green, blue, filter, transm);
					return RGB2Gray(red, green, blue);
				default:
					return 0.0f;
			}
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			float red, green, blue, filter, transm;
			alpha = ALPHA_OPAQUE; // (unless noted otherwise)
			switch(colormaptype)
			{
				case RGBColourMap:
					GetRGBValue(x, y, red, green, blue);
					gray = RGB2Gray(red, green, blue);
					return;
				case RGBAColourMap:
					GetRGBAValue(x, y, red, green, blue, alpha);
					gray = RGB2Gray(red, green, blue);
					return;
				case RGBFTColourMap:
					GetRGBFTValue(x, y, red, green, blue, filter, transm);
					gray = RGB2Gray(red, green, blue);
                    alpha = Colour::FTtoA(filter, transm);
					return;
				default:
					gray = 0.0f;
					return;
			}
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			CHECK_BOUNDS(x, y);
			MapEntry e(colormap[pixels[x + y * width]]);
			red = e.red;
			green = e.green;
			blue = e.blue;
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			CHECK_BOUNDS(x, y);
			MapEntry e(colormap[pixels[x + y * width]]);
			red = e.red;
			green = e.green;
			blue = e.blue;
			alpha = ALPHA_OPAQUE; // (unless noted otherwise)
			switch(colormaptype)
			{
				case RGBAColourMap:
					alpha = e.filter; // with RGBAColourMap, .filter is actually alpha
					return;
				case RGBFTColourMap:
					alpha = Colour::FTtoA(e.filter, e.transm);
					return;
				default:
					return;
			}
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			CHECK_BOUNDS(x, y);
			MapEntry e(colormap[pixels[x + y * width]]);
			red = e.red;
			green = e.green;
			blue = e.blue;
			filter = FT_OPAQUE; // (unless noted otherwise)
			transm = FT_OPAQUE; // (unless noted otherwise)
			switch(colormaptype)
			{
				case RGBAColourMap:
                    Colour::AtoFT(e.filter, filter, transm); // with RGBAColourMap, .filter is actually alpha
					return;
				case RGBFTColourMap:
					filter = e.filter;
					transm = e.transm;
					return;
				default:
					return;
			}
		}
		unsigned char GetIndexedValue(unsigned int x, unsigned int y)
		{
			CHECK_BOUNDS(x, y);
			return pixels[x + y * width];
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = Bit2Map(bit);
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = Gray2Map(gray);
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = Gray2Map(float(gray) / 255.0);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = GrayA2Map(gray, alpha);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = GrayA2Map(float(gray) / 255.0, float(alpha) / 255.0);
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = RGB2Map(red, green, blue);
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = RGB2Map(float(red) / 255.0, float(green) / 255.0, float(blue) / 255.0);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = RGBA2Map(red, green, blue, alpha);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = RGBA2Map(float(red) / 255.0, float(green) / 255.0, float(blue) / 255.0, float(alpha) / 255.0);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float filter, float transm)
		{
			CHECK_BOUNDS(x, y);
			// [CLi 2009-09] this was dividing by 255 - which I presume to have been a bug.
			pixels[x + y * width] = RGBFT2Map(red, green, blue, filter, transm);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
			CHECK_BOUNDS(x, y);
			// [CLi 2009-09] this was dividing by 255 - which I presume to have been a bug.
			pixels[x + y * width] = RGBFT2Map(col.red(), col.green(), col.blue(), col.filter(), col.transm());
		}
		void SetIndexedValue(unsigned int x, unsigned int y, unsigned char index)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = index;
		}

		void FillBitValue(bool bit)
		{
			fill(pixels.begin(), pixels.end(), Bit2Map(bit));
		}
		void FillGrayValue(float gray)
		{
			fill(pixels.begin(), pixels.end(), Gray2Map(gray));
		}
		void FillGrayValue(unsigned int gray)
		{
			fill(pixels.begin(), pixels.end(), Gray2Map(float(gray) / 255.0));
		}
		void FillGrayAValue(float gray, float alpha)
		{
			fill(pixels.begin(), pixels.end(), GrayA2Map(gray, alpha));
		}
		void FillGrayAValue(unsigned int gray, unsigned int alpha)
		{
			fill(pixels.begin(), pixels.end(), GrayA2Map(float(gray) / 255.0, float(alpha) / 255.0));
		}
		void FillRGBValue(float red, float green, float blue)
		{
			fill(pixels.begin(), pixels.end(), RGB2Map(red, green, blue));
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
			fill(pixels.begin(), pixels.end(), RGB2Map(float(red) / 255.0, float(green) / 255.0, float(blue) / 255.0));
		}
		void FillRGBAValue(float red, float green, float blue, float alpha)
		{
			fill(pixels.begin(), pixels.end(), RGBA2Map(red, green, blue, alpha));
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
			fill(pixels.begin(), pixels.end(), RGBA2Map(float(red) / 255.0, float(green) / 255.0, float(blue) / 255.0, float(alpha) / 255.0));
		}
		void FillRGBFTValue(float red, float green, float blue, float filter, float transm)
		{
			// [CLi 2009-09] this was dividing by 255 - which I presume to have been a bug.
			fill(pixels.begin(), pixels.end(), RGBFT2Map(red, green, blue, filter, transm));
		}
		void FillIndexedValue(unsigned char index)
		{
			fill(pixels.begin(), pixels.end(), index);
		}
	private:
		vector<unsigned char, Allocator> pixels;

		unsigned char Bit2Map(bool bit) const
		{
			if(bit == true)
				return Gray2Map(1.0f);
			else
				return Gray2Map(0.0f);
		}
		unsigned char Gray2Map(float gray) const
		{
			switch(colormaptype)
			{
				case RGBColourMap:
					return FindBestRGB(gray, gray, gray);
				case RGBAColourMap:
					return FindBestRGBA(gray, gray, gray, ALPHA_OPAQUE);
				case RGBFTColourMap:
					return FindBestRGBFT(gray, gray, gray, FT_OPAQUE, FT_OPAQUE);
				default:
					return 0;
			}
		}
		unsigned char GrayA2Map(float gray, float alpha) const
		{
            float filter, transm;
			switch(colormaptype)
			{
				case RGBColourMap:
					return FindBestRGB(gray, gray, gray);
				case RGBAColourMap:
					return FindBestRGBA(gray, gray, gray, alpha);
				case RGBFTColourMap:
                    Colour::AtoFT(alpha, filter, transm);
					return FindBestRGBFT(gray, gray, gray, filter, transm);
				default:
					return 0;
			}
		}
		unsigned char RGB2Map(float red, float green, float blue) const
		{
			switch(colormaptype)
			{
				case RGBColourMap:
					return FindBestRGB(red, green, blue);
				case RGBAColourMap:
					return FindBestRGBA(red, green, blue, ALPHA_OPAQUE);
				case RGBFTColourMap:
					return FindBestRGBFT(red, green, blue, FT_OPAQUE, FT_OPAQUE);
				default:
					return 0;
			}
		}
		unsigned char RGBA2Map(float red, float green, float blue, float alpha) const
		{
            float filter, transm;
			switch(colormaptype)
			{
				case RGBColourMap:
					return FindBestRGB(red, green, blue);
				case RGBAColourMap:
					return FindBestRGBA(red, green, blue, alpha);
				case RGBFTColourMap:
                    Colour::AtoFT(alpha, filter, transm);
					return FindBestRGBFT(red, green, blue, filter, transm);
				default:
					return 0;
			}
		}
		unsigned char RGBFT2Map(float red, float green, float blue, float filter, float transm) const
		{
			switch(colormaptype)
			{
				case RGBColourMap:
					return FindBestRGB(red, green, blue);
				case RGBAColourMap:
					return FindBestRGBA(red, green, blue, Colour::FTtoA(filter, transm));
				case RGBFTColourMap:
					return FindBestRGBFT(red, green, blue, filter, transm);
				default:
					return 0;
			}
		}
		unsigned char FindBestRGB(float red, float green, float blue) const
		{
			unsigned char best = 0;
			float diff = 3.0f;

			for(size_t i = 0; i < colormap.size(); i++)
			{
				float d(RGB2Gray(fabs(colormap[i].red - red), fabs(colormap[i].green - green), fabs(colormap[i].red - blue)));
				if(d < diff)
				{
					d = diff;
					best = (unsigned char)i;
				}
			}

			return best;
		}
		unsigned char FindBestRGBA(float red, float green, float blue, float alpha) const
		{
			unsigned char best = 0;
			float diff = 3.0f;

			for(size_t i = 0; i < colormap.size(); i++)
			{
				float d((RGB2Gray(fabs(colormap[i].red - red), fabs(colormap[i].green - green), fabs(colormap[i].red - blue)) * 3.0f +
				         fabs(colormap[i].filter - alpha)) / 4.0f);  // with RGBAColourMap, .filter is actually alpha
				if(d < diff)
				{
					d = diff;
					best = (unsigned char)i;
				}
			}

			return best;
		}
		unsigned char FindBestRGBFT(float red, float green, float blue, float filter, float transm) const
		{
			unsigned char best = 0;
			float diff = 3.0f;

			for(size_t i = 0; i < colormap.size(); i++)
			{
				float d((RGB2Gray(fabs(colormap[i].red - red), fabs(colormap[i].green - green), fabs(colormap[i].red - blue)) * 3.0f +
				         fabs(colormap[i].filter - filter) + fabs(colormap[i].transm - transm)) / 5.0f);
				if(d < diff)
				{
					d = diff;
					best = (unsigned char)i;
				}
			}

			return best;
		}
};

typedef ColourMapImage<> MemoryColourMapImage;

template<typename T, unsigned int TMAX, int IDT, class Allocator = allocator<T> >
class GrayImage : public Image
{
	public:
		GrayImage(unsigned int w, unsigned int h) :
			Image(w, h, ImageDataType(IDT)) { pixels.resize(w * h); FillBitValue(false); }
		GrayImage(unsigned int w, unsigned int h, const vector<RGBMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h); FillBitValue(false); }
		GrayImage(unsigned int w, unsigned int h, const vector<RGBAMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h); FillBitValue(false); }
		GrayImage(unsigned int w, unsigned int h, const vector<RGBFTMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h); FillBitValue(false); }
		~GrayImage() { }

		bool IsOpaque() const
		{
			return true;
		}
		bool IsGrayscale() const
		{
			return true;
		}
		bool IsColour() const
		{
			return false;
		}
		bool IsFloat() const
		{
			return false;
		}
		bool IsInt() const
		{
			return true;
		}
		bool IsGammaEncoded() const
		{
			return false;
		}
		bool IsIndexed() const
		{
			return false;
		}
		bool HasAlphaChannel() const
		{
			return false;
		}
		bool HasFilterTransmit() const
		{
			return false;
		}
		unsigned int GetMaxIntValue() const
		{
			return TMAX;
		}
		bool TryDeferDecoding(GammaCurvePtr&, unsigned int)
		{
			return false;
		}

		bool GetBitValue(unsigned int x, unsigned int y) const
		{
			CHECK_BOUNDS(x, y);
			return (pixels[x + y * width] != 0);
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			CHECK_BOUNDS(x, y);
			return float(pixels[x + y * width]) / float(TMAX);
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			CHECK_BOUNDS(x, y);
			gray = float(pixels[x + y * width]) / float(TMAX);
			alpha = ALPHA_OPAQUE;
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			red = green = blue = GetGrayValue(x, y);
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			red = green = blue = GetGrayValue(x, y);
			alpha = ALPHA_OPAQUE;
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			red = green = blue = GetGrayValue(x, y);
			filter = transm = FT_OPAQUE;
		}
		unsigned char GetIndexedValue(unsigned int x, unsigned int y)
		{
			CHECK_BOUNDS(x, y);
			return (unsigned char)(int(pixels[x + y * width]) / ((TMAX + 1) >> 8));
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			if(bit == true)
				SetGrayValue(x, y, TMAX);
			else
				SetGrayValue(x, y, (unsigned int)0);
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = T(gray * float(TMAX));
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = T(gray);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = T(gray * float(TMAX));
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = T(gray);
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			SetGrayValue(x, y, RGB2Gray(red, green, blue));
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
			SetGrayValue(x, y, RGB2Gray(float(red) / float(TMAX), float(green) / float(TMAX), float(blue) / float(TMAX)));
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float, float)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
			SetRGBValue(x, y, col.red(), col.green(), col.blue());
		}

		void FillBitValue(bool bit)
		{
			if(bit == true)
				FillGrayValue(TMAX);
			else
				FillGrayValue((unsigned int)0);
		}
		void FillGrayValue(float gray)
		{
			FillGrayValue((unsigned int)(gray * float(TMAX)));
		}
		void FillGrayValue(unsigned int gray)
		{
			fill(pixels.begin(), pixels.end(), T(gray));
		}
		void FillGrayAValue(float gray, float)
		{
			FillGrayValue(gray);
		}
		void FillGrayAValue(unsigned int gray, unsigned int)
		{
			FillGrayValue(gray);
		}
		void FillRGBValue(float red, float green, float blue)
		{
			FillGrayValue(RGB2Gray(red, green, blue));
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
			FillGrayValue(RGB2Gray(float(red) / float(TMAX), float(green) / float(TMAX), float(blue) / float(TMAX)));
		}
		void FillRGBAValue(float red, float green, float blue, float)
		{
			FillRGBValue(red, green, blue);
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int)
		{
			FillRGBValue(red, green, blue);
		}
		void FillRGBFTValue(float red, float green, float blue, float, float)
		{
			FillRGBValue(red, green, blue);
		}
	private:
		vector<T, Allocator> pixels;
};

typedef GrayImage<unsigned char, 255, Image::Gray_Int8> MemoryGray8Image;

typedef GrayImage<unsigned short, 65535, Image::Gray_Int16> MemoryGray16Image;

template<typename T, unsigned int TMAX, int IDT, class Allocator = allocator<T> >
class GrayAImage : public Image
{
	public:
		GrayAImage(unsigned int w, unsigned int h) :
			Image(w, h, ImageDataType(IDT)) { pixels.resize(w * h * 2); FillBitValue(false); }
		GrayAImage(unsigned int w, unsigned int h, const vector<RGBMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h * 2); FillBitValue(false); }
		GrayAImage(unsigned int w, unsigned int h, const vector<RGBAMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h * 2); FillBitValue(false); }
		GrayAImage(unsigned int w, unsigned int h, const vector<RGBFTMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h * 2); FillBitValue(false); }
		~GrayAImage() { }

		bool IsOpaque() const
		{
			for(typename vector<T, Allocator>::const_iterator i(pixels.begin()); i != pixels.end(); i += 2)
			{
				if(i[1] < TMAX)
					return false;
			}

			return true;
		}
		bool IsGrayscale() const
		{
			return true;
		}
		bool IsColour() const
		{
			return false;
		}
		bool IsFloat() const
		{
			return false;
		}
		bool IsInt() const
		{
			return true;
		}
		bool IsIndexed() const
		{
			return false;
		}
		bool IsGammaEncoded() const
		{
			return false;
		}
		bool HasAlphaChannel() const
		{
			return true;
		}
		bool HasFilterTransmit() const
		{
			return false;
		}
		unsigned int GetMaxIntValue() const
		{
			return TMAX;
		}
		bool TryDeferDecoding(GammaCurvePtr&, unsigned int)
		{
			return false;
		}

		bool GetBitValue(unsigned int x, unsigned int y) const
		{
            // TODO FIXME - [CLi] This ignores opacity information; other bit-based code doesn't.
			CHECK_BOUNDS(x, y);
			return (pixels[(x + y * width) * 2] != 0);
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			CHECK_BOUNDS(x, y);
			return float(pixels[(x + y * width) * 2]) / float(TMAX);
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			CHECK_BOUNDS(x, y);
			gray = float(pixels[(x + y * width) * 2]) / float(TMAX);
			alpha = float(pixels[(x + y * width) * 2 + 1]) / float(TMAX);
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			red = green = blue = GetGrayValue(x, y);
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			GetGrayAValue(x, y, red, alpha);
			green = blue = red;
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			float alpha;
			GetGrayAValue(x, y, red, alpha);
			green = blue = red;
            Colour::AtoFT(alpha, filter, transm);
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			if(bit == true)
				SetGrayAValue(x, y, TMAX, ALPHA_OPAQUE_INT(TMAX));
			else
				SetGrayAValue(x, y, (unsigned int)0, ALPHA_OPAQUE_INT(TMAX));
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			SetGrayAValue(x, y, gray, ALPHA_OPAQUE);
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			SetGrayAValue(x, y, gray, ALPHA_OPAQUE_INT(TMAX));
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 2] = T(gray * float(TMAX));
			pixels[(x + y * width) * 2 + 1] = T(alpha * float(TMAX));
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 2] = gray;
			pixels[(x + y * width) * 2 + 1] = alpha;
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			SetGrayValue(x, y, RGB2Gray(red, green, blue));
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
			SetGrayValue(x, y, RGB2Gray(float(red) / float(TMAX), float(green) / float(TMAX), float(blue) / float(TMAX)));
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float alpha)
		{
			SetGrayAValue(x, y, RGB2Gray(red, green, blue), alpha);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
			SetGrayAValue(x, y, RGB2Gray(float(red) / float(TMAX), float(green) / float(TMAX), float(blue) / float(TMAX)), float(alpha) / float(TMAX));
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float filter, float transm)
		{
            SetGrayAValue(x, y, RGB2Gray(red, green, blue), Colour::FTtoA(filter, transm));
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
			SetGrayAValue(x, y, RGB2Gray(col.red(), col.green(), col.blue()), col.FTtoA());
		}

		void FillBitValue(bool bit)
		{
			if(bit == true)
				FillGrayValue(TMAX);
			else
				FillGrayValue((unsigned int)0);
		}
		void FillGrayValue(float gray)
		{
			FillGrayValue((unsigned int)(gray * float(TMAX)));
		}
		void FillGrayValue(unsigned int gray)
		{
			FillGrayAValue(gray, ALPHA_OPAQUE_INT(TMAX));
		}
		void FillGrayAValue(float gray, float alpha)
		{
			// [CLi 2009-09] this was dividing by float(TMAX) - which I presume to have been a bug.
			T g(gray * float(TMAX)), a(alpha * float(TMAX));
			for(typename vector<T, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = g;
				i++;
				*i = a;
			}
		}
		void FillGrayAValue(unsigned int gray, unsigned int alpha)
		{
			for(typename vector<T, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = T(gray);
				i++;
				*i = T(alpha);
			}
		}
		void FillRGBValue(float red, float green, float blue)
		{
			FillGrayValue(RGB2Gray(red, green, blue));
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
			FillGrayValue(RGB2Gray(float(red) / float(TMAX), float(green) / float(TMAX), float(blue) / float(TMAX)));
		}
		void FillRGBAValue(float red, float green, float blue, float alpha)
		{
			FillGrayAValue(RGB2Gray(red, green, blue), alpha);
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
			FillGrayAValue(RGB2Gray(float(red) / float(TMAX), float(green) / float(TMAX), float(blue) / float(TMAX)), float(alpha) / float(TMAX));
		}
		void FillRGBFTValue(float red, float green, float blue, float filter, float transm)
		{
            FillGrayAValue(RGB2Gray(red, green, blue), Colour::FTtoA(filter, transm));
		}
	private:
		vector<T, Allocator> pixels;
};

typedef GrayAImage<unsigned char, 255, Image::GrayA_Int8> MemoryGrayA8Image;

typedef GrayAImage<unsigned short, 65535, Image::GrayA_Int16> MemoryGrayA16Image;

template<typename T, unsigned int TMAX, int IDT, class Allocator = allocator<T> >
class RGBImage : public Image
{
	public:
		RGBImage(unsigned int w, unsigned int h) :
			Image(w, h, ImageDataType(IDT)) { pixels.resize(w * h * 3); FillBitValue(false); }
		RGBImage(unsigned int w, unsigned int h, const vector<RGBMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h * 3); FillBitValue(false); }
		RGBImage(unsigned int w, unsigned int h, const vector<RGBAMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h * 3); FillBitValue(false); }
		RGBImage(unsigned int w, unsigned int h, const vector<RGBFTMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h * 3); FillBitValue(false); }
		~RGBImage() { }

		bool IsOpaque() const
		{
			return true;
		}
		bool IsGrayscale() const
		{
			return false;
		}
		bool IsColour() const
		{
			return true;
		}
		bool IsFloat() const
		{
			return false;
		}
		bool IsInt() const
		{
			return true;
		}
		bool IsIndexed() const
		{
			return false;
		}
		bool IsGammaEncoded() const
		{
			return false;
		}
		bool HasAlphaChannel() const
		{
			return false;
		}
		bool HasFilterTransmit() const
		{
			return false;
		}
		unsigned int GetMaxIntValue() const
		{
			return TMAX;
		}
		bool TryDeferDecoding(GammaCurvePtr&, unsigned int)
		{
			return false;
		}

		bool GetBitValue(unsigned int x, unsigned int y) const
		{
			float red, green, blue;
			GetRGBValue(x, y, red, green, blue);
			return IS_NONZERO_RGB(red, green, blue);
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			float red, green, blue;
			GetRGBValue(x, y, red, green, blue);
			return RGB2Gray(red, green, blue);
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			gray = GetGrayValue(x, y);
			alpha = ALPHA_OPAQUE;
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			CHECK_BOUNDS(x, y);
			red = float(pixels[(x + y * width) * 3]) / float(TMAX);
			green = float(pixels[(x + y * width) * 3 + 1]) / float(TMAX);
			blue = float(pixels[(x + y * width) * 3 + 2]) / float(TMAX);
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			GetRGBValue(x, y, red, green, blue);
			alpha = ALPHA_OPAQUE;
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			GetRGBValue(x, y, red, green, blue);
			filter = transm = FT_OPAQUE;
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			if(bit == true)
				SetGrayValue(x, y, TMAX);
			else
				SetGrayValue(x, y, (unsigned int)0);
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			SetRGBValue(x, y, gray, gray, gray);
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = gray;
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = T(gray * float(TMAX));
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = gray;
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 3] = T(red * float(TMAX));
			pixels[(x + y * width) * 3 + 1] = T(green * float(TMAX));
			pixels[(x + y * width) * 3 + 2] = T(blue * float(TMAX));
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 3] = T(red);
			pixels[(x + y * width) * 3 + 1] = T(green);
			pixels[(x + y * width) * 3 + 2] = T(blue);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float, float)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
			SetRGBValue(x, y, col.red(), col.green(), col.blue());
		}

		void FillBitValue(bool bit)
		{
			if(bit == true)
				FillGrayValue(TMAX);
			else
				FillGrayValue((unsigned int)0);
		}
		void FillGrayValue(float gray)
		{
			FillGrayValue((unsigned int)(gray * float(TMAX)));
		}
		void FillGrayValue(unsigned int gray)
		{
			fill(pixels.begin(), pixels.end(), gray);
		}
		void FillGrayAValue(float gray, float)
		{
			FillRGBValue(gray, gray, gray);
		}
		void FillGrayAValue(unsigned int gray, unsigned int)
		{
			FillRGBValue(gray, gray, gray);
		}
		void FillRGBValue(float red, float green, float blue)
		{
			// [CLi 2009-09] this was dividing by float(TMAX) - which I presume to have been a bug.
			T r(red * float(TMAX)), g(green * float(TMAX)), b(blue * float(TMAX));
			for(typename vector<T, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = r;
				i++;
				*i = g;
				i++;
				*i = b;
			}
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
			for(typename vector<T, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = T(red);
				i++;
				*i = T(green);
				i++;
				*i = T(blue);
			}
		}
		void FillRGBAValue(float red, float green, float blue, float)
		{
			FillRGBValue(red, green, blue);
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int)
		{
			FillRGBValue(red, green, blue);
		}
		void FillRGBFTValue(float red, float green, float blue, float, float)
		{
			FillRGBValue(red, green, blue);
		}
	private:
		vector<T, Allocator> pixels;
};

typedef RGBImage<unsigned char, 255, Image::RGB_Int8> MemoryRGB8Image;

typedef RGBImage<unsigned short, 65535, Image::RGB_Int16> MemoryRGB16Image;

template<typename T, unsigned int TMAX, int IDT, class Allocator = allocator<T> >
class RGBAImage : public Image
{
	public:
		RGBAImage(unsigned int w, unsigned int h) :
			Image(w, h, ImageDataType(IDT)) { pixels.resize(w * h * 4); FillBitValue(false); }
		RGBAImage(unsigned int w, unsigned int h, const vector<RGBMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h * 4); FillBitValue(false); }
		RGBAImage(unsigned int w, unsigned int h, const vector<RGBAMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h * 4); FillBitValue(false); }
		RGBAImage(unsigned int w, unsigned int h, const vector<RGBFTMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m) { pixels.resize(w * h * 4); FillBitValue(false); }
		~RGBAImage() { }

		bool IsOpaque() const
		{
			for(typename vector<T, Allocator>::const_iterator i(pixels.begin()); i != pixels.end(); i += 4)
			{
				if(i[3] < TMAX)
					return false;
			}

			return true;
		}
		bool IsGrayscale() const
		{
			return false;
		}
		bool IsColour() const
		{
			return true;
		}
		bool IsFloat() const
		{
			return false;
		}
		bool IsInt() const
		{
			return true;
		}
		bool IsIndexed() const
		{
			return false;
		}
		bool IsGammaEncoded() const
		{
			return false;
		}
		bool HasAlphaChannel() const
		{
			return true;
		}
		bool HasFilterTransmit() const
		{
			return false;
		}
		unsigned int GetMaxIntValue() const
		{
			return TMAX;
		}
		bool TryDeferDecoding(GammaCurvePtr&, unsigned int)
		{
			return false;
		}

		bool GetBitValue(unsigned int x, unsigned int y) const
		{
            // TODO FIXME - [CLi] This ignores opacity information; other bit-based code doesn't.
			float red, green, blue, alpha;
			GetRGBAValue(x, y, red, green, blue, alpha);
			return IS_NONZERO_RGB(red, green, blue);
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			float red, green, blue, alpha;
			GetRGBAValue(x, y, red, green, blue, alpha);
			return RGB2Gray(red, green, blue);
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			float red, green, blue;
			GetRGBAValue(x, y, red, green, blue, alpha);
			gray = RGB2Gray(red, green, blue);
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			float alpha;
			GetRGBAValue(x, y, red, green, blue, alpha);
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			CHECK_BOUNDS(x, y);
			red = float(pixels[(x + y * width) * 4]) / float(TMAX);
			green = float(pixels[(x + y * width) * 4 + 1]) / float(TMAX);
			blue = float(pixels[(x + y * width) * 4 + 2]) / float(TMAX);
			alpha = float(pixels[(x + y * width) * 4 + 3]) / float(TMAX);
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			float alpha;
			GetRGBAValue(x, y, red, green, blue, alpha);
            Colour::AtoFT(alpha, filter, transm);
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			if(bit == true)
				SetGrayValue(x, y, TMAX);
			else
				SetGrayValue(x, y, (unsigned int)0);
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			SetRGBAValue(x, y, gray, gray, gray, ALPHA_OPAQUE);
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			SetRGBAValue(x, y, gray, gray, gray, ALPHA_OPAQUE_INT(TMAX));
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float alpha)
		{
			SetRGBAValue(x, y, gray, gray, gray, alpha);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int alpha)
		{
			SetRGBAValue(x, y, gray, gray, gray, alpha);
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			SetRGBAValue(x, y, red, green, blue, ALPHA_OPAQUE);
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
			SetRGBAValue(x, y, red, green, blue, ALPHA_OPAQUE_INT(TMAX));
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 4] = T(red * float(TMAX));
			pixels[(x + y * width) * 4 + 1] = T(green * float(TMAX));
			pixels[(x + y * width) * 4 + 2] = T(blue * float(TMAX));
			pixels[(x + y * width) * 4 + 3] = T(alpha * float(TMAX));
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 4] = T(red);
			pixels[(x + y * width) * 4 + 1] = T(green);
			pixels[(x + y * width) * 4 + 2] = T(blue);
			pixels[(x + y * width) * 4 + 3] = T(alpha);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float filter, float transm)
		{
            SetRGBAValue(x, y, red, green, blue, Colour::FTtoA(filter, transm));
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 4] = T(col.red());
			pixels[(x + y * width) * 4 + 1] = T(col.green());
			pixels[(x + y * width) * 4 + 2] = T(col.blue());
			pixels[(x + y * width) * 4 + 3] = T(col.FTtoA());
		}

		void FillBitValue(bool bit)
		{
			if(bit == true)
				FillGrayValue(TMAX);
			else
				FillGrayValue((unsigned int)0);
		}
		void FillGrayValue(float gray)
		{
			FillRGBAValue(gray, gray, gray, ALPHA_OPAQUE);
		}
		void FillGrayValue(unsigned int gray)
		{
			FillRGBAValue(gray, gray, gray, ALPHA_OPAQUE_INT(TMAX));
		}
		void FillGrayAValue(float gray, float alpha)
		{
			FillRGBAValue(gray, gray, gray, alpha);
		}
		void FillGrayAValue(unsigned int gray, unsigned int alpha)
		{
			FillRGBAValue(gray, gray, gray, alpha);
		}
		void FillRGBValue(float red, float green, float blue)
		{
			FillRGBAValue(red, green, blue, ALPHA_OPAQUE);
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
			FillRGBAValue(red, green, blue, ALPHA_OPAQUE_INT(TMAX));
		}
		void FillRGBAValue(float red, float green, float blue, float alpha)
		{
			// [CLi 2009-09] this was dividing by float(TMAX) - which I presume to have been a bug.
			T r(red * float(TMAX)), g(green * float(TMAX)), b(blue * float(TMAX)), a(alpha * float(TMAX));
			for(typename vector<T, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = r;
				i++;
				*i = g;
				i++;
				*i = b;
				i++;
				*i = a;
			}
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
			for(typename vector<T, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = T(red);
				i++;
				*i = T(green);
				i++;
				*i = T(blue);
				i++;
				*i = T(alpha);
			}
		}
		void FillRGBFTValue(float red, float green, float blue, float filter, float transm)
		{
            FillRGBAValue(red, green, blue, Colour::FTtoA(filter, transm));
		}
	private:
		vector<T, Allocator> pixels;
};

typedef RGBAImage<unsigned char, 255, Image::RGBA_Int8> MemoryRGBA8Image;

typedef RGBAImage<unsigned short, 65535, Image::RGBA_Int16> MemoryRGBA16Image;

template<class PixelContainer = vector<float, allocator<float> > >
class RGBFTImage : public Image
{
	public:
		RGBFTImage(unsigned int w, unsigned int h) :
			Image(w, h, RGBFT_Float) { pixels.resize(w * h * 5); FillBitValue(false); }
		RGBFTImage(unsigned int w, unsigned int h, const vector<RGBMapEntry>& m) :
			Image(w, h, RGBFT_Float, m) { pixels.resize(w * h * 5); FillBitValue(false); }
		RGBFTImage(unsigned int w, unsigned int h, const vector<RGBAMapEntry>& m) :
			Image(w, h, RGBFT_Float, m) { pixels.resize(w * h * 5); FillBitValue(false); }
		RGBFTImage(unsigned int w, unsigned int h, const vector<RGBFTMapEntry>& m) :
			Image(w, h, RGBFT_Float, m) { pixels.resize(w * h * 5); FillBitValue(false); }
		~RGBFTImage() { }

		bool IsOpaque() const
		{
			for(typename PixelContainer::const_iterator i(pixels.begin()); i != pixels.end(); i += 5)
			{
				if(i[4] > 0.0f) // TODO FIXME - this ignores filter
					return false;
			}
			return true;
		}
		bool IsGrayscale() const
		{
			return false;
		}
		bool IsColour() const
		{
			return true;
		}
		bool IsFloat() const
		{
			return true;
		}
		bool IsInt() const
		{
			return false;
		}
		bool IsIndexed() const
		{
			return false;
		}
		bool IsGammaEncoded() const
		{
			return false;
		}
		bool HasAlphaChannel() const
		{
			return false;
		}
		bool HasFilterTransmit() const
		{
			return true;
		}
		unsigned int GetMaxIntValue() const
		{
			return 255;
		}
		bool TryDeferDecoding(GammaCurvePtr&, unsigned int)
		{
			return false;
		}

		bool GetBitValue(unsigned int x, unsigned int y) const
		{
            // TODO FIXME - [CLi] This ignores opacity information; other bit-based code doesn't.
			float red, green, blue, filter, transm;
			GetRGBFTValue(x, y, red, green, blue, filter, transm);
			return IS_NONZERO_RGB(red, green, blue);
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			float red, green, blue, filter, transm;
			GetRGBFTValue(x, y, red, green, blue, filter, transm);
			return RGB2Gray(red, green, blue);
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			float red, green, blue, filter, transm;
			GetRGBFTValue(x, y, red, green, blue, filter, transm);
			gray = RGB2Gray(red, green, blue);
            alpha = Colour::FTtoA(filter, transm);
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			float filter, transm;
			GetRGBFTValue(x, y, red, green, blue, filter, transm);
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			float filter, transm;
			GetRGBFTValue(x, y, red, green, blue, filter, transm);
            alpha = Colour::FTtoA(filter, transm);
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			CHECK_BOUNDS(x, y);
			red = pixels[(x + y * width) * 5];
			green = pixels[(x + y * width) * 5 + 1];
			blue = pixels[(x + y * width) * 5 + 2];
			filter = pixels[(x + y * width) * 5 + 3];
			transm = pixels[(x + y * width) * 5 + 4];
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			if(bit == true)
				SetGrayValue(x, y, 1.0f);
			else
				SetGrayValue(x, y, 0.0f);
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			SetRGBFTValue(x, y, gray, gray, gray, FT_OPAQUE, FT_OPAQUE);
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			SetGrayValue(x, y, float(gray) / 255.0f);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float alpha)
		{
            float filter, transm;
            Colour::AtoFT(alpha, filter, transm);
			SetRGBFTValue(x, y, gray, gray, gray, filter, transm);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int alpha)
		{
			float c = float(gray) / 255.0f;
            float filter, transm;
            Colour::AtoFT(float(alpha) / 255.0f, filter, transm);
			SetRGBFTValue(x, y, c, c, c, filter, transm);
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			SetRGBFTValue(x, y, red, green, blue, FT_OPAQUE, FT_OPAQUE);
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
			SetRGBFTValue(x, y, float(red) / 255.0f, float(green) / 255.0f, float(blue) / 255.0f, FT_OPAQUE, FT_OPAQUE);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float alpha)
		{
            float filter, transm;
            Colour::AtoFT(alpha, filter, transm);
			SetRGBFTValue(x, y, red, green, blue, filter, transm);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
            float filter, transm;
            Colour::AtoFT(float(alpha) / 255.0f, filter, transm);
			SetRGBFTValue(x, y, float(red) / 255.0f, float(green) / 255.0f, float(blue) / 255.0f, filter, transm);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float filter, float transm)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 5] = red;
			pixels[(x + y * width) * 5 + 1] = green;
			pixels[(x + y * width) * 5 + 2] = blue;
			pixels[(x + y * width) * 5 + 3] = filter;
			pixels[(x + y * width) * 5 + 4] = transm;
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 5] = col.red();
			pixels[(x + y * width) * 5 + 1] = col.green();
			pixels[(x + y * width) * 5 + 2] = col.blue();
			pixels[(x + y * width) * 5 + 3] = col.filter();
			pixels[(x + y * width) * 5 + 4] = col.transm();
		}

		void FillBitValue(bool bit)
		{
			if(bit == true)
				FillGrayValue(1.0f);
			else
				FillGrayValue(0.0f);
		}
		void FillGrayValue(float gray)
		{
			FillRGBFTValue(gray, gray, gray, FT_OPAQUE, FT_OPAQUE);
		}
		void FillGrayValue(unsigned int gray)
		{
			FillGrayValue(float(gray) / 255.0f);
		}
		void FillGrayAValue(float gray, float alpha)
		{
            float filter, transm;
            Colour::AtoFT(alpha, filter, transm);
			FillRGBFTValue(gray, gray, gray, filter, transm);
		}
		void FillGrayAValue(unsigned int gray, unsigned int alpha)
		{
			FillGrayAValue(float(gray) / 255.0f, float(alpha) / 255.0f);
		}
		void FillRGBValue(float red, float green, float blue)
		{
			FillRGBFTValue(red, green, blue, FT_OPAQUE, FT_OPAQUE);
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
			FillRGBFTValue(float(red) / 255.0f, float(green) / 255.0f, float(blue) / 255.0f, FT_OPAQUE, FT_OPAQUE);
		}
		void FillRGBAValue(float red, float green, float blue, float alpha)
		{
            float filter, transm;
            Colour::AtoFT(alpha, filter, transm);
			FillRGBFTValue(red, green, blue, filter, transm);
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
            float filter, transm;
            Colour::AtoFT(float(alpha) / 255.0f, filter, transm);
			FillRGBFTValue(float(red) / 255.0f, float(green) / 255.0f, float(blue) / 255.0f, filter, transm);
		}
		void FillRGBFTValue(float red, float green, float blue, float filter, float transm)
		{
			for(typename PixelContainer::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = red;
				i++;
				*i = green;
				i++;
				*i = blue;
				i++;
				*i = filter;
				i++;
				*i = transm;
			}
		}
	private:
		PixelContainer pixels;
};

typedef RGBFTImage<> MemoryRGBFTImage;

template<typename T, unsigned int TMAX, int IDT, class Allocator = allocator<T> >
class NonlinearGrayImage : public Image
{
	public:
		NonlinearGrayImage(unsigned int w, unsigned int h) :
			Image(w, h, ImageDataType(IDT)), gamma(NeutralGammaCurve::Get()) { gammaLUT = gamma->GetLookupTable(TMAX); pixels.resize(w * h); FillBitValue(false); }
		NonlinearGrayImage(unsigned int w, unsigned int h, const vector<RGBMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m), gamma(NeutralGammaCurve::Get()) { gammaLUT = gamma->GetLookupTable(TMAX); pixels.resize(w * h); FillBitValue(false); }
		NonlinearGrayImage(unsigned int w, unsigned int h, const vector<RGBAMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m), gamma(NeutralGammaCurve::Get()) { gammaLUT = gamma->GetLookupTable(TMAX); pixels.resize(w * h); FillBitValue(false); }
		NonlinearGrayImage(unsigned int w, unsigned int h, const vector<RGBFTMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m), gamma(NeutralGammaCurve::Get()) { gammaLUT = gamma->GetLookupTable(TMAX); pixels.resize(w * h); FillBitValue(false); }
		~NonlinearGrayImage() { }

		bool IsOpaque() const
		{
			return true;
		}
		bool IsGrayscale() const
		{
			return true;
		}
		bool IsColour() const
		{
			return false;
		}
		bool IsFloat() const
		{
			return false;
		}
		bool IsInt() const
		{
			return true;
		}
		bool IsIndexed() const
		{
			return false;
		}
		bool IsGammaEncoded() const
		{
			return true;
		}
		bool HasAlphaChannel() const
		{
			return false;
		}
		bool HasFilterTransmit() const
		{
			return false;
		}
		unsigned int GetMaxIntValue() const
		{
			return TMAX;
		}
		bool TryDeferDecoding(GammaCurvePtr& g, unsigned int max)
		{
			if (max != TMAX) return false;
			if (!GammaCurve::IsNeutral(gamma)) return !g;
			gamma.swap(g);
			gammaLUT = gamma->GetLookupTable(TMAX);
			return true;
		}

		bool GetBitValue(unsigned int x, unsigned int y) const
		{
			CHECK_BOUNDS(x, y);
			return (pixels[x + y * width] != 0);
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			CHECK_BOUNDS(x, y);
			return gammaLUT[pixels[x + y * width]];
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			CHECK_BOUNDS(x, y);
			gray = gammaLUT[pixels[x + y * width]];
			alpha = ALPHA_OPAQUE;
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			red = green = blue = GetGrayValue(x, y);
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			red = green = blue = GetGrayValue(x, y);
			alpha = ALPHA_OPAQUE;
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			red = green = blue = GetGrayValue(x, y);
			filter = transm = FT_OPAQUE;
		}
		unsigned char GetIndexedValue(unsigned int x, unsigned int y)
		{
			CHECK_BOUNDS(x, y);
			return (unsigned char)(int(pixels[x + y * width]) / ((TMAX + 1) >> 8));
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			if(bit == true)
				SetGrayValue(x, y, TMAX);
			else
				SetGrayValue(x, y, (unsigned int)0);
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = IntEncode(gamma, gray, TMAX);
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = T(gray);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = IntEncode(gamma, gray, TMAX);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = T(gray);
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			SetGrayValue(x, y, RGB2Gray(red, green, blue));
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
			SetGrayValue(x, y, RGB2Gray(gammaLUT[red], gammaLUT[green], gammaLUT[blue]));
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float, float)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
			SetRGBValue(x, y, col.red(), col.green(), col.blue());
		}

		void FillBitValue(bool bit)
		{
			if(bit == true)
				FillGrayValue(TMAX);
			else
				FillGrayValue((unsigned int)0);
		}
		void FillGrayValue(float gray)
		{
			FillGrayValue(IntEncode(gamma, gray, TMAX));
		}
		void FillGrayValue(unsigned int gray)
		{
			fill(pixels.begin(), pixels.end(), T(gray));
		}
		void FillGrayAValue(float gray, float)
		{
			FillGrayValue(gray);
		}
		void FillGrayAValue(unsigned int gray, unsigned int)
		{
			FillGrayValue(gray);
		}
		void FillRGBValue(float red, float green, float blue)
		{
			FillGrayValue(RGB2Gray(red, green, blue));
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
			FillGrayValue(RGB2Gray(gammaLUT[red], gammaLUT[green], gammaLUT[blue]));
		}
		void FillRGBAValue(float red, float green, float blue, float)
		{
			FillRGBValue(red, green, blue);
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int)
		{
			FillRGBValue(red, green, blue);
		}
		void FillRGBFTValue(float red, float green, float blue, float, float)
		{
			FillRGBValue(red, green, blue);
		}
	private:
		vector<T, Allocator> pixels;
		GammaCurvePtr gamma;
		const float* gammaLUT;
};

typedef NonlinearGrayImage<unsigned char, 255, Image::Gray_Gamma8> MemoryNonlinearGray8Image;

typedef NonlinearGrayImage<unsigned short, 65535, Image::Gray_Gamma16> MemoryNonlinearGray16Image;

template<typename T, unsigned int TMAX, int IDT, class Allocator = allocator<T> >
class NonlinearGrayAImage : public Image
{
	public:
		NonlinearGrayAImage(unsigned int w, unsigned int h) :
			Image(w, h, ImageDataType(IDT)), gamma(NeutralGammaCurve::Get()) { gammaLUT = gamma->GetLookupTable(TMAX); pixels.resize(w * h * 2); FillBitValue(false); }
		NonlinearGrayAImage(unsigned int w, unsigned int h, const vector<RGBMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m), gamma(NeutralGammaCurve::Get()) { gammaLUT = gamma->GetLookupTable(TMAX); pixels.resize(w * h * 2); FillBitValue(false); }
		NonlinearGrayAImage(unsigned int w, unsigned int h, const vector<RGBAMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m), gamma(NeutralGammaCurve::Get()) { gammaLUT = gamma->GetLookupTable(TMAX); pixels.resize(w * h * 2); FillBitValue(false); }
		NonlinearGrayAImage(unsigned int w, unsigned int h, const vector<RGBFTMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m), gamma(NeutralGammaCurve::Get()) { gammaLUT = gamma->GetLookupTable(TMAX); pixels.resize(w * h * 2); FillBitValue(false); }
		~NonlinearGrayAImage() { }

		bool IsOpaque() const
		{
			for(typename vector<T, Allocator>::const_iterator i(pixels.begin()); i != pixels.end(); i += 2)
			{
				if(i[1] < TMAX)
					return false;
			}

			return true;
		}
		bool IsGrayscale() const
		{
			return true;
		}
		bool IsColour() const
		{
			return false;
		}
		bool IsFloat() const
		{
			return false;
		}
		bool IsInt() const
		{
			return true;
		}
		bool IsIndexed() const
		{
			return false;
		}
		bool IsGammaEncoded() const
		{
			return true;
		}
		bool HasAlphaChannel() const
		{
			return true;
		}
		bool HasFilterTransmit() const
		{
			return false;
		}
		unsigned int GetMaxIntValue() const
		{
			return TMAX;
		}
		bool TryDeferDecoding(GammaCurvePtr& g, unsigned int max)
		{
			if (max != TMAX) return false;
			if (!GammaCurve::IsNeutral(gamma)) return !g;
			gamma.swap(g);
			gammaLUT = gamma->GetLookupTable(TMAX);
			return true;
		}

		bool GetBitValue(unsigned int x, unsigned int y) const
		{
            // TODO FIXME - [CLi] This ignores opacity information; other bit-based code doesn't.
			CHECK_BOUNDS(x, y);
			return (pixels[(x + y * width) * 2] != 0);
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			CHECK_BOUNDS(x, y);
			return gammaLUT[pixels[(x + y * width) * 2]];
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			CHECK_BOUNDS(x, y);
			gray = gammaLUT[pixels[(x + y * width) * 2]];
			alpha = pixels[(x + y * width) * 2 + 1] / float(TMAX);
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			red = green = blue = GetGrayValue(x, y);
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			GetGrayAValue(x, y, red, alpha);
			green = blue = red;
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			float alpha;
			GetGrayAValue(x, y, red, alpha);
			green = blue = red;
            Colour::AtoFT(alpha, filter, transm);
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			if(bit == true)
				SetGrayAValue(x, y, TMAX, ALPHA_OPAQUE_INT(TMAX));
			else
				SetGrayAValue(x, y, (unsigned int)0, ALPHA_OPAQUE_INT(TMAX));
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			SetGrayAValue(x, y, gray, ALPHA_OPAQUE);
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			SetGrayAValue(x, y, gray, ALPHA_OPAQUE_INT(TMAX));
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 2] = IntEncode(gamma, gray, TMAX);
			pixels[(x + y * width) * 2 + 1] = T(alpha * float(TMAX));
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 2] = gray;
			pixels[(x + y * width) * 2 + 1] = alpha;
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			SetGrayValue(x, y, RGB2Gray(red, green, blue));
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
            // not really pretty here, but we're doing color math, so we need to decode and re-encode
			SetGrayValue(x, y, RGB2Gray(gammaLUT[red], gammaLUT[green], gammaLUT[blue]));
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float alpha)
		{
			SetGrayAValue(x, y, RGB2Gray(red, green, blue), alpha);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
            // not really pretty here, but we're doing color math, so we need to decode and re-encode
			SetGrayAValue(x, y, RGB2Gray(gammaLUT[red], gammaLUT[green], gammaLUT[blue]), float(alpha) / float(TMAX));
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float filter, float transm)
		{
            SetGrayAValue(x, y, RGB2Gray(red, green, blue), Colour::FTtoA(filter, transm));
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
			SetGrayAValue(x, y, RGB2Gray(col.red(), col.green(), col.blue()), col.FTtoA());
		}

		void FillBitValue(bool bit)
		{
			if(bit == true)
				FillGrayValue(TMAX);
			else
				FillGrayValue((unsigned int)0);
		}
		void FillGrayValue(float gray)
		{
			FillGrayValue(IntEncode(gamma, gray, TMAX));
		}
		void FillGrayValue(unsigned int gray)
		{
			FillGrayAValue(gray, ALPHA_OPAQUE_INT(TMAX));
		}
		void FillGrayAValue(float gray, float alpha)
		{
			// [CLi 2009-09] this was dividing by float(TMAX) - which I presume to have been a bug.
			T g(IntEncode(gamma, gray, TMAX)), a(IntEncode(alpha, TMAX));
			for(typename vector<T, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = g;
				i++;
				*i = a;
			}
		}
		void FillGrayAValue(unsigned int gray, unsigned int alpha)
		{
			for(typename vector<T, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = T(gray);
				i++;
				*i = T(alpha);
			}
		}
		void FillRGBValue(float red, float green, float blue)
		{
			FillGrayValue(RGB2Gray(red, green, blue));
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
            // not really pretty here, but we're doing color math, so we need to decode and re-encode
			FillGrayValue(RGB2Gray(gammaLUT[red], gammaLUT[green], gammaLUT[blue]));
		}
		void FillRGBAValue(float red, float green, float blue, float alpha)
		{
			FillGrayAValue(RGB2Gray(red, green, blue), alpha);
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
            // not really pretty here, but we're doing color math, so we need to decode and re-encode
			FillGrayAValue(RGB2Gray(gammaLUT[red], gammaLUT[green], gammaLUT[blue]), float(alpha) / float(TMAX));
		}
		void FillRGBFTValue(float red, float green, float blue, float filter, float transm)
		{
			FillGrayAValue(RGB2Gray(red, green, blue), Colour::FTtoA(filter, transm));
		}
	private:
		vector<T, Allocator> pixels;
		GammaCurvePtr gamma; 
		const float* gammaLUT;
};

typedef NonlinearGrayAImage<unsigned char, 255, Image::GrayA_Gamma8> MemoryNonlinearGrayA8Image;

typedef NonlinearGrayAImage<unsigned short, 65535, Image::GrayA_Gamma16> MemoryNonlinearGrayA16Image;

template<typename T, unsigned int TMAX, int IDT, class Allocator = allocator<T> >
class NonlinearRGBImage : public Image
{
	public:
		NonlinearRGBImage(unsigned int w, unsigned int h) :
			Image(w, h, ImageDataType(IDT)), gamma(NeutralGammaCurve::Get()) { gammaLUT = gamma->GetLookupTable(TMAX); pixels.resize(w * h * 3); FillBitValue(false); }
		NonlinearRGBImage(unsigned int w, unsigned int h, const vector<RGBMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m), gamma(NeutralGammaCurve::Get()) { gammaLUT = gamma->GetLookupTable(TMAX); pixels.resize(w * h * 3); FillBitValue(false); }
		NonlinearRGBImage(unsigned int w, unsigned int h, const vector<RGBAMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m), gamma(NeutralGammaCurve::Get()) { gammaLUT = gamma->GetLookupTable(TMAX); pixels.resize(w * h * 3); FillBitValue(false); }
		NonlinearRGBImage(unsigned int w, unsigned int h, const vector<RGBFTMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m), gamma(NeutralGammaCurve::Get()) { gammaLUT = gamma->GetLookupTable(TMAX); pixels.resize(w * h * 3); FillBitValue(false); }
		~NonlinearRGBImage() { }

		bool IsOpaque() const
		{
			return true;
		}
		bool IsGrayscale() const
		{
			return false;
		}
		bool IsColour() const
		{
			return true;
		}
		bool IsFloat() const
		{
			return false;
		}
		bool IsInt() const
		{
			return true;
		}
		bool IsIndexed() const
		{
			return false;
		}
		bool IsGammaEncoded() const
		{
			return true;
		}
		bool HasAlphaChannel() const
		{
			return false;
		}
		bool HasFilterTransmit() const
		{
			return false;
		}
		unsigned int GetMaxIntValue() const
		{
			return TMAX;
		}
		bool TryDeferDecoding(GammaCurvePtr& g, unsigned int max)
		{
			if (max != TMAX) return false;
			if (!GammaCurve::IsNeutral(gamma)) return !g;
			gamma.swap(g);
			gammaLUT = gamma->GetLookupTable(TMAX);
			return true;
		}

		bool GetBitValue(unsigned int x, unsigned int y) const
		{
			float red, green, blue;
			GetRGBValue(x, y, red, green, blue);
			return IS_NONZERO_RGB(red, green, blue);
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			float red, green, blue;
			GetRGBValue(x, y, red, green, blue);
			return RGB2Gray(red, green, blue);
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			gray = GetGrayValue(x, y);
			alpha = ALPHA_OPAQUE;
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			CHECK_BOUNDS(x, y);
			red = gammaLUT[pixels[(x + y * width) * 3]];
			green = gammaLUT[pixels[(x + y * width) * 3 + 1]];
			blue = gammaLUT[pixels[(x + y * width) * 3 + 2]];
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			GetRGBValue(x, y, red, green, blue);
			alpha = ALPHA_OPAQUE;
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			GetRGBValue(x, y, red, green, blue);
			filter = transm = FT_OPAQUE;
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			if(bit == true)
				SetGrayValue(x, y, TMAX);
			else
				SetGrayValue(x, y, (unsigned int)0);
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			SetRGBValue(x, y, gray, gray, gray);
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = gray;
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = IntEncode(gamma, gray, TMAX);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int)
		{
			CHECK_BOUNDS(x, y);
			pixels[x + y * width] = gray;
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 3] = IntEncode(gamma, red, TMAX);
			pixels[(x + y * width) * 3 + 1] = IntEncode(gamma, green, TMAX);
			pixels[(x + y * width) * 3 + 2] = IntEncode(gamma, blue, TMAX);
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 3] = T(red);
			pixels[(x + y * width) * 3 + 1] = T(green);
			pixels[(x + y * width) * 3 + 2] = T(blue);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float, float)
		{
			SetRGBValue(x, y, red, green, blue);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
			SetRGBValue(x, y, col.red(), col.green(), col.blue());
		}

		void FillBitValue(bool bit)
		{
			if(bit == true)
				FillGrayValue(TMAX);
			else
				FillGrayValue((unsigned int)0);
		}
		void FillGrayValue(float gray)
		{
			FillGrayValue((unsigned int)(IntEncode(gamma, gray, TMAX)));
		}
		void FillGrayValue(unsigned int gray)
		{
			fill(pixels.begin(), pixels.end(), gray);
		}
		void FillGrayAValue(float gray, float)
		{
			FillRGBValue(gray, gray, gray);
		}
		void FillGrayAValue(unsigned int gray, unsigned int)
		{
			FillRGBValue(gray, gray, gray);
		}
		void FillRGBValue(float red, float green, float blue)
		{
			// [CLi 2009-09] this was dividing by float(TMAX) - which I presume to have been a bug.
			T r(IntEncode(gamma, red, TMAX)), g(IntEncode(gamma, green, TMAX)), b(IntEncode(gamma, blue, TMAX));
			for(typename vector<T, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = r;
				i++;
				*i = g;
				i++;
				*i = b;
			}
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
			for(typename vector<T, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = T(red);
				i++;
				*i = T(green);
				i++;
				*i = T(blue);
			}
		}
		void FillRGBAValue(float red, float green, float blue, float)
		{
			FillRGBValue(red, green, blue);
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int)
		{
			FillRGBValue(red, green, blue);
		}
		void FillRGBFTValue(float red, float green, float blue, float, float)
		{
			FillRGBValue(red, green, blue);
		}
	private:
		vector<T, Allocator> pixels;
		GammaCurvePtr gamma; 
		const float* gammaLUT;
};

typedef NonlinearRGBImage<unsigned char, 255, Image::RGB_Gamma8> MemoryNonlinearRGB8Image;

typedef NonlinearRGBImage<unsigned short, 65535, Image::RGB_Gamma16> MemoryNonlinearRGB16Image;

template<typename T, unsigned int TMAX, int IDT, class Allocator = allocator<T> >
class NonlinearRGBAImage : public Image
{
	public:
		NonlinearRGBAImage(unsigned int w, unsigned int h) :
			Image(w, h, ImageDataType(IDT)), gamma(NeutralGammaCurve::Get()) { gammaLUT = gamma->GetLookupTable(TMAX); pixels.resize(w * h * 4); FillBitValue(false); }
		NonlinearRGBAImage(unsigned int w, unsigned int h, const vector<RGBMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m), gamma(NeutralGammaCurve::Get()) { gammaLUT = gamma->GetLookupTable(TMAX); pixels.resize(w * h * 4); FillBitValue(false); }
		NonlinearRGBAImage(unsigned int w, unsigned int h, const vector<RGBAMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m), gamma(NeutralGammaCurve::Get()) { gammaLUT = gamma->GetLookupTable(TMAX); pixels.resize(w * h * 4); FillBitValue(false); }
		NonlinearRGBAImage(unsigned int w, unsigned int h, const vector<RGBFTMapEntry>& m) :
			Image(w, h, ImageDataType(IDT), m), gamma(NeutralGammaCurve::Get()) { gammaLUT = gamma->GetLookupTable(TMAX); pixels.resize(w * h * 4); FillBitValue(false); }
		~NonlinearRGBAImage() { }

		bool IsOpaque() const
		{
			for(typename vector<T, Allocator>::const_iterator i(pixels.begin()); i != pixels.end(); i += 4)
			{
				if(i[3] < TMAX)
					return false;
			}

			return true;
		}
		bool IsGrayscale() const
		{
			return false;
		}
		bool IsColour() const
		{
			return true;
		}
		bool IsFloat() const
		{
			return false;
		}
		bool IsInt() const
		{
			return true;
		}
		bool IsIndexed() const
		{
			return false;
		}
		bool IsGammaEncoded() const
		{
			return true;
		}
		bool HasAlphaChannel() const
		{
			return true;
		}
		bool HasFilterTransmit() const
		{
			return false;
		}
		unsigned int GetMaxIntValue() const
		{
			return TMAX;
		}
		bool TryDeferDecoding(GammaCurvePtr& g, unsigned int max)
		{
			if (max != TMAX) return false;
			if (!GammaCurve::IsNeutral(gamma)) return !g;
			gamma.swap(g);
			gammaLUT = gamma->GetLookupTable(TMAX);
			return true;
		}

		bool GetBitValue(unsigned int x, unsigned int y) const
		{
            // TODO FIXME - [CLi] This ignores opacity information; other bit-based code doesn't.
			float red, green, blue, alpha;
			GetRGBAValue(x, y, red, green, blue, alpha);
			return IS_NONZERO_RGB(red, green, blue);
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			float red, green, blue, alpha;
			GetRGBAValue(x, y, red, green, blue, alpha);
			return RGB2Gray(red, green, blue);
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			float red, green, blue;
			GetRGBAValue(x, y, red, green, blue, alpha);
			gray = RGB2Gray(red, green, blue);
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			float alpha;
			GetRGBAValue(x, y, red, green, blue, alpha);
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			CHECK_BOUNDS(x, y);
			red = gammaLUT[pixels[(x + y * width) * 4]];
			green = gammaLUT[pixels[(x + y * width) * 4 + 1]];
			blue = gammaLUT[pixels[(x + y * width) * 4 + 2]];
			alpha = float(pixels[(x + y * width) * 4 + 3]) / float(TMAX);
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			float alpha;
			GetRGBAValue(x, y, red, green, blue, alpha);
            Colour::AtoFT(alpha, filter, transm);
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			if(bit == true)
				SetGrayValue(x, y, TMAX);
			else
				SetGrayValue(x, y, (unsigned int)0);
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			SetRGBAValue(x, y, gray, gray, gray, ALPHA_OPAQUE);
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			SetRGBAValue(x, y, gray, gray, gray, ALPHA_OPAQUE_INT(TMAX));
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float alpha)
		{
			SetRGBAValue(x, y, gray, gray, gray, alpha);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int alpha)
		{
			SetRGBAValue(x, y, gray, gray, gray, alpha);
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			SetRGBAValue(x, y, red, green, blue, ALPHA_OPAQUE);
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
			SetRGBAValue(x, y, red, green, blue, ALPHA_OPAQUE_INT(TMAX));
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 4] = T(IntEncode(gamma, red, TMAX));
			pixels[(x + y * width) * 4 + 1] = T(IntEncode(gamma, green, TMAX));
			pixels[(x + y * width) * 4 + 2] = T(IntEncode(gamma, blue, TMAX));
			pixels[(x + y * width) * 4 + 3] = T(IntEncode(alpha, TMAX));
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 4] = T(red);
			pixels[(x + y * width) * 4 + 1] = T(green);
			pixels[(x + y * width) * 4 + 2] = T(blue);
			pixels[(x + y * width) * 4 + 3] = T(alpha);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float filter, float transm)
		{
            SetRGBAValue(x, y, red, green, blue, Colour::FTtoA(filter, transm));
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
			CHECK_BOUNDS(x, y);
			pixels[(x + y * width) * 4] = T(col.red());
			pixels[(x + y * width) * 4 + 1] = T(col.green());
			pixels[(x + y * width) * 4 + 2] = T(col.blue());
			pixels[(x + y * width) * 4 + 3] = T(col.FTtoA());
		}

		void FillBitValue(bool bit)
		{
			if(bit == true)
				FillGrayValue(TMAX);
			else
				FillGrayValue((unsigned int)0);
		}
		void FillGrayValue(float gray)
		{
			FillRGBAValue(gray, gray, gray, ALPHA_OPAQUE);
		}
		void FillGrayValue(unsigned int gray)
		{
			FillRGBAValue(gray, gray, gray, ALPHA_OPAQUE_INT(TMAX));
		}
		void FillGrayAValue(float gray, float alpha)
		{
			FillRGBAValue(gray, gray, gray, alpha);
		}
		void FillGrayAValue(unsigned int gray, unsigned int alpha)
		{
			FillRGBAValue(gray, gray, gray, alpha);
		}
		void FillRGBValue(float red, float green, float blue)
		{
			FillRGBAValue(red, green, blue, ALPHA_OPAQUE);
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
			FillRGBAValue(red, green, blue, ALPHA_OPAQUE_INT(TMAX));
		}
		void FillRGBAValue(float red, float green, float blue, float alpha)
		{
			// [CLi 2009-09] this was dividing by float(TMAX) - which I presume to have been a bug.
			T r(IntEncode(gamma, red, TMAX)), g(IntEncode(gamma, green, TMAX)), b(IntEncode(gamma, blue, TMAX)), a(IntEncode(alpha, TMAX));
			for(typename vector<T, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = r;
				i++;
				*i = g;
				i++;
				*i = b;
				i++;
				*i = a;
			}
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
			for(typename vector<T, Allocator>::iterator i(pixels.begin()); i != pixels.end(); i++)
			{
				*i = T(red);
				i++;
				*i = T(green);
				i++;
				*i = T(blue);
				i++;
				*i = T(alpha);
			}
		}
		void FillRGBFTValue(float red, float green, float blue, float filter, float transm)
		{
            FillRGBAValue(red, green, blue, Colour::FTtoA(filter, transm));
		}
	private:
		vector<T, Allocator> pixels;
		GammaCurvePtr gamma; 
		const float* gammaLUT;
};

typedef NonlinearRGBAImage<unsigned char, 255, Image::RGBA_Gamma8> MemoryNonlinearRGBA8Image;

typedef NonlinearRGBAImage<unsigned short, 65535, Image::RGBA_Gamma16> MemoryNonlinearRGBA16Image;

// sample basic file-based pixel container. not very efficient and limited to 2gb files.
// it is expected that for performance reasons, platforms will provide their own specific
// implementation of this; hence, this should be considered only a fall-back default.
class FileBackedPixelContainer
{
	public:
		typedef long size_type;
		enum
		{
			RED    = 0,
			GREEN  = 1,
			BLUE   = 2,
			FILTER = 3,
			TRANSM = 4
		};
		class pixel_type
		{
			public:
				pixel_type()
				{
					elements[RED] = 0.0;
					elements[GREEN] = 0.0;
					elements[BLUE] = 0.0;
					elements[FILTER] = 0.0;
					elements[TRANSM] = 0.0;
				}
				pixel_type(const COLC *vals)
				{
					memcpy(elements, vals, sizeof(elements));
				}
				pixel_type(float r, float g, float b, float f, float t)
				{
					elements[RED] = r;
					elements[GREEN] = g;
					elements[BLUE] = b;
					elements[FILTER] = f;
					elements[TRANSM] = t;
				}
				~pixel_type() {}
				operator COLC *() { return elements; }
				operator const COLC *() const { return elements; }
				bool operator==(float val) const { return elements[RED] == val && elements[GREEN] == val && elements[BLUE] == val && elements[FILTER] == val && elements[TRANSM] == val; }
				bool operator!=(float val) const { return !(*this == val); }

			protected:
				COLC elements[5];
		};

		FileBackedPixelContainer(size_type width, size_type height):
			m_File(NULL), m_Width(width), m_Height(height), m_xPos(0), m_yPos(0), m_Dirty(false), m_Path(POV_PLATFORM_BASE.CreateTemporaryFile())
		{
			if ((m_File = fopen(UCS2toASCIIString(m_Path).c_str(), "wb+")) == NULL)
				throw POV_EXCEPTION(kCannotOpenFileErr, "Cannot open backing file for intermediate image storage.");
			m_Buffer.resize(width);
			m_Committed.resize(height);
		}

		virtual ~FileBackedPixelContainer()
		{
			Flush();
			fclose(m_File);
			POV_PLATFORM_BASE.DeleteTemporaryFile(m_Path);
		}

		void Flush(void)
		{
			if (m_Dirty)
			{
				SetLine(m_yPos, &m_Buffer[0]);
				m_Dirty = false;
			}
		}

		void SetPixel(size_type x, size_type y, const pixel_type& pixel)
		{
			SetPos(x, y);
			if (memcmp(m_Buffer[m_xPos], &pixel, sizeof(pixel)))
			{
				memcpy(m_Buffer[m_xPos], &pixel, sizeof(pixel));
				m_Dirty = true;
			}
			NextPixel();
		}

		void SetPixel(size_type x, size_type y, float red, float green, float blue, float filter, float transm)
		{
			SetPos(x, y);
			COLC *p = m_Buffer[m_xPos];
			if (p[RED] != red || p[GREEN] != green || p[BLUE] != blue || p[FILTER] != filter || p[TRANSM] != transm)
			{
				p[RED] = red;
				p[GREEN] = green;
				p[BLUE] = blue;
				p[FILTER] = filter;
				p[TRANSM] = transm;
				m_Dirty = true;
			}
		}

		void SetPixel(float red, float green, float blue, float filter, float transm)
		{
			COLC *p = m_Buffer[m_xPos];
			if (p[RED] != red || p[GREEN] != green || p[BLUE] != blue || p[FILTER] != filter || p[TRANSM] != transm)
			{
				p[RED] = red;
				p[GREEN] = green;
				p[BLUE] = blue;
				p[FILTER] = filter;
				p[TRANSM] = transm;
				m_Dirty = true;
			}
			NextPixel();
		}

		void GetPixel(pixel_type& pixel)
		{
			memcpy(&pixel, m_Buffer[m_xPos], sizeof(pixel));
			NextPixel();
		}
		
		void GetPixel(float& red, float& green, float& blue, float& filter, float& transm)
		{
			red = m_Buffer[m_xPos][RED];
			green = m_Buffer[m_xPos][GREEN];
			blue = m_Buffer[m_xPos][BLUE];
			filter = m_Buffer[m_xPos][FILTER];
			transm = m_Buffer[m_xPos][TRANSM];
			NextPixel();
		}

		void GetPixel(size_type x, size_type y, pixel_type& pixel)
		{
			SetPos(x, y);
			memcpy(&pixel, m_Buffer[m_xPos], sizeof(pixel));
		}
		
		void GetPixel(size_type x, size_type y, float& red, float& green, float& blue, float& filter, float& transm)
		{
			SetPos(x, y);
			red = m_Buffer[m_xPos][RED];
			green = m_Buffer[m_xPos][GREEN];
			blue = m_Buffer[m_xPos][BLUE];
			filter = m_Buffer[m_xPos][FILTER];
			transm = m_Buffer[m_xPos][TRANSM];
		}

		void ClearCache(const pixel_type& pixel = pixel_type())
		{
			Flush();
			for (int i = 0; i < m_Width; i++)
				memcpy(&m_Buffer[i], &pixel, sizeof(pixel));
		}

		void FillLine(size_type y, pixel_type& pixel)
		{
			ClearCache(pixel);
			SetLine(y, &m_Buffer[0]);
		}

		void Fill(pixel_type& pixel)
		{
			bool notBlank(pixel != 0.0);

			ClearCache(pixel);
			for (int i = 0; i < m_Height; i++)
				if (notBlank || m_Committed[i])
					SetLine(i, &m_Buffer[0]);
		}

		void Fill(float red, float green, float blue, float filter, float transm)
		{
			pixel_type pixel(red, green, blue, filter, transm);
			Fill(pixel);
		}

	protected:
		FILE				*m_File;
		bool				m_Dirty;
		size_type			m_Width;
		size_type			m_Height;
		size_type			m_xPos;
		size_type			m_yPos;
		UCS2String			m_Path;
		vector<bool>		m_Committed;
		vector<pixel_type>	m_Buffer;

		void SetPos(size_type x, size_type y, bool cache = true)
		{
			if (x < 0 || x >= m_Width || y < 0 || y >= m_Height)
				throw POV_EXCEPTION(kFileDataErr, "Invalid coordinates in intermediate image file seek.");
			if (y == m_yPos)
			{
				m_xPos = x;
				return;
			}
			Flush();
			if (cache)
				GetLine(y, &m_Buffer[0]);
			m_xPos = x;
			m_yPos = y;
		}

		void NextPixel(void)
		{
			if (m_xPos == m_Width - 1)
			{
				if (m_yPos == m_Height - 1)
					return;
				SetPos(0, m_yPos + 1);
			}
			else
				m_xPos++;
		}

		void SetLine(size_type y, const pixel_type *src)
		{
			size_type pos = sizeof(pixel_type) * m_Width * y;
			if (fseek(m_File, pos, SEEK_SET) != 0)
				throw POV_EXCEPTION(kFileDataErr, "Intermediate image storage backing file seek failed.");
			if (fwrite(src, sizeof(pixel_type), m_Width, m_File) != m_Width)
				throw POV_EXCEPTION(kFileDataErr, "Intermediate image storage backing file write failed.");
			m_Committed[y] = true;
		}

		void GetLine(size_type y, pixel_type *dest)
		{
			if (m_Committed[y] == false)
			{
				COLC pixel[5] = {0.0, 0.0, 0.0, 0.0, 0.0};
				for (int i = 0; i < m_Width; i++)
					memcpy(dest++, pixel, sizeof(pixel));
				return;
			}
			size_type pos = sizeof(pixel_type) * m_Width * y;
			if (fseek(m_File, pos, SEEK_SET) != 0)
				throw POV_EXCEPTION(kFileDataErr, "Intermediate image storage backing file seek failed.");
			if (fread(dest, sizeof(pixel_type), m_Width, m_File) != m_Width)
				throw POV_EXCEPTION(kFileDataErr, "Intermediate image storage backing file read failed.");
		}

	private:
		// not available
		FileBackedPixelContainer(void) {}
};

class FileRGBFTImage : public Image
{
	public:
		FileRGBFTImage(unsigned int w, unsigned int h): Image(w, h, RGBFT_Float), pixels(width, height) { }
		~FileRGBFTImage() { }

		bool IsGrayscale() const { return false; }
		bool IsColour() const { return true; }
		bool IsFloat() const { return true; }
		bool IsInt() const { return false; }
		bool IsIndexed() const { return false; }
		bool IsGammaEncoded() const { return false; }
		bool HasAlphaChannel() const { return false; }
		bool HasFilterTransmit() const { return true; }
		unsigned int GetMaxIntValue() const { return 255; }
		void SetEncodingGamma(GammaCurvePtr gamma) { ; }
		bool TryDeferDecoding(GammaCurvePtr&, unsigned int) { return false; }
		bool IsOpaque() const { throw POV_EXCEPTION(kUncategorizedError, "Internal error: IsOpaque() not supported in FileRGBFTImage"); }
		bool GetBitValue(unsigned int x, unsigned int y) const
		{
            // TODO FIXME - [CLi] This ignores opacity information; other bit-based code doesn't.
			float red, green, blue, filter, transm;
			GetRGBFTValue(x, y, red, green, blue, filter, transm);
			return IS_NONZERO_RGB(red, green, blue);
		}
		float GetGrayValue(unsigned int x, unsigned int y) const
		{
			float red, green, blue, filter, transm;
			GetRGBFTValue(x, y, red, green, blue, filter, transm);
			return RGB2Gray(red, green, blue);
		}
		void GetGrayAValue(unsigned int x, unsigned int y, float& gray, float& alpha) const
		{
			float red, green, blue, filter, transm;
			GetRGBFTValue(x, y, red, green, blue, filter, transm);
			gray = RGB2Gray(red, green, blue);
            alpha = Colour::FTtoA(filter, transm);
		}
		void GetRGBValue(unsigned int x, unsigned int y, float& red, float& green, float& blue) const
		{
			float filter, transm;
			GetRGBFTValue(x, y, red, green, blue, filter, transm);
		}
		void GetRGBAValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& alpha) const
		{
			float filter, transm;
			GetRGBFTValue(x, y, red, green, blue, filter, transm);
            alpha = Colour::FTtoA(filter, transm);
		}
		void GetRGBFTValue(unsigned int x, unsigned int y, float& red, float& green, float& blue, float& filter, float& transm) const
		{
			CHECK_BOUNDS(x, y);
			pixels.GetPixel(x, y, red, green, blue, filter, transm);
		}

		void SetBitValue(unsigned int x, unsigned int y, bool bit)
		{
			if(bit == true)
				SetGrayValue(x, y, 1.0f);
			else
				SetGrayValue(x, y, 0.0f);
		}
		void SetGrayValue(unsigned int x, unsigned int y, float gray)
		{
			SetRGBFTValue(x, y, gray, gray, gray, FT_OPAQUE, FT_OPAQUE);
		}
		void SetGrayValue(unsigned int x, unsigned int y, unsigned int gray)
		{
			SetGrayValue(x, y, float(gray) / 255.0f);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, float gray, float alpha)
		{
			// TODO - should alpha be converted to filter and transm? [trf]
            float filter, transm;
            Colour::AtoFT(alpha, filter, transm);
			SetRGBFTValue(x, y, gray, gray, gray, filter, transm);
		}
		void SetGrayAValue(unsigned int x, unsigned int y, unsigned int gray, unsigned int alpha)
		{
			// TODO - should alpha be converted to filter and transm? [trf]
			float c = float(gray) / 255.0f;
            float filter, transm;
            Colour::AtoFT(float(alpha) / 255.0f, filter, transm);
			SetRGBFTValue(x, y, c, c, c, filter, transm);
		}
		void SetRGBValue(unsigned int x, unsigned int y, float red, float green, float blue)
		{
			SetRGBFTValue(x, y, red, green, blue, FT_OPAQUE, FT_OPAQUE);
		}
		void SetRGBValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue)
		{
			SetRGBFTValue(x, y, float(red) / 255.0f, float(green) / 255.0f, float(blue) / 255.0f, FT_OPAQUE, FT_OPAQUE);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, float red, float green, float blue, float alpha)
		{
			// TODO - should alpha be converted to filter and transm? [trf]
			float filter, transm;
            Colour::AtoFT(alpha, filter, transm);
            SetRGBFTValue(x, y, red, green, blue, filter, transm);
		}
		void SetRGBAValue(unsigned int x, unsigned int y, unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
			// TODO - should alpha be converted to filter and transm? [trf]
            float filter, transm;
            Colour::AtoFT(float(alpha) / 255.0f, filter, transm);
			SetRGBFTValue(x, y, float(red) / 255.0f, float(green) / 255.0f, float(blue) / 255.0f, filter, transm);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, float red, float green, float blue, float filter, float transm)
		{
			CHECK_BOUNDS(x, y);
			pixels.SetPixel(x, y, red, green, blue, filter, transm);
		}
		void SetRGBFTValue(unsigned int x, unsigned int y, const Colour& col)
		{
			CHECK_BOUNDS(x, y);
			pixels.SetPixel(x, y, *col);
		}
		void FillBitValue(bool bit)
		{
			if(bit == true)
				FillGrayValue(1.0f);
			else
				FillGrayValue(0.0f);
		}
		void FillGrayValue(float gray)
		{
			FillRGBFTValue(gray, gray, gray, FT_OPAQUE, FT_OPAQUE);
		}
		void FillGrayValue(unsigned int gray)
		{
			FillGrayValue(float(gray) / 255.0f);
		}
		void FillGrayAValue(float gray, float alpha)
		{
			// TODO - should alpha be converted to filter and transm? [trf]
			float filter, transm;
            Colour::AtoFT(alpha, filter, transm);
			FillRGBFTValue(gray, gray, gray, filter, transm);
		}
		void FillGrayAValue(unsigned int gray, unsigned int alpha)
		{
			// TODO - should alpha be converted to filter and transm? [trf]
			FillGrayAValue(float(gray) / 255.0f, float(alpha) / 255.0f);
		}
		void FillRGBValue(float red, float green, float blue)
		{
			FillRGBFTValue(red, green, blue, FT_OPAQUE, FT_OPAQUE);
		}
		void FillRGBValue(unsigned int red, unsigned int green, unsigned int blue)
		{
			FillRGBFTValue(float(red) / 255.0f, float(green) / 255.0f, float(blue) / 255.0f, FT_OPAQUE, FT_OPAQUE);
		}
		void FillRGBAValue(float red, float green, float blue, float alpha)
		{
			// TODO - should alpha be converted to filter and transm? [trf]
			float filter, transm;
            Colour::AtoFT(alpha, filter, transm);
			FillRGBFTValue(red, green, blue, filter, transm);
		}
		void FillRGBAValue(unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha)
		{
			// TODO - should alpha be converted to filter and transm? [trf]
			float filter, transm;
            Colour::AtoFT(alpha, filter, transm);
			FillRGBFTValue(red, green, blue, filter, transm);
		}
		void FillRGBFTValue(float red, float green, float blue, float filter, float transm)
		{
			pixels.Fill(red, green, blue, filter, transm);
		}

	protected:
		mutable FileBackedPixelContainer pixels;
};

void RGBMap2RGBAMap(const vector<Image::RGBMapEntry>& m, vector<Image::RGBAMapEntry>& n);
void RGBMap2RGBFTMap(const vector<Image::RGBMapEntry>& m, vector<Image::RGBFTMapEntry>& n);
void RGBAMap2RGBMap(const vector<Image::RGBAMapEntry>& m, vector<Image::RGBMapEntry>& n);
void RGBAMap2RGBFTMap(const vector<Image::RGBAMapEntry>& m, vector<Image::RGBFTMapEntry>& n);
void RGBFTMap2RGBMap(const vector<Image::RGBFTMapEntry>& m, vector<Image::RGBMapEntry>& n);
void RGBFTMap2RGBAMap(const vector<Image::RGBFTMapEntry>& m, vector<Image::RGBAMapEntry>& n);

void RGBMap2RGBAMap(const vector<Image::RGBMapEntry>& m, vector<Image::RGBAMapEntry>& n)
{
	n.clear();
	n.reserve(m.size());

	for(vector<Image::RGBMapEntry>::const_iterator i(m.begin()); i != m.end(); i++)
		n.push_back(Image::RGBAMapEntry(i->red, i->green, i->blue, ALPHA_OPAQUE));
}

void RGBMap2RGBFTMap(const vector<Image::RGBMapEntry>& m, vector<Image::RGBFTMapEntry>& n)
{
	n.clear();
	n.reserve(m.size());

	for(vector<Image::RGBMapEntry>::const_iterator i(m.begin()); i != m.end(); i++)
		n.push_back(Image::RGBFTMapEntry(i->red, i->green, i->blue, FT_OPAQUE, FT_OPAQUE));
}

void RGBAMap2RGBMap(const vector<Image::RGBAMapEntry>& m, vector<Image::RGBMapEntry>& n)
{
	n.clear();
	n.reserve(m.size());

	for(vector<Image::RGBAMapEntry>::const_iterator i(m.begin()); i != m.end(); i++)
		n.push_back(Image::RGBMapEntry(i->red, i->green, i->blue));
}

void RGBAMap2RGBFTMap(const vector<Image::RGBAMapEntry>& m, vector<Image::RGBFTMapEntry>& n)
{
	n.clear();
	n.reserve(m.size());

	for(vector<Image::RGBAMapEntry>::const_iterator i(m.begin()); i != m.end(); i++)
    {
		float filter, transm;
        Colour::AtoFT(i->alpha, filter, transm);
		n.push_back(Image::RGBFTMapEntry(i->red, i->green, i->blue, filter, transm));
    }
}

void RGBFTMap2RGBMap(const vector<Image::RGBFTMapEntry>& m, vector<Image::RGBMapEntry>& n)
{
	n.clear();
	n.reserve(m.size());

	for(vector<Image::RGBFTMapEntry>::const_iterator i(m.begin()); i != m.end(); i++)
		n.push_back(Image::RGBMapEntry(i->red, i->green, i->blue));
}

void RGBFTMap2RGBAMap(const vector<Image::RGBFTMapEntry>& m, vector<Image::RGBAMapEntry>& n)
{
	n.clear();
	n.reserve(m.size());

	for(vector<Image::RGBFTMapEntry>::const_iterator i(m.begin()); i != m.end(); i++)
        n.push_back(Image::RGBAMapEntry(i->red, i->green, i->blue, Colour::FTtoA(i->filter, i->transm)));
}

Image *Image::Create(unsigned int w, unsigned int h, ImageDataType t, bool f)
{
	try
	{
		switch(t)
		{
			case Bit_Map:
				return new MemoryBitMapImage(w, h);
			case Gray_Int8:
				return new MemoryGray8Image(w, h);
			case Gray_Int16:
				return new MemoryGray16Image(w, h);
			case GrayA_Int8:
				return new MemoryGrayA8Image(w, h);
			case GrayA_Int16:
				return new MemoryGrayA16Image(w, h);
			case RGB_Int8:
				return new MemoryRGB8Image(w, h);
			case RGB_Int16:
				return new MemoryRGB16Image(w, h);
			case RGBA_Int8:
				return new MemoryRGBA8Image (w, h);
			case RGBA_Int16:
				return new MemoryRGBA16Image(w, h);
			case RGBFT_Float:
				#ifdef FILE_MAPPED_IMAGE_ALLOCATOR
				if (f == true)
				{
					try
					{
						return new RGBFTImage<FILE_MAPPED_IMAGE_ALLOCATOR<float> >(w, h);
					}
					catch(std:bad_alloc&)
					{
						// if there isn't enough virtual address space to allocate a memory-backed image, try a file-backed one
						return new FileRGBFTImage(w, h);
					}
				}
				#endif
				try
				{
					return new MemoryRGBFTImage(w, h);
				}
				catch(std::bad_alloc&)
				{
					// if there isn't the RAM to allocate a memory-backed image, try a file-backed one
					if (f == true)
						return new FileRGBFTImage(w, h);
					throw;
				}
			case RGB_Gamma8:
				return new MemoryNonlinearRGB8Image(w, h);
			case RGB_Gamma16:
				return new MemoryNonlinearRGB16Image(w, h);
			case RGBA_Gamma8:
				return new MemoryNonlinearRGBA8Image (w, h);
			case RGBA_Gamma16:
				return new MemoryNonlinearRGBA16Image(w, h);
			case Gray_Gamma8:
				return new MemoryNonlinearGray8Image(w, h);
			case Gray_Gamma16:
				return new MemoryNonlinearGray16Image(w, h);
			case GrayA_Gamma8:
				return new MemoryNonlinearGrayA8Image(w, h);
			case GrayA_Gamma16:
				return new MemoryNonlinearGrayA16Image(w, h);
			default:
				throw POV_EXCEPTION_STRING("Undefined image format in Image::Create");
		}
	}
	catch(std::bad_alloc&)
	{
		throw POV_EXCEPTION(kOutOfMemoryErr, "Insufficient memory to allocate intermediate image storage.");
	}
}

Image *Image::Create(unsigned int w, unsigned int h, ImageDataType t, const vector<RGBMapEntry>& m, bool f)
{
	try
	{
		switch(t)
		{
			case Bit_Map:
				return new MemoryBitMapImage(w, h, m);
			case Colour_Map:
				return new MemoryColourMapImage(w, h, m);
			case Gray_Int8:
				return new MemoryGray8Image(w, h, m);
			case Gray_Int16:
				return new MemoryGray16Image(w, h, m);
			case GrayA_Int8:
				return new MemoryGrayA8Image(w, h, m);
			case GrayA_Int16:
				return new MemoryGrayA16Image(w, h, m);
			case RGB_Int8:
				return new MemoryRGB8Image(w, h, m);
			case RGB_Int16:
				return new MemoryRGB16Image(w, h, m);
			case RGBA_Int8:
				return new MemoryRGBA8Image (w, h, m);
			case RGBA_Int16:
				return new MemoryRGBA16Image(w, h, m);
			case RGBFT_Float:
				#ifdef FILE_MAPPED_IMAGE_ALLOCATOR
					if(f == true)
						return new RGBFTImage<FILE_MAPPED_IMAGE_ALLOCATOR<float> >(w, h, m);
				#endif
				return new MemoryRGBFTImage(w, h, m);
			case RGB_Gamma8:
				return new MemoryNonlinearRGB8Image(w, h, m);
			case RGB_Gamma16:
				return new MemoryNonlinearRGB16Image(w, h, m);
			case RGBA_Gamma8:
				return new MemoryNonlinearRGBA8Image (w, h, m);
			case RGBA_Gamma16:
				return new MemoryNonlinearRGBA16Image(w, h, m);
			case Gray_Gamma8:
				return new MemoryNonlinearGray8Image(w, h, m);
			case Gray_Gamma16:
				return new MemoryNonlinearGray16Image(w, h, m);
			case GrayA_Gamma8:
				return new MemoryNonlinearGrayA8Image(w, h, m);
			case GrayA_Gamma16:
				return new MemoryNonlinearGrayA16Image(w, h, m);
			default:
				throw POV_EXCEPTION_STRING("Image::Create Exception TODO"); // TODO FIXME WIP
		}
	}
	catch(std::bad_alloc&)
	{
		throw POV_EXCEPTION(kOutOfMemoryErr, "Insufficient memory to allocate intermediate image storage.");
	}
}

Image *Image::Create(unsigned int w, unsigned int h, ImageDataType t, const vector<RGBAMapEntry>& m, bool f)
{
	try
	{
		switch(t)
		{
			case Bit_Map:
				return new MemoryBitMapImage(w, h, m);
			case Colour_Map:
				return new MemoryColourMapImage(w, h, m);
			case Gray_Int8:
				return new MemoryGray8Image(w, h, m);
			case Gray_Int16:
				return new MemoryGray16Image(w, h, m);
			case GrayA_Int8:
				return new MemoryGrayA8Image(w, h, m);
			case GrayA_Int16:
				return new MemoryGrayA16Image(w, h, m);
			case RGB_Int8:
				return new MemoryRGB8Image(w, h, m);
			case RGB_Int16:
				return new MemoryRGB16Image(w, h, m);
			case RGBA_Int8:
				return new MemoryRGBA8Image (w, h, m);
			case RGBA_Int16:
				return new MemoryRGBA16Image(w, h, m);
			case RGBFT_Float:
				#ifdef FILE_MAPPED_RGBFT_IMAGE_ALLOCATOR
					if(f == true)
						return new RGBFTImage<FILE_MAPPED_RGBFT_IMAGE_ALLOCATOR<float> >(w, h, m);
				#endif
				return new MemoryRGBFTImage(w, h, m);
			case RGB_Gamma8:
				return new MemoryNonlinearRGB8Image(w, h, m);
			case RGB_Gamma16:
				return new MemoryNonlinearRGB16Image(w, h, m);
			case RGBA_Gamma8:
				return new MemoryNonlinearRGBA8Image (w, h, m);
			case RGBA_Gamma16:
				return new MemoryNonlinearRGBA16Image(w, h, m);
			case Gray_Gamma8:
				return new MemoryNonlinearGray8Image(w, h, m);
			case Gray_Gamma16:
				return new MemoryNonlinearGray16Image(w, h, m);
			case GrayA_Gamma8:
				return new MemoryNonlinearGrayA8Image(w, h, m);
			case GrayA_Gamma16:
				return new MemoryNonlinearGrayA16Image(w, h, m);
			default:
				throw POV_EXCEPTION_STRING("Image::Create Exception TODO"); // TODO FIXME WIP
		}
	}
	catch(std::bad_alloc&)
	{
		throw POV_EXCEPTION(kOutOfMemoryErr, "Insufficient memory to allocate intermediate image storage.");
	}
}

Image *Image::Create(unsigned int w, unsigned int h, ImageDataType t, const vector<RGBFTMapEntry>& m, bool f)
{
	try
	{
		switch(t)
		{
			case Bit_Map:
				return new MemoryBitMapImage(w, h, m);
			case Colour_Map:
				return new MemoryColourMapImage(w, h, m);
			case Gray_Int8:
				return new MemoryGray8Image(w, h, m);
			case Gray_Int16:
				return new MemoryGray16Image(w, h, m);
			case GrayA_Int8:
				return new MemoryGrayA8Image(w, h, m);
			case GrayA_Int16:
				return new MemoryGrayA16Image(w, h, m);
			case RGB_Int8:
				return new MemoryRGB8Image(w, h, m);
			case RGB_Int16:
				return new MemoryRGB16Image(w, h, m);
			case RGBA_Int8:
				return new MemoryRGBA8Image (w, h, m);
			case RGBA_Int16:
				return new MemoryRGBA16Image(w, h, m);
			case RGBFT_Float:
				#ifdef FILE_MAPPED_IMAGE_ALLOCATOR
					if(f == true)
						return new RGBFTImage<FILE_MAPPED_IMAGE_ALLOCATOR<float> >(w, h, m);
				#endif
				return new MemoryRGBFTImage(w, h, m);
			case RGB_Gamma8:
				return new MemoryNonlinearRGB8Image(w, h, m);
			case RGB_Gamma16:
				return new MemoryNonlinearRGB16Image(w, h, m);
			case RGBA_Gamma8:
				return new MemoryNonlinearRGBA8Image (w, h, m);
			case RGBA_Gamma16:
				return new MemoryNonlinearRGBA16Image(w, h, m);
			case Gray_Gamma8:
				return new MemoryNonlinearGray8Image(w, h, m);
			case Gray_Gamma16:
				return new MemoryNonlinearGray16Image(w, h, m);
			case GrayA_Gamma8:
				return new MemoryNonlinearGrayA8Image(w, h, m);
			case GrayA_Gamma16:
				return new MemoryNonlinearGrayA16Image(w, h, m);
			default:
				throw POV_EXCEPTION_STRING("Image::Create Exception TODO"); // TODO FIXME WIP
		}
	}
	catch(std::bad_alloc&)
	{
		throw POV_EXCEPTION(kOutOfMemoryErr, "Insufficient memory to allocate intermediate image storage.");
	}
}

Image *Image::Read(ImageFileType type, IStream *file, const ReadOptions& options)
{
	#ifdef SYS_TO_STANDARD
		if (type == SYS)
			type = SYS_TO_STANDARD ;
	#endif

	switch (type)
	{
		case HDR:
			throw POV_EXCEPTION(kParamErr, POV_SPEC_NOT_SUPPORT);
		case EXR:
			throw POV_EXCEPTION(kParamErr, POV_SPEC_NOT_SUPPORT);
		case PNG:
			throw POV_EXCEPTION(kParamErr, POV_SPEC_NOT_SUPPORT);
		case GIF:
			throw POV_EXCEPTION(kParamErr, POV_SPEC_NOT_SUPPORT);
		case POT:
			throw POV_EXCEPTION(kParamErr, POV_SPEC_NOT_SUPPORT);
		case TGA:
			return (Targa::Read(file, options));
		case JPEG:
			throw POV_EXCEPTION(kParamErr, POV_SPEC_NOT_SUPPORT);
		case IFF:
			return (Iff::Read(file, options));
		case PGM:
			return (Pgm::Read(file, options));
		case PPM:
			return (Ppm::Read(file, options));
		case BMP:
			return (Bmp::Read(file, options));
		case TIFF:
			throw POV_EXCEPTION(kParamErr, POV_SPEC_NOT_SUPPORT);
		case SYS:
			throw POV_EXCEPTION(kCannotOpenFileErr, "This platform has not defined a SYS file type");
		default:
			throw POV_EXCEPTION(kParamErr, "Invalid file type");
	}
}

void Image::Write(ImageFileType type, OStream *file, const Image *image, const WriteOptions& options)
{
	if (image->GetWidth() == 0 || image->GetHeight() == 0)
		throw POV_EXCEPTION(kParamErr, "Invalid image size for output");
	
	if (file == NULL)
		throw POV_EXCEPTION(kCannotOpenFileErr, "Invalid image file");
	
	#ifdef SYS_TO_STANDARD
		if (type == SYS)
			type = SYS_TO_STANDARD;
	#endif
	
	switch (type)
	{
		case GIF:
		case IFF:
		case PGM:
		case TIFF:
		case POT:
			throw POV_EXCEPTION(kParamErr, "Unsupported file type for output");
		case SYS:
			throw POV_EXCEPTION(kCannotOpenFileErr, "This platform has not defined a SYS file type");
		case HDR:
			throw POV_EXCEPTION(kParamErr, POV_SPEC_NOT_SUPPORT);
		case EXR:
			break;
		case PNG:
			throw POV_EXCEPTION(kParamErr, POV_SPEC_NOT_SUPPORT);
		case TGA:
			Targa::Write(file, image, options);
			break;
		case PPM:
			Ppm::Write(file, image, options);
			break;
		case BMP:
			Bmp::Write(file, image, options);
			break;
		case JPEG:
			throw POV_EXCEPTION(kParamErr, POV_SPEC_NOT_SUPPORT);
		default :
			throw POV_EXCEPTION(kParamErr, "Invalid file type");
	}
}

void Image::GetRGBIndexedValue(unsigned char index, float& red, float& green, float& blue) const
{
	switch(colormaptype)
	{
		case NoColourMap:
			red = 0.0f;
			green = 0.0f;
			blue = 0.0f;
			break;
		case RGBColourMap:
		case RGBAColourMap:
		case RGBFTColourMap:
			red = colormap[index].red;
			green = colormap[index].green;
			blue = colormap[index].blue;
			break;
	}
}

void Image::GetRGBAIndexedValue(unsigned char index, float& red, float& green, float& blue, float& alpha) const
{
	switch(colormaptype)
	{
		case NoColourMap:
			red = 0.0f;
			green = 0.0f;
			blue = 0.0f;
			alpha = ALPHA_OPAQUE;
			break;
		case RGBColourMap:
			red = colormap[index].red;
			green = colormap[index].green;
			blue = colormap[index].blue;
			alpha = ALPHA_OPAQUE;
			break;
		case RGBAColourMap:
			red = colormap[index].red;
			green = colormap[index].green;
			blue = colormap[index].blue;
			alpha = colormap[index].filter; // with RGBAColourMap, .filter is actually alpha
			break;
		case RGBFTColourMap:
			red = colormap[index].red;
			green = colormap[index].green;
			blue = colormap[index].blue;
            alpha = Colour::FTtoA(colormap[index].filter, colormap[index].transm);
			break;
	}
}

void Image::GetRGBFTIndexedValue(unsigned char index, float& red, float& green, float& blue, float& filter, float& transm) const
{
	switch(colormaptype)
	{
		case NoColourMap:
			red = 0.0f;
			green = 0.0f;
			blue = 0.0f;
			filter = FT_OPAQUE;
			transm = FT_OPAQUE;
			break;
		case RGBColourMap:
			red = colormap[index].red;
			green = colormap[index].green;
			blue = colormap[index].blue;
			filter = transm = FT_OPAQUE;
			break;
		case RGBAColourMap:
			red = colormap[index].red;
			green = colormap[index].green;
			blue = colormap[index].blue;
            Colour::AtoFT(colormap[index].filter, filter, transm); // with RGBAColourMap, .filter is actually alpha
			break;
		case RGBFTColourMap:
			red = colormap[index].red;
			green = colormap[index].green;
			blue = colormap[index].blue;
			filter = colormap[index].filter;
			transm = colormap[index].transm;
			break;
	}
}

void Image::SetRGBIndexedValue(unsigned char index, float red, float green, float blue)
{
	switch(colormaptype)
	{
		case NoColourMap:
			break;
		case RGBColourMap:
			colormap[index].red = red;
			colormap[index].green = green;
			colormap[index].blue = blue;
			colormap[index].filter = 0.0f; // not used with RGBColourMap
			colormap[index].transm = 0.0f; // not used with RGBColourMap
			break;
		case RGBAColourMap:
			colormap[index].red = red;
			colormap[index].green = green;
			colormap[index].blue = blue;
			colormap[index].filter = ALPHA_OPAQUE; // with RGBAColourMap, .filter is actually alpha
			colormap[index].transm = 0.0f; // not used with RGBAColourMap
			break;
		case RGBFTColourMap:
			colormap[index].red = red;
			colormap[index].green = green;
			colormap[index].blue = blue;
			colormap[index].filter = FT_OPAQUE;
			colormap[index].transm = FT_OPAQUE;
			break;
	}
}

void Image::SetRGBAIndexedValue(unsigned char index, float red, float green, float blue, float alpha)
{
	switch(colormaptype)
	{
		case NoColourMap:
			break;
		case RGBColourMap:
			colormap[index].red = red;
			colormap[index].green = green;
			colormap[index].blue = blue;
			colormap[index].filter = 0.0f; // not used with RGBColourMap
			colormap[index].transm = 0.0f; // not used with RGBColourMap
			break;
		case RGBAColourMap:
			colormap[index].red = red;
			colormap[index].green = green;
			colormap[index].blue = blue;
			colormap[index].filter = alpha; // with RGBAColourMap, .filter is actually alpha
			colormap[index].transm = 0.0f; // not used with RGBAColourMap
			break;
		case RGBFTColourMap:
			colormap[index].red = red;
			colormap[index].green = green;
			colormap[index].blue = blue;
            Colour::AtoFT(alpha, colormap[index].filter, colormap[index].transm);
			break;
	}
}

void Image::SetRGBFTIndexedValue(unsigned char index, float red, float green, float blue, float filter, float transm)
{
	switch(colormaptype)
	{
		case NoColourMap:
			break;
		case RGBColourMap:
			colormap[index].red = red;
			colormap[index].green = green;
			colormap[index].blue = blue;
			colormap[index].filter = 0.0f; // not used with RGBColourMap
			colormap[index].transm = 0.0f; // not used with RGBColourMap
			break;
		case RGBAColourMap:
			colormap[index].red = red;
			colormap[index].green = green;
			colormap[index].blue = blue;
            colormap[index].filter = Colour::FTtoA(filter, transm); // note: filter is alpha in RGBA maps
			colormap[index].transm = 0.0f; // not used with RGBAColourMap
			break;
		case RGBFTColourMap:
			colormap[index].red = red;
			colormap[index].green = green;
			colormap[index].blue = blue;
			colormap[index].filter = filter;
			colormap[index].transm = transm;
			break;
	}
}

void Image::SetRGBFTIndexedValue(unsigned char index, const Colour& col)
{
	switch(colormaptype)
	{
		case NoColourMap:
			break;
		case RGBColourMap:
			colormap[index].red = col.red();
			colormap[index].green = col.green();
			colormap[index].blue = col.blue();
			colormap[index].filter = 0.0f; // not used with RGBColourMap
			colormap[index].transm = 0.0f; // not used with RGBColourMap
			break;
		case RGBAColourMap:
			colormap[index].red = col.red();
			colormap[index].green = col.green();
			colormap[index].blue = col.blue();
			colormap[index].filter = col.FTtoA(); // note: filter is alpha in RGBA maps
			colormap[index].transm = 0.0f; // not used with RGBAColourMap
			break;
		case RGBFTColourMap:
			colormap[index].red = col.red();
			colormap[index].green = col.green();
			colormap[index].blue = col.blue();
			colormap[index].filter = col.filter();
			colormap[index].transm = col.transm();
			break;
	}
}

unsigned char Image::GetIndexedValue(unsigned int, unsigned int)
{
	return 0;
}

void Image::SetIndexedValue(unsigned int x, unsigned int y, unsigned char index)
{
	CHECK_BOUNDS(x, y);
	switch(colormaptype)
	{
		case NoColourMap:
			SetBitValue(x,y, false);
			break;
		case RGBColourMap:
			SetRGBValue(x, y, colormap[index].red, colormap[index].green, colormap[index].blue);
			break;
		case RGBAColourMap:
			SetRGBAValue(x, y, colormap[index].red, colormap[index].green, colormap[index].blue, colormap[index].filter); // with RGBAColourMap, .filter is actually alpha
			break;
		case RGBFTColourMap:
			SetRGBFTValue(x, y, colormap[index].red, colormap[index].green, colormap[index].blue, colormap[index].filter, colormap[index].transm);
			break;
	}
}

void Image::FillIndexedValue(unsigned char index)
{
	switch(colormaptype)
	{
		case NoColourMap:
			FillBitValue(false);
			break;
		case RGBColourMap:
			FillRGBValue(colormap[index].red, colormap[index].green, colormap[index].blue);
			break;
		case RGBAColourMap:
			FillRGBAValue(colormap[index].red, colormap[index].green, colormap[index].blue, colormap[index].filter); // with RGBAColourMap, .filter is actually alpha
			break;
		case RGBFTColourMap:
			FillRGBFTValue(colormap[index].red, colormap[index].green, colormap[index].blue, colormap[index].filter, colormap[index].transm);
			break;
	}
}

void Image::SetEncodedGrayValue(unsigned int x, unsigned int y, const GammaCurvePtr& g, unsigned int max, unsigned int gray)
{
	if (!IsIndexed() && GetMaxIntValue() == max && GammaCurve::IsNeutral(g))
        // avoid potential re-quantization in case we have a pretty match between encoded data and container
		SetGrayValue(x, y, gray);
    else
        SetGrayValue(x, y, IntDecode(g,gray,max));
}
void Image::SetEncodedGrayAValue(unsigned int x, unsigned int y, const GammaCurvePtr& g, unsigned int max, unsigned int gray, unsigned int alpha, bool premul)
{
    bool doPremultiply   = !premul && (premultiplied || !HasTransparency()); // need to apply premultiplication if encoded data isn't PM'ed but container content should be
    bool doUnPremultiply = premul && !premultiplied && HasTransparency(); // need to undo premultiplication if other way round
	if (!doPremultiply && !doUnPremultiply && !IsIndexed() && GetMaxIntValue() == max && GammaCurve::IsNeutral(g))
        // avoid potential re-quantization in case we have a pretty match between encoded data and container
		SetGrayAValue(x, y, gray, alpha);
    else
    {
        float fAlpha = IntDecode(alpha,max);
        float fGray  = IntDecode(g,gray,max);
        if (doPremultiply)
            fGray *= fAlpha;
        else if (doUnPremultiply && alpha != 0)
            fGray /= fAlpha;
        // else no need to worry about premultiplication (or can't compensate anyway)
        SetGrayAValue(x, y, fGray, fAlpha);
    }
}
void Image::SetEncodedRGBValue(unsigned int x, unsigned int y, const GammaCurvePtr& g, unsigned int max, unsigned int red, unsigned int green, unsigned int blue)
{
    if (!IsIndexed() && GetMaxIntValue() == max && GammaCurve::IsNeutral(g))
        // avoid potential re-quantization in case we have a pretty match between encoded data and container
		SetRGBValue(x, y, red, green, blue);
	else
		SetRGBValue(x, y, IntDecode(g,max,red), IntDecode(g,max,green), IntDecode(g,max,blue));
}
void Image::SetEncodedRGBAValue(unsigned int x, unsigned int y, const GammaCurvePtr& g, unsigned int max, unsigned int red, unsigned int green, unsigned int blue, unsigned int alpha, bool premul)
{
    bool doPremultiply   = !premul && (premultiplied || !HasTransparency()); // need to apply premultiplication if encoded data isn't PM'ed but container content should be
    bool doUnPremultiply = premul && !premultiplied && HasTransparency(); // need to undo premultiplication if other way round
	if (!doPremultiply && !doUnPremultiply && !IsIndexed() && GetMaxIntValue() == max && GammaCurve::IsNeutral(g))
        // avoid potential re-quantization in case we have a pretty match between encoded data and container
		SetRGBAValue(x, y, red, green, blue, alpha);
    else
    {
        float fAlpha = IntDecode(alpha,  max);
        float fRed   = IntDecode(g,red,  max);
        float fGreen = IntDecode(g,green,max);
        float fBlue  = IntDecode(g,blue, max);
        if (doPremultiply)
        {
            fRed   *= fAlpha;
            fGreen *= fAlpha;
            fBlue  *= fAlpha;
        }
        else if (doUnPremultiply && alpha != 0)
        {
            fRed   /= fAlpha;
            fGreen /= fAlpha;
            fBlue  /= fAlpha;
        }
        // else no need to worry about premultiplication (or can't compensate anyway)
        SetRGBAValue(x, y, fRed, fGreen, fBlue, fAlpha);
    }
}
void Image::SetEncodedGrayValue(unsigned int x, unsigned int y, const GammaCurvePtr& g, float gray)
{
    SetGrayValue(x, y, GammaCurve::Decode(g,gray));
}
void Image::SetEncodedGrayAValue(unsigned int x, unsigned int y, const GammaCurvePtr& g, float gray, float alpha, bool premul)
{
    bool doPremultiply   = !premul && (premultiplied || !HasTransparency()); // need to apply premultiplication if encoded data isn't PM'ed but container content should be
    bool doUnPremultiply = premul && !premultiplied && HasTransparency(); // need to undo premultiplication if other way round
    gray = GammaCurve::Decode(g,gray);
    if (doPremultiply)
        gray *= alpha;
    else if (doUnPremultiply && alpha != 0) // TODO maybe use some epsilon?!
        gray /= alpha;
    // else no need to worry about premultiplication (or can't compensate anyway)
    SetGrayAValue(x, y, gray, alpha);
}
void Image::SetEncodedRGBValue(unsigned int x, unsigned int y, const GammaCurvePtr& g, float red, float green, float blue)
{
    SetRGBValue(x, y, GammaCurve::Decode(g,red), GammaCurve::Decode(g,green), GammaCurve::Decode(g,blue));
}
void Image::SetEncodedRGBAValue(unsigned int x, unsigned int y, const GammaCurvePtr& g, float red, float green, float blue, float alpha, bool premul)
{
    bool doPremultiply   = !premul && (premultiplied || !HasTransparency()); // need to apply premultiplication if encoded data isn't PM'ed but container content should be
    bool doUnPremultiply = premul && !premultiplied && HasTransparency(); // need to undo premultiplication if other way round
    red   = GammaCurve::Decode(g,red);
    green = GammaCurve::Decode(g,green);
    blue  = GammaCurve::Decode(g,blue);
    if (doPremultiply)
    {
        red   *= alpha;
        green *= alpha;
        blue  *= alpha;
    }
    else if (doUnPremultiply && alpha != 0.0) // TODO maybe use some epsilon?!
    {
        red   /= alpha;
        green /= alpha;
        blue  /= alpha;
    }
    // else no need to worry about premultiplication (or can't compensate anyway)
    SetRGBAValue(x, y, red, green, blue, alpha);
}

unsigned int Image::GetEncodedGrayValue(unsigned int x, unsigned int y, const GammaCurvePtr& g, unsigned int max) const
{
	float fGray;
    if (!premultiplied && HasTransparency())
    {
        // data has transparency and is stored non-premultiplied; precompose against a black background
        float fAlpha;
        GetGrayAValue(x, y, fGray, fAlpha);
        fGray *= fAlpha;
    }
    else
    {
        // no need to worry about premultiplication
        fGray = GetGrayValue(x, y);
    }
	return IntEncode(g,fGray,max);
}
void Image::GetEncodedGrayAValue(unsigned int x, unsigned int y, const GammaCurvePtr& g, unsigned int max, unsigned int& gray, unsigned int& alpha, bool premul) const
{
    bool doPremultiply   = premul && !premultiplied && HasTransparency(); // need to apply premultiplication if encoded data should be premul'ed but container content isn't
    bool doUnPremultiply = !premul && premultiplied && HasTransparency(); // need to undo premultiplication if other way round
	float fGray, fAlpha;
	GetGrayAValue(x, y, fGray, fAlpha);
    if (doPremultiply)
        fGray *= fAlpha;
    else if (doUnPremultiply && fAlpha != 0) // TODO maybe use some epsilon?!
        fGray /= fAlpha;
    // else no need to worry about premultiplication
    if (!premul)
    {
        // Data is to be encoded un-premultiplied, so clipping will happen /before/ multiplying with alpha (because the latter is done in the viewer),
        // which is equivalent to clipping pre-multiplied components to be no greater than alpha; compensate for this by increasing opacity
        // of any exceptionally bright pixels.
        if (fGray > 1.0)
        {
            float fFactor = fGray;
            if (fFactor * fAlpha > 1.0)
                fFactor = 1.0/fAlpha;
            // this keeps the product of alpha*color constant
            fAlpha *= fFactor;
            fGray  /= fFactor;
        }
    }
	gray  = IntEncode(g,fGray,max);
	alpha = IntEncode(fAlpha,max);
}
void Image::GetEncodedRGBValue(unsigned int x, unsigned int y, const GammaCurvePtr& g, unsigned int max, unsigned int& red, unsigned int& green, unsigned int& blue) const
{
	float fRed, fGreen, fBlue;
    if (!premultiplied && HasTransparency())
    {
        float fAlpha;
        // data has transparency and is stored non-premultiplied; precompose against a black background
        GetRGBAValue(x, y, fRed, fGreen, fBlue, fAlpha);
        fRed   *= fAlpha;
        fGreen *= fAlpha;
        fBlue  *= fAlpha;
    }
    else
    {
        // no need to worry about premultiplication
        GetRGBValue(x, y, fRed, fGreen, fBlue);
    }
	red   = IntEncode(g,fRed,  max);
	green = IntEncode(g,fGreen,max);
	blue  = IntEncode(g,fBlue, max);
}
void Image::GetEncodedRGBAValue(unsigned int x, unsigned int y, const GammaCurvePtr& g, unsigned int max, unsigned int& red, unsigned int& green, unsigned int& blue, unsigned int& alpha, bool premul) const
{
    bool doPremultiply   = premul && !premultiplied && HasTransparency(); // need to apply premultiplication if encoded data should be premul'ed but container content isn't
    bool doUnPremultiply = !premul && premultiplied && HasTransparency(); // need to undo premultiplication if other way round
	float fRed, fGreen, fBlue, fAlpha;
	GetRGBAValue(x, y, fRed, fGreen, fBlue, fAlpha);
    if (doPremultiply)
    {
        fRed   *= fAlpha;
        fGreen *= fAlpha;
        fBlue  *= fAlpha;
    }
    else if (doUnPremultiply && fAlpha != 0) // TODO maybe use some epsilon?!
    {
        fRed   /= fAlpha;
        fGreen /= fAlpha;
        fBlue  /= fAlpha;
    }
    // else no need to worry about premultiplication
    if (!premul)
    {
        // Data is to be encoded un-premultiplied, so clipping will happen /before/ multiplying with alpha (because the latter is done in the viewer),
        // which is equivalent to clipping pre-multiplied components to be no greater than alpha; compensate for this by increasing opacity
        // of any exceptionally bright pixels.
        float fBright = min(fRed, min(fGreen, fBlue));
        if (fBright > 1.0)
        {
            float fFactor = fBright;
            if (fFactor * fAlpha > 1.0)
                fFactor = 1.0/fAlpha;
            // this keeps the product of alpha*color constant
            fAlpha *= fFactor;
            fRed   /= fFactor;
            fGreen /= fFactor;
            fBlue  /= fFactor;
        }
    }
	red   = IntEncode(g,fRed,  max);
	green = IntEncode(g,fGreen,max);
	blue  = IntEncode(g,fBlue, max);
	alpha = IntEncode(fAlpha,max);
}

float Image::GetEncodedGrayValue(unsigned int x, unsigned int y, const GammaCurvePtr& g) const
{
	float gray;
    if (!premultiplied && HasTransparency())
    {
        // data has transparency and is stored non-premultiplied; precompose against a black background
        float alpha;
        GetGrayAValue(x, y, gray, alpha);
        gray *= alpha;
    }
    else
    {
        // no need to worry about premultiplication
        gray = GetGrayValue(x, y);
    }
    return GammaCurve::Encode(g,gray);
}
void Image::GetEncodedGrayAValue(unsigned int x, unsigned int y, const GammaCurvePtr& g, float& gray, float& alpha, bool premul) const
{
    bool doPremultiply   = premul && !premultiplied && HasTransparency(); // need to apply premultiplication if encoded data should be premul'ed but container content isn't
    bool doUnPremultiply = !premul && premultiplied && HasTransparency(); // need to undo premultiplication if other way round
	GetGrayAValue(x, y, gray, alpha);
    if (doPremultiply)
        gray *= alpha;
    else if (doUnPremultiply && alpha != 0) // TODO maybe use some epsilon?!
        gray /= alpha;
    // else no need to worry about premultiplication
    gray = GammaCurve::Encode(g,gray);
}
void Image::GetEncodedRGBValue(unsigned int x, unsigned int y, const GammaCurvePtr& g, float& red, float& green, float& blue) const
{
    if (!premultiplied && HasTransparency())
    {
        // data has transparency and is stored non-premultiplied; precompose against a black background
        float alpha;
        GetRGBAValue(x, y, red, green, blue, alpha);
        red   *= alpha;
        green *= alpha;
        blue  *= alpha;
    }
    else
    {
        // no need to worry about premultiplication
        GetRGBValue(x, y, red, green, blue);
    }
    red   = GammaCurve::Encode(g,red);
	green = GammaCurve::Encode(g,green);
	blue  = GammaCurve::Encode(g,blue);
}
void Image::GetEncodedRGBAValue(unsigned int x, unsigned int y, const GammaCurvePtr& g, float& red, float& green, float& blue, float& alpha, bool premul) const
{
    bool doPremultiply   = premul && !premultiplied && HasTransparency(); // need to apply premultiplication if encoded data should be premul'ed but container content isn't
    bool doUnPremultiply = !premul && premultiplied && HasTransparency(); // need to undo premultiplication if other way round
	GetRGBAValue(x, y, red, green, blue, alpha);
    if (doPremultiply)
    {
        red   *= alpha;
        green *= alpha;
        blue  *= alpha;
    }
    else if (doUnPremultiply && alpha != 0) // TODO maybe use some epsilon?!
    {
        red   /= alpha;
        green /= alpha;
        blue  /= alpha;
    }
    // else no need to worry about premultiplication
	red   = GammaCurve::Encode(g,red);
	green = GammaCurve::Encode(g,green);
	blue  = GammaCurve::Encode(g,blue);
}

void Image::GetRGBValue(unsigned int x, unsigned int y, RGBColour& colour, bool premul) const
{
    if (premul && !premultiplied && HasTransparency())
    {
        // data is non-premultiplied, but caller expects premultiplied data
        float alpha;
	    GetRGBAValue(x, y, colour.red(), colour.green(), colour.blue(), alpha);
        colour *= alpha;
    }
    if (!premul && premultiplied && HasTransparency())
    {
        // data is premultiplied, but caller expects non-premultiplied data
        float alpha;
	    GetRGBAValue(x, y, colour.red(), colour.green(), colour.blue(), alpha);
        if (alpha != 0.0) // TODO maybe use some epsilon?!
            colour /= alpha;
    }
    else
    {
	    GetRGBValue(x, y, colour.red(), colour.green(), colour.blue());
    }
}
void Image::GetRGBFTValue(unsigned int x, unsigned int y, Colour& colour, bool premul) const
{
	GetRGBFTValue(x, y, colour.red(), colour.green(), colour.blue(), colour.filter(), colour.transm());
    if (premul && !premultiplied && HasTransparency())
    {
        // data is non-premultiplied, but caller expects premultiplied data
        float alpha = colour.FTtoA();
        colour.red()   *= alpha;
        colour.green() *= alpha;
        colour.blue()  *= alpha;
    }
    else if (!premul && premultiplied && HasTransparency())
    {
        // data is premultiplied, but caller expects non-premultiplied data
        float alpha = colour.FTtoA();
        if (alpha != 0.0) // TODO maybe use some epsilon?!
        {
            colour.red()   /= alpha;
            colour.green() /= alpha;
            colour.blue()  /= alpha;
        }
    }
}

unsigned int Image::GetColourMapSize() const
{
	return colormap.size();
}

void Image::GetColourMap(vector<RGBMapEntry>& m) const
{
	m.resize(colormap.size());
	for(size_t i = 0; i < colormap.size(); i++)
		GetRGBIndexedValue((unsigned char)(i), m[i].red, m[i].green, m[i].blue);
}

void Image::GetColourMap(vector<RGBAMapEntry>& m) const
{
	m.resize(colormap.size());
	for(size_t i = 0; i < colormap.size(); i++)
		GetRGBAIndexedValue((unsigned char)(i), m[i].red, m[i].green, m[i].blue, m[i].alpha);
}

void Image::GetColourMap(vector<RGBFTMapEntry>& m) const
{
	m.resize(colormap.size());
	for(size_t i = 0; i < colormap.size(); i++)
		GetRGBFTIndexedValue((unsigned char)(i), m[i].red, m[i].green, m[i].blue, m[i].filter, m[i].transm);
}

void Image::SetColourMap(const vector<RGBMapEntry>& m)
{
	colormap.resize(max(m.size(), sizeof(unsigned char) * 256));
	colormaptype = RGBColourMap;
	colormap.assign(m.begin(), m.end());
}

void Image::SetColourMap(const vector<RGBAMapEntry>& m)
{
	colormap.resize(max(m.size(), sizeof(unsigned char) * 256));
	colormaptype = RGBAColourMap;
	colormap.assign(m.begin(), m.end());
}

void Image::SetColourMap(const vector<RGBFTMapEntry>& m)
{
	colormap.resize(max(m.size(), sizeof(unsigned char) * 256));
	colormaptype = RGBFTColourMap;
	colormap.assign(m.begin(), m.end());
}

}
