/*******************************************************************************
 * thread.cpp
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/backend/support/bsptree.cpp $
 * $Revision: #1 $
 * $Change: 5014 $
 * $DateTime: 2010/06/13 03:51:51 $
 * $Author: thorsten $
 *******************************************************************************/

#include "base/thread.h"

// include the magic system headers needed to support threads
#if(POV_USE_POSIX_THREADS == 1)
	#include <pthread.h>
#elif(POV_USE_WINDOWS_THREADS == 1)
	#include <Windows.h>
#endif

// this must be the last file included
#include "base/povdebug.h"

#include <algorithm>

namespace pov_base
{

Mutex Thread::threadMutex;
Thread::Threads Thread::threads;
Thread::TLSHelper Thread::platformThreadLocalStorageHandle;

Mutex::Mutex()
{
	#if(POV_USE_POSIX_THREADS == 1)
		pthread_mutex_init(&platformMutexHandle, NULL);
	#elif(POV_USE_WINDOWS_THREADS == 1)
		InitializeCriticalSection(platformMutexHandle.handle);
		platformMutexHandle.entered = false;
	#endif
}

Mutex::~Mutex()
{
	#if(POV_USE_POSIX_THREADS == 1)
		pthread_mutex_destroy(&platformMutexHandle);
	#elif(POV_USE_WINDOWS_THREADS == 1)
		DeleteCriticalSection(platformMutexHandle.handle);
	#endif
}

void Mutex::Lock()
{
	#if(POV_USE_POSIX_THREADS == 1)
		pthread_mutex_lock(&platformMutexHandle);
	#elif(POV_USE_WINDOWS_THREADS == 1)
		EnterCriticalSection(platformMutexHandle.handle);
		while(platformMutexHandle.entered == true) { } // WARNING: If your code hangs here, use a recursive mutex! This code is intentional! [trf]
		platformMutexHandle.entered = true;
	#endif
}

bool Mutex::TryLock()
{
	#if(POV_USE_POSIX_THREADS == 1)
		return (pthread_mutex_trylock(&platformMutexHandle) == 0);
	#elif(POV_USE_WINDOWS_THREADS == 1)
		bool result = (TryEnterCriticalSection(platformMutexHandle.handle) != 0);

		if((result == true) && (platformMutexHandle.entered == true)
		{
			LeaveCriticalSection(platformMutexHandle.handle);
			result = false;
		}
		else
			platformMutexHandle.entered = true;

		return result;
	#endif
}

void Mutex::Unlock()
{
	#if(POV_USE_POSIX_THREADS == 1)
		pthread_mutex_unlock(&platformMutexHandle);
	#elif(POV_USE_WINDOWS_THREADS == 1)
		platformMutexHandle.entered = false;
		LeaveCriticalSection(platformMutexHandle.handle);
	#endif
}

RecursiveMutex::RecursiveMutex()
{
	#if(POV_USE_POSIX_THREADS == 1)
		pthread_mutexattr_init(&platformMutexHandle.attr);
		pthread_mutexattr_settype(&platformMutexHandle.attr, PTHREAD_MUTEX_RECURSIVE);
		pthread_mutex_init(&platformMutexHandle.mutex, &platformMutexHandle.attr);
	#elif(POV_USE_WINDOWS_THREADS == 1)
		InitializeCriticalSection(platformMutexHandle);
	#endif
}

RecursiveMutex::~RecursiveMutex()
{
	#if(POV_USE_POSIX_THREADS == 1)
		pthread_mutex_destroy(&platformMutexHandle.mutex);
		pthread_mutexattr_destroy(&platformMutexHandle.attr); // TODO - Is this needed? [trf]
	#elif(POV_USE_WINDOWS_THREADS == 1)
		DeleteCriticalSection(platformMutexHandle);
	#endif
}

void RecursiveMutex::Lock()
{
	#if(POV_USE_POSIX_THREADS == 1)
		pthread_mutex_lock(&platformMutexHandle.mutex);
	#elif(POV_USE_WINDOWS_THREADS == 1)
		EnterCriticalSection(platformMutexHandle);
	#endif
}

bool RecursiveMutex::TryLock()
{
	#if(POV_USE_POSIX_THREADS == 1)
		return (pthread_mutex_trylock(&platformMutexHandle.mutex) == 0);
	#elif(POV_USE_WINDOWS_THREADS == 1)
		return (TryEnterCriticalSection(platformMutexHandle) != 0);
	#endif
}

void RecursiveMutex::Unlock()
{
	#if(POV_USE_POSIX_THREADS == 1)
		pthread_mutex_unlock(&platformMutexHandle.mutex);
	#elif(POV_USE_WINDOWS_THREADS == 1)
		LeaveCriticalSection(platformMutexHandle);
	#endif
}

Thread::Thread(boost::function0<void> fn, size_t stack) :
	threadFunction(fn),
	threadDataPtr(NULL)
{
	if(stack < POV_MINIMUM_THREAD_STACK_SIZE)
		stack = POV_MINIMUM_THREAD_STACK_SIZE;

	joinable = false;

	Mutex::ScopedLock lock(threadMutex);

	#if(POV_USE_POSIX_THREADS == 1)
		pthread_attr_init(&platformThreadHandle.attr);
		pthread_attr_setstacksize(&platformThreadHandle.attr, stack);

		if(pthread_create(&platformThreadHandle.thread, &platformThreadHandle.attr, &Thread::PlatformRunFunction, reinterpret_cast<void *>(this)) != 0)
			throw -1; // TODO FIXME
	#elif(POV_USE_WINDOWS_THREADS == 1)
		platformThreadHandle = CreateThread(NULL, stack, &Thread::PlatformRunFunction, reinterpret_cast<LDVOID>(this), 0, NULL);

		if(platformThreadHandle == NULL)
			throw -1; // TODO FIXME
	#endif

	threads.insert(this);
	joinable = true;
}

Thread::Thread(PlatformThreadHandle& pth) :
	threadDataPtr(NULL)
{
	platformThreadHandle = pth;

	joinable = false;

	Mutex::ScopedLock lock(threadMutex);

	threads.insert(this);
	joinable = true;
}

Thread::~Thread()
{
	Join();

	Mutex::ScopedLock lock(threadMutex);

	#if(POV_USE_POSIX_THREADS == 1)
		pthread_detach(platformThreadHandle.thread);
		pthread_attr_destroy(&platformThreadHandle.attr); // TODO - Is this needed? [trf]
	#elif(POV_USE_WINDOWS_THREADS == 1)
		CloseHandle(platformThreadHandle);
	#endif

	threads.erase(this);
}

void Thread::Join()
{
	if(joinable == true)
	{
		#if(POV_USE_POSIX_THREADS == 1)
			pthread_join(platformThreadHandle.thread, NULL);
		#elif(POV_USE_WINDOWS_THREADS == 1)
			WaitForSingleObject(platformThreadHandle, INFINITE);
		#endif
	}
}

bool Thread::operator==(const Thread& other) const
{
	return (this == &other);
}

bool Thread::operator!=(const Thread& other) const
{
	return (this != &other);
}

bool Thread::operator<(const Thread& other) const
{
	return (this < &other);
}

void *Thread::GetThreadDataPtr() const
{
	return threadDataPtr;
}

void Thread::SetThreadDataPtr(void *ptr)
{
	threadDataPtr = ptr;
}

void Thread::Run()
{
	#if(POV_USE_POSIX_THREADS == 1)
		if(pthread_setspecific(platformThreadLocalStorageHandle.handle, reinterpret_cast<void *>(this)) != 0)
			throw -1; // TODO FIXME
	#elif(POV_USE_WINDOWS_THREADS == 1)
		if(TlsSetValue(platformThreadLocalStorageHandle.handle, reinterpret_cast<LPVOID>(this)) == 0)
			throw -1; // TODO FIXME
	#endif

	threadFunction();

	joinable = false;
}

void Thread::Yield()
{
	#if(POV_USE_POSIX_THREADS == 1)
		sched_yield();
	#elif(POV_USE_WINDOWS_THREADS == 1)
		Sleep(INFINITE);
	#endif
}

void Thread::Sleep(unsigned int msec)
{
	#if(POV_USE_POSIX_THREADS == 1)
		// Note: By definition in POSIX standard, usleep value is limited in range!
		if(msec < 1000000)
			usleep(msec);
		else
			sleep(msec / 1000);
	#elif(POV_USE_WINDOWS_THREADS == 1)
		Sleep(msec);
	#endif
}

Thread& Thread::GetCurrentThread()
{
	#if(POV_USE_POSIX_THREADS == 1)
		void *ptr = pthread_getspecific(platformThreadLocalStorageHandle.handle);

		if(ptr == NULL)
			throw -1; // TODO FIXME

		return *reinterpret_cast<Thread *>(ptr);
	#elif(POV_USE_WINDOWS_THREADS == 1)
		LPVOID ptr = TlsGetValue(platformThreadLocalStorageHandle.handle);

		if(ptr == NULL)
			throw -1; // TODO FIXME

		return *reinterpret_cast<Thread *>(ptr);
	#endif
}

Thread::ThreadList Thread::GetAllThreads()
{
	Mutex::ScopedLock lock(threadMutex);
	ThreadList tl;

	std::copy(threads.begin(), threads.end(), tl.begin());

	return tl;
}

Thread *Thread::AdoptThread()
{
	#if(POV_USE_POSIX_THREADS == 1)
		PlatformThreadHandle pth;

		pthread_attr_init(&pth.attr);
		pth.thread = pthread_self();

		return new Thread(pth);
	#elif(POV_USE_WINDOWS_THREADS == 1)
		return new Thread(GetCurrentThread());
	#endif
}

void Thread::ModifyThread(ThreadHandle th, boost::function1<void, PlatformThreadHandle *> fn)
{
	Mutex::ScopedLock lock(threadMutex);
	Threads::iterator ti = threads.find(reinterpret_cast<Thread *>(th));

	if(ti != threads.end())
		fn(&((*ti)->platformThreadHandle));
}

POV_PLATFORM_THREAD_RUN_FUNCTION_DECL Thread::PlatformRunFunction(POV_PLATFORM_THREAD_RUN_FUNCTION_ARGS)
{
	#if(POV_USE_POSIX_THREADS == 1)
		reinterpret_cast<Thread *>(threadptr)->Run();
		pthread_exit(NULL);
		return NULL;
	#elif(POV_USE_WINDOWS_THREADS == 1)
		reinterpret_cast<Thread *>(threadptr)->Run();
		return 0;
	#endif
}

Thread::TLSHelper::TLSHelper()
{
	#if(POV_USE_POSIX_THREADS == 1)
		handle = NULL;
		if(pthread_key_create(&handle, NULL) != 0)
			throw -1; // TODO FIXME
	#elif(POV_USE_WINDOWS_THREADS == 1)
		handle = TlsAlloc();
		if(handle == TLS_OUT_OF_INDEXES)
			throw -1; // TODO FIXME
	#endif
}

Thread::TLSHelper::~TLSHelper()
{
	#if(POV_USE_POSIX_THREADS == 1)
		pthread_key_delete(handle);
	#elif(POV_USE_WINDOWS_THREADS == 1)
		if(handle != TLS_OUT_OF_INDEXES)
			TlsFree(handle);
	#endif
}

}
