/*******************************************************************************
 * animationprocessing.cpp
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/frontend/animationprocessing.cpp $
 * $Revision: #2 $
 * $Change: 5047 $
 * $DateTime: 2010/06/30 07:58:31 $
 * $Author: thorsten $
 *******************************************************************************/

#include <string>
#include <cctype>

// configbase.h must always be the first POV file included within base *.cpp files
#include "base/configbase.h"
#include "base/types.h"
#include "base/povmscpp.h"
#include "base/povmsgid.h"

#include "frontend/configfrontend.h"
#include "frontend/animationprocessing.h"

// this must be the last file included
#include "base/povdebug.h"

namespace pov_frontend
{

AnimationProcessing::AnimationProcessing(POVMS_Object& options) :
	renderOptions(options)
{
	bool cyclic = renderOptions.TryGetBool(kPOVAttrib_CyclicAnimation, false);

	initialFrame = max(renderOptions.TryGetInt(kPOVAttrib_InitialFrame, 1), 0);
	initialClock = renderOptions.TryGetFloat(kPOVAttrib_InitialClock, 0.0);

	finalFrame = max(renderOptions.TryGetInt(kPOVAttrib_FinalFrame, 1), initialFrame);
	finalClock = renderOptions.TryGetFloat(kPOVAttrib_FinalClock, 1.0);

	if(cyclic == true)
		finalFrame++;

	if(renderOptions.Exist(kPOVAttrib_SubsetStartFrame) == true)
	{
		POVMSFloat subsetStartPercent = max(renderOptions.GetFloat(kPOVAttrib_SubsetStartFrame), POVMSFloat(0.0));
		if((subsetStartPercent == 0.0) || (subsetStartPercent >= 1.0))
			subsetStartFrame = POVMSInt(subsetStartPercent);
		else
			subsetStartFrame = POVMSFloat((finalFrame - initialFrame + 1) * subsetStartPercent);
		if (subsetStartFrame < initialFrame)
			subsetStartFrame = initialFrame; // TODO: should we be issuing a warning or throwing an error here?
	}
	else
		subsetStartFrame = initialFrame;

	if(options.Exist(kPOVAttrib_SubsetEndFrame) == true)
	{
		POVMSFloat subsetEndPercent = max(renderOptions.GetFloat(kPOVAttrib_SubsetEndFrame), POVMSFloat(0.0));
		if((subsetEndPercent == 0.0) || (subsetEndPercent >= 1.0))
			subsetEndFrame = POVMSInt(subsetEndPercent);
		else
			subsetEndFrame = POVMSFloat((finalFrame - initialFrame + 1) * subsetEndPercent);
		if (subsetEndFrame < subsetStartFrame)
			subsetEndFrame = subsetStartFrame; // TODO: should we be issuing a warning or throwing an error here?
	}
	else
	{
		subsetEndFrame = finalFrame;
		if(cyclic == true)
			subsetEndFrame--;
	}

	fieldRenderFlag = renderOptions.TryGetBool(kPOVAttrib_FieldRender, false);
	oddFieldFlag = renderOptions.TryGetBool(kPOVAttrib_OddField, false);

	clockDelta = double(finalClock - initialClock) / double(finalFrame - initialFrame);
	frameNumber = subsetStartFrame;
	clockValue = POVMSFloat(double(clockDelta * double(subsetStartFrame - initialFrame)) + double(initialClock));

	frameNumberDigits = 1;
	for(POVMSInt i = finalFrame; i >= 10; i /= 10)
		frameNumberDigits++;
}

POVMS_Object AnimationProcessing::GetFrameRenderOptions()
{
	POVMS_Object opts(renderOptions);

	opts.SetFloat(kPOVAttrib_Clock, clockValue);

	POVMS_List declares;
	if(opts.Exist(kPOVAttrib_Declare) == true)
		opts.Get(kPOVAttrib_Declare, declares);

	POVMS_Object clock_delta(kPOVMSType_WildCard);
	clock_delta.SetString(kPOVAttrib_Identifier, "clock_delta");
	clock_delta.SetFloat(kPOVAttrib_Value, POVMSFloat(clockDelta));
	declares.Append(clock_delta);

	POVMS_Object final_clock(kPOVMSType_WildCard);
	final_clock.SetString(kPOVAttrib_Identifier, "final_clock");
	final_clock.SetFloat(kPOVAttrib_Value, finalClock);
	declares.Append(final_clock);

	POVMS_Object final_frame(kPOVMSType_WildCard);
	final_frame.SetString(kPOVAttrib_Identifier, "final_frame");
	final_frame.SetFloat(kPOVAttrib_Value, finalFrame);
	declares.Append(final_frame);

	POVMS_Object frame_number(kPOVMSType_WildCard);
	frame_number.SetString(kPOVAttrib_Identifier, "frame_number");
	frame_number.SetFloat(kPOVAttrib_Value, frameNumber);
	declares.Append(frame_number);

	POVMS_Object initial_clock(kPOVMSType_WildCard);
	initial_clock.SetString(kPOVAttrib_Identifier, "initial_clock");
	initial_clock.SetFloat(kPOVAttrib_Value, initialClock);
	declares.Append(initial_clock);

	POVMS_Object initial_frame(kPOVMSType_WildCard);
	initial_frame.SetString(kPOVAttrib_Identifier, "initial_frame");
	initial_frame.SetFloat(kPOVAttrib_Value, initialFrame);
	declares.Append(initial_frame);

	opts.Set(kPOVAttrib_Declare, declares);

	return opts;
}

void AnimationProcessing::ComputeNextFrame()
{
	frameNumber++;
	clockValue = POVMSFloat(double(clockDelta * double(frameNumber - initialFrame)) + double(initialClock));
}

bool AnimationProcessing::MoreFrames()
{
	return (frameNumber < subsetEndFrame);
}

POVMSInt AnimationProcessing::GetFrameNumber()
{
	return frameNumber;
}

POVMSInt AnimationProcessing::GetStartFrame()
{
	return subsetStartFrame;
}

POVMSInt AnimationProcessing::GetEndFrame()
{
	return subsetEndFrame;
}

POVMSInt AnimationProcessing::GetTotalFrames()
{
	return subsetEndFrame - subsetStartFrame + 1;
}

POVMSFloat AnimationProcessing::GetClockValue()
{
	return clockValue;
}

int AnimationProcessing::GetFrameNumberDigits()
{
	return frameNumberDigits;
}

}
