/*******************************************************************************
 * filemessagehandler.cpp
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2010 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/spec-3.7/source/frontend/filemessagehandler.cpp $
 * $Revision: #2 $
 * $Change: 5047 $
 * $DateTime: 2010/06/30 07:58:31 $
 * $Author: thorsten $
 *******************************************************************************/

// configbase.h must always be the first POV file included within base *.cpp files
#include "base/configbase.h"
#include "base/types.h"
#include "base/path.h"

#include "frontend/configfrontend.h"
#include "frontend/renderfrontend.h"
#include "frontend/filemessagehandler.h"

// this must be the last file included
#include "base/povdebug.h"

namespace pov_frontend
{

FileMessageHandler::FileMessageHandler()
{
}

FileMessageHandler::~FileMessageHandler()
{
}

void FileMessageHandler::HandleMessage(const SceneData& sd, POVMSType ident, POVMS_Object& msg, POVMS_Object& result)
{
	switch(ident)
	{
		case kPOVMsgIdent_FindFile:
			(void)FindFile(sd.searchpaths, msg, result);
			break;
		case kPOVMsgIdent_ReadFile:
			if(ReadFile(sd.searchpaths, msg, result) == true)
				sd.readfiles.push_back(result);
			break;
		case kPOVMsgIdent_CreatedFile:
			CreatedFile(msg);
			sd.createdfiles.push_back(msg);
			break;
	}
}

bool FileMessageHandler::FindFile(const list<Path>& lps, POVMS_Object& msg, POVMS_Object& result)
{
	POVMS_List files;
	Path path;

	msg.Get(kPOVAttrib_ReadFile, files);

	for(int i = 1; i <= files.GetListSize(); i++)
	{
		POVMS_Attribute attr;

		files.GetNth(i, attr);

		path = FindFilePath(lps, Path(attr.GetUCS2String()));

		if(path.Empty() == false)
			break;
	}

	result.SetUCS2String(kPOVAttrib_ReadFile, path().c_str());

	return (path.Empty() == false);
}

bool FileMessageHandler::ReadFile(const list<Path>& lps, POVMS_Object& msg, POVMS_Object& result)
{
	Path path(FindFilePath(lps, Path(msg.GetUCS2String(kPOVAttrib_ReadFile))));

	if(path.Empty() == false)
		result.SetUCS2String(kPOVAttrib_LocalFile, path().c_str());

	return (path.Empty() == false);
}

void FileMessageHandler::CreatedFile(POVMS_Object&)
{
}

Path FileMessageHandler::FindFilePath(const list<Path>& lps, const Path& f)
{
	// check the current working dir (or full path if supplied) first
	// note that if the file doesn't have a path and it is found in the
	// CWD, the CWD is not returned as part of the path.
	if(CheckIfFileExists(f) == true)
		return f;

	// if the path is absolute there's no point in checking the include paths;
	// given it wasn't found above we can just return an empty path
	if(f.HasVolume() == true)
		return Path();

	Path path(f);

	for(list<Path>::const_iterator i(lps.begin()); i != lps.end(); i++)
	{
		Path path (*i, f);

		if(CheckIfFileExists(path) == true)
			return path;
	}

	// not found so return empty path
	return Path();
}

}
