/****************************************************************************
 *               averageimage.cpp
 *
 * This program implements PPM image averaging.
 *
 * from Persistence of Vision(tm) Ray Tracer version 3.6.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2004 Persistence of Vision Raytracer Pty. Ltd.
 *---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 *---------------------------------------------------------------------------
 * This program is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 *****************************************************************************/

namespace std { }

#include <exception>
#include <bitset>
#include <cstring>
#include <cstdio>
#include <cstdlib>
#include <cstdarg>
#include <climits>
#include <cctype>
#include <cmath>

#include "fileinputoutput.h"

USING_POV_BASE_NAMESPACE

using namespace std;

#define POV_MALLOC(size,msg) pov_malloc ((size), __FILE__, __LINE__, (msg))

const int IS16BITIMAGE = 0x00020000;

typedef struct Image8_Line_Struct IMAGE8_LINE;
typedef struct Image16_Line_Struct IMAGE16_LINE;

struct Image8_Line_Struct
{
  unsigned char *red, *green, *blue, *transm;
};

struct Image16_Line_Struct
{
  unsigned short *red, *green, *blue, *transm;
};

typedef struct Image_Struct IMAGE;

struct Image_Struct
{
  int References; /* Keeps track of number of pointers to this structure */
  int Map_Type;
  int File_Type;
  int Image_Type; /* What this image is being used for */
  int iwidth, iheight;
  union
  {
    IMAGE8_LINE *rgb8_lines;
    IMAGE16_LINE *rgb16_lines;
    unsigned short **gray16_lines;
    unsigned char **map_lines;
  } data;
};

void Write_PPM_Image(IMAGE *Image, char *name);
void Read_PPM_Image(IMAGE *Image, char *name);
int Read_ASCII_File_Number(IStream *filep);
void Get_Pixel(IMAGE *Image, unsigned int ixcoor, unsigned int iycoor, unsigned int& red, unsigned int& green, unsigned int& blue);
void Set_Pixel(IMAGE *Image, unsigned int ixcoor, unsigned int iycoor, unsigned int red, unsigned int green, unsigned int blue);
int Error(const char *format,...);
void *pov_malloc(size_t size, const char *file, int line, const char *msg);

int main(int argc, char **argv)
{
	try
	{
		if(argc < 4)
		{
			puts("Usage:   averageimages <outimage> <inimage1> [<inimage>*]");
			puts("Returns:  0 in case of successful processing\n"
			     "         -2 in case of an error\n");
			return -2;
		}

		IMAGE outputImage;
		IMAGE inputImage[256];
		int imageCount = argc - 2;

		Read_PPM_Image(&outputImage, argv[2]);

		for(int i = 0; i < imageCount; i++)
		{
			fprintf(stdout, "Reading image '%s'\n", argv[2 + i]);
			Read_PPM_Image(&inputImage[i], argv[2 + i]);

			if((outputImage.iwidth != inputImage[i].iwidth) || (outputImage.iheight != inputImage[i].iheight))
				Error("Image size does not match.");	

			if(outputImage.Image_Type != inputImage[i].Image_Type)
				Error("Image color depth does not match.");	
		}

		fprintf(stdout, "Averaging %d images...\n", imageCount);

		for(unsigned int y = 0; y < outputImage.iheight; y++)
		{
			for(unsigned int x = 0; x < outputImage.iwidth; x++)
			{
				unsigned int sum_red = 0, sum_green = 0, sum_blue = 0;

				for(int i = 0; i < imageCount; i++)
				{
					unsigned int red, green, blue;

					Get_Pixel(&inputImage[i], x, y, red, green, blue);

					sum_red += red;
					sum_green += green;
					sum_blue += blue;
				}

				Set_Pixel(&outputImage, x, y, sum_red / imageCount, sum_green / imageCount, sum_blue / imageCount);
			}
		}

		fprintf(stdout, "Writing image...\n");

		Write_PPM_Image(&outputImage, argv[1]);

		fprintf(stdout, "Done!\n");
	}
	catch(exception& e)
	{
		Error("Caught C++ exception '%s'.", e.what());
	}
	catch(...)
	{
		Error("Caught unknown C++ exception!");
	}

	// unreachable
	return -2;
}

void Write_PPM_Image(IMAGE *Image, char *name)
{
	OStream *filep;

	// --- Start by trying to open the file --- 
	filep = new OStream(0);
	if(filep->open(name) == 0)
		Error("Cannot open PPM image %s.", name);

	// --- Write header --- 
    if((Image->Image_Type & IS16BITIMAGE) != 0)
		filep->printf("P6\n%d %d\n65535\n", Image->iwidth, Image->iheight);
	else
		filep->printf("P6\n%d %d\n255\n", Image->iwidth, Image->iheight);

	// --- Write image data --- 
	for(unsigned int y = 0; y < Image->iheight; y++)
	{
		for(unsigned int x = 0; x < Image->iwidth; x++)
		{
			unsigned int rval, gval, bval;

			Get_Pixel(Image, x, y, rval, gval, bval);

		    if((Image->Image_Type & IS16BITIMAGE) != 0)
		    {
				filep->Write_Byte(rval >> 8);
				filep->Write_Byte(rval & 0xFF);
				filep->Write_Byte(gval >> 8);
				filep->Write_Byte(gval & 0xFF);
				filep->Write_Byte(bval >> 8);
			}
			else
			{
				filep->Write_Byte((rval / 256) & 0xFF);
				filep->Write_Byte((gval / 256) & 0xFF);
				filep->Write_Byte((bval / 256) & 0xFF);
			}
		}
	}

	// Close the image file 

	delete filep;
}

void Read_PPM_Image(IMAGE *Image, char *name)
{
  IStream *filep;
  unsigned char header[2];
  char line[1024];
  char *ptr;
  int nbr;

  int width, height;
  unsigned int depth;

  IMAGE8_LINE *line_data;
  IMAGE16_LINE *line_16_data;
  int data_hi, data_lo;
  int x, i;

  // --- Start by trying to open the file --- 
  filep = new IStream(0);
  if(filep->open(name) == 0)
    Error ("Cannot open PPM image %s.", name);

  // --- Read Header --- 
  if (!filep->read((char *)header, 2))
    Error ("Cannot read header of PPM image %s.", name);

  if(header[0] != 'P') Error ("File is not in PPM format.");

  if((header[1] != '3') && (header[1] != '6'))
    Error ("File is not in PPM format (type %d).", header[1]);

  do
  {
    filep->getline (line, 1024);
    line[1023] = '\0';
    if ((ptr = strchr(line, '#')) != NULL) *ptr = '\0';  // remove comment 
  }
  while (line[0]=='\0');  // read until line without comment from beginning 

  // --- First: two numbers: with and height --- 
  if (sscanf(line,"%d %d",&width, &height) != 2)
    Error ("Cannot read width and height from PPM image.");

  if (width <= 0 || height <= 0)
    Error ("Invalid width or height read from PPM image.");

  do
  {
    filep->getline (line, 1024) ;
    line[1023] = '\0';
    if ((ptr = strchr(line, '#')) != NULL) *ptr = '\0';  // remove comment 
  }
  while (line[0]=='\0');  // read until line without comment from beginning 

  // --- Second: one number: color depth --- 
  if (sscanf(line,"%d",&depth) != 1)
    Error ("Cannot read color depth from PPM image.");

  if ((depth > 65535) || (depth < 1))
    Error ("Unsupported number of colors (%d) in PPM image.", depth);

  Image->Image_Type = 0;
  Image->iwidth = width;
  Image->iheight = height;

  if (depth < 256)
  {
    Image->data.rgb8_lines = (IMAGE8_LINE *)POV_MALLOC(height * sizeof(IMAGE8_LINE), "PPM image");

    for (i = 0; i < height; i++)
    {
      line_data = &Image->data.rgb8_lines[i];

      line_data->red    = (unsigned char *)POV_MALLOC(width, "PPM image line");
      line_data->green  = (unsigned char *)POV_MALLOC(width, "PPM image line");
      line_data->blue   = (unsigned char *)POV_MALLOC(width, "PPM image line");
      line_data->transm = (unsigned char *)NULL;

      if (header[1] == '3') // --- ASCII PPM file (type 3) --- 
      {
        for (x = 0; x < width; x++)
        {
          nbr = Read_ASCII_File_Number(filep);
          if (nbr >= 0) line_data->red[x] = (nbr*255)/depth;
          else Error ("Cannot read image data from PPM image.");
          nbr = Read_ASCII_File_Number(filep);
          if (nbr >= 0) line_data->green[x] = (nbr*255)/depth;
          else Error ("Cannot read image data from PPM image.");
          nbr = Read_ASCII_File_Number(filep);
          if (nbr >= 0) line_data->blue[x] = (nbr*255)/depth;
          else Error ("Cannot read image data from PPM image.");
        }
      }
      else                  // --- binary PPM file (type 6) --- 
      {
        for (x = 0; x < width; x++)
        {
          if ((nbr = filep->Read_Byte ()) == EOF)
            Error("Cannot read data from PPM image.");

          line_data->red[x] = (nbr*255)/depth;

          if ((nbr = filep->Read_Byte ()) == EOF)
            Error("Cannot read data from PPM image.");

          line_data->green[x] = (nbr*255)/depth;

          if ((nbr = filep->Read_Byte ()) == EOF)
            Error("Cannot read data from PPM image.");

          line_data->blue[x] = (nbr*255)/depth;
        }
      }
    }
  }
  else // --- 16 bit PPM (binary or ASCII) --- 
  {
    Image->Image_Type |= IS16BITIMAGE;

    Image->data.rgb16_lines = (IMAGE16_LINE *)POV_MALLOC(height * sizeof(IMAGE16_LINE), "PPM image");

    for (i = 0; i < height; i++)
    {
      line_16_data = &Image->data.rgb16_lines[i];

      line_16_data->red    = (unsigned short *)POV_MALLOC(width * sizeof(unsigned short), "PPM image line");
      line_16_data->green  = (unsigned short *)POV_MALLOC(width * sizeof(unsigned short), "PPM image line");
      line_16_data->blue   = (unsigned short *)POV_MALLOC(width * sizeof(unsigned short), "PPM image line");
      line_16_data->transm = (unsigned short *)NULL;

      if (header[1] == '3') // --- ASCII PPM file (type 3) --- 
      {
        for (x = 0; x < width; x++)
        {
          nbr = Read_ASCII_File_Number(filep);
          if (nbr >= 0) line_16_data->red[x] = (nbr*65535)/depth;
          else Error ("Cannot read image data from PPM image.");

          nbr = Read_ASCII_File_Number(filep);
          if (nbr >= 0) line_16_data->green[x] = (nbr*65535)/depth;
          else Error ("Cannot read image data from PPM image.");

          nbr = Read_ASCII_File_Number(filep);
          if (nbr >= 0) line_16_data->blue[x] = (nbr*65535)/depth;
          else Error ("Cannot read image data from PPM image.");
        }
      }
      else                  // --- binary PPM file (type 6) --- 
      {
        for (x = 0; x < width; x++)
        {
          if ((data_hi = filep->Read_Byte ()) == EOF)
            Error ("Cannot read data from PPM image.");
          if ((data_lo = filep->Read_Byte ()) == EOF)
            Error ("Cannot read data from PPM image.");
          line_16_data->red[x] = (256*data_hi + data_lo)*65535/depth;

          if ((data_hi = filep->Read_Byte ()) == EOF)
            Error ("Cannot read data from PPM image.");
          if ((data_lo = filep->Read_Byte ()) == EOF)
            Error ("Cannot read data from PPM image.");
          line_16_data->green[x] = (256*data_hi + data_lo)*65535/depth;

          if ((data_hi = filep->Read_Byte ()) == EOF)
            Error ("Cannot read data from PPM image.");
          if ((data_lo = filep->Read_Byte ()) == EOF)
            Error ("Cannot read data from PPM image.");
          line_16_data->blue[x] = (256*data_hi + data_lo)*65535/depth;
        }
      }
    }
  }

  // Close the image file 

  delete filep;
}

int Read_ASCII_File_Number(IStream *filep)
{
  int value;
  int pos = 0;
  char buffer[50];

  do
  {
    value = filep->Read_Byte();
  }
  while ( isspace(value) );

  if ( isdigit(value) ) buffer[pos] = (char)value;
  else return -1;

  while ( !isspace(value = filep->Read_Byte()) && (pos<48))
  {
    if ( isdigit(value) )
    {
      pos++;
      buffer[pos] = (char)value;
    }
    else return -1;
  }

  buffer[pos+1] = '\0';

  value = atoi(buffer);

  return value;
}

void Get_Pixel(IMAGE *Image, unsigned int ixcoor, unsigned int iycoor, unsigned int& red, unsigned int& green, unsigned int& blue)
{
	IMAGE16_LINE *line16 = NULL;
	IMAGE8_LINE *line8 = NULL;

	if((Image->Image_Type & IS16BITIMAGE) == IS16BITIMAGE)
	{
		line16 = &Image->data.rgb16_lines[iycoor];
		red = (unsigned int)line16->red[ixcoor];
		green = (unsigned int)line16->green[ixcoor];
		blue = (unsigned int)line16->blue[ixcoor];
	}
	else
	{
		line8 = &Image->data.rgb8_lines[iycoor];
		red = ((unsigned int)line8->red[ixcoor]) * 256;
		green = ((unsigned int)line8->green[ixcoor]) * 256;
		blue = ((unsigned int)line8->blue[ixcoor]) * 256;
	}
}

void Set_Pixel(IMAGE *Image, unsigned int ixcoor, unsigned int iycoor, unsigned int red, unsigned int green, unsigned int blue)
{
	IMAGE16_LINE *line16 = NULL;
	IMAGE8_LINE *line8 = NULL;

	if((Image->Image_Type & IS16BITIMAGE) == IS16BITIMAGE)
	{
		line16 = &Image->data.rgb16_lines[iycoor];
		line16->red[ixcoor] = (unsigned short)red;
		line16->green[ixcoor] = (unsigned short)green;
		line16->blue[ixcoor] = (unsigned short)blue;
	}
	else
	{
		line8 = &Image->data.rgb8_lines[iycoor];
		line8->red[ixcoor] = (unsigned char)(red / 256);
		line8->green[ixcoor] = (unsigned char)(green / 256);
		line8->blue[ixcoor] = (unsigned char)(blue / 256);
	}
}

int Error(const char *format,...)
{
	va_list marker;
	char localvsbuffer[4096];

	sprintf(localvsbuffer, "Error: ");

	va_start(marker, format);
	vsnprintf(localvsbuffer + strlen(localvsbuffer), 4095 - strlen(localvsbuffer), format, marker);
	va_end(marker);

	fprintf(stderr, "%s\n\n", localvsbuffer);

	exit(-2);
}

void *pov_malloc(size_t size, const char *file, int line, const char *msg)
{
	void *block;

	if(size == 0)
		Error("Attempt to malloc zero size block (File: %s Line: %d).", file, line);
	block = malloc(size);

	if(block == NULL)
		Error("Out of memory.  Cannot allocate %ld bytes for %s (File: %s Line: %d).", (long)size, msg, file, line);

	return block;
}
