#include <stdio.h>
#include <string.h>
#include "Manual.h"

/***=======================================================================***/
/*** HorizontalRule: prints a horizontal bar to whatever file (including   ***/
/***                 stdout) specified.                                    ***/
/***                                                                       ***/
/*** Arguments:                                                            ***/
/***   outp:   the target file                                             ***/
/***   int n:  the number of carriage returns to add after the rule        ***/
/***=======================================================================***/
void HorizontalRule(FILE *outp, int n)
{
  int i;

  fprintf(outp, "<++>---------------------------------------------------------"
	 "--------------<++>\n");
  for (i = 0; i < n; i++) {
    fprintf(outp, "\n");
  }
}

/***=======================================================================***/
/*** PrintSplash: print the splash lines for mdgx.  This is where to put   ***/
/***              information about the program's primary authorship and   ***/
/***              copyrights.                                              ***/
/***                                                                       ***/
/*** Arguments:                                                            ***/
/***   outp:  the target file                                              ***/
/***=======================================================================***/
void PrintSplash(FILE *outp)
{
  HorizontalRule(outp, 0);
  fprintf(outp,
          "<++> mdgx: A molecular dynamics engine in the AMBER suite of "
          "programs      <++>\n"
          "<++>                                                             "
          "          <++>\n"
          "<++> Written by David S. Cerutti, Case Group (2009)              "
          "          <++>\n");
  HorizontalRule(outp, 1);
}

/***=======================================================================***/
/*** PrintVADesc: this function prints a variable name, alias, and         ***/
/***              description using the specified formatting.              ***/
/***                                                                       ***/
/*** Arguments:                                                            ***/
/***   leadspace: the number of leading white space characters             ***/
/***   vname:     the variable name                                        ***/
/***   vnlen:     the amount of space to give the variable name            ***/
/***   valias:    the variable alias                                       ***/
/***   valen:     the amount of space to give the variable alias           ***/
/***   vdesc:     the variable description                                 ***/
/***   vdlen:     the amount of space to give the variable description (if ***/
/***                the description is longer than the amount of space     ***/
/***                alotted, additional lines will be used)                ***/
/***   vdindent:  the amount of indentation to apply to extra lines of the ***/
/***                variable description                                   ***/
/***=======================================================================***/
static void PrintVADesc(int leadspace, char* vname, int vnlen, char* valias,
			int valen, char* vdesc, int vdlen, int vdindent)
{
  int i, j, endfound, vpivot;
  char scratch[4096];

  /*** Print leading spaces ***/
  for (i = 0; i < leadspace; i++) {
    printf(" ");
  }

  /*** Extend the variable name, then print it ***/
  j = strlen(vname);
  strcpy(scratch, vname);
  for (i = j; i < vnlen; i++) {
    scratch[i] = ' ';
  }
  scratch[vnlen] = '\0';
  printf("%s", scratch);

  /*** Extend the alias name, then print it ***/
  j = strlen(valias);
  strcpy(scratch, valias);
  for (i = j; i < valen; i++) {
    scratch[i] = ' ';
  }
  scratch[valen] = '\0';
  printf("%s", scratch);

  /*** Parse the description ***/
  endfound = 0;
  j = 0;
  while (endfound == 0) {
    for (i = 0; i < vdlen; i++) {
      if (vdesc[j+i] == ' ') {
	i++;
	vpivot = j+i;
      }
      if (vdesc[j+i] == '\0') {
	vpivot = j+i;
	endfound = 1;
	break;
      }
    }
    strncpy(scratch, &vdesc[j], vpivot-j);
    scratch[vpivot-j] = '\0';
    printf("%s\n", scratch);
    if (j == 0 && endfound == 0) {
      vdlen -= vdindent;
    }
    j = vpivot;
    if (endfound == 0) {
      for (i = 0; i < leadspace + vnlen + valen + vdindent; i++) {
	printf(" ");
      }
    }
  }
}

/***=======================================================================***/
/*** PrintParagraph: prints a paragraph within a specified width.  No      ***/
/***                 leading white space or other columns are provided.    ***/
/***                 However, each paragraph is terminated by printing one ***/
/***                 additional carriage return (so that paragaphs are     ***/
/***                 separated by a full blank line).                      ***/
/***                                                                       ***/
/*** Arguments:                                                            ***/
/***   vpar:   the paragraph string (carriage returns should not be        ***/
/***           included in the string)                                     ***/
/***   width:  the width of the text to print (the text will be broken     ***/
/***           up at whitespace characters to prevent lines from exceeding ***/
/***           this width)                                                 ***/
/***   outp:   the target file for this paragraph                          ***/
/***=======================================================================***/
void PrintParagraph(char *vpar, int width, FILE *outp)
{
  int i, j, vpivot, endfound;
  char scratch[4096];

  endfound = 0;
  j = 0;
  while (endfound == 0) {
    for (i = 0; i < width; i++) {
      if (vpar[j+i] == ' ') {
	i++;
        vpivot = j+i;
      }
      if (vpar[j+i] == '\0') {
        vpivot = j+i;
        endfound = 1;
        break;
      }
    }
    strncpy(scratch, &vpar[j], vpivot-j);
    scratch[vpivot-j] = '\0';
    fprintf(outp, "%s\n", scratch);
    j = vpivot;
  }
  fprintf(outp, "\n");
}

/***=======================================================================***/
/*** PrintUsage: print a brief set of usage instructions, to help users    ***/
/***             get started.                                              ***/
/***=======================================================================***/
void PrintUsage()
{
  printf("Usage:\n"
         ">> mdgx -i <input file>         (simplest operation)\n"
         ">> mdgx <arguments>             (operation similar to SANDER)\n"
         ">> mdgx <documentation name>    (prints documentation)\n\n");
  PrintParagraph("Command-line information may be entered using the same "
		 "arguments as the SANDER and PMEMD programs.  Alternatively, "
		 "all input may be provided in a single file using the -i "
		 "argument.  Any of the following arguments may also be used "
		 "to obtain additional documentation:", 79, stdout);
  printf("  -INPUT:   print all command line input options\n"
         "  -IFILE:   documentation on input file format\n"
	 "  -FILES:   print descriptions of &files namelist variables (these "
	 "may also\n"
	 "            be entered as command line input)\n"
         "  -CNTRL:   print descriptions of &cntrl namelist variables (most "
         "are similar\n"
         "            to SANDER variables, but some are unique to mdgx and "
         "some SANDER\n"
         "            variables are not supported)\n"
         "  -EWALD:   print descriptions of &ewald namelist variables\n"
         "  -FORCE:   print descriptions of &force namelist variables\n"
         "  -FIT:     print descriptions of &fit namelist variables\n"
         "  -ATTR:    attributions of certain aspects of the code, with "
	 "references\n\n");
}

/***=======================================================================***/
/*** PrintCommandLineInputOptions: this function essentially reproduces    ***/
/***                               what the AMBER manual already has for   ***/
/***                               SANDER and PMEMD command line input,    ***/
/***                               but since mdgx has some new features    ***/
/***                               it is necessary to have independent     ***/
/***                               documentation.                          ***/
/***=======================================================================***/
void PrintCommandLineInputOptions()
{
  PrintSplash(stdout);
  PrintVADesc(2, "-O", 5, " ", 2, "Overwrite output files if they exist "
	      "(appending files with the -A option found in SANDER is "
	      "currently not supported)", 71, 0);
  PrintVADesc(2, "-i", 5, " ", 2, "(input) control data for an energy "
	      "minimization / molecular dynamics run", 71, 0);
  PrintVADesc(2, "-o", 5, " ", 2, "(output) human-readable state information "
	      "and diagnostics", 71, 0);
  PrintVADesc(2, "-p", 5, " ", 2, "(input) molecular topology file (AMBER "
	      "prmtop format)", 71, 0);
  PrintVADesc(2, "-p2", 5, " ", 2, "(input) molecular topology file; if "
	      "thermodynamic integration is active, this topology describes "
	      "the final state while the original topology describes the "
	      "initial state", 71, 0);
  PrintVADesc(2, "-xpt", 5, " ", 2, "(input) extra points rule file directing "
	      "mdgx to add extra points to the topology at run time", 71, 0);
  PrintVADesc(2, "-xpt2", 5, " ", 2, "(input) extra points rule file for the "
	      "topology specified by the -p2 flag", 71, 0);
  PrintVADesc(2, "-c", 5, " ", 2, "(input) initial coordinates (and, "
	      "optionally, velocities) and periodic box dimensions", 71, 0);
  PrintVADesc(2, "-c2", 5, " ", 2, "(input) initial coordinates; if "
	      "thermodynamic integration is active, this second set of input "
	      "coordinates pertains to the initial coordinates of the system "
	      "in its final state as the mixing parameter lambda goes to 1",
	      71, 0);
  PrintVADesc(2, "-d", 5, " ", 2, "(output) comprehensive force / energy "
	      "report file", 71, 0);
  PrintVADesc(2, "-x", 5, " ", 2, "(output) coordinate trajectory file", 71,
	      0);
  PrintVADesc(2, "-x2", 5, " ", 2, "(output) coordinate trajectory file; only "
	      "used when thermodynamic integration is active", 71, 0);
  PrintVADesc(2, "-v", 5, " ", 2, "(output) velocity trajectory file", 71, 0);
  PrintVADesc(2, "-v2", 5, " ", 2, "(output) velocity trajectory file; only "
	      "used when thermodynamic integration is active", 71, 0);
  PrintVADesc(2, "-e", 5, " ", 2, "(output) energy data over trajectory", 71,
	      0);
  PrintVADesc(2, "-r", 5, " ", 2, "(output) checkpoint (and final) "
	      "coordinates and periodic unit cell dimensions from energy "
	      "minimization runs, plus final velocities from molecular "
	      "dynamics runs\n", 71, 0);
  PrintVADesc(2, "-r2", 5, " ", 2, "(output) checkpoint file; only used when "
	      "thermodynamic integration is active\n", 71, 0);
}

/***=======================================================================***/
/*** PrintInputFormat: helpful documentation on the format of mdgx input   ***/
/***                   files.                                              ***/
/***=======================================================================***/
void PrintInputFormat()
{
  PrintSplash(stdout);
  PrintParagraph("The typical mdgx input file is designed to look very much "
		 "like a typical SANDER input file.  However, there are some "
		 "key differences implemented to make the mdgx input file "
		 "format more flexible and the control data more intuitive.",
		 79, stdout);
  PrintParagraph("The only significant restriction introduced to the mdgx "
		 "input file format is that different segments of the input "
		 "file must begin with the namelist identifier (e.g. &cntrl, "
		 "&ewald) on its own line, and be terminated with the "
		 "identifier &end, again on its own line.", 79, stdout);
  printf("Here is an example input file:\n\n"
	 "&files\n"
	 "  -p       Tests/ions.top\n"
	 "  -c       Tests/ions.min\n"
	 "  -rst     Tests/ionsMDGX\n"
	 "  -rstsuff .rst\n"
	 "&end\n\n"
	 "&cntrl\n"
	 "  DoRATTLE = 1,   LJ14Fac = 0.5,   Elec14Fac = 0.8333,\n"
	 "  ElecCut = 9.0,  vdw_cutoff = 15.0,\n"
	 "  dt = 0.001,   nstlim 500000,  nfistep = 1000,\n"
	 "  ntpr = 1000,   ntwr 1000,  ntwf = 1000,\n"
	 "  Temperature = 0.0,\n"
	 "  SplnSpc = 0.015625,\n"
	 "&end\n\n"
	 "&ewald\n"
	 "  ordr1 = 4,\n"
	 "  ordr2 = 4,\n"
	 "  ordr3 = 4,\n"
	 "  nfft1 = 64,\n"
	 "  nfft2 = 64,\n"
	 "  nfft3 = 64;\n"
	 "&end\n\n");
  PrintParagraph("Note the presence of the familiar <namelist identifier> "
		 "<arguments> <&end> format, carried over from SANDER. "
		 "However, mdgx includes new namelists such as &files (which "
		 "allows the bulk of the command line information to be given "
		 "in the input file) and new variables such as file suffixes "
		 "(for specifying multiple files in a single run).  There are "
		 "also aliases for the familiar (though sometimes "
		 "unintelligible) SANDER namelist variables, to provide a "
		 "format that will accept SANDER input while also permitting "
		 "users to write less cryptic command files.", 79, stdout);
  PrintParagraph("Another important change to the mdgx file format is that = "
		 "signs are no longer required between a variable name and "
		 "the desired value.  In fact, commas are no longer required "
		 "either, though they are useful for separating different "
		 "attributes in each namelist.", 79, stdout);
}

/***=======================================================================***/
/*** PrintFilesNamelistVariables: this function provides documentation on  ***/
/***                              &files namelist variables, with whatever ***/
/***                              aliases may be available.                ***/
/***=======================================================================***/
void PrintFilesNamelistVariables()
{
  PrintSplash(stdout);
  PrintParagraph("The files namelist is a means of specifying many input and "
		 "output files, which would otherwise be given on the command "
		 "line, in the input file along with other namelists. "
		 "However, if files are specified on the command line, they "
		 "will take precedence over any data in the &files namelist.",
		 79, stdout);
  PrintParagraph("The &files namelist can also be used to specify suffixes "
		 "for output files, in the event that multiple sequential "
		 "output files are to be generated in a single run.  This can "
		 "be useful for runs on managed resources that do not allow "
		 "multiple initializations of a program in a single job "
		 "submission, or for managing many segments of a very long "
		 "trajectory.  The suffixes only come into use if the "
		 "variable nfistep (alias FileStepCount) is set in the &cntrl "
		 "namelist.  Otherwise, only one file of each output type "
		 "will be written and only the base file names will be used. "
		 "File overwriting has slightly different behavior when "
		 "multiple sequential output files are specified.  If "
		 "overwriting is activated in such a case, mdgx will search "
		 "for the lowest file number such that a complete state "
		 "information file (specified by the -o variable) or a "
		 "complete restart file (specified by the -r variable) are "
		 "unavailable.  The dynamics will begin, or resume, at that "
		 "point.", 79, stdout);
  printf("  Name      Alias      Description\n"
	 " ------ ------------- ---------------------------------------------"
	 "-------------\n");
  PrintVADesc(1, "-p", 7, "Topology", 14, "(input) molecular topology file "
	      "(AMBER prmtop format)", 58, 2);
  PrintVADesc(1, "-p2", 7, "Topology2", 14, "(input) molecular topology file; "
	      "if thermodynamic integration is active, this topology "
	      "describes the final state while the original topology "
	      "describes the initial state", 58, 2);
  PrintVADesc(1, "-xpt", 7, "EPRules", 14, "(input) extra points rule file "
	      "directing mdgx to add extra points to the topology at run time",
	      58, 2);
  PrintVADesc(1, "-xpt2", 7, "EPRules2", 14, "(input) extra points rule file "
	      "for the topology specified by the -p2 flag", 58, 2);
  PrintVADesc(1, "-c", 7, "StartCrd", 14, "(input) initial coordinates (and, "
	      "optionally, velocities) and periodic unit cell size (mdgx does "
	      "not run with non-periodic unit cells)", 58, 2);
  PrintVADesc(1, "-d", 7, "ForceDump", 14, "(output) a comprehensive force "
	      "and energy report; this file is analogous to forcedump.dat as "
	      "produced by SANDER", 58, 2);
  PrintVADesc(1, "-rrp", 7, "ResReport", 14, "(output) a complete description "
	      "of the various residue types in the system; does not include "
	      "information on connections between residues", 58, 2);
  PrintVADesc(1, "-o", 7, "OutputBase", 14, "(output) human-readable state "
	      "information and diagnostics", 58, 2);
  PrintVADesc(1, "-e", 7, "EnergyBase", 14, "(output) energy data over "
	      "trajectory", 58, 2);
  PrintVADesc(1, "-x", 7, "CrdTrajBase", 14, "(output) coordinate trajectory "
	      "file; coordinate sets saved at specified time intervals", 58,
	      2);
  PrintVADesc(1, "-v", 7, "VelTrajBase", 14, "(output) velocity trajectory "
	      "file; velocity sets saved at specified time intervals", 58, 2);
  PrintVADesc(1, "-f", 7, "FrcTrajBase", 14, "(output) force trajectory "
	      "file; forces on all particles saved at specified time"
	      "intervals", 58, 2);
  PrintVADesc(1, "-r", 7, "RestartBase", 14, "(output) checkpoint (and final) "
	      "coordinates and periodic unit cell dimensions from energy "
	      "minimization runs, plus final velocities from molecular"
	      "dynamics runs", 58, 2);
  PrintVADesc(1, "-osf", 7, "OutputSuff", 14, "output state information data "
	      "file suffix (if this or other suffixes are not specified, the "
	      "base name is taken to be the complete file name)", 58, 2);
  PrintVADesc(1, "-esf", 7, "EnergySuff", 14, "Energy data file suffix", 58,
	      2);
  PrintVADesc(1, "-xsf", 7, "CrdTrajSuff", 14, "Coordinate trajectory file "
	      "suffix", 58, 2);
  PrintVADesc(1, "-vsf", 7, "VelTrajSuff", 14, "Velocity trajectory file "
	      "suffix", 58, 2);
  PrintVADesc(1, "-fsf", 7, "FrcTrajSuff", 14, "Force trajectory file suffix",
	      58, 2);
  PrintVADesc(1, "-rsf", 7, "RestartSuff", 14, "Restart file suffix\n", 58, 2);
}

/***=======================================================================***/
/*** PrintCntrlNamelistVariables: this function provides documentation on  ***/
/***                              &cntrl namelist variables, with whatever ***/
/***                              aliases may be available.                ***/
/***=======================================================================***/
void PrintCntrlNamelistVariables()
{
  PrintSplash(stdout);
  PrintParagraph("The &cntrl namelist is required for any SANDER run, and is "
		 "present with little modification in mdgx.  Most SANDER "
		 "input files can therefore be read directly by mdgx or "
		 "adapted without much effort.  However, there are some "
		 "variables that have either been replaced or discarded.  The "
		 "following list describes all variables available in the "
		 "mdgx &cntrl namelist.", 79, stdout);
  PrintParagraph("Molecular Dynamics Variables:", 79, stdout);
  printf("    Name        Alias      Description\n"
         " ---------- ------------- -----------------------------------------"
         "-------------\n");
  PrintVADesc(1, "imin", 11, "RunMode", 14, "The run mode (0 = molecular "
	      "dynamics, 1 = energy minimization, 2 = force computation for "
	      "the input coordinates)", 54, 2);
  PrintVADesc(1, "irest", 11, "RestartMD", 14, "Set to 1 to request that "
	      "molecular dynamics be restarted using velocities given in the "
	      "input coordiantes file; set to 0 to request that initial " 
	      "velocities be assigned to random values in a Maxwell "
	      "distribution", 54, 2);
  PrintVADesc(1, "ioutfm", 11, "CoordFormat", 14, "The format of trajectory "
              "output coordinates, 0 (default) being ascii format with three "
	      "decimal places and 1 being binary NetCDF format, specifying "
	      "all coordinates to single floating point precision", 54, 2);
  PrintVADesc(1, "nstlim", 11, "StepCount", 14, "Number of MD steps to be "
	      "performed", 54, 2);
  PrintVADesc(1, "nfistep", 11, "FileStepCount", 14, "Length of each segment "
              "of the trajectory; the number of segments is nstlim / nfistep",
	      54, 2);
  PrintVADesc(1, "nscm", 11, "ZeroMomentum", 14, "Removal of translational "
	      "and rotational center-of-mass (COM) motion at regular "
	      "intervals", 54, 2);
  PrintVADesc(1, "t", 11, "StartTime", 14, "Time at the start of the "
	      "simulation (picoseconds); this parameter is for user reference "
	      "and otherwise only affects the time entered in the diagnostics "
	      "files", 54, 2);
  PrintVADesc(1, "dt", 11, "TimeStep", 14, "Time step (picoseconds); the "
	      "recommended MAXIMUM is 0.0025 if bonds to hydrogen are "
	      "constrained, or 0.001 otherwise", 54, 2);
  PrintVADesc(1, "temp0", 11, "Temperature", 14, "Target temperature for a "
              "constant temperature simulation.  The default is 298.0.  This "
	      "value can be used to initialize velocities if a specific "
	      "initial temperature is not set.", 54, 2);
  PrintVADesc(1, "tempi", 11, "StartTemp", 14, "Initial temperature for a "
              "simulation.  The default is -100.0, which commands mdgx to use "
	      "temp0 as the initial temperature for things such as velocity "
	      "initialization.  If a positive value of tempi is specified, "
	      "tempi will be used to initialize velocities.", 54, 2);
  PrintVADesc(1, "ntt", 11, "Thermostat", 14, "Thermostat specification.  "
	      "Numerical values of 0 (default, no thermostat), 1 (Berendsen "
	      "thermostat), 2 (Andersen thermostat), and 4 (Nose-Hoover "
	      "thermostat) are supported.", 54, 2);
  PrintVADesc(1, "ntp", 11, "Barostat", 14, "Barostat and coordinate "
	      "rescaling specification.  Numerical values of 0 (default, no "
	      "barostat, constant volume), 1 (Berendsen barostat, isotropic "
	      "rescaling), and 5 (Monte-Carlo barostat, with the option of "
	      "anisotropic rescaling) are supported.", 54, 2);
  PrintVADesc(1, "pres0", 11, "Pressure", 14, "Target pressure for a constant "
              "pressure simulation.  The default is 1.0 bar.", 54, 2);
  PrintVADesc(1, "tautp", 11, "BerendsenTC", 14, "Time constant for "
	      "Berendsen temperature coupling.  Default value is 0.4 ps.", 54,
              2);
  PrintVADesc(1, "taup", 11, "BerendsenPC", 14, "Compressibility constant for "
	      "Berendsen pressure coupling.  Default value is 4.4e-5 / bar.",
	      54, 2);
  PrintVADesc(1, "tauthv", 11, "HooverTC", 14, "Time constant for Hoover "
	      "temperature coupling.  Default value is 1.0 ps.", 54, 2);
  PrintVADesc(1, "tauphv", 11, "HooverPC", 14, "Compressibility constant for "
              "Hoover pressure coupling.  Default value is 1.0 / bar.", 54, 2);
  PrintVADesc(1, "mccomp", 11, "MCBarostatPC", 14, "Coordinate rescaling "
	      "factor for isotropic Monte-Carlo pressure coupling.  Default "
	      "is 2.0e-3, to rescale the volume by +/- 1/10th of 1%.", 54, 2);
  PrintVADesc(1, "mccompx", 11, "MCBarostatPCX", 14, "Coordinate rescaling "
	      "factor for anisotropic Monte-Carlo pressure coupling in the X "
	      "direction.  Default is 2.0e-3, to rescale the volume by +/- "
	      "1/10th of 1%.", 54, 2);
  PrintVADesc(1, "mccompy", 11, "MCBarostatPCY", 14, "Coordinate rescaling "
	      "factor for anisotropic Monte-Carlo pressure coupling in the Y "
	      "direction.  Default is to perform only isotropic rescaling "
	      "based on the factor for the X direction.", 54, 2);
  PrintVADesc(1, "mccompz", 11, "MCBarostatPCZ", 14, "Coordinate rescaling "
	      "factor for anisotropic Monte-Carlo pressure coupling in the Z "
	      "direction.  Default is to perform only isotropic rescaling "
	      "based on the factor for the X direction.", 54, 2);
  PrintVADesc(1, "mcbfrq", 11, "MCBarostatFrq", 14, "Step frequency for "
	      "applying the Monte-Carlo barostat, if this barostat is "
	      "activated.  Default 100.", 54, 2);
  PrintVADesc(1, "vrand", 11, "RandomReset", 14, "Time constant for Andersen "
	      "temperature coupling.  This is specified as an integer "
	      "denoting the number of time steps, and as such is related to "
	      "the time step size dt.  Default value is 1000.", 54, 2);
  PrintVADesc(1, "ig", 11, "RandomSeed", 14, "The random seed for velocity "
	      "initialization and thermostats which may require it", 54, 2);
  PrintVADesc(1, "es_cutoff", 11, "ElecCut", 14, "The electrostatic direct "
	      "space cutoff (Angstroms)", 54, 2);
  PrintVADesc(1, "vdw_cutoff", 11, "VdwCut", 14, "The van-der Waals (direct "
	      "space) cutoff (Angstroms)", 54, 2);
  PrintVADesc(1, "cut", 11, "DirectCutoff", 14, "The general (van-der Waals "
              "and electrostatic) direct space cutoff (Angstroms).  This "
	      "value, if indicated, will override es_cutoff and vdw_cutoff.",
	      54, 2);
  PrintVADesc(1, "rigidbond", 11, "DoRATTLE", 14, "Set to 1 to activate "
	      "RATTLE bond length constraints", 54, 2);
  PrintVADesc(1, "rigidwat", 11, "DoSETTLE", 14, "Set to 1 to activate "
	      "SETTLE water geometry constraints", 54, 2);
  PrintVADesc(1, "tol", 11, "RattleTol", 14, "Tolerance for RATTLE bond "
	      "length constraints", 54, 2);
  PrintVADesc(1, "scee", 11, "Elec14Fac", 14, "The electrostatic 1-4 "
	      "interaction scaling factor", 54, 2);
  PrintVADesc(1, "scnb", 11, "Vdw14Fac", 14, "The van-der Waals 1-4 "
	      "interaction scaling factor", 54, 2);
  printf("\n");
  PrintParagraph("Thermodynamic integration control variables:", 79, stdout);
  printf("  Name        Alias        Description\n"
         " ------ ----------------- -----------------------------------------"
         "-------------\n");
  PrintVADesc(1, "icfe", 11, "RunTI", 14, "Flag to activate thermodynamic "
	      "integration.  Default 0 (no TI); set to 1 for active.", 54, 2);
  PrintVADesc(1, "clambda", 11, "MixFactor", 14, "The mixing parameter, "
	      "(1-L)^k of state 1 and 1-(1-L)^k of state 2", 54, 2);
  PrintVADesc(1, "klambda", 11, "MixOrder", 14, "The order of the mixing "
	      "parameter, (1-L)^k of state 1 and 1-(1-L)^k of state 2", 54, 2);
  PrintVADesc(1, "nsynch", 11, "SynchTI", 14, "Explicit synchronization of "
	      "trajectory coordinates will occur every nsynch steps (default "
	      "1000).  Set nsynch to 0 to disable this feature.", 54, 2);
  printf("\n");
  PrintParagraph("Output control variables:", 79, stdout);
  printf("  Name        Alias        Description\n"
         " ------ ----------------- -----------------------------------------"
         "-------------\n");
  PrintVADesc(1, "ntpr", 7, "WriteDiagnostics", 18, "Diagnostics and state "
	      "information output frequency", 54, 2);
  PrintVADesc(1, "ntwx", 7, "WriteCoordinates", 18, "Trajectory coordinates "
	      "will be written at this frequency", 54, 2);
  PrintVADesc(1, "ntwv", 7, "WriteCoordinates", 18, "Trajectory velocities "
	      "will be written at this frequency", 54, 2);
  PrintVADesc(1, "ntwf", 7, "WriteCoordinates", 18, "Trajectory forces will "
	      "be written at this frequency", 54, 2);
  PrintVADesc(1, "ntwr", 7, "WriteCoordinates", 18, "Trajectory forces will "
	      "be written at this frequency", 54, 2);
  PrintVADesc(1, "tchk", 7, "TopologyCheck", 18, "Active by default, set to 0 "
	      "to turn off topology and conformation checking at the start of "
	      "each run segment", 54, 2);
}

/***=======================================================================***/
/*** PrintEwaldNamelistVariables: this function provides documentation on  ***/
/***                              &ewald namelist variables, with whatever ***/
/***                              aliases may be available.                ***/
/***=======================================================================***/
void PrintEwaldNamelistVariables()
{
  PrintSplash(stdout);
  PrintParagraph("All variables in the &ewald namelist have default values, "
		 "so use of this namelist is optional, but optimization of "
		 "the parameters in this section can be very helpful for "
		 "performing the most efficient molecular simulations.  The "
		 "SANDER and PMEMD programs warn users not to modify "
		 "variables in this section without significant experience; "
		 "what is most important is a clear understanding of all the "
		 "variables and how they will affect the accuracy of "
		 "computed forces.  The state information / diagnostics "
		 "output file will print verbose explanations of the "
		 "consequences of any variables that are changed, so "
		 "modification of these variables, with careful reading of "
		 "the output and checks on the accuracy of computed forces, "
		 "should be safe.", 79, stdout);
  PrintParagraph("Smooth Particle Mesh Ewald (SPME) Variables:", 79, stdout);
  printf("    Name        Alias      Description\n"
         " ---------- ------------- ------------------------------------------"
         "-----------\n");
  PrintVADesc(1, "dsum_tol", 11, "DSumTol", 14, "The direct sum tolerance; "
	      "at the direct space cutoff es_cutoff (see the &cntrl namelist)"
	      ", the ratio between the interaction energy of two point "
	      "charges and two Gaussian smeared charges of the same magnitude "
	      "will differ from 1 by dsum_tol; this variable controls the "
	      "accuracy of the electrostatic direct space sum", 54, 2);
  PrintVADesc(1, "sigma", 11, "Sigma", 14, "The width (root mean squared "
	      "deviation) of spherical Gaussians used to smooth out the "
	      "charge density; sigma = ew_coeff / 2; if this value is "
	      "supplied by the user it will supercede any entry for ew_coeff "
	      "and dsum_tol will be calculated from sigma; otherwise, sigma "
	      "and ew_coeff will be calculated from dsum_tol", 54, 2);
  PrintVADesc(1, "ew_coeff", 11, "EwaldCof", 14, "The Ewald coefficient; this "
	      "quantity has a name only because it appears frequently in "
	      "the Smooth Particle Mesh Ewald formulas; physically it makes "
	      "more sense to consider the Gaussian charge width sigma", 54, 2);
  PrintVADesc(1, "eetbdns", 11, "SplnSpc", 14, "The discretization of the "
	      "erfc(beta*r)/r force and energy spline computation tables", 54,
	      2);
  PrintVADesc(1, "rho", 11, "MaxDensity", 14, "The maximum expected density "
	      "of the system, g/mL.  Default 2.0, increase to raise the "
	      "maximum storage in the direct-space decomposition cell grid.",
	      54, 2);
  PrintVADesc(1, "nfft[123]", 11, "MeshDim[XYZ]", 14, "The number of mesh "
	      "points in the X, Y, or Z dimensions, respectively (if the unit "
	      "cell / simulaton box is orthorhombic), or otherwise the number "
	      "of mesh points along the 1st, 2nd, and 3rd unit cell vectors",
	      54, 2);
  PrintVADesc(1, "ordr[123]", 11, "Order[XYZ]", 14, "The order of particle "
	      "<--> mesh interpolation along the 1st, 2nd, and 3rd unit cell "
	      "vectors", 54, 2);
  PrintVADesc(1, "order", 11, "Order", 14, "Sets the interpolation order "
	      "isotropically along all unit cell vectors to the specified "
	      "value (supercedes ordr[123])", 54, 2);
  printf("\n");
  PrintParagraph("Long-ranged van-der Waals control parameters:", 79, stdout);
  printf("    Name        Alias      Description\n"
         " ---------- ------------- ------------------------------------------"
         "-----------\n");
  PrintVADesc(1, "vdwmeth", 11, "vdwMethod", 14, "The method for computing "
	      "long-ranged van der Waals interactions.  The default of 1 "
	      "implies the inclusion of a homogeneity assumption in the "
	      "long-ranged component of the van-der Waals interactions; this "
	      "correction changes the computed energy and pressure, but not "
	      "forces, and would therefore not affect dynamics in a "
	      "simulation at constant volume. A value of zero removes any "
	      "such correction and make the van-der Waals energy depend "
	      "solely on the pairwise interactions.", 54, 2);
  printf("\n");
  PrintParagraph("Multi-Level Ewald (MLE) Variables:", 79, stdout);
  printf("    Name        Alias      Description\n"
         " ---------- ------------- ------------------------------------------"
         "-----------\n");
  PrintVADesc(1, "nlev", 11, "EwaldLevels", 14, "The number of levels in the "
	      "multi-level mesh hierarchy (standard Smooth Particle Mesh "
	      "Ewald has one level); setting this variable to any value "
	      "greater than one will activate MLE (maximum 4)", 54, 2);
  PrintVADesc(1, "lpad[123]", 11, "Padding[123]", 14, "The number of layers "
	      "of \"padding\" for meshes at levels 1, 2, and 3; the "
	      "reciprocal space pair potential will be represented exactly on "
	      "mesh level 1 up to (and including) lpad1 grid points from the "
	      "source, and will be represented to different degrees of "
	      "resolution (see cfac[234]) on higher mesh levels, up to "
	      "(and including) lpad1 + lpad2 or lpad1 + lpad2 + lpad3 layers "
	      "from the source (note that the highest mesh level is not "
	      "padded as it involves only one convolution over the entire "
	      "simulation box); higher values of lpad[123] will produce more "
	      "accurate results (see also ggordr)", 54, 2);
  PrintVADesc(1, "cfac[234]", 11, "Spread[234]", 14, "The coarsening factor "
	      "for higher mesh levels; by definition, cfac1 is 1; generally, "
	      "it is advisable to set cfac2 to 1.5-2.0, and to set cfac3 or "
	      "cfac4 (if even higher mesh levels are in use) to increasingly "
	      "large numbers; although cfac is a real number, but it must be "
	      "specified such that the mesh size is an integer multiple of "
	      "cfac in every dimension", 54, 2);
  PrintVADesc(1, "ggordr", 11, "GridOrder", 14, "The order of grid <--> grid "
	      "B-Spline interpolation; higher orders of interpolation will "
	      "produce more accurate results for given values of lpad[123]",
	      54, 2);
}

/***=======================================================================***/
/*** PrintForceNamelistVariables: the force namelist was added to support  ***/
/***                              customization of detailed force report   ***/
/***                              files.                                   ***/
/***=======================================================================***/
void PrintForceNamelistVariables()
{
  PrintSplash(stdout);
  PrintParagraph("By default, all available information is printed to the "
		 "force report file; all of the variables listed below are "
		 "set to 1 by default.  Most of this information is not "
		 "needed, however, so much of the output can be suppressed by "
		 "setting these variables to 0.", 79, stdout);
  printf("   Name       Alias      Description\n"
         " -------- ------------- --------------------------------------------"
         "-----------\n");
  PrintVADesc(1, "var", 11, "VarName", 14, "Information is dumped into a "
	      "Matlab-readable force report file; this specifies the name of "
	      "the variable that Matlab will use to store the force "
	      "information if it reads the report; it may be useful to read "
	      "multiple reports at once, and compare the results using Matlab",
	      56, 2);
  PrintVADesc(1, "dumpcrd", 11, "DumpCoord", 14, "Flag to dump the "
	      "coordinates of the system", 56, 2);
  PrintVADesc(1, "dumpbond", 11, "DumpBond", 14, "Flag to dump the forces due "
	      "to bonded (1-2) interactions", 56, 2);
  PrintVADesc(1, "dumpangl", 11, "DumpAngl", 14, "Flag to dump the forces due "
	      "to angle interactions", 56, 2);
  PrintVADesc(1, "dumpdihe", 11, "DumpDihe", 14, "Flag to dump the dihedral "
	      "forces", 56, 2);
  PrintVADesc(1, "dumpdelec", 11, "DumpDElec", 14, "Flag to dump direct sum "
	      "electrostatic forces", 56, 2);
  PrintVADesc(1, "dumprelec", 11, "DumpRElec", 14, "Flag to dump reciprocal "
	      "sum electrostatic forces", 56, 2);
  PrintVADesc(1, "dumpvdw", 11, "DumpVdw", 14, "Flag to dump van-der Waals "
	      "forces", 56, 2);
  PrintVADesc(1, "dumpall", 11, "DumpAll", 14, "Flag to dump total (summed) "
	      "forces", 56, 2);
}

/***=======================================================================***/
/*** PrintFitNamelistVariables: the fit namelist was added to support the  ***/
/***                            greatly expanded extra points features.    ***/
/***=======================================================================***/
void PrintFitNamelistVariables()
{
  PrintSplash(stdout);
  PrintParagraph("The basic concept of fitting charges to reproduce the "
		 "electrostatic potential of a molecule, by finding the "
		 "solution with least squared error, in the presence of "
		 "restraints, is carried over from the original Kollmann "
		 "RESP.  This namelist provides tools for fitting charges in "
		 "a comprehensible manner, with exceptional user control over "
		 "the range of the fitting data.  Because mdgx is unique "
		 "among the current AMBER molecular dynamics engines for its "
		 "ability to use certain types of extra points (virtual "
		 "sites), this namelist is also useful for fitting new charge "
		 "models to accelerate parameter development in mdgx.", 79,
		 stdout);
  PrintParagraph("Many of these variables can be appended with a # identifier "
		 "which specifies multiple instances of the same type of "
		 "input.  The list must be continuous (i.e. 1 3 4 5 6 is not "
		 "allowed), and not specifying a # tag implies the first "
		 "element of the list.", 79, stdout);
  printf("   Name       Alias      Description\n"
         " -------- ------------- --------------------------------------------"
         "-----------\n");
  PrintVADesc(1, "phi#", 11, "QMPhi#", 14, "Names of additional electrostatic "
	      "potentials to use in fitting.  The files are read as formatted "
	      "Gaussian cubegen output, containing electrostatic potentials "
	      "sampled on a regular grid and a list of molecular coordinates "
	      "which is expected to match the atoms found in the prmtop.", 56,
	      2);
  PrintVADesc(1, "auxphi#", 11, "AuxPhi#", 14, "An auxiliary electrostatic "
	      "potential to use in fitting, also in formatted Gaussian "
	      "cubegen output, corresponding to phi#.  The effect of "
	      "specifying an auxiliary potential is to have a single set "
	      "of charges fit to reproduce the average of the two potentials."
	      "  This feature supports development of fixed-charge force "
	      "fields if one posits that the correct charges of a "
	      "non-polarizable model would sit halfway between the charges "
	      "of a fully polarized molecule in some solvent reaction field "
	      "and the charges of an unpolarized molecule in the gas phase.",
	      56, 2);
  PrintVADesc(1, "conf", 11, "ConfFile", 14, "If specified, mdgx, will output "
	      "the first molecular conformation, complete with any added "
	      "virtual sites, in PDB format for inspection.  This is useful "
	      "for understanding exactly what model is being fitted.", 56, 2);
  PrintVADesc(1, "eprules", 11, "EPRules", 14, "If specified, mdgx, will "
	      "output all fitted charges in the form of a Virtual Sites rule "
	      "file, which can be given as input to subsequent simulations to "
	      "modify the original prmtp and apply the fitted charge model.",
	      56, 2);
  PrintVADesc(1, "hist", 11, "HistFile", 14, "If specified, mdgx will print "
	      "a histogram of the distance of all points from the nearest "
	      "atom of the molecule.", 56, 2);
  PrintVADesc(1, "qtot", 11, "TotalQ", 14, "The total charge constraint in "
	      "units of the proton charge; the sum of all fitted charges is "
	      "required to equal this value.  Default 0.0.", 56, 2);
  PrintVADesc(1, "minq#", 11, "MinimizeQ#", 14, "Restrain the charges of a "
	      "group of atoms to zero by the weight given in minqwt.  The "
	      "groups are specified in ambmask format.", 56, 2);
  PrintVADesc(1, "equalq#", 11, "EqualizeQ#", 14, "Restrain the charges of a "
	      "group of atoms to have the same values.  Groups are specified "
	      "in ambmask format.", 56, 2);
  PrintVADesc(1, "minqwt", 11, "MinQWeight", 14, "Weight used for restraining "
	      "values of charges to zero; as more and more fitting data is "
	      "included (either through a higher sampling density of the "
	      "electrostatic potential due to each molecular conformation or "
	      "additional molecular conformations) higher values of minqwt "
	      "may be needed to keep the fitted charges small.  However, with "
	      "more data the need to restrain charges may diminish as well.",
	      56, 2);
  PrintVADesc(1, "phiwt#", 11, "PhiWeight#", 14, "The weights assigned to "
	      "electrostatic potentials specified by phi#.  This modulates "
	      "the importance of one molecular configuration, and the "
	      "electrostatic potential it implies, in the fit.", 56, 2);
  PrintVADesc(1, "nfpt", 11, "FitPoints", 14, "The number of fitting points "
	      "to select from each electrostatic potential grid.  The points "
	      "nearest the molecule, which satisfy the limits set by the "
	      "solvent probe and point-to-point distances as defined below, "
	      "will be selected for the fit.  Default 1000.", 56, 2);
  PrintVADesc(1, "psig", 11, "ProbeSig", 14, "The Lennard-Jones sigma "
	      "parameter of the solvent probe.  Default 3.16435 (TIP4P "
	      "oxygen).", 56, 2);
  PrintVADesc(1, "peps", 11, "ProbeEps", 14, "The Lennard-Jones parameter of "
	      "the solvent probe.  Default 0.16275 (TIP4P oxygen).", 56, 2);
  PrintVADesc(1, "parm", 11, "ProbeArm", 14, "The probe arm; points on the "
	      "electrostatic potential grid that would be inaccessible to the "
	      "solvent probe may still be included in the fit if they are "
	      "within the probe arm's reach.", 56, 2);
  PrintVADesc(1, "pnrg", 11, "StericLimit", 14, "The maximum Lennard-Jones "
	      "energy of the solvent probe at which a point will qualify for "
	      "inclusion in the fit.  Default 3.0 kcal/mol.", 56, 2);
  PrintVADesc(1, "maxmem", 11, "MaxMemory", 14, "The amount of memory "
	      "available to this job.  The test is actually the amount "
	      "allocated to the fitting and testing matrices, which is the "
	      "number of fitting data points times the number of independent "
	      "charges being fitted times 16 bytes.  The actual memory usage "
	      "will be slightly higher, but the matrices comprise the bulk of "
	      "it.\n", 56, 2);
  PrintVADesc(1, "flim", 11, "Proximity", 14, "The minimum proximity of any "
	      "two points to be included in the fit.  Default 0.4A.", 56, 2);
  PrintVADesc(1, "hbin", 11, "HistogramBin", 14, "If hist is specified, mdgx "
	      "will print a histogram reporting the number of fitting points "
	      "falling within any particular distance of some atom of the "
	      "molecule.  This parameter controls the discretization of the "
	      "histogram.", 56, 2);
  PrintVADesc(1, "verbose", 11, "Verbose", 14, "Print information relating to "
	      "progress on the fitting run.  Default is to print such data.  "
	      "Set to zero to suppress output.", 56, 2);
}

/***=======================================================================***/
/*** PrintAttributions: users are provided with a straightforward means of ***/
/***                    seeing which aspects of this code were taken from  ***/
/***                    other sources.  Attributions are also provided in  ***/
/***                    comments to the source code, where appropriate.    ***/
/***=======================================================================***/
void PrintAttributions()
{
  PrintSplash(stdout);
  PrintParagraph("Attributions:", 79, stdout);
  PrintParagraph("- Implementation of the SETTLE algorithm (J. Comput. Chem. "
		 "13:952-966, 1992) was adapted from the NAMD program source, "
		 "v2.6, developed by the Theoretical and Computational "
		 "Biophysics Group in the Beckman Institute for Advanced "
		 "Science and Technology at the University of Illinois at "
		 "Urbana-Champaign.", 79, stdout);
  PrintParagraph("- Implementation of the Smooth Particle Mesh Ewald "
		 "algorithm (J. Chem. Phys. 103, 8577-8593, 1995) includes "
		 "code developed by Thomas A. Darden for optimization of the "
		 "convolution kernel.", 79, stdout);
  PrintParagraph("- Dr. Robert E. Duke is acknowledged for outstanding advice "
		 "and insights into the problem of efficient and scalable "
		 "molecular dynamics.  The mdgx program would not exist "
		 "without his support.", 79, stdout);
}
