

      module euvac

      use shr_kind_mod, only : r8 => shr_kind_r8
      use abortutils,   only : endrun
      use wrap_nf
      use cam_logfile,  only : iulog
      implicit none

      private
      public :: euvac_init
      public :: euvac_set_etf
      public :: euvac_etf

      save

      integer               :: nstruct
      integer               :: nbins
      real(r8), allocatable :: wc(:)                ! wave interval center (nm)
      real(r8), allocatable :: we(:)                ! wave interval edges (nm)
      real(r8), allocatable :: wlintv(:)            ! wave interval (nm)
      real(r8), allocatable :: wlintvi(:)           ! inverse wave interval (nm)
      real(r8), allocatable :: refmin(:)
      real(r8), allocatable :: afac(:)
      real(r8), allocatable :: euvac_etf(:)

      contains

      subroutine euvac_init (euvac_file)
!---------------------------------------------------------------
!	... initialize euvac etf module
!---------------------------------------------------------------

      use spmd_utils,     only : masterproc
      use error_messages, only : alloc_err
      use ioFileMod,      only : getfil
#ifdef SPMD
      use mpishorthand,   only : mpiint, mpir8, mpicom
#endif

      implicit none

      character(len=*), intent(in) :: euvac_file

!---------------------------------------------------------------
!	... local variables
!---------------------------------------------------------------
      integer  :: ncid
      integer  :: n
      integer  :: dimid
      integer  :: varid
      integer  :: astat
      character(len=256) :: locfn

!-----------------------------------------------------------------------
!	... readin the etf data
!-----------------------------------------------------------------------
masterproc_only : &
      if( masterproc ) then
         call getfil( euvac_file, locfn, 0 )
         call wrap_open (trim(locfn), NF_NOWRITE, ncid)
!-----------------------------------------------------------------------
!	... check primary dimension consistency
!-----------------------------------------------------------------------
         call wrap_inq_dimid( ncid, 'dim1_WC', dimid )
         call wrap_inq_dimlen( ncid, dimid, nbins )
         call wrap_inq_dimid( ncid, 'dim1_WLINT', dimid )
         call wrap_inq_dimlen( ncid, dimid, n )
         if( n /= nbins ) then
            write(iulog,*) 'euvac_init: WLINT dimension(',n,') does not match bin count ',nbins
            call endrun
         end if
         call wrap_inq_dimid( ncid, 'dim1_REFMIN', dimid )
         call wrap_inq_dimlen( ncid, dimid, n )
         if( n /= nbins ) then
            write(iulog,*) 'euvac_init: REFMIN dimension(',n,') does not match bin count ',nbins
            call endrun
         end if
         call wrap_inq_dimid( ncid, 'dim1_AFAC', dimid )
         call wrap_inq_dimlen( ncid, dimid, n )
         if( n /= nbins ) then
            write(iulog,*) 'euvac_init: AFAC dimension(',n,') does not match bin count ',nbins
            call endrun
         end if

!-----------------------------------------------------------------------
!	... allocate primary arrays
!-----------------------------------------------------------------------
         allocate( wc(nbins), we(nbins+1), wlintv(nbins), wlintvi(nbins), &
                   refmin(nbins), afac(nbins), euvac_etf(nbins), stat=astat )
         if( astat /= 0 ) then
	    call alloc_err( astat, 'euvac_init', 'wc ... euvac_etf', nbins )
         end if
!-----------------------------------------------------------------------
!	... read primary arrays
!-----------------------------------------------------------------------
         call wrap_inq_varid( ncid, 'WC', varid )
         call wrap_get_var_realx( ncid, varid, wc )
         call wrap_inq_varid( ncid, 'WLINT', varid )
         call wrap_get_var_realx( ncid, varid, wlintv )
         call wrap_inq_varid( ncid, 'REFMIN', varid )
         call wrap_get_var_realx( ncid, varid, refmin )
         call wrap_inq_varid( ncid, 'AFAC', varid )
         call wrap_get_var_realx( ncid, varid, afac )

         call wrap_close( ncid )
      end if masterproc_only

#ifdef SPMD
      call mpibcast( nbins, 1, mpiint, 0, mpicom )
      if( .not. masterproc ) then
         allocate( wc(nbins), we(nbins+1), wlintv(nbins), wlintvi(nbins), &
                   refmin(nbins), afac(nbins), euvac_etf(nbins), stat=astat )
         if( astat /= 0 ) then
	    call alloc_err( astat, 'euvac_init', 'wc ... euvac_etf', nbins )
         end if
      end if
      call mpibcast( wc, nbins, mpir8, 0, mpicom )
      call mpibcast( wlintv, nbins, mpir8, 0, mpicom )
      call mpibcast( refmin, nbins, mpir8, 0, mpicom )
      call mpibcast( afac, nbins, mpir8, 0, mpicom )
#endif

      wlintvi(:)   = 1._r8/wlintv(:)
      we(:nbins)   = wc(:nbins) - .5_r8*wlintv(:nbins)
      we(nbins+1)  = wc(nbins) + .5_r8*wlintv(nbins)

      end subroutine euvac_init

      subroutine euvac_set_etf( f107, f107a )
!---------------------------------------------------------------
!	... set euvac etf
!---------------------------------------------------------------

      use spmd_utils,     only : masterproc

      implicit none

!---------------------------------------------------------------
!	... dummy arguments
!---------------------------------------------------------------
      real(r8), intent(in) :: f107
      real(r8), intent(in) :: f107a

!---------------------------------------------------------------
!	... local variables
!---------------------------------------------------------------
      real(r8), parameter :: factor = 80._r8
      integer  :: w
      real(r8) :: pindex

      pindex = .5_r8*(f107 + f107a) - factor
      euvac_etf(:) = refmin(:) * max( .8_r8,(1._r8 + afac(:)*pindex) )

      if( masterproc ) then
         write(iulog,*) ' '
         write(iulog,*) '--------------------------------------------------------'
         write(iulog,*) 'euvac_set_etf: f107,f107a = ',f107,f107a
#ifdef EUVAC_DIAGS
         write(iulog,*) 'euvac_set_etf:  wc, etf'
         do w = 1,nbins
            write(iulog,'(1p,2g15.7)') wc(w),euvac_etf(w)
         end do
#endif
         write(iulog,*) '--------------------------------------------------------'
         write(iulog,*) ' '
      end if

      end subroutine euvac_set_etf

      end module euvac
