module mo_lightning
  !----------------------------------------------------------------------
  ! ... the lightning module
  !----------------------------------------------------------------------

  use shr_kind_mod,  only : r8 => shr_kind_r8
  use shr_const_mod, only : pi => shr_const_pi
  use ppgrid,        only : begchunk, endchunk, pcols, pver
  use commap,        only : latwts =>  w, phi => clat, clon
  use abortutils,    only : endrun
#ifdef SPMD
  use mpishorthand,  only : mpicom, mpir8
#endif
  use cam_logfile,   only : iulog
  use spmd_utils,    only : masterproc

  implicit none

  private
  public  :: lightning_inti
  public  :: lightning_no_prod
  public  :: prod_no

  save

  real(r8) :: csrf
  real(r8) :: factor = 0.1_r8              ! user-controlled scaling factor to achieve arbitrary no prod.
  real(r8) :: geo_factor                   ! latitude grid cell factor
  real(r8) :: vdist(16,3)                  ! vertical distribution of lightning
  real(r8), allocatable :: prod_no(:,:,:)
  real(r8), allocatable :: glob_prod_no_col(:,:)
  real(r8), allocatable :: flash_freq(:,:)
  real(r8) :: twopi
  integer :: no_ndx,xno_ndx

contains

  subroutine lightning_inti( lght_no_prd_factor )
    !----------------------------------------------------------------------
    !       ... initialize the lightning module
    !----------------------------------------------------------------------

    use shr_const_mod, only : rearth => shr_const_rearth
    use mo_regrider,   only : regrid_inti, regrid_2d
    use ioFileMod,     only : getfil
    use mo_chem_utls,  only : get_spc_ndx
    use dycore,        only : get_resolution
    use cam_history,   only : addfld, phys_decomp
    use dyn_grid,      only : get_dyn_grid_parm

    implicit none

    !----------------------------------------------------------------------
    !	... dummy args
    !----------------------------------------------------------------------
    real(r8), intent(in), optional :: lght_no_prd_factor        ! lightning no production factor

    !----------------------------------------------------------------------
    !	... local variables
    !----------------------------------------------------------------------
    integer  :: astat
    integer  :: ncid
    integer  :: dimid
    integer  :: vid
    integer  :: gndx
    integer  :: jl, ju
    integer  :: nlat, nlon
    integer  :: plon, plat
    real(r8), allocatable :: lats(:)
    real(r8), allocatable :: lons(:)
    real(r8), allocatable :: landmask(:,:)
    character(len=256) :: locfn
    real(r8)    ::  d2r

    no_ndx = get_spc_ndx('NO')
    xno_ndx = get_spc_ndx('XNO')

    if ( no_ndx < 0 .and. xno_ndx < 0 ) return

    twopi = 2._r8 * pi
    d2r = pi/180._r8
    
    if( present( lght_no_prd_factor ) ) then
       if( lght_no_prd_factor /= 1._r8 ) then
          factor = factor*lght_no_prd_factor
       end if
    else
       select case( get_resolution() )
       case( '1.9x2.5' )
          factor = .38_r8
       case( '2x2.5' )
          factor = .4_r8
       case( '4x5' )
          factor = 1.6_r8
       case( '10x15' )
          factor = 11._r8
       case default
          write(iulog,*) 'lightning_inti: unknown resolution'
          call endrun
       end select
    end if

    if (masterproc) write(iulog,*) 'lght_inti: lightning no production scaling factor = ',factor

    plon = get_dyn_grid_parm('plon')
    plat = get_dyn_grid_parm('plat')

    csrf = twopi*rearth*rearth/real(plon)                            ! rearth in m
    !----------------------------------------------------------------------
    !       ... vdist(kk,itype) = % of lightning nox between (kk-1) and (kk)
    !           km for profile itype
    !----------------------------------------------------------------------
    vdist(:,1) = (/  3.0, 3.0, 3.0, 3.0, 3.4, 3.5, 3.6, 4.0, &       ! midlat cont
                     5.0, 7.0, 9.0, 14.0, 16.0, 14.0, 8.0, 0.5 /)
    vdist(:,2) = (/  2.5, 2.5, 2.5, 2.5, 2.5, 2.5, 2.5, 6.1, &       ! trop marine
                     17.0, 15.4, 14.5, 13.0, 12.5, 2.8, 0.9, 0.3 /)
    vdist(:,3) = (/  2.0, 2.0, 2.0, 1.5, 1.5, 1.5, 3.0, 5.8, &       ! trop cont
                     7.6, 9.6, 11.0, 14.0, 14.0, 14.0, 8.2, 2.3 /)

    allocate( prod_no(pcols,pver,begchunk:endchunk),stat=astat )
    if( astat /= 0 ) then
       write(iulog,*) 'lght_inti: failed to allocate prod_no; error = ',astat
       call endrun
    end if
    allocate( flash_freq(pcols,begchunk:endchunk),stat=astat )
    if( astat /= 0 ) then
       write(iulog,*) 'lght_inti: failed to allocate flash_freq; error = ',astat
       call endrun
    end if
    allocate( glob_prod_no_col(pcols,begchunk:endchunk),stat=astat )
    if( astat /= 0 ) then
       write(iulog,*) 'lght_inti: failed to allocate glob_prod_no_col; error = ',astat
       call endrun
    end if
    prod_no(:,:,:)   = 0._r8
    flash_freq(:,:)  = 0._r8

    geo_factor = 1._r8/(sum(latwts)/real(plat))

    call addfld( 'LNO_COL_PROD','TG N/YR', 1,    'I', 'lighting column NO source', phys_decomp )
    call addfld( 'LNO_PROD',    '/cm3/s',  pver, 'I', 'lighting insitu NO source', phys_decomp )
    call addfld( 'FLASHFRQ',    '1/MIN',   1,    'I', 'lighting flash rate', phys_decomp )        ! flash frequency in grid box per minute (PPP)
    call addfld( 'FLASHENGY',   '   ',     1,    'I', 'lighting flash rate', phys_decomp )        ! flash frequency in grid box per minute (PPP)
    call addfld( 'CLDHGT',      'KM',      1,    'I', 'cloud top height', phys_decomp )           ! cloud top height
    call addfld( 'DCHGZONE',    'KM',      1,    'I', 'depth of discharge zone', phys_decomp )           ! depth of discharge zone
    call addfld( 'CGIC',        'RATIO',   1,    'I', 'ratio of cloud-ground/intracloud discharges', phys_decomp )        ! ratio of cloud-ground/intracloud discharges

  end subroutine lightning_inti

  subroutine lightning_no_prod( state, pbuf, srfflx_state2d )
    !----------------------------------------------------------------------
    !	... set no production from lightning
    !----------------------------------------------------------------------
    use physics_types,    only : physics_state
    use phys_buffer,      only : pbuf_size_max, pbuf_fld, pbuf_get_fld_idx
    use physconst,        only : rga
    use phys_grid,        only : get_rlat_all_p, get_lat_all_p, get_lon_all_p
    use cam_history,      only : outfld
    use camsrfexch_types, only : srfflx_state

    implicit none

    !----------------------------------------------------------------------
    !	... dummy args
    !----------------------------------------------------------------------
    type(physics_state), intent(in) :: state(begchunk:endchunk) ! physics state
    type(pbuf_fld), intent(in)      :: pbuf(pbuf_size_max)      ! physics buffer
    type(srfflx_state), intent(in) :: srfflx_state2d(begchunk:endchunk) ! physics state

    !----------------------------------------------------------------------
    !	... local variables
    !----------------------------------------------------------------------
    real(r8), parameter    :: land   = 1._r8
    real(r8), parameter    :: secpyr = 365._r8 * 8.64e4_r8

    integer :: i, c
    integer :: cldtind             ! level index for cloud top
    integer :: cldbind             ! level index for cloud base > 273k
    integer :: surf_type
    integer :: file                ! file index
    integer :: k, kk, zlow_ind, zhigh_ind, itype
    real(r8) :: glob_flashfreq     ! global flash frequency [s-1]
    real(r8) :: glob_noprod        ! global rate of no production [as tgn/yr]
    real(r8) :: frac_sum           ! work variable
    real(r8) :: zlow
    real(r8) :: zhigh
    real(r8) :: zlow_scal
    real(r8) :: zhigh_scal
    real(r8) :: fraction
    real(r8) :: dchgz
    real(r8) :: dchgzone(pcols,begchunk:endchunk)           ! depth of discharge zone [km]
    real(r8) :: cldhgt(pcols,begchunk:endchunk)             ! cloud top height [km]
    real(r8) :: cgic(pcols,begchunk:endchunk)               ! cloud-ground/intracloud discharge ratio
    real(r8) :: flash_energy(pcols,begchunk:endchunk)       ! energy of flashes per second
    real(r8) :: prod_no_col(pcols,begchunk:endchunk)        ! global no production rate for diagnostics
    real(r8) :: wrk, wrk1
    integer  :: ncol                                    ! columns per chunk
    integer  :: lchnk                                   ! columns per chunk
    real(r8) :: cldtop(pcols)                           ! cloud top level index
    real(r8) :: cldbot(pcols)                           ! cloud bottom level index
    real(r8) :: zmid(pcols,pver)                        ! geopot height above surface at midpoints (m)
    real(r8) :: zint(pcols,pver+1,begchunk:endchunk)    ! geopot height above surface at interfaces (m)
    real(r8) :: zsurf(pcols)                            ! geopot height above surface at interfaces (m)
    real(r8) :: rlats(pcols,begchunk:endchunk)          ! column latitudes in chunks

    !----------------------------------------------------------------------
    ! 	... parameters to determine cg/ic ratio [price and rind, 1993]
    !----------------------------------------------------------------------
    real(r8), parameter  :: ca = .021_r8
    real(r8), parameter  :: cb = -.648_r8
    real(r8), parameter  :: cc = 7.49_r8
    real(r8), parameter  :: cd = -36.54_r8
    real(r8), parameter  :: ce = 64.09_r8
    real(r8), parameter  :: t0 = 273._r8
    real(r8), parameter  :: m2km  = 1.e-3_r8
    real(r8), parameter  :: km2cm = 1.e5_r8
    real(r8) :: lat25
    integer  :: cldtop_ndx, cldbot_ndx
    integer  :: latndx(pcols,begchunk:endchunk)
    integer  :: lonndx(pcols,begchunk:endchunk)
    integer  :: istat
      real(r8) :: flash_freq_land, flash_freq_ocn

#ifdef SPMD
#include <mpif.h>
#endif

    if ( no_ndx < 0 .and. xno_ndx < 0 ) return

    lat25 = twopi * 25._r8 / 360._r8
    !----------------------------------------------------------------------
    !	... initialization
    !----------------------------------------------------------------------

    flash_freq(:,:)       = 0._r8
    prod_no(:,:,:)        = 0._r8
    prod_no_col(:,:)      = 0._r8
    cldhgt(:,:)           = 0._r8
    dchgzone(:,:)         = 0._r8
    cgic(:,:)             = 0._r8
    flash_energy(:,:)     = 0._r8
    glob_prod_no_col(:,:) = 0._r8

    cldtop_ndx = pbuf_get_fld_idx( 'CLDTOP' )
    cldbot_ndx = pbuf_get_fld_idx( 'CLDBOT' )

    !--------------------------------------------------------------------------------
    !	... estimate flash frequency and resulting no emissions
    !           [price, penner, prather, 1997 (jgr)]
    !    lightning only occurs in convective clouds with a discharge zone, i.e.
    !    an altitude range where liquid water, ice crystals, and graupel coexist.
    !    we test this by examining the temperature at the cloud base.
    !    it is assumed that only one thunderstorm occurs per grid box, and its
    !    flash frequency is determined by the maximum cloud top height (not the
    !    depth of the discharge zone). this is somewhat speculative but yields
    !    reasonable results.
    !
    !       the cg/ic ratio is determined by an empirical formula from price and
    !    rind [1993]. the average energy of a cg flash is estimated as 6.7e9 j,
    !    and the average energy of a ic flash is assumed to be 1/10 of that value.
    !       the no production rate is assumed proportional to the discharge energy
    !    with 1e17 n atoms per j. the total number of n atoms is then distributed
    !    over the complete column of grid boxes.
    !--------------------------------------------------------------------------------
    Chunk_loop : do c = begchunk,endchunk
       ncol  = state(c)%ncol
       lchnk = state(c)%lchnk
       cldtop(:ncol) = pbuf(cldtop_ndx)%fld_ptr(1,1:ncol,1,lchnk,1)
       cldbot(:ncol) = pbuf(cldbot_ndx)%fld_ptr(1,1:ncol,1,lchnk,1)
       zsurf(:ncol) = state(c)%phis(:ncol)*rga
       call get_rlat_all_p( c, ncol, rlats(1,c) )
       call get_lat_all_p( c, ncol, latndx(1,c) )
       call get_lon_all_p( c, ncol, lonndx(1,c) )
       do k = 1,pver
          zmid(:ncol,k)   = state(c)%zm(:ncol,k) + zsurf(:ncol)
          zint(:ncol,k,c) = state(c)%zi(:ncol,k) + zsurf(:ncol)
       end do
       zint(:ncol,pver+1,c) = state(c)%zi(:ncol,pver+1) + zsurf(:ncol)

       col_loop : do i = 1,ncol
          !--------------------------------------------------------------------------------
          ! 	... find cloud top and bottom level above 273k
          !--------------------------------------------------------------------------------
          cldtind = nint( cldtop(i) )
          cldbind = nint( cldbot(i) )
          do
             if( cldbind <= cldtind .or. state(c)%t(i,cldbind) < t0 ) then
                exit
             end if
             cldbind = cldbind - 1
          end do
          cloud_layer : if( cldtind < pver .and. cldtind > 0 .and. cldtind < cldbind ) then
             !--------------------------------------------------------------------------------
             !       ... compute cloud top height and depth of charging zone
             !--------------------------------------------------------------------------------
             cldhgt(i,c)   = m2km*max( 0._r8,zint(i,cldtind,c) )
             dchgz = cldhgt(i,c) - m2km*zmid(i,cldbind)
             dchgzone(i,c) = dchgz
             !--------------------------------------------------------------------------------
             !       ... compute flash frequency for given cloud top height
             !           (flashes storm^-1 min^-1)
             !--------------------------------------------------------------------------------
             flash_freq_land = 3.44e-5_r8 * cldhgt(i,c)**4.9_r8 
             flash_freq_ocn  = 6.40e-4_r8 * cldhgt(i,c)**1.7_r8
             flash_freq(i,c) = srfflx_state2d(c)%landfrac(i)*flash_freq_land + &
                               srfflx_state2d(c)%ocnfrac(i) *flash_freq_ocn

             !--------------------------------------------------------------------------------
             !       ... compute cg/ic ratio
             !           cgic = proportion of cg flashes (=pg from ppp paper)
             !--------------------------------------------------------------------------------
             cgic(i,c) = 1./((((ca*dchgz + cb)*dchgz + cc) *dchgz + cd)*dchgz + ce)
             if( dchgz < 5.5_r8 ) then
                cgic(i,c) = 0._r8
             else if( dchgz > 14._r8 ) then
                cgic(i,c) = .02_r8
             end if
             !--------------------------------------------------------------------------------
             !       ... compute flash energy (cg*6.7e9 + ic*6.7e8)
             !           and convert to total energy per second
             !           set ic = cg
             !--------------------------------------------------------------------------------
             flash_energy(i,c) = 6.7e9_r8 * flash_freq(i,c)/60._r8
             !--------------------------------------------------------------------------------
             !       ... LKE Aug 23, 2005: scale production to account for different grid
             !           box sizes. This requires a reduction in the overall fudge factor 
             !           (e.g., from 1.2 to 0.5) 
             !--------------------------------------------------------------------------------
             flash_energy(i,c) =  flash_energy(i,c) * latwts(latndx(i,c)) * geo_factor

             !--------------------------------------------------------------------------------
             ! 	... compute number of n atoms produced per second
             !           and convert to n atoms per second per cm2 and apply fudge factor
             !--------------------------------------------------------------------------------
             prod_no_col(i,c) = 1.e17_r8*flash_energy(i,c)/(1.e4_r8*csrf*latwts(latndx(i,c))) * factor
             !--------------------------------------------------------------------------------
             ! 	... compute global no production rate in tgn/yr:
             !           tgn per second: * 14.00674 * 1.65979e-24 * 1.e-12
             !             nb: 1.65979e-24 = 1/avo
             !           tgn per year: * secpyr
             !--------------------------------------------------------------------------------
             glob_prod_no_col(i,c) = 1.e17_r8*flash_energy(i,c) &
                  * 14.00674_r8 * 1.65979e-24_r8 * 1.e-12_r8 * secpyr * factor
          end if cloud_layer
       end do Col_loop
    end do Chunk_loop

    !--------------------------------------------------------------------------------
    ! 	... Accumulate global total, convert to flashes per second
    !--------------------------------------------------------------------------------
    glob_flashfreq = sum( flash_freq(:,:) )/60._r8

    !--------------------------------------------------------------------------------
    ! 	... Accumulate global NO production rate
    !--------------------------------------------------------------------------------
    glob_noprod = sum( glob_prod_no_col(:,:) )

    !--------------------------------------------------------------------------------
    !	... Gather,scatter global sum
    !--------------------------------------------------------------------------------

#ifdef SPMD
    call mpi_allreduce( glob_flashfreq, wrk, 1, mpi_double_precision, mpi_sum, mpicom, istat )
    if( istat /= MPI_SUCCESS ) then
       write(iulog,*) 'wetdep: MPI_ALLREDUCE for flashfreq failed; error = ',istat
       call endrun
    end if
    glob_flashfreq = wrk
    call mpi_allreduce( glob_noprod, wrk, 1, mpi_double_precision, mpi_sum, mpicom, istat )
    if( istat /= MPI_SUCCESS ) then
       write(iulog,*) 'MOZ_HOOK: MPI_ALLREDUCE for noprod failed; error = ',istat
       call endrun
    end if
    glob_noprod = wrk
#endif

    if( masterproc ) then
       write(iulog,*) ' '
       write(iulog,'(''Global flash freq (/s), lightning NOx (TgN/y) = '',2f10.4)') &
            glob_flashfreq, glob_noprod
    end if

    if( glob_noprod > 0._r8 ) then
       !--------------------------------------------------------------------------------
       !	... Distribute production up to cloud top [Pickering et al., 1998 (JGR)]
       !--------------------------------------------------------------------------------
       do c = begchunk,endchunk
          ncol  = state(c)%ncol
          lchnk = state(c)%lchnk
          cldtop(:ncol) = pbuf(cldtop_ndx)%fld_ptr(1,1:ncol,1,lchnk,1)
          do i = 1,ncol
             cldtind = nint( cldtop(i) )
             if( prod_no_col(i,c) > 0._r8 ) then
                if( cldhgt(i,c) > 0._r8 ) then
                   if( abs( rlats(i,c) ) > lat25 ) then
                      itype = 1                                                    ! midlatitude continental
                   else if( nint( srfflx_state2d(c)%landfrac(i) ) == land ) then
                      itype = 3                                                    ! tropical continental
                   else
                      itype = 2                                                    ! topical marine
                   end if
                   frac_sum = 0._r8
                   do k = cldtind,pver
                      zlow       = zint(i,k+1,c) * m2km                            ! lower interface height (km)
                      zlow_scal  = zlow * 16._r8/cldhgt(i,c)                       ! scale to 16 km convection height
                      zlow_ind   = max( 1,INT(zlow_scal)+1 )                       ! lowest vdist index to include in layer
                      zhigh      = zint(i,k,c) * m2km                              ! upper interface height (km)
                      zhigh_scal = zhigh * 16._r8/cldhgt(i,c)                      ! height (km) scaled to 16km convection height
                      zhigh_ind  = max( 1,MIN( 16,INT(zhigh_scal)+1 ) )            ! highest vdist index to include in layer
                      do kk = zlow_ind,zhigh_ind
                         wrk  = kk
                         wrk1 = kk - 1
                         fraction = min( zhigh_scal,wrk ) &                         ! fraction of vdist in this model layer
                              - max( zlow_scal,wrk1 )
                         fraction = max( 0._r8, min( 1._r8,fraction ) )
                         frac_sum = frac_sum + fraction*vdist(kk,itype)
                         prod_no(i,k,c) = prod_no(i,k,c) &                         ! sum the fraction of column NOx in layer k
                              + fraction*vdist(kk,itype)*.01_r8
                      end do
                      prod_no(i,k,c) = prod_no_col(i,c) * prod_no(i,k,c) &         ! multiply fraction by column amount
                           / (km2cm*(zhigh - zlow))                    ! and convert to atom N cm^-3 s^-1
                   end do
                end if
             end if
          end do
       end do
    end if

    !--------------------------------------------------------------------------------
    !       ... output lightning no production to history file
    !--------------------------------------------------------------------------------
    do c = begchunk,endchunk
       lchnk = state(c)%lchnk
       call outfld( 'LNO_PROD',     prod_no(:,:,c),        pcols, lchnk )
       call outfld( 'LNO_COL_PROD', glob_prod_no_col(:,c), pcols, lchnk )
       call outfld( 'FLASHFRQ',     flash_freq(:,c),       pcols, lchnk )
       call outfld( 'FLASHENGY',    flash_energy(:,c),     pcols, lchnk )
       call outfld( 'CLDHGT',       cldhgt(:,c),           pcols, lchnk )
       call outfld( 'DCHGZONE',     dchgzone(:,c),         pcols, lchnk )
       call outfld( 'CGIC',         cgic(:,c),             pcols, lchnk )
    enddo

  end subroutine lightning_no_prod

endmodule mo_lightning
