!-------------------------------------------------------------------------------
! This module uses the Lean solar irradiance data to provide a solar cycle
! scaling factor used in heating rate calculations 
!-------------------------------------------------------------------------------
module rad_solar_var

  use shr_kind_mod , only : r8 => shr_kind_r8
  use solar_data,    only : sol_irrad, we, nbins, sol_tsi, has_spectrum
  use abortutils,    only : endrun

  implicit none
  save

  private
  public :: rad_solar_var_init
  public :: get_variability

  real(r8), allocatable :: ref_irrad(:)
  real(r8), allocatable :: irrad(:)

  real(r8), allocatable :: radbinmax(:)
  real(r8), allocatable :: radbinmin(:)
  integer :: nradbins
  real(r8) :: ref_tsi

contains

!-------------------------------------------------------------------------------
!-------------------------------------------------------------------------------
  subroutine rad_solar_var_init( )
    use radconstants,  only : get_number_sw_bands
    use radconstants,  only : get_sw_spectral_boundaries
    use radconstants,  only : get_ref_solar_band_irrad
    use radconstants,  only : get_ref_total_solar_irrad

    integer :: i
    integer :: ierr
    integer :: yr, mon, tod

    call get_number_sw_bands(nradbins)

    if ( has_spectrum ) then
       allocate (radbinmax(nradbins),stat=ierr)
       if (ierr /= 0) then
          call endrun('init_solvar_data: Error allocating space for radbinmax')
       end if

       allocate (radbinmin(nradbins),stat=ierr)
       if (ierr /= 0) then
          call endrun('init_solvar_data: Error allocating space for radbinmin')
       end if

       allocate (ref_irrad(nradbins), stat=ierr)
       if (ierr /= 0) then
          call endrun('init_solvar_data: Error allocating space for ref_irrad')
       end if

       allocate (irrad(nradbins), stat=ierr)
       if (ierr /= 0) then
          call endrun('init_solvar_data: Error allocating space for irrad')
       end if

       call get_sw_spectral_boundaries(radbinmin, radbinmax, 'nm')

       call get_ref_solar_band_irrad( ref_irrad )
    else
       call get_ref_total_solar_irrad( ref_tsi )
    endif

  endsubroutine rad_solar_var_init

!-------------------------------------------------------------------------------
!-------------------------------------------------------------------------------
  subroutine get_variability( sfac )

    real(r8), intent(out) :: sfac(nradbins)       ! scaling factors for CAM heating

    integer :: yr, mon, day, tod

    if ( has_spectrum ) then

      call integrate_spectrum( nbins, nradbins, we, radbinmin, radbinmax, sol_irrad, irrad)

      sfac(:nradbins) = irrad(:nradbins)/ref_irrad(:nradbins)

    else
       sfac(:nradbins) = sol_tsi/ref_tsi
    endif

  endsubroutine get_variability

!-------------------------------------------------------------------------------
! private method.........
!-------------------------------------------------------------------------------

  subroutine integrate_spectrum( nsrc, ntrg, src_x, min_trg, max_trg, src, trg )

    use mo_util, only : rebin

    implicit none

    !---------------------------------------------------------------
    !	... dummy arguments
    !---------------------------------------------------------------
    integer,  intent(in)  :: nsrc                  ! dimension source array
    integer,  intent(in)  :: ntrg                  ! dimension target array
    real(r8), intent(in)  :: src_x(nsrc+1)         ! source coordinates
    real(r8), intent(in)  :: max_trg(ntrg)         ! target coordinates
    real(r8), intent(in)  :: min_trg(ntrg)         ! target coordinates
    real(r8), intent(in)  :: src(nsrc)             ! source array
    real(r8), intent(out) :: trg(ntrg)             ! target array
 
    !---------------------------------------------------------------
    !	... local variables
    !---------------------------------------------------------------
    real(r8) :: trg_x(2), targ(1)         ! target coordinates
    integer  :: i

    do i = 1, ntrg

       trg_x(1) = min_trg(i)
       trg_x(2) = max_trg(i)

       call rebin( nsrc, 1, src_x, trg_x, src(1:nsrc), targ(:) )
       ! W/m2/nm --> W/m2
       trg( i ) = targ(1)*(trg_x(2)-trg_x(1))

    enddo


  end subroutine integrate_spectrum

endmodule rad_solar_var
