#undef DEBUG

module stratiform

!---------------------------------------------------------------------------------
! Purpose:
!
! Provides the CAM interface to the prognostic cloud water and ice parameterization
!
!---------------------------------------------------------------------------------

  use shr_kind_mod,  only: r8=>shr_kind_r8
  use ppgrid,        only: pcols, pver, pverp
  use physconst,     only: gravit, latvap, latice
  use abortutils,    only: endrun
  use chemistry,     only: chem_is
  use phys_control,  only: phys_getopts

  use perf_mod
  use cam_logfile,   only: iulog

  implicit none
  private
  save

  public :: stratiform_register, stratiform_init_cnst, stratiform_implements_cnst
  public :: stratiform_init
  public :: stratiform_tend

! Private module data

  integer, parameter :: ncnstmax=4                      ! number of constituents
  character(len=8), dimension(ncnstmax), parameter :: & ! constituent names
     cnst_names = (/'CLDLIQ', 'CLDICE','NUMLIQ','NUMICE'/)
  integer :: ncnst 					! number of constituents (can vary)

  logical :: use_shfrc  !local copy of flag from convect_shallow_use_shfrc

  character(len=16) :: microp_scheme  ! microphysics scheme

  integer :: &
       cldo_idx,     &! old cld index in physics buffer
       kvh_idx,      &! kvh index in physics buffer
       ixcldliq,     &! cloud liquid amount index
       ixcldice,     &! cloud ice amount index
       ixnumliq,     &! cloud liquid number index
       ixnumice,     &! cloud ice water index
       qcwat_idx,    &! qcwat index in physics buffer
       lcwat_idx,    &! lcwat index in physics buffer
       tcwat_idx,    &! tcwat index in physics buffer
       cld_idx,      &! cld index in physics buffer
       concld_idx     ! concld index in physics buffer

contains


!===============================================================================

  subroutine stratiform_register
!----------------------------------------------------------------------
!
! Register the constituents (cloud liquid and cloud ice) and the fields
! in the physics buffer.
! 
!-----------------------------------------------------------------------
    use constituents, only: cnst_add, pcnst
    use physconst,    only: mwdry, cpair
    use phys_buffer,  only: pbuf_times, pbuf_add

    integer idx
!-----------------------------------------------------------------------

    call phys_getopts(microp_scheme_out=microp_scheme)

    ncnst = 2

    ! Register cloud water and determine index.
    call cnst_add(cnst_names(1), mwdry, cpair, 0._r8, ixcldliq, &
       longname='Grid box averaged cloud liquid amount')
    call cnst_add(cnst_names(2), mwdry, cpair, 0._r8, ixcldice, &
       longname='Grid box averaged cloud ice amount')

    ! Request physics buffer space for fields that persist across timesteps.
    call pbuf_add('QCWAT', 'global', 1,pver,pbuf_times, qcwat_idx)
    call pbuf_add('LCWAT', 'global', 1,pver,pbuf_times, lcwat_idx)
    call pbuf_add('TCWAT', 'global', 1,pver,pbuf_times, tcwat_idx)
    call pbuf_add('CLD',   'global', 1,pver,pbuf_times, cld_idx)
    call pbuf_add('CONCLD','global', 1,pver,pbuf_times, concld_idx)
    call pbuf_add('CLDO',  'global', 1,pver,pbuf_times, cldo_idx)

    call pbuf_add('QINI' , 'physpkg', 1,pver,  pcnst, idx)
    call pbuf_add('TINI' , 'physpkg', 1,pver,      1, idx)

    call pbuf_add('QME' , 'physpkg', 1,pver,      1, idx)
    call pbuf_add('PRAIN' , 'physpkg', 1,pver,      1, idx)
    call pbuf_add('NEVAPR' , 'physpkg', 1,pver,      1, idx)

    call pbuf_add('REI',    'physpkg', 1, pver, 1, idx)
    call pbuf_add('REL',    'physpkg', 1, pver, 1, idx)
    call pbuf_add('REL_FN', 'physpkg', 1, pver, 1, idx)  ! REL at fixed number for indirect rad forcing

    call pbuf_add('DEI',    'physpkg', 1, pver, 1, idx)       !mitchell ice effective diameter for radiation
    call pbuf_add('MU',     'physpkg', 1, pver, 1, idx)       !size distribution shape parameter for radiation
    call pbuf_add('LAMBDAC','physpkg', 1, pver, 1, idx)       !size distribution shape parameter for radiation
    call pbuf_add('ICIWP',  'physpkg', 1, pver, 1, idx)       !in cloud ice water path for radiation
    call pbuf_add('ICLWP',  'physpkg', 1, pver, 1, idx)       !in cloud liquid water path for radiation

  end subroutine stratiform_register

!===============================================================================

  function stratiform_implements_cnst(name)
!----------------------------------------------------------------------- 
! 
! Purpose: return true if specified constituent is implemented by this package
! 
! Author: B. Eaton
! 
!-----------------------------------------------------------------------
     implicit none
!-----------------------------Arguments---------------------------------

     character(len=*), intent(in) :: name   ! constituent name
     logical :: stratiform_implements_cnst     ! return value
!---------------------------Local workspace-----------------------------
     integer :: m
!-----------------------------------------------------------------------

     stratiform_implements_cnst = .false.

     do m = 1, ncnst
        if (name == cnst_names(m)) then
           stratiform_implements_cnst = .true.
           return
        end if
     end do
  end function stratiform_implements_cnst

!===============================================================================
  subroutine stratiform_init_cnst(name, q, gcid)
!----------------------------------------------------------------------- 
! 
! Initialize the cloud water mixing ratios (liquid and ice), if they are
! not read from the initial file
! 
!-----------------------------------------------------------------------
    implicit none

! Arguments
    character(len=*), intent(in)  :: name                ! constituent name
    real(r8),         intent(out) :: q(:,:)   ! mass mixing ratio  (gcol, plev)
    integer,          intent(in)  :: gcid(:)  ! global column id
!-----------------------------------------------------------------------

    if ( name == 'CLDLIQ' ) then
       q = 0.0_r8
       return
    else if ( name == 'CLDICE' ) then
       q = 0.0_r8
       return
    else if ( name == 'NUMLIQ' ) then
       q = 0.0_r8
       return
    else if ( name == 'NUMICE' ) then
       q = 0.0_r8
       return
    end if

  end subroutine stratiform_init_cnst

!===============================================================================
  subroutine stratiform_init
!----------------------------------------------------------------------
!
! Initialize the cloud water parameterization
! 
!-----------------------------------------------------------------------
    use cldwat,        only: inimc
    use constituents,  only: cnst_get_ind, cnst_name, cnst_longname, sflxnam, apcnst
    use cam_history,   only: addfld, add_default, phys_decomp
    use physconst,     only: tmelt, rh2o, rhodair
    use convect_shallow, only : convect_shallow_use_shfrc
    use dycore,       only: dycore_is

    integer :: m, mm
    character(len=16) :: deep_scheme
!-----------------------------------------------------------------------

    call phys_getopts(deep_scheme_out=deep_scheme)

    ! initialization routine for prognostic cloud water
    call inimc (tmelt, rhodair/1000.0_r8, gravit, rh2o, deep_scheme )

    ! find out whether shfrc from convect_shallow will be used in cldfrc
    if ( convect_shallow_use_shfrc() ) then
       use_shfrc = .true.
    else 
       use_shfrc = .false.
    endif

    ! register history variables
    do m = 1,ncnst
       call cnst_get_ind(cnst_names(m), mm)
       call addfld (cnst_name(mm), 'kg/kg   ', pver, 'A', cnst_longname(mm), phys_decomp)
       call addfld (sflxnam(mm),   'kg/m2/s ',    1, 'A', trim(cnst_name(mm))//' surface flux', phys_decomp)
       call add_default (cnst_name(mm), 1, ' ')
       call add_default (sflxnam(mm),   1, ' ')
       if ( dycore_is('LR') )then
          call addfld (apcnst(mm), 'kg/kg   ', pver, 'A', cnst_longname(mm)//' after physics' , phys_decomp)
       end if
    end do

    call addfld ('FWAUT   ','fraction',pver, 'A','Relative importance of liquid autoconversion' ,phys_decomp)
    call addfld ('FSAUT   ','fraction',pver, 'A','Relative importance of ice autoconversion'    ,phys_decomp)
    call addfld ('FRACW   ','fraction',pver, 'A','Relative  importance of rain accreting liquid',phys_decomp)
    call addfld ('FSACW   ','fraction',pver, 'A','Relative  importance of snow accreting liquid',phys_decomp)
    call addfld ('FSACI   ','fraction',pver, 'A','Relative  importance of snow accreting ice'   ,phys_decomp)
    call addfld ('CME     ','kg/kg/s ',pver, 'A','Rate of cond-evap within the cloud'           ,phys_decomp)
    call addfld ('CMEICE  ','kg/kg/s ',pver, 'A','Rate of cond-evap of ice within the cloud'    ,phys_decomp)
    call addfld ('CMELIQ  ','kg/kg/s ',pver, 'A','Rate of cond-evap of liq within the cloud'    ,phys_decomp)
    call addfld ('ICE2PR  ','kg/kg/s ',pver, 'A','Rate of conversion of ice to precip'          ,phys_decomp)
    call addfld ('LIQ2PR  ','kg/kg/s ',pver, 'A','Rate of conversion of liq to precip'          ,phys_decomp)
    call addfld ('ZMDLF   ','kg/kg/s ',pver, 'A','Detrained liquid water from ZM convection'    ,phys_decomp)
    call addfld ('PRODPREC','kg/kg/s ',pver, 'A','Rate of conversion of condensate to precip'   ,phys_decomp)
    call addfld ('EVAPPREC','kg/kg/s ',pver, 'A','Rate of evaporation of falling precip'        ,phys_decomp)
    call addfld ('EVAPSNOW','kg/kg/s ',pver, 'A','Rate of evaporation of falling snow'          ,phys_decomp)
    call addfld ('HPROGCLD','W/kg'    ,pver, 'A','Heating from prognostic clouds'               ,phys_decomp)
    call addfld ('HCME    ','W/kg'    ,pver, 'A','Heating from cond-evap within the cloud'      ,phys_decomp)
    call addfld ('HEVAP   ','W/kg'    ,pver, 'A','Heating from evaporation of falling precip'   ,phys_decomp)
    call addfld ('HFREEZ  ','W/kg'    ,pver, 'A','Heating rate due to freezing of precip'       ,phys_decomp)
    call addfld ('HMELT   ','W/kg'    ,pver, 'A','Heating from snow melt'                       ,phys_decomp)
    call addfld ('HREPART ','W/kg'    ,pver, 'A','Heating from cloud ice/liquid repartitioning' ,phys_decomp)
    call addfld ('REPARTICE','kg/kg/s',pver, 'A','Cloud ice tendency from cloud ice/liquid repartitioning' ,phys_decomp)
    call addfld ('REPARTLIQ','kg/kg/s',pver, 'A','Cloud liq tendency from cloud ice/liquid repartitioning' ,phys_decomp)
    call addfld ('FICE    ','fraction',pver, 'A','Fractional ice content within cloud'          ,phys_decomp)
    call addfld ('ICWMR   ','kg/kg   ',pver, 'A','Prognostic in-cloud water mixing ratio'       ,phys_decomp)
    call addfld ('ICIMR   ','kg/kg   ',pver, 'A','Prognostic in-cloud ice mixing ratio'         ,phys_decomp)
    call addfld ('PCSNOW  ','m/s'     ,1   , 'A','Snow fall from prognostic clouds'             ,phys_decomp)
    call addfld ('DQSED   ','kg/kg/s' ,pver, 'A','Water vapor tendency from cloud sedimentation',phys_decomp)
    call addfld ('DLSED   ','kg/kg/s' ,pver, 'A','Cloud liquid tendency from sedimentation'     ,phys_decomp)
    call addfld ('DISED   ','kg/kg/s' ,pver, 'A','Cloud ice tendency from sedimentation'        ,phys_decomp)
    call addfld ('HSED    ','W/kg'    ,pver, 'A','Heating from cloud sediment evaporation'      ,phys_decomp)
    call addfld ('SNOWSED ','m/s'     ,1   , 'A','Snow from cloud ice sedimentation'            ,phys_decomp)
    call addfld ('RAINSED ','m/s'     ,1   , 'A','Rain from cloud liquid sedimentation'         ,phys_decomp)
    call addfld ('PRECSED ','m/s'     ,1   , 'A','Precipitation from cloud sedimentation'       ,phys_decomp)

    call add_default ('FICE    ', 1, ' ')

    call addfld ('CNVCLD  ','fraction',1,    'A','Vertically integrated convective cloud amount',phys_decomp)
    call addfld ('CLDST   ','fraction',pver, 'A','Stratus cloud fraction',phys_decomp)
    call addfld ('CONCLD  ','fraction',pver, 'A','Convective cloud cover',phys_decomp)
    call add_default ('CONCLD  ', 1, ' ')

    ! history variables for new microphysics
    call addfld ('MPDT   ','K/s     ',pver, 'A','T tendency - Morrison microphysics',phys_decomp)
    call addfld ('MPDQ   ','kg/kg/s     ',pver, 'A','Q tendency - Morrison microphysics',phys_decomp)
    call addfld ('LIQCLDF','fraction',pver, 'A','Stratus Liquid cloud fraction',phys_decomp)
    call addfld ('ICECLDF','fraction',pver, 'A','Stratus ICE cloud fraction',phys_decomp)
    call addfld ('IWC   ','kg/m3   ',pver, 'A','Grid box average ice water content'         ,phys_decomp)
    call addfld ('LWC   ','kg/m3   ',pver, 'A','Grid box average liquid water content'         ,phys_decomp)
    call addfld ('ICWNC   ','m-3     ',pver, 'A','Prognostic in-cloud water number conc'       ,phys_decomp)
    call addfld ('ICINC   ','m-3     ',pver, 'A','Prognostic in-cloud ice number conc'         ,phys_decomp)
    call addfld ('EFFLIQ  ','Micron  ',pver, 'A','Prognostic droplet effective radius'       ,phys_decomp)
    call addfld ('EFFLIQ_IND','Micron  ',pver, 'A','Prognostic droplet effective radius (indirect effect)'       ,phys_decomp)
    call addfld ('EFFICE  ','Micron  ',pver, 'A','Prognostic ice effeictive radius'         ,phys_decomp)
    call addfld ('WSUB  ','m/s     ',pver, 'A','Diagnostic sub-grid vertical velocity'         ,phys_decomp)
    call addfld ('CDNUMC  ','#/m2    ',1, 'A','Vertically-integrated droplet concentration'         ,phys_decomp)
    call addfld ('WTKE    ','m/s     ',pver, 'A','Standard deviation of updraft velocity',phys_decomp)
    call addfld ('LCLOUD  ','        ',pver, 'A','Liquid cloud fraction',phys_decomp)
    call addfld ('NDROPMIX','#/kg/s  ',pver, 'A','Droplet number mixing',phys_decomp)
    call addfld ('NDROPSRC','#/kg/s  ',pver, 'A','Droplet number source',phys_decomp)
    call addfld ('NLSED   ','#/kg/s  ',pver, 'A','Droplet number tendency from sedimentation',phys_decomp)
    call addfld ('NDROPSNK','#/kg/s  ',pver, 'A','Droplet number loss by microphysics',phys_decomp)
    call addfld ('NDROPCOL','#/m2    ',1,    'A','Column droplet number',phys_decomp)
    call add_default ('WTKE    ', 1, ' ')
    call add_default ('LCLOUD  ', 1, ' ')
    call add_default ('NDROPSRC', 1, ' ')
    call add_default ('NDROPMIX', 1, ' ')
    call add_default ('NDROPSNK', 1, ' ')
    call add_default ('NDROPCOL', 1, ' ')

    ! averaging for cloud particle number and size
    call addfld ('AWNC   ','m-3     ',pver, 'A','Average cloud water number conc'         ,phys_decomp)
    call addfld ('AWNI   ','m-3     ',pver, 'A','Average cloud ice number conc'         ,phys_decomp)
    call addfld ('AREL  ','Micron  ',pver, 'A','Average droplet effective radius'       ,phys_decomp)
    call addfld ('AREI  ','Micron  ',pver, 'A','Average ice effective radius'       ,phys_decomp)
    ! frequency arrays for above
    call addfld ('FREQL  ','fraction  ',pver, 'A','Fractional occurance of liquid'       ,phys_decomp)
    call addfld ('FREQI  ','fraction  ',pver, 'A','Fractional occurance of ice'       ,phys_decomp)

    ! Average cloud top particle size and number (liq, ice) and frequency
    call addfld ('ACTREL  ','Micron  ',1, 'A','Average Cloud Top droplet effective radius'       ,phys_decomp)
    call addfld ('ACTREI  ','Micron  ',1, 'A','Average Cloud Top ice effective radius'       ,phys_decomp)
    call addfld ('ACTNL  ','Micron  ',1, 'A','Average Cloud Top droplet number'       ,phys_decomp)
    call addfld ('ACTNI  ','Micron  ',1, 'A','Average Cloud Top ice number'       ,phys_decomp)

    call addfld ('FCTL  ','fraction  ',1, 'A','Fractional occurance of cloud top liquid'       ,phys_decomp)
    call addfld ('FCTI  ','fraction  ',1, 'A','Fractional occurance of cloud top ice'       ,phys_decomp)

  end subroutine stratiform_init

!===============================================================================

  subroutine stratiform_tend(state, ptend_all, dtime, &
       icefrac, landfrac, ocnfrac, landm, snowh, &
       dlf, rliq ,   &  
       cmfmc,   cmfmc2,   &
       ts,      sst,      zdu,  &
       prec_str,snow_str, prec_sed, snow_sed, prec_pcw, snow_pcw, & 
       pbuf )

  !----------------------------------------------------------------------- 
  ! 
  ! Purpose: 
  !
  ! Interface to sedimentation, detrain, cloud fraction and 
  !        microphysics subroutines
  !
  ! Author: D.B. Coleman
  ! Date: Apr 2004
  ! 
  !-----------------------------------------------------------------------

  use shr_kind_mod,     only: r8 => shr_kind_r8
  use ppgrid
  use cloud_fraction,   only: cldfrc
  use physics_types,    only: physics_state, physics_ptend, physics_tend
  use physics_types,    only: physics_ptend_init, physics_update, physics_tend_init
  use physics_types,    only: physics_ptend_sum,  physics_state_copy
  use cam_history,      only: outfld
  use phys_buffer,      only: pbuf_size_max, pbuf_fld, pbuf_old_tim_idx, pbuf_get_fld_idx
  use constituents,     only: cnst_get_ind, pcnst
  use pkg_cld_sediment, only: cld_sediment_vel, cld_sediment_tend
  use cldwat,           only: pcond, cldwat_fice
  use rad_constituents, only: rad_cnst_get_clim_info, rad_cnst_get_clim_aer
  use time_manager,     only: is_first_step
  use pkg_cldoptics,    only: cldefr
  use phys_buffer,  only: pbuf_times
  use time_manager,  only: is_first_step

!
! Parameters
!
  real(r8) pnot                  ! reference pressure
  parameter (pnot = 1.e5_r8)


!
! Input arguments
!

  type(physics_state), intent(in)    :: state   ! state variables
  type(physics_ptend), intent(out) :: ptend_all   ! package tendencies
  type(pbuf_fld),      intent(inout), dimension(pbuf_size_max) :: pbuf

  real(r8), intent(in)  :: dtime                ! timestep
  real(r8), intent(in)  :: icefrac (pcols)      ! sea ice fraction (fraction)
  real(r8), intent(in)  :: landfrac(pcols)      ! land fraction (fraction)
  real(r8), intent(in)  :: ocnfrac (pcols)      ! ocean fraction (fraction)
  real(r8), intent(in) :: landm(pcols)          ! land fraction ramped over water
  real(r8), intent(in) :: snowh(pcols)         ! Snow depth over land, water equivalent (m)

  real(r8), intent(in) :: dlf(pcols,pver)       ! detrained water from ZM
  real(r8), intent(in) :: rliq(pcols)                    ! vertical integral of liquid not yet in q(ixcldliq)
  real(r8), intent(in) :: cmfmc(pcols,pverp)    ! convective mass flux--m sub c
  real(r8), intent(in) :: cmfmc2(pcols,pverp)   ! shallow convective mass flux--m sub c

  real(r8), intent(in) :: ts(pcols)             ! surface temperature
  real(r8), intent(in) :: sst(pcols)            ! sea surface temperature
  real(r8), intent(in) :: zdu(pcols,pver)       ! detrainment rate from deep convection

  real(r8), intent(out)  :: prec_str(pcols)  ! [Total] sfc flux of precip from stratiform (m/s) 
  real(r8), intent(out)  :: snow_str(pcols)  ! [Total] sfc flux of snow from stratiform   (m/s)
  real(r8), intent(out)  :: prec_sed(pcols)  ! surface flux of total cloud water from sedimentation
  real(r8), intent(out)  :: snow_sed(pcols) ! surface flux of cloud ice from sedimentation
  real(r8), intent(out)  :: prec_pcw(pcols) ! sfc flux of precip from microphysics(m/s)
  real(r8), intent(out)  :: snow_pcw(pcols) ! sfc flux of snow from microphysics (m/s)

  !
  ! Local variables
  !

  type(physics_state)  :: state1   ! local copy of the state variable
  type(physics_tend ) :: tend          ! Physics tendencies (empty, needed for physics_update call)
  type(physics_ptend)  :: ptend_loc   ! package tendencies


  integer i,k,m
  integer :: lchnk                  ! chunk identifier
  integer :: ncol                   ! number of atmospheric columns

  ! physics buffer fields
  integer itim, ifld
  real(r8), pointer, dimension(:,:) :: qcwat   ! cloud water old q
  real(r8), pointer, dimension(:,:) :: tcwat   !cloud water old temperature
  real(r8), pointer, dimension(:,:) :: lcwat   ! cloud liquid water old q
  real(r8), pointer, dimension(:,:) ::  cld    ! cloud fraction
  real(r8), pointer, dimension(:,:) ::  concld ! convective cloud fraction
   real(r8), pointer, dimension(:,:) :: qme
   real(r8), pointer, dimension(:,:) :: prain   ! total precipitation (rain + snow)
   real(r8), pointer, dimension(:,:) :: nevapr  ! evaporation of total precipitation (rain + snow)
   real(r8), pointer, dimension(:,:) :: rel     ! liquid effective drop radius (microns)
   real(r8), pointer, dimension(:,:) :: rei     ! ice effective drop size (microns)
   real(r8), pointer, dimension(:,:) ::  cldo    ! old cloud fraction
   real(r8), pointer, dimension(:,:) :: kkvh   ! heat flux for cldwat
   real(r8) :: shfrc(pcols,pver)        !  cloud fraction from shallow convection scheme
   real(r8), pointer, dimension(:,:) :: rel_fn     ! ice effective drop size at fixed number (indirect effect) (microns)

   ! physics buffer fields for radiation
   real(r8), pointer, dimension(:,:) :: dei     ! ice effective diameter (meters)
   real(r8), pointer, dimension(:,:) :: mu      ! size distribution shape parameter for radiation
   real(r8), pointer, dimension(:,:) :: lambdac ! size distribution slope parameter for radiation
   real(r8), pointer, dimension(:,:) :: iciwp   ! in-cloud ice water path for radiation
   real(r8), pointer, dimension(:,:) :: iclwp   ! in-cloud liquid water path for radiation

  ! local variables for stratiform_sediment
  real(r8) :: rain(pcols)                       ! surface flux of cloud liquid
  real(r8) :: pvliq(pcols,pver+1)               ! vertical velocity of cloud liquid drops (Pa/s)
  real(r8) :: pvice(pcols,pver+1)               ! vertical velocity of cloud ice particles (Pa/s)

  ! local variables for cldfrc
  real(r8)  cldst(pcols,pver)     ! cloud fraction
  real(r8)  clc(pcols)            ! column convective cloud amount
  real(r8) rhdfda(pcols,pver)    ! d_RH/d_cloud_fraction    ====wlin
  real(r8) rhu00(pcols,pver)     ! RH limit, U00             ====wlin
  real(r8) relhum(pcols,pver)         ! RH, output to determine drh/da
  real(r8) rhu002(pcols,pver)         ! same as rhu00 but for perturbed rh 
  real(r8) cld2(pcols,pver)           ! same as cld but for perturbed rh
  real(r8) concld2(pcols,pver)        ! same as concld but for perturbed rh 
  real(r8) cldst2(pcols,pver)         ! same as cldst but for perturbed rh 
  real(r8) relhum2(pcols,pver)        ! RH after  perturbation            
  real(r8) :: pmid(pcols,pver)      ! midpoint pressures
  real(r8) :: t(pcols,pver)         ! temperature
  real(r8) :: q(pcols,pver)         ! specific humidity
  real(r8) :: omga(pcols,pver)      ! vertical pressure velocity
  real(r8) :: phis(pcols)           ! surface geopotential
  real(r8) :: pdel(pcols,pver)      ! pressure depth of layer
  real(r8) :: ps(pcols)             ! surface pressure
  real(r8) :: icecldf(pcols,pver)    ! ice cloud fraction
  real(r8) :: liqcldf(pcols,pver)    ! liquid cloud fraction (combined into cloud)

  ! local variables for microphysics
  real(r8) :: rdtime                          ! 1./dtime
  real(r8) :: qtend (pcols,pver)            ! moisture tendencies
  real(r8) :: ttend (pcols,pver)            ! temperature tendencies
  real(r8) :: ltend (pcols,pver)            ! cloud liquid water tendencies
  real(r8) :: evapheat(pcols,pver)          ! heating rate due to evaporation of precip
  real(r8) :: evapsnow(pcols,pver)          ! local evaporation of snow
  real(r8) :: prfzheat(pcols,pver)          ! heating rate due to freezing of precip (W/kg)
  real(r8) :: meltheat(pcols,pver)          ! heating rate due to phase change of precip
  real(r8) :: cmeheat (pcols,pver)          ! heating rate due to phase change of precip
  real(r8) :: prodsnow(pcols,pver)          ! local production of snow
  real(r8) :: totcw   (pcols,pver)          ! total cloud water mixing ratio
  real(r8) :: fice    (pcols,pver)          ! Fractional ice content within cloud
  real(r8) :: fsnow   (pcols,pver)          ! Fractional snow production
  real(r8) :: repartht(pcols,pver)          ! heating rate due to phase repartition of input precip
  real(r8) :: icimr(pcols,pver)             ! in cloud ice mixing ratio
  real(r8) :: icwmr(pcols,pver)             ! in cloud water mixing ratio
  real(r8) fwaut(pcols,pver)              
  real(r8) fsaut(pcols,pver)              
  real(r8) fracw(pcols,pver)              
  real(r8) fsacw(pcols,pver)              
  real(r8) fsaci(pcols,pver)              
  real(r8) cmeice(pcols,pver)   ! Rate of cond-evap of ice within the cloud
  real(r8) cmeliq(pcols,pver)   ! Rate of cond-evap of liq within the cloud
  real(r8) ice2pr(pcols,pver)   ! rate of conversion of ice to precip
  real(r8) liq2pr(pcols,pver)   ! rate of conversion of liquid to precip
  real(r8) liq2snow(pcols,pver)   ! rate of conversion of liquid to snow
  real(r8) temp(pcols)
  real(r8) res(pcols,pver)
  real(r8) droprad ! radius of droplets detrained from cumulus (m)
  real(r8) invdropmass ! inverse of mean droplet mass (#/kg)

  ! variables for morrison microphysics
  real(r8) :: dum1,dum2
  real(r8) :: qc(pcols,pver)
  real(r8) :: qi(pcols,pver)
  real(r8) :: nc(pcols,pver)
  real(r8) :: ni(pcols,pver)
  real(r8) :: icinc(pcols,pver)             ! in cloud ice number conc
  real(r8) :: cdnumc(pcols)             ! vertically-integrated droplet concentration
  real(r8) :: icwnc(pcols,pver)             ! in cloud water number conc
  real(r8) :: iwc(pcols,pver)             ! grid box average ice water content
  real(r8) :: lwc(pcols,pver)             ! grid box average liquid water content  
  real(r8) :: effliq(pcols,pver)             ! in cloud liq eff rad
  real(r8) :: effice(pcols,pver)             ! in cloud ice eff rad
  real(r8) :: effliq_fn(pcols,pver)             ! in cloud liq eff rad at fixed number concentration	
  real(r8) :: wsub(pcols,pver)             ! sub-grid vertical velocity (m/s)

  ! output from mmicro_pcond
  real(r8) :: tlat(pcols,pver)
  real(r8) :: qvlat(pcols,pver)
  real(r8) :: qcten(pcols,pver)
  real(r8) :: qiten(pcols,pver)
  real(r8) :: ncten(pcols,pver)
  real(r8) :: niten(pcols,pver)
  real(r8) :: effc(pcols,pver)
  real(r8) :: effc_fn(pcols,pver)     ! liquid effective radius at fixed number (for indirect calc)
  real(r8) :: effi(pcols,pver)
  real(r8) :: prect(pcols)
  real(r8) :: preci(pcols)

  ! averaging arrays for effective radius and number....
  real(r8) :: efiout(pcols,pver)
  real(r8) :: efcout(pcols,pver)
  real(r8) :: ncout(pcols,pver)
  real(r8) :: niout(pcols,pver)

  real(r8) :: freqi(pcols,pver)
  real(r8) :: freql(pcols,pver)

  ! average cloud top radius & number
  real(r8) :: ctrel(pcols)
  real(r8) :: ctrei(pcols)
  real(r8) :: ctnl(pcols)
  real(r8) :: ctni(pcols)
  real(r8) :: fcti(pcols)
  real(r8) :: fctl(pcols)

  ! gather mass mixing ratio for all aerosols affecting the climate
  integer :: naer_all
  real(r8), pointer :: aermmr1(:,:)
  real(r8), allocatable :: aer_mmr(:,:,:) ! aerosol mass mixing ratio

  !======================================================================
  lchnk = state%lchnk
  ncol  = state%ncol


  call physics_state_copy(state,state1)   ! copy state to local state1.
  call physics_ptend_init(ptend_loc)  ! initialize local ptend type
  call physics_ptend_init(ptend_all)  ! initialize output ptend type
  call physics_tend_init(tend)   ! tend here is just a null place holder


  ! Associate pointers with physics buffer fields
  itim = pbuf_old_tim_idx()
  ifld = pbuf_get_fld_idx('QCWAT')
  qcwat => pbuf(ifld)%fld_ptr(1,1:pcols,1:pver,lchnk,itim)
  ifld = pbuf_get_fld_idx('TCWAT')
  tcwat => pbuf(ifld)%fld_ptr(1,1:pcols,1:pver,lchnk,itim)
  ifld = pbuf_get_fld_idx('LCWAT')
  lcwat => pbuf(ifld)%fld_ptr(1,1:pcols,1:pver,lchnk,itim)
  ifld = pbuf_get_fld_idx('CLD')
  cld => pbuf(ifld)%fld_ptr(1,1:pcols,1:pver,lchnk,itim)
  ifld = pbuf_get_fld_idx('CONCLD')  
  concld => pbuf(ifld)%fld_ptr(1,1:pcols,1:pver,lchnk,itim)
  ifld = pbuf_get_fld_idx('CLDO')
  do m=1,pbuf_times
      cldo => pbuf(ifld)%fld_ptr(1,1:pcols,1:pver,lchnk,m)
  enddo

   ifld = pbuf_get_fld_idx('QME')
   qme  => pbuf(ifld)%fld_ptr(1,1:pcols,1:pver,lchnk, 1)
   ifld = pbuf_get_fld_idx('PRAIN')
   prain  => pbuf(ifld)%fld_ptr(1,1:pcols,1:pver,lchnk, 1)
   ifld = pbuf_get_fld_idx('NEVAPR')
   nevapr  => pbuf(ifld)%fld_ptr(1,1:pcols,1:pver,lchnk, 1)

   ifld = pbuf_get_fld_idx('REL')
   rel  => pbuf(ifld)%fld_ptr(1,1:pcols,1:pver,lchnk, 1)
   ifld = pbuf_get_fld_idx('REI')
   rei  => pbuf(ifld)%fld_ptr(1,1:pcols,1:pver,lchnk, 1)

   ifld = pbuf_get_fld_idx('REL_FN')
   rel_fn  => pbuf(ifld)%fld_ptr(1,1:pcols,1:pver,lchnk, 1)

   ifld = pbuf_get_fld_idx('KVH')
   kkvh => pbuf(ifld)%fld_ptr(1,1:pcols,1:pverp,lchnk, 1)

   ifld = pbuf_get_fld_idx('DEI')
   dei  => pbuf(ifld)%fld_ptr(1,1:pcols,1:pver,lchnk, 1)
   ifld = pbuf_get_fld_idx('MU')
   mu  => pbuf(ifld)%fld_ptr(1,1:pcols,1:pver,lchnk, 1)
   ifld = pbuf_get_fld_idx('LAMBDAC')
   lambdac  => pbuf(ifld)%fld_ptr(1,1:pcols,1:pver,lchnk, 1)
   ifld = pbuf_get_fld_idx('ICIWP')
   iciwp  => pbuf(ifld)%fld_ptr(1,1:pcols,1:pver,lchnk, 1)
   ifld = pbuf_get_fld_idx('ICLWP')
   iclwp  => pbuf(ifld)%fld_ptr(1,1:pcols,1:pver,lchnk, 1)

   ! if first timestep, initialize heatflux....in pbuf at all time levels.
   if (is_first_step()) then
      kkvh(:,:)= 0.0_r8
   endif

   !+++sediment ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

   ! Allow the cloud liquid drops and ice particles to sediment
   ! Occurs before adding convectively detrained cloud water, because the phase of the
   ! of the detrained water is unknown.
   call t_startf('stratiform_sediment')

   ptend_loc%name         = 'pcwsediment'
   ptend_loc%ls           = .TRUE.
   ptend_loc%lq(1)        = .TRUE.
   ptend_loc%lq(ixcldice) = .TRUE.
   ptend_loc%lq(ixcldliq) = .TRUE.

   call cld_sediment_vel (ncol,                                    &
         icefrac, landfrac, ocnfrac, state1%pmid, state1%pdel, state1%t, &
         cld, state1%q(:,:,ixcldliq), state1%q(:,:,ixcldice), pvliq, pvice, landm, snowh)

   call cld_sediment_tend (ncol, dtime ,                                       &
         state1%pint, state1%pmid           , state1%pdel            , state1%t     , &
         cld  ,    state1%q(:,:,ixcldliq), state1%q(:,:,ixcldice) , pvliq, pvice, &
         ptend_loc%q(:,:,ixcldliq), ptend_loc%q(:,:,ixcldice), ptend_loc%q(:,:,1), ptend_loc%s  , &
         rain   , snow_sed   )

   ! convert rain and snow from kg/m2 to m/s
   snow_sed(:ncol) = snow_sed(:ncol)/1000._r8
   rain(:ncol) = rain(:ncol)/1000._r8
   ! compute total precip (m/s)
   prec_sed(:ncol) = rain(:ncol) + snow_sed(:ncol)

   ! record history variables
   lchnk = state1%lchnk
   call outfld('DQSED'   ,ptend_loc%q(:,:,1)       , pcols,lchnk)
   call outfld('DISED'   ,ptend_loc%q(:,:,ixcldice), pcols,lchnk)
   call outfld('DLSED'   ,ptend_loc%q(:,:,ixcldliq), pcols,lchnk)
   call outfld('HSED'    ,ptend_loc%s              , pcols,lchnk)
   call outfld('PRECSED' ,prec_sed                 , pcols,lchnk)
   call outfld('SNOWSED' ,snow_sed                 , pcols,lchnk)
   call outfld('RAINSED' ,rain                 , pcols,lchnk)

   ! add tendency from this process to tend from other processes here
   call physics_ptend_sum(ptend_loc,ptend_all, state)

   ! update physics state type state1 with ptend_loc 
   call physics_update(state1, tend, ptend_loc, dtime)
   call physics_ptend_init(ptend_loc)

   call t_stopf('stratiform_sediment')

   ! accumulate prec and snow
   prec_str(:ncol) = prec_sed(:ncol)
   snow_str(:ncol) = snow_sed(:ncol)

   !++detrain ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

   ! Put the detraining cloud water from convection into the cloud and environment. 

   ptend_loc%name         = 'pcwdetrain'
   ptend_loc%lq(ixcldliq) = .TRUE.

   ! Put all of the detraining cloud water from convection into the large scale cloud.
   ! It all goes in liquid for the moment.
   do k = 1,pver
      do i = 1,state1%ncol
         ptend_loc%q(i,k,ixcldliq) = dlf(i,k)
      end do
   end do

   call outfld('ZMDLF' ,dlf  , pcols,state1%lchnk)

   ! add tendency from this process to tend from other processes here
   call physics_ptend_sum(ptend_loc, ptend_all, state)

   call physics_update(state1, tend, ptend_loc, dtime)
   call physics_ptend_init(ptend_loc)

   ! accumulate prec and snow, reserved liquid has now been used
   prec_str(:ncol) = prec_str(:ncol) - rliq(:ncol)  ! ( snow contribution is zero )

   !++++ cldfrc ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

   ! cloud fraction after transport and convection,
   ! derive the relationship between rh and cld from 
   ! the employed c7loud scheme

   pmid(:ncol,:pver) = state1%pmid(:ncol,:pver)
   t(:ncol,:pver) = state1%t(:ncol,:pver)
   q(:ncol,:pver) = state1%q(:ncol,:pver,1)
   qi(:ncol,:pver) = state1%q(:ncol,:pver,ixcldice)
   omga(:ncol,:pver) = state1%omega(:ncol,:pver)
   pdel(:ncol,:pver) =  state1%pdel(:ncol,:pver)
   ps(:ncol) = state1%pint(:ncol,pverp)
   phis(:ncol) = state1%phis(:ncol)

   if ( use_shfrc ) then
      ifld = pbuf_get_fld_idx('shfrc')
      shfrc(:pcols,:pver) = pbuf(ifld)%fld_ptr(1,1:pcols,1:pver,lchnk,1)
   else 
      shfrc(:,:) = 0._r8
   endif

   call t_startf("cldfrc")
   call cldfrc(lchnk,   ncol,                                &
	pmid,      t,        q,     omga, phis, &
	shfrc   ,use_shfrc, &
        cld,    clc,     pdel,   &
        cmfmc,   cmfmc2,  landfrac,snowh,   concld,  cldst,    &
	ts,      sst, ps,      zdu,     ocnfrac, rhu00, &
        qi, icecldf, liqcldf, &
	relhum,  0  )    

   ! re-calculate cloud with perturbed rh             add call cldfrc  
   call cldfrc(lchnk,   ncol,                                &
	pmid,       t,      q,      omga, phis, &
	shfrc   ,use_shfrc, &
        cld2,   clc,     pdel,   &
        cmfmc,   cmfmc2   ,landfrac,snowh,   concld2, cldst2,   &
	ts,      sst, ps,        zdu,   ocnfrac, rhu002, &
        qi, icecldf, liqcldf, &
	relhum2, 1  )              
   call t_stopf("cldfrc")

   ! cldfrc does not define layer cloud for model layer at k=1
   ! so set rhu00(k=1)=2.0 to not calculate cme for this layer

   rhu00(:ncol,1) = 2.0_r8 

   ! Add following to estimate rhdfda                       

   do k=1,pver
      do i=1,ncol
         if(relhum(i,k) < rhu00(i,k) ) then
            rhdfda(i,k)=0.0_r8
         else if (relhum(i,k) >= 1.0_r8 ) then
            rhdfda(i,k)=0.0_r8
         else
            !under certain circumstances, rh+ cause cld not to changed
            !when at an upper limit, or w/ strong subsidence
            !need to further check whether this if-block is necessary

            if((cld2(i,k) - cld(i,k) ) < 1.e-4_r8 ) then
               rhdfda(i,k) = 0.01_r8*relhum(i,k)*1.e+4_r8   !instead of 0.0
            else
               rhdfda(i,k)=0.01_r8*relhum(i,k)/(cld2(i,k)-cld(i,k))
            endif
         endif
      enddo
   enddo

   !+ mp +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

   ! cloud water and ice parameterizations
   call t_startf('stratiform_microphys')

   ! Initialize chunk id and size
   lchnk = state1%lchnk
   ncol  = state1%ncol

   rdtime = 1._r8/dtime

   ! Define fractional amount of cloud condensate in ice phase
   call cldwat_fice(ncol, state1%t, fice, fsnow)

   ! compute total cloud water
   totcw(:ncol,:pver) = state1%q(:ncol,:pver,ixcldice) + state1%q(:ncol,:pver,ixcldliq)

   ! save input cloud ice
   repartht(:ncol,:pver) = state1%q(:ncol,:pver,ixcldice)

   ptend_loc%q(:ncol,:pver,ixcldice) = &
         ( totcw(:ncol,:pver) * fice(:ncol,:pver)            - state1%q(:ncol,:pver,ixcldice) ) * rdtime
   ptend_loc%q(:ncol,:pver,ixcldliq) = &
         ( totcw(:ncol,:pver) * (1.0_r8 - fice(:ncol,:pver)) - state1%q(:ncol,:pver,ixcldliq) ) * rdtime

   call outfld('REPARTICE'   ,ptend_loc%q(:,:,ixcldice), pcols,lchnk)
   call outfld('REPARTLIQ'   ,ptend_loc%q(:,:,ixcldliq), pcols,lchnk)

   ! Set output flags
   ptend_loc%name         = 'cldwat-repartition'
   ptend_loc%lq(ixcldice) = .true.
   ptend_loc%lq(ixcldliq) = .true.

   ! add tendency from this process to tend from other processes here
   call physics_ptend_sum(ptend_loc,ptend_all, state)

   ! update state for use below
   call physics_update (state1, tend, ptend_loc, dtime)
   call physics_ptend_init(ptend_loc)

   ! Set output flags for final cldwat update
   ptend_loc%name         = 'cldwat'
   ptend_loc%ls           = .true.
   ptend_loc%lq(1)        = .true.
   ptend_loc%lq(ixcldice) = .true.
   ptend_loc%lq(ixcldliq) = .true.

   ! Determine heating from change in cloud ice
   repartht(:ncol,:pver) = latice/dtime * (state1%q(:ncol,:pver,ixcldice) - repartht(:ncol,:pver))

   ! calculate the tendencies for moisture, temperature and cloud fraction
   qtend(:ncol,:pver) = (state1%q(:ncol,:pver,1) - qcwat(:ncol,:pver))*rdtime
   ttend(:ncol,:pver) = (state1%t(:ncol,:pver)   - tcwat(:ncol,:pver))*rdtime
   ltend(:ncol,:pver) = (totcw  (:ncol,:pver)   - lcwat(:ncol,:pver))*rdtime

   ! call microphysics package to calculate tendencies
   call t_startf('pcond')
   call pcond (lchnk,   ncol, &
         state1%t  , ttend      , state1%q(1,1,1), qtend   , state1%omega, &
         totcw    , state1%pmid , state1%pdel , cld       , fice       , fsnow, &
         qme      , prain   , prodsnow, nevapr   , evapsnow   , evapheat   , prfzheat, &
         meltheat , prec_pcw     , snow_pcw       , dtime         , fwaut      , &
         fsaut    , fracw      , fsacw      , fsaci      , ltend      , &
         rhdfda   , rhu00      , icefrac    , state1%zi   , ice2pr, liq2pr, liq2snow, snowh)
   call t_stopf('pcond')

   ! make it interactive
   do i = 1,ncol
      do k = 1,pver
         ptend_loc%s(i,k)          = qme(i,k) * (latvap + latice*fice(i,k)) &
               + evapheat(i,k) + prfzheat(i,k) + meltheat(i,k) + repartht(i,k)
            
         ptend_loc%q(i,k,1)        =-qme(i,k) + nevapr(i,k)

         ptend_loc%q(i,k,ixcldice) =qme(i,k)*fice(i,k)      - ice2pr(i,k)
         ptend_loc%q(i,k,ixcldliq) =qme(i,k)*(1._r8-fice(i,k)) - liq2pr(i,k)
      end do
   end do

   ! convert precipitation from kg/m2 to m/s
   snow_pcw  (:ncol) = snow_pcw  (:ncol)/1000._r8
   prec_pcw(:ncol) = prec_pcw(:ncol)/1000._r8

   do i = 1,ncol
      do k = 1,pver
         cmeheat(i,k) = qme(i,k) * (latvap + latice*fice(i,k))
         cmeice (i,k) = qme(i,k) * fice(i,k)
         cmeliq (i,k) = qme(i,k) * (1._r8 - fice(i,k))
      end do
   end do
  
   ! record history variables
   call outfld('FWAUT',fwaut,    pcols,lchnk)
   call outfld('FSAUT',fsaut,    pcols,lchnk)
   call outfld('FRACW',fracw,    pcols,lchnk)
   call outfld('FSACW',fsacw,    pcols,lchnk)
   call outfld('FSACI',fsaci,    pcols,lchnk)

   call outfld('PCSNOW'  ,snow_pcw    , pcols,lchnk)
   call outfld('FICE'    ,fice    , pcols,lchnk)
   call outfld('CMEICE'  ,cmeice  , pcols,lchnk)
   call outfld('CMELIQ'  ,cmeliq  , pcols,lchnk)
   call outfld('ICE2PR'  ,ice2pr  , pcols,lchnk)
   call outfld('LIQ2PR'  ,liq2pr  , pcols,lchnk)
   call outfld('HPROGCLD',ptend_loc%s , pcols,lchnk)
   call outfld('HEVAP   ',evapheat, pcols,lchnk)
   call outfld('HMELT'   ,meltheat, pcols,lchnk)
   call outfld('HCME'    ,cmeheat , pcols,lchnk)
   call outfld('HFREEZ'  ,prfzheat, pcols,lchnk)
   call outfld('HREPART' ,repartht, pcols,lchnk)

   ! add tendency from this process to tend from other processes here
   call physics_ptend_sum(ptend_loc,ptend_all, state)

   ! Set the name of the final package tendencies. Note that this
   ! is a special case in physics_update, so a change here must be 
   ! matched there.
   ptend_all%name = 'stratiform'

   ! used below
   call physics_update (state1, tend, ptend_loc, dtime)

   call physics_ptend_init(ptend_loc) 


   ! Jan.22.2006. Added by Sungsu
   ! Re-calculate (update) net cloud fraction for correct and consistent 
   ! use in the subsequent radiation and other physics. 
   ! Jan.3.2008. AG - cloud fraction moved here
   call t_startf("cldfrc")
   qi(:ncol,:pver) = state1%q(:ncol,:pver,ixcldice)
   call cldfrc(lchnk,   ncol,                                &
      state1%pmid, state1%t, state1%q(:,:,1), state1%omega, state1%phis, &
      shfrc   ,use_shfrc, &
      cld,    clc,  state1%pdel,   &
      cmfmc,   cmfmc2,  landfrac,snowh,   concld,  cldst,    &
      ts,      sst, state1%pint(:,pverp),      zdu,     ocnfrac, rhu00, &
      qi, icecldf, liqcldf, &
      relhum,  0  )    
   call t_stopf("cldfrc")

   !??? should cld be outfld'ed here ???

   call outfld('CONCLD  ',concld, pcols,lchnk)
   call outfld('CLDST   ',cldst,  pcols,lchnk)
   call outfld('CNVCLD  ',clc,    pcols,lchnk)
   call outfld('ICECLDF',icecldf, pcols,lchnk)
   call outfld('LIQCLDF',liqcldf, pcols,lchnk)

   ! Compute in cloud ice and liquid mixing ratios
   do k=1,pver
      do i = 1,ncol
         iwc(i,k)=(state1%q(i,k,ixcldice) + dtime*ptend_loc%q(i,k,ixcldice))*state1%pmid(i,k)/(287.15_r8*state1%t(i,k))
         lwc(i,k)=(state1%q(i,k,ixcldliq) + dtime*ptend_loc%q(i,k,ixcldliq))*state1%pmid(i,k)/(287.15_r8*state1%t(i,k))

         icimr(i,k) = (state1%q(i,k,ixcldice) + dtime*ptend_loc%q(i,k,ixcldice)) / max(0.01_r8,cld(i,k))
         icwmr(i,k) = (state1%q(i,k,ixcldliq) + dtime*ptend_loc%q(i,k,ixcldliq)) / max(0.01_r8,cld(i,k))
      end do
   end do

   ! common outfield calls for either microphysics
   call outfld('IWC',iwc,    pcols,lchnk)
   call outfld('LWC',lwc,    pcols,lchnk)
   call outfld('ICIMR',icimr,    pcols,lchnk)
   call outfld('ICWMR',icwmr,    pcols,lchnk)
   call outfld('CME'  ,qme     , pcols,lchnk)
   call outfld('PRODPREC',prain, pcols,lchnk)
   call outfld('EVAPPREC',nevapr, pcols,lchnk)
   call outfld('EVAPSNOW',evapsnow, pcols,lchnk)

   call t_stopf('stratiform_microphys')

   ! accumulate prec and snow
   prec_str(:ncol) = prec_str(:ncol) + prec_pcw(:ncol)
   snow_str(:ncol) = snow_str(:ncol) + snow_pcw(:ncol)

   ! Save off q and t after cloud water
   call cnst_get_ind('CLDLIQ', ixcldliq)
   call cnst_get_ind('CLDICE', ixcldice)

   do k=1,pver
      qcwat(:ncol,k) = state1%q(:ncol,k,1)
      tcwat(:ncol,k) = state1%t(:ncol,k)
      lcwat(:ncol,k) = state1%q(:ncol,k,ixcldice) + state1%q(:ncol,k,ixcldliq)
   end do
  
   ! Cloud water and ice particle sizes, saved in physics buffer for radiation
   call cldefr(lchnk, ncol, landfrac, state1%t, rel, rei, state1%ps, state1%pmid, landm, icefrac, snowh)
      
end subroutine stratiform_tend

!===============================================================================

end module stratiform

