#! /bin/tcsh -f
#
#=======================================================================
#
#  run-lightning.csh
#
#  Generic batch submission script for lightning using LSF.  
#
#-----------------------------------------------------------------------
# Batch options for machine with LSF batch system. 
# Usage for PathScale compiler (default): 
#   bsub < run-lightning.csh
#-----------------------------------------------------------------------
#
#BSUB -a mpich_gm
#BSUB -x
#BSUB -n 4
#BSUB -R "span[ptile=2]"
#BSUB -o cam.o%J
#BSUB -e cam.e%J
#BSUB -q regular
#BSUB -W 0:10
# #BSUB -P <please_set>

#=======================================================================

##extract number of tasks from batch environment
set ntasks = `echo $LSB_HOSTS | wc -w`

setenv INC_NETCDF /contrib/2.6/netcdf/3.6.0-p1-pathscale-2.4-64/include
setenv LIB_NETCDF /contrib/2.6/netcdf/3.6.0-p1-pathscale-2.4-64/lib
set mpich=/contrib/2.6/mpich-gm/1.2.6..14a-pathscale-2.4-64
setenv INC_MPI ${mpich}/include
setenv LIB_MPI ${mpich}/lib
set ps=/contrib/2.6/pathscale/2.4
setenv PATH ${mpich}/bin:${ps}/bin:${PATH}
setenv LD_LIBRARY_PATH ${ps}/lib/2.4:/opt/pathscale/lib/2.4/32:${LD_LIBRARY_PATH}

## ROOT OF CAM DISTRIBUTION - probably needs to be customized.
## Contains the source code for the CAM distribution.
## (the root directory contains the subdirectory "models")
set camroot      = /fis/cgd/...

## ROOT OF CAM DATA DISTRIBUTION - needs to be customized unless running at NCAR.
## Contains the initial and boundary data for the CAM distribution.
## (the root directory contains the subdirectories "atm" and "lnd")
setenv CSMDATA /fis/cgd/cseg/csm/inputdata    

## LOGNAME - used in default settings, must be set if not available
## setenv LOGNAME <username>
if !($?LOGNAME) then
    echo "environment variable, LOGNAME required for setting of defaults - exiting"
    exit 1
endif

## Default namelist settings:
## $case is the case identifier for this run. It will be placed in the namelist.
## $runtype is the run type: startup, continue, or branch.
## $stop_n is the number of days to integrate (units depends on stop_option)
set case         = camrun
set runtype      = startup
set stop_n       = 1

## $wrkdir is a working directory where the model will be built and run.
## $blddir is the directory where model will be compiled.
## $rundir is the directory where the model will be run.
## $cfgdir is the directory containing the CAM configuration scripts.
set wrkdir       = /ptmp/$LOGNAME
set blddir       = $wrkdir/$case/bld
set rundir       = $wrkdir/$case
set cfgdir       = $camroot/models/atm/cam/bld
set archdir      = $camroot/scripts/ccsm_utils/Tools/archiving

## Ensure that run and build directories exist
mkdir -p $rundir                || echo "cannot create $rundir" && exit 1
mkdir -p $blddir                || echo "cannot create $blddir" && exit 1

## If an executable doesn't exist, build one.
if ( ! -x $blddir/cam ) then
    cd $blddir                  || echo "cd $blddir failed" && exit 1
    $cfgdir/configure -spmd -nosmp -fc pathf90 -linker mpif90 -ntasks $ntasks || echo "configure failed" && exit 1
    echo "building CAM in $blddir ..."
    rm -f Depends
    gmake >&! MAKE.out      || echo "CAM build failed: see $blddir/MAKE.out" && exit 1
endif

## Create the namelist
cd $blddir                      || echo "cd $blddir failed" && exit 1
$cfgdir/build-namelist -s -case $case -runtype $runtype \
 -namelist "&camexp stop_option='ndays', stop_n=$stop_n /"  || echo "build-namelist failed" && exit 1

## Run CAM
cd $rundir                      || echo "cd $rundir failed" && exit 1
mv $blddir/*in .
echo "running CAM in $rundir"
mpirun.lsf -np 2 $blddir/cam             || echo "CAM run failed" && exit 1

## POST-PROCESSING vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
## see the README in the archiving dir for more info how to use these scripts

## SHORT-TERM ARCHIVE SETTINGS
setenv STA_ROOT $wrkdir               #root path - required
setenv STA_SAVE_INTERIM_RFILES FALSE  #false->will archive only restart files from end of block run

## LONG-TERM ARCHIVE SETTINGS
set uc_logname = `sh -c 'echo ${LOGNAME} | tr [a-z] [A-Z]'`
setenv LTA_ROOT /${uc_logname}/csm    #root path - required
setenv LTA_RETENTION 365              #retention period (days)
setenv LTA_INTEGRITY HIGH             #archival validated by file comparison (vs NORMAL where its by file size only)
setenv LTA_COS rel=ec                 #ec->economy class of service (just one copy archived)

## GENERAL ARCHIVING SETTINGS
setenv ARCH_CASE $case                #casename - required

## call to short-term archive script
$archdir/st_archive.sh

## call to long-term archive script - will spawn batch job
$archdir/lt_archive.sh -f

exit 0
