#!/bin/sh 
#

# lt_archiver.sh:  top-level script for the archiving of
#                  CAM model output to long-term storage
#
# usage on bangkok, calgary, tempest, bluevista, blueice, bluefire, lightning: 
# ./lt_archiver.sh
#
# valid arguments: 
# -i    interactive usage
# -f    force batch submission (avoids user prompt)
# -h    displays this help message
#
# **pass environment variables by preceding above commands 
#   with 'env var1=setting var2=setting '
# **more details in the README of the archiving directory 


#validate required env var settings
if [ -z "$STA_ROOT" ]; then
    echo ""
    echo "lt_archive.sh: error, environment variable STA_ROOT is required "
    echo "               for root location of short-term archive"
    echo "lt_archive.sh: exiting"
    exit 1
fi

if [ -z "$ARCH_CASE" ]; then
    echo ""
    echo "lt_archive.sh: error, environment variable ARCH_CASE is required "
    echo "               for casename of model run to archive"
    echo "lt_archive.sh: exiting"
    exit 1
fi

sta="${STA_ROOT}/st_archive/${ARCH_CASE}"

if [ ! -d "$sta" ]; then
    echo ""
    echo "lt_archive.sh: error, expected location of short-term archive is not a "
    echo "               valid directory:  ${sta}"
    echo "lt_archive.sh: exiting"
    exit 1
fi

if [ -z "$LTA_ROOT" ]; then
    echo ""
    echo "lt_archive.sh: error, environment variable LTA_ROOT is required "
    echo "               for root location of long-term archive"
    echo "lt_archive.sh: exiting"
    exit 1
fi

lta="${LTA_ROOT}/${ARCH_CASE}"
#strip off any prefix if specified
lta=${lta#*:}

#return to directory of job initiation
if [ -n "$LSB_JOBID" ]; then
    cd ${LS_SUBCWD}
elif [ -n "$PBS_JOBID" ]; then
    cd ${PBS_O_WORKDIR}
elif [ -n "$QSUB_REQID" ]; then  
    cd ${QSUB_WORKDIR}
fi

hostname=`hostname`
case $hostname in

    ##bluevista
    bv* )

    submit_script="lta_${ARCH_CASE}_bluevista.sh"
    site="NCAR"

    #do this check only if batch submission intended?
    if [ -z "$CAM_ACCOUNT" ]; then
	export CAM_ACCOUNT=`grep -i "^${LOGNAME}:" /etc/project.ncar | cut -f 1 -d "," | cut -f 2 -d ":" `
	if [ -z "${CAM_ACCOUNT}" ]; then
	    echo ""
	    echo "ERROR: unable to locate an account number to charge for this job under user: $LOGNAME"
	    exit 2
	fi
    fi


##vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv writing to batch script vvvvvvvvvvvvvvvvvvv
cat > ./${submit_script} << EOF
#!/bin/sh
#

#BSUB -n 1                       # total tasks needed
#BSUB -J ${ARCH_CASE}.lta
#BSUB -o lta_${ARCH_CASE}.o%J              # output filename
#BSUB -e lta_${ARCH_CASE}.o%J              # error filename
#BSUB -q share                  # queue
#BSUB -W 3:00
#BSUB -P $CAM_ACCOUNT      

EOF
##^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ writing to batch script ^^^^^^^^^^^^^^^^^^^
    ;;

    ##blueice
    bl* )

    submit_script="lta_${ARCH_CASE}_blueice.sh"
    site="NCAR"

    #do this check only if batch submission intended?
    if [ -z "$CAM_ACCOUNT" ]; then
	export CAM_ACCOUNT=`grep -i "^${LOGNAME}:" /etc/project.ncar | cut -f 1 -d "," | cut -f 2 -d ":" `
	if [ -z "${CAM_ACCOUNT}" ]; then
	    echo ""
	    echo "ERROR: unable to locate an account number to charge for this job under user: $LOGNAME"
	    exit 2
	fi
    fi


##vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv writing to batch script vvvvvvvvvvvvvvvvvvv
cat > ./${submit_script} << EOF
#!/bin/sh
#

#BSUB -n 1                       # total tasks needed
#BSUB -J ${ARCH_CASE}.lta
#BSUB -o lta_${ARCH_CASE}.o%J              # output filename
#BSUB -e lta_${ARCH_CASE}.o%J              # error filename
#BSUB -q share                  # queue
#BSUB -W 3:00
#BSUB -P $CAM_ACCOUNT      

EOF
##^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ writing to batch script ^^^^^^^^^^^^^^^^^^^
    ;;

    ##bluefire
    be* )

    submit_script="lta_${ARCH_CASE}_bluefire.sh"
    site="NCAR"

    #do this check only if batch submission intended?
    if [ -z "$CAM_ACCOUNT" ]; then
	export CAM_ACCOUNT=`grep -i "^${LOGNAME}:" /etc/project.ncar | cut -f 1 -d "," | cut -f 2 -d ":" `
	if [ -z "${CAM_ACCOUNT}" ]; then
	    echo ""
	    echo "ERROR: unable to locate an account number to charge for this job under user: $LOGNAME"
	    exit 2
	fi
    fi


##vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv writing to batch script vvvvvvvvvvvvvvvvvvv
cat > ./${submit_script} << EOF
#!/bin/sh
#

#BSUB -n 1                       # total tasks needed
#BSUB -J ${ARCH_CASE}.lta
#BSUB -o lta_${ARCH_CASE}.o%J              # output filename
#BSUB -e lta_${ARCH_CASE}.o%J              # error filename
#BSUB -q share                  # queue
#BSUB -W 3:00
#BSUB -P $CAM_ACCOUNT      

EOF
##^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ writing to batch script ^^^^^^^^^^^^^^^^^^^
    ;;

    ##lightning
    ln* )

    submit_script="lta_${ARCH_CASE}_lightning.sh"
    site="NCAR"

    #do this check only if batch submission intended?
    if [ -z "$CAM_ACCOUNT" ]; then
	CAM_ACCOUNT=`grep -i "^${LOGNAME}:" /etc/project | cut -f 1 -d "," | cut -f 2 -d ":" `
	if [ -z "${CAM_ACCOUNT}" ]; then
	    echo ""
	    echo "ERROR: unable to locate an account number to charge for this job under user: $LOGNAME"
	    exit 2
	fi
    fi


##vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv writing to batch script vvvvvvvvvvvvvvvvvvv
cat > ./${submit_script} << EOF
#!/bin/sh
#

#BSUB -n 1                       # total tasks needed
#BSUB -J ${ARCH_CASE}.lta
#BSUB -o lta_${ARCH_CASE}.o%J              # output filename
#BSUB -e lta_${ARCH_CASE}.o%J              # error filename
#BSUB -q share                  # queue
#BSUB -W 3:00
#BSUB -P $CAM_ACCOUNT      

EOF
##^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ writing to batch script ^^^^^^^^^^^^^^^^^^^
    ;;

    ##bangkok
    ba* | b0* )

    export PATH=/cluster/torque/bin:${PATH}

    submit_script="lta_${ARCH_CASE}_bangkok.sh"
    site="NCAR"


##vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv writing to batch script vvvvvvvvvvvvvvvvvvv
cat > ./${submit_script} << EOF
#!/bin/sh
#

#PBS -l nodes=1:ppn=1
#PBS -N ${ARCH_CASE}.lta
#PBS -j oe
#PBS -q small
#PBS -V

EOF
##^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ writing to batch script ^^^^^^^^^^^^^^^^^^^
    ;;


    ##calgary
    ca* | c0* )

    export PATH=/cluster/torque/bin:${PATH}

    submit_script="lta_${ARCH_CASE}_calgary.sh"
    site="NCAR"


##vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv writing to batch script vvvvvvvvvvvvvvvvvvv
cat > ./${submit_script} << EOF
#!/bin/sh
#

#PBS -l nodes=1:ppn=1
#PBS -N ${ARCH_CASE}.lta
#PBS -j oe
#PBS -q small
#PBS -V

EOF
##^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ writing to batch script ^^^^^^^^^^^^^^^^^^^
    ;;


    ##tempest
    te* )

    submit_script="lta_${ARCH_CASE}_tempest.sh"
    site="NCAR"


##vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv writing to batch script vvvvvvvvvvvvvvvvvvv
cat > ./${submit_script} << EOF
#!/bin/sh
#

#QSUB -q sb -l mpp_p=4            # select batch queue
#QSUB -lT 10800 -lt 10800         # set CPU time limits
#QSUB -mb -me -eo                 # combine stderr & stout
#QSUB -x                          # Export all Environment variables

EOF
##^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ writing to batch script ^^^^^^^^^^^^^^^^^^^
    ;;

    * ) 

    echo ""
    echo "lt_archive.sh: error, machine $hostname not currently supported"
    echo "lt_archive.sh: exiting"
    exit 1 
    ;;

esac

if [ "$site" == "NCAR" ]; then
    if [ -z "$LTA_INTEGRITY" ]; then
	echo "lt_archive.sh: warning, environment variable LTA_INTEGRITY is not "
	echo "               set - using "HIGH" as default to validate file transfer by file comparison"
	export LTA_INTEGRITY=HIGH
    fi

    if [ -z "$LTA_ANN_TARRING" ]; then
	echo "lt_archive.sh: warning, environment variable LTA_ANN_TARRING is not "
	echo "               set - using "TRUE" as default for whether to bundle "
	echo "               certain history files into annual tar files"
	export LTA_ANN_TARRING=TRUE
    fi

    if [ -z "$LTA_RETENTION" ]; then
	echo "lt_archive.sh: warning, environment variable LTA_RETENTION is not "
	echo "               set - using "365" as default retention period"
	lta_opts="-pe 365"
    else
	lta_opts="-pe $LTA_RETENTION"
    fi

    if [ -z "$LTA_COS" ]; then
	echo "lt_archive.sh: warning, environment variable LTA_COS is not "
	echo "               set - using "rel=ec" as default for economy class of service"
	lta_opts="$lta_opts -class rel=ec"
    else
	lta_opts="$lta_opts -class $LTA_COS"
    fi

    if [ -z "$LTA_WPWD" ]; then
	echo "lt_archive.sh: warning, environment variable LTA_WPWD is not "
	echo "               set - will not use a write password for archived files"
    else
	lta_opts="$lta_opts -wpwd $LTA_WPWD"
    fi

    if [ -z "$LTA_PROJ" ]; then
	echo "lt_archive.sh: warning, environment variable LTA_PROJ is not "
	echo "               set - using default project number for access charging"
    else
	lta_opts="$lta_opts -proj $LTA_PROJ"
    fi

##vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv writing to batch script vvvvvvvvvvvvvvvvvvv
cat >> ./${submit_script} << EOF

site="NCAR"
integrity=$LTA_INTEGRITY
ann_tarring=$LTA_ANN_TARRING
lta_opts="$lta_opts"

EOF
##^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ writing to batch script ^^^^^^^^^^^^^^^^^^^

fi



##vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv writing to batch script vvvvvvvvvvvvvvvvvvv
cat >> ./${submit_script} << EOF

#function to get the size of a file from its long listing
extract_file_size() {
    if [ \$# -lt 5 ]; then
        echo "0"
    else
	echo \$5
    fi
}

#function to do a long listing of a file within a long-term archive
lta_ls () {
    if [ "\$1" == "NCAR" ]; then
	if ! msls -l /\$2 2> /dev/null; then
	    echo "0 0 0 0 0"
	fi
    fi
}

#function to copy a local directory to a long-term archive
#arg1 is site, arg2 is sourcedir, arg3 is destdir, remaining are archiving options
lta_put () {
    if [ "\$1" == "NCAR" ]; then
	shift
	source=\$1
	shift
	dest=\$1
	shift
	echo msrcp -R \$* \$source mss:/\$dest
	msrcp -R \$* \$source mss:/\$dest
    fi
}

#function to copy a file from long-term archive
lta_get () {
    if [ "\$1" == "NCAR" ]; then
	echo msrcp mss:/\$2 \$3 2> /dev/null
	msrcp mss:/\$2 \$3 2> /dev/null
        # work-around for bluevista, blueice to ensure file arrives before continuing - will correct later
	sleep 30
    fi
}


#avoid running multiple lt_archive jobs
if [ -d ${sta}/lt_archive_in_progress ]; then
    echo ""
    echo "error, long-term archiver already running for $ARCH_CASE"
    echo "if this is KNOWN TO BE FALSE, remove the directory "
    echo "${sta}/lt_archive_in_progress "
    echo "and try again"
    echo "exiting"
    exit 1
else
    mkdir ${sta}/lt_archive_in_progress
fi

#first bundle any monthly history files into yearly tar files
echo ""
set atm lnd ice
for model in \$*; do
    echo ""
    echo "looking to bundle \${model} history files into yearly tar files"
    cd ${sta}/\${model}/hist

    list_of_partial_files=\`ls partial.*.tar 2> /dev/null\`
    for file in \${list_of_partial_files}; do
	echo tar -xf \$file
	tar -xf \$file
	rm \$file
    done

    still_tarring=\$ann_tarring

    while [ "\$still_tarring" == "TRUE" ]; do
	first_file=\`ls -1 *.h0.????-??.nc *.h.????-??.nc 2> /dev/null | head -1\`
	if [ -n "\$first_file" ]; then
	    #first_file=\${list_of_partial_files%% *}
	    loc_listing=\`ls -l \${first_file}\`
	    loc_size=\`extract_file_size \$loc_listing\`
	    if [ \$loc_size -lt 1000000000 ]; then
		trunc_filename=\`echo \$first_file | sed "s/-..\.nc//;"\`
		if [ -f \${trunc_filename}-12.nc ]; then  
		    if [ -f \${trunc_filename}-11.nc ]; then
			#december file is not alone - will archive for the year
			echo ""
			echo tar -cf \${trunc_filename}.tar \${trunc_filename}*.nc
			tar -cf \${trunc_filename}.tar \${trunc_filename}*.nc
			if [ \$? -eq 0 ]; then
			    echo ""
			    echo rm \${trunc_filename}*.nc
			    rm \${trunc_filename}*.nc
			else
			    echo "error trying to bundle history files into yearly tar file - exiting"
			    rmdir ${sta}/lt_archive_in_progress
			    exit -1
			fi
		    else
			#only have the december file, no longer needed for staging
			echo ""
			echo "removing files used for initial staging of data"
			echo rm \${trunc_filename}*.nc
			rm \${trunc_filename}*.nc
		    fi
		else
		    tar_filename=\`echo \$trunc_filename | sed "s/.*\.h0*\./partial\./;"\`
		    echo ""
		    echo tar -cf \${tar_filename}.tar \${trunc_filename}*.nc
		    tar -cf \${tar_filename}.tar \${trunc_filename}*.nc
		    if [ \$? -eq 0 ]; then
			echo ""
			echo rm \${trunc_filename}*.nc
			rm \${trunc_filename}*.nc
		    else
			echo "error trying to bundle history files into partial tar file - exiting"
			rmdir ${sta}/lt_archive_in_progress
			exit -1
		    fi
		fi
	    else
		still_tarring="FALSE"
	    fi
	else
	    still_tarring="FALSE"
	fi
    done
done
echo "finished bundling all history files into yearly tar files"
echo ""

echo cd ${sta}
cd ${sta}

#temporarily change permissions on partial files to avoid archiving them
list_of_partial_files=\`find . -name 'partial*' -type f\`
for file in \$list_of_partial_files; do
    chmod a-r \$file
done

#save list of files to archive for validation later
#list_of_files_to_archive=\`find . -name '*' -type f\`
list_of_files_to_archive=\`find . ! -name 'partial*' -type f\`

#now copy files to long-term archive, must be in sta dir
lta_put \$site . ${lta} \$lta_opts
rc=\$?

#replace read permissions on partial files
for file in \$list_of_partial_files; do
    chmod a+r \$file
done

if [ "\$integrity" == "NORMAL" ]; then
    #verify successful copy to mass store by file size
    for file in \$list_of_files_to_archive; do
	echo ""
	echo "now verifying \$file"
	#just use mass store listing for verification
	lta_listing=\`lta_ls \$site ${lta}/\${file} 2>&1 | tail -1\`
	lta_size=\`extract_file_size \$lta_listing\`
	loc_listing=\`ls -l \${file}\`
	loc_size=\`extract_file_size \$loc_listing\`
	if [ \$loc_size -gt 0 ] && [ \$loc_size -eq \$lta_size ]; then
	    echo "file in long-term archive is the same size as local - remove local"
	    echo rm \${file}
	    rm \${file}
	    continue
	else
	    echo "file in long-term archive is NOT the same size as local - leaving local copy"
	    rc=1
	fi
    done
else
    #verify successful copy to mass store by file comparison
    for file in \$list_of_files_to_archive; do
	echo ""
	echo "now verifying \$file"
	lta_get \$site ${lta}/\${file} ./file_to_compare
	cmp -s \${file} ./file_to_compare
	if [ \$? -eq 0 ]; then
	    echo "file in long-term archive is identical to local - deleting local"
	    rm \${file}
	    rm ./file_to_compare
	else
	    echo "file in long-term archive is NOT identical to local - leaving local copy"
	    rc=2
	fi
    done
fi

#must remove flag
rmdir ${sta}/lt_archive_in_progress

if [ \$rc -ne 0 ]; then
    echo "long-term archiving completed - "
    echo "warning, at least one file did not make it to long-term archive and"
    echo "         remains in the short-term archiving directory"
else
    echo "long-term archiving completed successfully"
fi

exit \$rc

EOF
##^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ writing to batch script ^^^^^^^^^^^^^^^^^^^


chmod a+x $submit_script
arg1=${1##*-}
case $arg1 in
    [iI]* )
    ./${submit_script}
    exit 0
    ;;

    [fF]* )
    ;;

    "" )
    echo ""
    echo "**********************"
    echo "$submit_script has been created and will be submitted to the batch queue..."
    echo "(ret) to continue, (a) to abort"
    read ans
    case $ans in
	[aA]* ) 
	echo "aborting...type ./lt_archive.sh -h for help message"
	exit 0
	;;
    esac
    ;;

    * )
    echo ""
    echo "**********************"
    echo "usage on bangkok, calgary, tempest, bluevista, blueice, bluefire, lightning: "
    echo "./lt_archive.sh"
    echo ""
    echo "valid arguments: "
    echo "-i    interactive usage"
    echo "-f    force batch submission (avoids user prompt)"
    echo "-h    displays this help message"
    echo ""
    echo "**pass environment variables by preceding above commands "
    echo "  with 'env var1=setting var2=setting '"
    echo ""
    echo "**********************"
    exit 0
    ;;
esac

echo "submitting..."
case $hostname in
    ##bluevista
    bv* )  bsub < ${submit_script};;

    ##blueice
    bl* )  bsub < ${submit_script};;

    ##bluefire
    be* )  bsub < ${submit_script};;

    ##lightning
    ln* )  bsub < ${submit_script};;

    ##bangkok,calgary
    ba* | b0* | ca* | c0* )  qsub ${submit_script};;

    ##tempest
    te* )  qsub ${submit_script};;

esac
echo "lt_archive.sh: long-term archiver submitted to batch queue"
exit 0
