! <compile=optimized>
#include "copyright.h"
#include "../include/assert.fh"
#include "../include/dprec.fh"
module qmmm_struct_module
! ----------------------------------------------------------------------
! PURPOSE: Data type holding information for QMMM calculations
!          These data are required for all type of QMMM runs, that is
!          using the internal QM methods and external programs
! 
! Author: Andreas W. Goetz
!         <agoetz@sdsc.edu>
! Date  : November 2010
!
! Based on previous data type and subroutines contained in qmmm_module
! as written by Ross Walker and Mike Crowley
!
! ATTENTION: This data type is not self contained, yet
!            Parts of it are allocated outside of this module
!            See also comments below in new_qmmm_struct_type()
!            Take care when broadcasting / allocating!
!
! Methods:
!   new      : allocate
!   delete   : deallocate
!   broadcast: broadcast data type
!              also allocate if necessary (for example on slaves)
!   print    : print content of all variables to stdout
!              (useful for debugging)
! ----------------------------------------------------------------------

  use ElementOrbitalIndex, only : numberElements

  implicit none

  private
  
  public :: qmmm_struct_type
#ifdef MPI
  public :: broadcast
#endif
  public :: new, delete, print

  type qmmm_struct_type

     ! Calculated Core-Core energy for QM-QM nuclei-nuclei
     ! and QM nuclei - MM charge interactions in (eV)     
     _REAL_ :: enuclr_qmqm, enuclr_qmmm

     ! Electronic energy (in eV)
     _REAL_ :: elec_eng

     ! Dispersion and hydrogen bonding correction energies
     ! as computed in dh_correction_module (kcal/mol)
     _REAL_ :: dCorrection
     _REAL_ :: hCorrection

     ! nquant long - original resp charges for the QM atoms as read from the prmtop file
     _REAL_, dimension(:), pointer :: qm_resp_charges => null()

     ! Sum of resp charges making up the quantum region - In AMBERELECTROSTATIC UNITS
     _REAL_ :: qm_resp_charge_sum
     _REAL_, dimension(:), pointer :: mm_link_pair_resp_charges => null()

     ! Coordinates of the mm link pair atoms as they would be for a given MD step in amber's
     ! unimaged coordinate array. Extracted by position link atoms and restored back into
     ! the x array by restore_mm_link_pair_coords.
     _REAL_, dimension(:,:), pointer :: mm_link_pair_saved_coords => null()

     ! Cartesian coordinates of ALL (real+link) qm atoms [3*(nquant+nlink) long]
     _REAL_, dimension(:,:), pointer :: qm_coords => null()

     ! MM charges scaled by one scale to make them electron units
     _REAL_, dimension(:), pointer :: scaled_mm_charges => null()

     ! Forces generated by qm_mm before adding them to the main f array.
     _REAL_, dimension(:,:), pointer :: dxyzqm, dxyzcl => null()
     
     ! Imaged mm coordinates and scaled mm charges.
     _REAL_, dimension(:,:), pointer :: qm_xcrd => null()

     ! Switched electrostatic potential at each QM atom site due to MM atoms within cutoff
     _REAL_, dimension(:), pointer :: switched_mmpot => null()

     ! Total number of atoms
     integer :: natom
     
     ! Total number of quantum atoms (excluding link atoms)
     integer :: nquant

     ! Total number of link atoms
     integer :: nlink

     ! Total number of quantum atoms = nquant+nlink
     integer :: nquant_nlink

     ! The number of atom types present. 
     integer :: qm_ntypes

     ! The number of noshake QM atoms overlapping with noshakemask when qmshake=0
     integer :: noshake_overlap

     ! The id of each type, essentially the atomic number of that type, e.g. type 1 may have atomic number = 8.
     integer, dimension(numberElements) :: qm_type_id

     ! The type of each qm atom, essentially a re-basing of the atomic numbers to minimise memory usage.
     integer, dimension(:), pointer :: qm_atom_type => null()
  
     ! list of MM-QM atoms for which link atoms were added
     integer, dimension(:,:), pointer :: link_pairs => null()

     ! List of atom numbers of the qm atoms as numbered in prmtop
     ! nquant_nlink long, link atoms are on the end and contain the number of the
     ! mm pair making up the link atom
     integer, pointer :: iqmatoms(:) => null()

     ! integer list of atomic numbers for the qm atoms 
     integer, dimension(:), pointer :: iqm_atomic_numbers => null()

     ! Number of pairs per QM atom. - length of pair_list. 
     integer :: qm_mm_pairs

     ! Non bond pair list for each QM atom
     integer, dimension(:), pointer :: qm_mm_pair_list => null()

     ! atomic numbers of MM atoms included in QM-MM pairs (only used for PM3/MM*)
     ! qm_mm_pairs long
     ! allocated in read_qmmm_nm_and_alloc for SQM external charges
     ! allocated in ??? for sander QM/MM
     integer, dimension(:), pointer :: qm_mm_pair_atom_numbers  => null()

     ! Number of times qm_mm has been called - effectively nstep
     integer :: num_qmmm_calls

     ! True / false mask specifying if atom is a QM atom. True = QM atom (natom long)
     logical, dimension(:), pointer :: atom_mask => null()

     ! True / false mask specifying if atom is a MM link pair atom. True = MM link pair atom (natom long)
     logical, dimension(:), pointer :: mm_link_mask => null()

     ! Set to true at beginning of sander subroutine and then set to false at the end of qm_mm. Used for allocation purposes.
     logical :: qm_mm_first_call

     ! Set to true at beginning of sander subroutine and then set to false at the end of qm_mm. Used for allocation purposes.
     logical :: fock_first_call
     logical :: fock2_2atm_first_call 
     logical :: qm2_allocate_e_repul_first_call
     logical :: qm2_calc_rij_eqns_first_call
     logical :: qm2_scf_first_call
     logical :: zero_link_charges_first_call
     logical :: adj_mm_link_pair_crd_first_call

     ! Set to True if theory is AM1, PM3, PDDG/PM3, PM3CARB1, RM1, PM3ZNB
     logical :: AM1_OR_PM3

     ! Set to True if theory is PDDG/PM3 or PDDG/MNDO
     logical :: PDDG_IN_USE

     ! Set to true if the coordinates of the MM link pair atoms in ambers main coordinate array
     ! have been set to the link atom coordinates. This acts as a safety to ensure that the adj 
     ! link atoms is not called without having restored them.
     logical :: mmcoords_contains_lnk_coords

     ! Set to True if theory is PM3 and qmmm_int == 3 or 4
     logical :: PM3MMX_INTERFACE

  end type qmmm_struct_type

  interface new
     module procedure new_qmmm_struct_type
  end interface

  interface delete
     module procedure delete_qmmm_struct_type
  end interface

#ifdef MPI
  interface broadcast
     module procedure broadcast_qmmm_struct_type
  end interface
#endif

  interface print
     module procedure print_qmmm_struct_type
  end interface

  interface extend
     module procedure extend_integer_array
  end interface

contains


  subroutine new_qmmm_struct_type(self, qmmm_int, qmmm_switch)
    
    ! NOTE: self%nquant needs to be known before calling this subroutine
    !       self%nlink as well or set to zero and arrays re-allocated later on

    implicit none

    ! need inout here since scalar parts may already have data
    type(qmmm_struct_type), intent(inout) :: self
    integer, intent(in) :: qmmm_int
    logical, intent(in) :: qmmm_switch

    integer :: natom, nquant, nlink, npairmax, ier

    ! this data type is still scattered...
    !
    ! self%qm_atom_type               allocated in qm_assign_atom_types()          (qm_assign_atom_types.f) - for master
    !                                          and broadcast()                                              - for slaves
    ! self%link_pairs                 allocated in identify_link_atoms()           (qm_link_atoms.f)        - for master
    !                                          and broadcast()                                              - for slaves
    ! self%mm_link_pair_saved_coords  allocated in adj_mm_link_pair_crd()          (qm_link_atoms.f)
    ! self%%mm_link_pair_resp_charges allocated in qm_zero_mm_link_pair_main_chg() (qm_zero_charges.f)
    ! self%qm_coords                  allocated in qm_mm()                         (qm_mm.f) and sqm (sqm.f)

    ! attention: this assumes that nquant and nlink are already assigned!
    natom  = self%natom
    nquant = self%nquant
    nlink  = self%nlink

    if ( .not. associated ( self%qm_resp_charges ) ) then
       allocate ( self%qm_resp_charges(nquant), stat=ier )
       REQUIRE(ier == 0)
    end if

    if ( .not. associated ( self%iqmatoms ) ) then
       allocate ( self%iqmatoms((nquant+nlink)), stat=ier )
       REQUIRE(ier == 0)
    else
       call extend(self%iqmatoms, nquant+nlink)
    end if

    if ( .not. associated ( self%iqm_atomic_numbers ) ) then
       allocate ( self%iqm_atomic_numbers((nquant+nlink)), stat=ier )
       REQUIRE(ier == 0)
    else
       call extend(self%iqm_atomic_numbers, nquant+nlink)
    end if

    if ( .not. associated ( self%atom_mask ) ) then
       allocate ( self%atom_mask(natom), stat=ier )
       REQUIRE(ier == 0)
    end if

    if ( .not. associated ( self%mm_link_mask ) ) then
       allocate ( self%mm_link_mask(natom), stat=ier )
       REQUIRE(ier == 0)
    end if

    if ( .not. associated (self%scaled_mm_charges ) ) then
       allocate (self%scaled_mm_charges(natom), stat=ier )
       REQUIRE(ier == 0)
    end if

    if ( .not. associated ( self%qm_xcrd ) ) then
       ! qm_xcrd only actually needs to be 4,qm_mm_pairs long...
       allocate ( self%qm_xcrd(4,natom), stat=ier )
       REQUIRE(ier == 0)
    end if

    ! All QM atoms share the same pair list:
    npairmax = natom - nquant + 1
    if ( .not. associated ( self%qm_mm_pair_list ) ) then
       allocate ( self%qm_mm_pair_list( npairmax ), stat=ier ) 
       REQUIRE(ier == 0)
    end if

    if (qmmm_int /= 0) then
       if ( .not. associated ( self%dxyzcl ) ) then
          ! dxyzcl array only actually needs to be 3,qm_mm_pairs long..
          allocate ( self%dxyzcl(3,natom), stat=ier )
          REQUIRE(ier == 0)
       end if
    end if

    if (qmmm_switch) then
       if ( .not. associated ( self%switched_mmpot ) ) then
          allocate ( self%switched_mmpot(nquant+nlink), stat=ier )
          REQUIRE(ier == 0) !Deallocated in deallocate qmmm
       end if
    end if

  end subroutine new_qmmm_struct_type

  subroutine extend_integer_array(iarr, new_size)

    ! extend an integer array if necessary and pad with zeros

    implicit none

    integer, pointer :: iarr(:)
    integer, intent(in) :: new_size

    integer :: ier, old_size
    integer, allocatable :: tmp(:)

    old_size = size (iarr)

    if (old_size < new_size) then

       allocate ( tmp(old_size), stat=ier )
       REQUIRE(ier == 0)

       tmp(:) = iarr(:)

       deallocate ( iarr, stat=ier )
       REQUIRE(ier == 0)

       allocate ( iarr(new_size), stat=ier )
       REQUIRE(ier == 0)

       iarr(1:old_size) = tmp(1:old_size)
       iarr(old_size+1:new_size) = 0

       deallocate ( tmp, stat=ier )
       REQUIRE(ier == 0)

    end if

  end subroutine extend_integer_array

  subroutine delete_qmmm_struct_type(self, qmmm_int, idc, qmmm_switch)

    implicit none

    type(qmmm_struct_type), intent(inout) :: self
    integer, intent(in) :: qmmm_int, idc
    logical, intent(in) :: qmmm_switch

    integer :: ier

    if ( associated ( self%qm_resp_charges ) ) then
       deallocate ( self%qm_resp_charges, stat=ier )
       REQUIRE(ier == 0)
    end if

    if ( associated ( self%iqmatoms) ) then
       deallocate ( self%iqmatoms, stat=ier )
       REQUIRE(ier == 0)
    end if

    if ( associated ( self%iqm_atomic_numbers) ) then
       deallocate ( self%iqm_atomic_numbers, stat=ier )
       REQUIRE(ier == 0)
    end if

    if ( associated ( self%atom_mask ) ) then
       deallocate ( self%atom_mask, stat=ier )
       REQUIRE(ier == 0)
    end if

    if ( associated ( self%mm_link_mask ) ) then
       deallocate ( self%mm_link_mask, stat=ier )
       REQUIRE(ier == 0)
    end if

    if ( associated (self%scaled_mm_charges ) ) then
       deallocate (self%scaled_mm_charges, stat=ier )
       REQUIRE(ier == 0) 
    end if

    if ( associated (self%qm_xcrd ) ) then
       deallocate (self%qm_xcrd, stat=ier )
       REQUIRE(ier == 0)
    end if

    if ( associated ( self%qm_mm_pair_list ) ) then
       deallocate ( self%qm_mm_pair_list, stat=ier ) 
       REQUIRE(ier == 0)
    end if

    if (qmmm_int /= 0) then
       if ( associated ( self%dxyzcl ) ) then
          deallocate ( self%dxyzcl, stat=ier )
          REQUIRE(ier == 0)
       end if
    end if

    !if (qmmm_int==3 .or. qmmm_int==4) then
    if (self%PM3MMX_INTERFACE) then
       ! needed for PM3/MM*
       if ( associated ( self%qm_mm_pair_atom_numbers ) ) then
          deallocate ( self%qm_mm_pair_atom_numbers, stat = ier)
          REQUIRE(ier == 0)
       end if
    end if

    if (qmmm_switch) then
       if ( associated ( self%switched_mmpot ) ) then
          deallocate ( self%switched_mmpot, stat=ier )
          REQUIRE(ier == 0) 
       end if
    end if

    ! Thes following parts have been allocated outside of this module
    ! So take special care...
    if ( associated ( self%qm_coords ) ) then
       deallocate ( self%qm_coords, stat = ier)
       REQUIRE(ier == 0)
    end if

    if ( associated ( self%dxyzqm ) ) then
       deallocate ( self%dxyzqm, stat = ier )
       REQUIRE(ier == 0)
    end if

    if ( associated ( self%qm_atom_type ) ) then
       deallocate ( self%qm_atom_type, stat = ier )
       REQUIRE(ier == 0)
    end if

    if (self%nlink > 0 .and. idc==0) then

       if ( associated ( self%link_pairs ) ) then
          deallocate ( self%link_pairs, stat = ier)
          REQUIRE(ier == 0)
       end if

       if ( associated ( self%mm_link_pair_resp_charges ) ) then
          deallocate ( self%mm_link_pair_resp_charges, stat = ier )
          REQUIRE(ier == 0)
       end if

       !Note for some cases, e.g. gas phase the adj_mm_link_pair_crd routine may
       !never have been called - so only do the deallocation if it has been called.
       if (.not. self%adj_mm_link_pair_crd_first_call) then
          if ( associated ( self%mm_link_pair_saved_coords ) ) then
             deallocate ( self%mm_link_pair_saved_coords, stat = ier )
             REQUIRE(ier == 0)
          end if
       end if

    end if

  end subroutine delete_qmmm_struct_type


#ifdef MPI
  subroutine broadcast_qmmm_struct_type(self, qmmm_int, qmmm_switch)

    implicit none
#include "parallel.h"
    include 'mpif.h'

    type(qmmm_struct_type), intent(inout) :: self
    integer, intent(in) :: qmmm_int
    logical, intent(in) :: qmmm_switch
    
    integer :: ier
    
    ! broadcast variables
    call mpi_bcast(self%natom        , 1             , mpi_integer, 0, commsander, ier) 
    call mpi_bcast(self%nquant       , 1             , mpi_integer, 0, commsander, ier) 
    call mpi_bcast(self%nlink        , 1             , mpi_integer, 0, commsander, ier) 
    call mpi_bcast(self%nquant_nlink , 1             , mpi_integer, 0, commsander, ier) 
    call mpi_bcast(self%qm_ntypes    , 1             , mpi_integer, 0, commsander, ier) 
    call mpi_bcast(self%qm_type_id   , numberElements, mpi_integer, 0, commsander, ier) 
    call mpi_bcast(self%noshake_overlap,  1          , mpi_integer, 0, commsander, ier) 
    call mpi_bcast(self%AM1_OR_PM3   , 1             , mpi_logical, 0, commsander, ier) 
    call mpi_bcast(self%PDDG_IN_USE  , 1             , mpi_logical, 0, commsander, ier) 
    call mpi_bcast(self%PM3MMX_INTERFACE, 1          , mpi_logical, 0, commsander, ier) 
  
    ! arrays may not have been allocated on slaves yet, therefore try to create object
    call new(self, qmmm_int, qmmm_switch)
    ! The following arrays are not allocated in new (still messed up due to historic reasons)
    if (self%nlink > 0) then
       if ( .not. associated ( self%link_pairs ) ) then
          allocate ( self%link_pairs(2,self%nlink), stat=ier )   
          REQUIRE(ier == 0)
       end if
    end if
    if ( .not. associated ( self%qm_atom_type ) ) then
       allocate ( self%qm_atom_type(self%nquant_nlink),stat=ier )
       REQUIRE(ier == 0)
    end if

    ! broadcast arrays
    call mpi_bcast(self%qm_atom_type     , self%nquant_nlink  , mpi_integer         , 0, commsander, ier)
    if (self%nlink > 0) then
       call mpi_bcast(self%link_pairs    , 2*self%nlink       , mpi_integer         , 0, commsander, ier)
    end if
    call mpi_bcast(self%iqmatoms          , self%nquant_nlink , mpi_integer         , 0, commsander, ier)
    call mpi_bcast(self%iqm_atomic_numbers, self%nquant_nlink , mpi_integer         , 0, commsander, ier)
    call mpi_bcast(self%atom_mask         , self%natom        , mpi_logical         , 0, commsander, ier)
    call mpi_bcast(self%mm_link_mask      , self%natom        , mpi_logical         , 0, commsander, ier)
    call mpi_bcast(self%scaled_mm_charges , self%natom        , mpi_double_precision, 0, commsander, ier)

  end subroutine broadcast_qmmm_struct_type
#endif


  subroutine print_qmmm_struct_type(self)

    implicit none

    type(qmmm_struct_type), intent(in) :: self

    write(6,'(a)') 'PRINTING OF QMMM_STRUCT_TYPE NOT IMPLEMENTED'
    
  end subroutine print_qmmm_struct_type

end module qmmm_struct_module
