module phys_control
!-----------------------------------------------------------------------
! Purpose:
!
! Provides a control interface to CAM physics packages
!
! Revision history:
! 2006-05-01  D. B. Coleman,  Creation of module
! 2009-02-13  Eaton           Replace *_{default,set}opts methods with module namelist.
!                             Add vars to indicate physics version and chemistry type.
!-----------------------------------------------------------------------

use spmd_utils,    only: masterproc
use cam_logfile,   only: iulog
use abortutils,    only: endrun
use shr_kind_mod,  only: r8 => shr_kind_r8

implicit none
private
save

! Public methods
public :: &
   phys_ctl_readnl,   &! read namelist from file
   phys_getopts,      &! generic query method
   phys_deepconv_pbl, &! return true if deep convection is allowed in the PBL
   phys_do_flux_avg    ! return true to average surface fluxes

! Private module data

character(len=16), parameter :: unset_str = 'UNSET'
integer,           parameter :: unset_int = huge(1)

! Namelist variables:
character(len=16) :: deep_scheme     = unset_str  ! deep convection package
character(len=16) :: shallow_scheme  = unset_str  ! shallow convection package
character(len=16) :: eddy_scheme     = unset_str  ! vertical diffusion package
character(len=16) :: microp_scheme   = unset_str  ! microphysics package
integer           :: srf_flux_avg    = unset_int  ! 1 => smooth surface fluxes, 0 otherwise
logical           :: atm_dep_flux    = .false.    ! true => deposition fluxes will be provided
                                                  !         to the coupler
logical           :: do_tms          = .false.    ! switch for turbulent mountain stress
real(r8)          :: tms_orocnst     = 4._r8      ! turbulent mountain stress parameter

!======================================================================= 
contains
!======================================================================= 

subroutine phys_ctl_readnl(nlfile)

   use namelist_utils,  only: find_group_name
   use units,           only: getunit, freeunit
   use mpishorthand

   character(len=*), intent(in) :: nlfile  ! filepath for file containing namelist input

   ! Local variables
   integer :: unitn, ierr
   character(len=*), parameter :: subname = 'phys_ctl_readnl'

   namelist /phys_ctl_nl/ deep_scheme, shallow_scheme, eddy_scheme, microp_scheme, &
      srf_flux_avg, atm_dep_flux, do_tms, tms_orocnst
   !-----------------------------------------------------------------------------

   if (masterproc) then
      unitn = getunit()
      open( unitn, file=trim(nlfile), status='old' )
      call find_group_name(unitn, 'phys_ctl_nl', status=ierr)
      if (ierr == 0) then
         read(unitn, phys_ctl_nl, iostat=ierr)
         if (ierr /= 0) then
            call endrun(subname // ':: ERROR reading namelist')
         end if
      end if
      close(unitn)
      call freeunit(unitn)
   end if

#ifdef SPMD
   ! Broadcast namelist variables
   call mpibcast(deep_scheme,    len(deep_scheme),    mpichar, 0, mpicom)
   call mpibcast(shallow_scheme, len(shallow_scheme), mpichar, 0, mpicom)
   call mpibcast(eddy_scheme,    len(eddy_scheme),    mpichar, 0, mpicom)
   call mpibcast(microp_scheme,  len(microp_scheme),  mpichar, 0, mpicom)
   call mpibcast(srf_flux_avg,   1,                   mpiint,  0, mpicom)
   call mpibcast(atm_dep_flux,   1,                   mpilog,  0, mpicom)
   call mpibcast(do_tms,         1,                   mpilog,  0, mpicom)
   call mpibcast(tms_orocnst,    1,                   mpir8,   0, mpicom)
#endif

   ! Error checking:

   ! check if srf_flux_avg was set by user.  If not then assign default based on
   ! value of eddy_scheme
   if (srf_flux_avg == unset_int) then
      srf_flux_avg = 0
      if (eddy_scheme .ne. 'HB') srf_flux_avg = 1
   end if

end subroutine phys_ctl_readnl

!===============================================================================

subroutine phys_getopts(deep_scheme_out, shallow_scheme_out, eddy_scheme_out, microp_scheme_out, &
                        atm_dep_flux_out, do_tms_out, tms_orocnst_out )
!-----------------------------------------------------------------------
! Purpose: Return runtime settings
!          deep_scheme_out   : deep convection scheme
!          shallow_scheme_out: shallow convection scheme
!          eddy_scheme_out   : vertical diffusion scheme
!	   microp_scheme_out : microphysics scheme
!-----------------------------------------------------------------------

   character(len=16), intent(out), optional :: deep_scheme_out
   character(len=16), intent(out), optional :: shallow_scheme_out
   character(len=16), intent(out), optional :: eddy_scheme_out
   character(len=16), intent(out), optional :: microp_scheme_out
   logical,           intent(out), optional :: atm_dep_flux_out
   logical,           intent(out), optional :: do_tms_out
   real(r8),          intent(out), optional :: tms_orocnst_out

   if ( present(deep_scheme_out) )      deep_scheme_out = deep_scheme
   if ( present(shallow_scheme_out) )   shallow_scheme_out = shallow_scheme
   if ( present(eddy_scheme_out) )      eddy_scheme_out = eddy_scheme
   if ( present(microp_scheme_out) )    microp_scheme_out = microp_scheme
   if ( present(atm_dep_flux_out) )     atm_dep_flux_out = atm_dep_flux
   if ( present(do_tms_out) )           do_tms_out = do_tms
   if ( present(tms_orocnst_out) )      tms_orocnst_out = tms_orocnst

end subroutine phys_getopts

!===============================================================================

function phys_deepconv_pbl()

  logical phys_deepconv_pbl

   ! Don't allow deep convection in PBL if running UW PBL scheme
   if ( (eddy_scheme .eq. 'diag_TKE' ) .or. (shallow_scheme .eq. 'UW' ) ) then
      phys_deepconv_pbl = .true.
   else
      phys_deepconv_pbl = .false.
   endif

   return

end function phys_deepconv_pbl

!===============================================================================

function phys_do_flux_avg()

   logical :: phys_do_flux_avg
   !----------------------------------------------------------------------

   phys_do_flux_avg = .false.
   if (srf_flux_avg == 1) phys_do_flux_avg = .true.

end function phys_do_flux_avg

!===============================================================================

end module phys_control
