! <compile=optimized>

!The 3D-RISM-KH software found here is copyright (c) 2011-2012 by 
!Andriy Kovalenko, Tyler Luchko, Takeshi Yamazaki and David A. Case.
!
!This program is free software: you can redistribute it and/or modify it
!under the terms of the GNU General Public License as published by the Free
!Software Foundation, either version 3 of the License, or (at your option)
!any later version.
!
!This program is distributed in the hope that it will be useful, but
!WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
!or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
!for more details.
!
!You should have received a copy of the GNU General Public License in the
!../../LICENSE file.  If not, see <http://www.gnu.org/licenses/>.
!
!Users of the 3D-RISM capability found here are requested to acknowledge
!use of the software in reports and publications.  Such acknowledgement
!should include the following citations:
!
!1) A. Kovalenko and F. Hirata. J. Chem. Phys., 110:10095-10112  (1999); 
!ibid. 112:10391-10417 (2000).   
!
!2) A. Kovalenko,  in:  Molecular  Theory  of  Solvation,  edited  by  
!F. Hirata  (Kluwer Academic Publishers, Dordrecht, 2003), pp.169-275.  
!
!3) T. Luchko, S. Gusarov, D.R. Roe, C. Simmerling, D.A. Case, J. Tuszynski,
!and  A. Kovalenko, J. Chem. Theory Comput., 6:607-624 (2010). 

#include "../include/dprec.fh"
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!Kovalenko-Hirata closure class for 1D-RISM.
!!!Kovalenko, A.; Hirata, F. J. Chem. Phys. 1999, 110, 10095–10112.
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
module rism1d_kh_c
  use safemem
  !the KH type is actually empty
  type rism1d_kh
     logical kh
  end type rism1d_kh

  public rism1d_kh_new, rism1d_kh_destroy, rism1d_kh_gvv, rism1d_kh_gvv_dT, rism1d_kh_solvene
contains
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!Initializes the KH closure
!!!IN:
!!!   this : KH object
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  subroutine rism1d_kh_new(this)
    implicit none
    type(rism1d_kh), intent(inout) :: this
  end subroutine rism1d_kh_new

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!Calculates Gvv from Uvv, Hvv, and Cvv using the KH closure
!!!IN:
!!!   this : the KH closure object
!!!   gvv  : site-site pair correlation function
!!!   uvv  : site-site potential
!!!   hvv  : site-site total correlation function
!!!   cvv  : site-site direct correlation function
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  subroutine rism1d_kh_gvv(this,gvv, uvv, hvv, cvv)
    implicit none
    type(rism1d_kh), intent(in) :: this
    _REAL_, intent(out) :: gvv(:,:)
    _REAL_, intent(in) :: uvv(:,:),hvv(:,:),cvv(:,:)
    integer :: ivv, ir
    _REAL_ :: tvv
    do ivv = 1, ubound(gvv,2)
       do ir = 1, ubound(gvv,1)
          tvv = -uvv(ir,ivv) +hvv(ir,ivv)-cvv(ir,ivv)
          if(tvv>=0d0)then
             gvv(ir,ivv) = 1d0 + tvv
          else
             gvv(ir,ivv) = exp(tvv)
          end if
       end do
    end do
  end subroutine rism1d_kh_gvv

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!Calculates temperature derivative Gvv from Gvv, Uvv, Hvv_dT, and Cvv_dT using
!!!the associated closure
!!!IN:
!!!   this   : the closure object
!!!   gvv_dT  : site-site temperature derivative pair correlation function
!!!   gvv    : site-site pair correlation function
!!!   hvv_dT  : site-site temperature derivative total correlation function
!!!   cvv_dT  : site-site temperature derivative direct correlation function
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  subroutine rism1d_kh_gvv_dT(this,gvv_dT, uvv, gvv, hvv_dT, cvv_dT)
    implicit none
    type(rism1d_kh), intent(in) :: this
    _REAL_, intent(out) :: gvv_dT(:,:)
    _REAL_, intent(in) :: uvv(:,:),gvv(:,:),hvv_dT(:,:),cvv_dT(:,:)
    integer :: ivv, ir
    _REAL_ :: tvv_dT
    do ivv = 1, ubound(gvv_dT,2)
       do ir = 1, ubound(gvv_dT,1)
          tvv_dT = uvv(ir,ivv) +hvv_dT(ir,ivv)-cvv_dT(ir,ivv)
          if(gvv(ir,ivv)>=1d0)then
             gvv_dT(ir,ivv) = tvv_dT
          else
             gvv_dT(ir,ivv) = tvv_dT*gvv(ir,ivv)
          end if
       end do
    end do
  end subroutine rism1d_kh_gvv_dT

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!Calculates Bvv (bridge function) from Uvv, Hvv, and Cvv using the KH closure
!!!IN:
!!!   this : the KH closure object
!!!   uvv  : site-site potential
!!!   gvv  : site-site pair correlation function
!!!   cvv  : site-site direct correlation function
!!!OUT:
!!!   bvv  : site-site bridge function
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  function rism1d_kh_bvv(this, uvv, gvv, cvv) result(bvv)
    implicit none
    type(rism1d_kh), intent(in) :: this
    _REAL_, pointer :: bvv(:,:)
    _REAL_, intent(in) :: uvv(:,:),gvv(:,:),cvv(:,:)
    integer :: ivv, ir
    _REAL_ :: tvv
    nullify(bvv)
    bvv =>safemem_realloc(bvv,ubound(gvv,1),ubound(gvv,2))
    do ivv = 1, ubound(bvv,2)
       do ir = 1, ubound(bvv,1)
          tvv = -uvv(ir,ivv) +gvv(ir,ivv)-1d0-cvv(ir,ivv)
          if(tvv>=0d0)then
             bvv(ir,ivv) = -tvv + log(1 + tvv)
          else
             bvv(ir,ivv) = 0
          end if
       end do
    end do
  end function rism1d_kh_bvv

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!Calculate the excess chemical potential in kT
!!!IN:
!!!   this : the closure object
!!!   gvv  : site-site pair correlation function
!!!   cvv  : site-site direct correlation function
!!!   mtv  : multiplicity of each site
!!!   rhov : number density of each site
!!!   rhotrgt : (optional) The final, physical density for thermodynamics.  This
!!!             can be used as an effective correction for some closures that
!!!             over estimate the pressure (e.g. HNC and KH)
!!!   dr   : r-space grid spacing
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  function rism1d_kh_exchem(this, gvv, cvv, mtv, jvv, rhov, rhotrgtv, dr) result(exchem)
    use constants, only : pi
    implicit none
    type(rism1d_kh), intent(in) :: this
    _REAL_, intent(in) :: gvv(:,:),cvv(:,:), rhov(:), rhotrgtv(:), dr
    integer, intent(in) :: mtv(:),jvv(:,:)
    _REAL_ :: exchem(ubound(mtv,1))
    _REAL_ :: r, h2c,hvv, exchemv
    integer :: ivv, iv1, iv2, ir
    
    do iv2=1,ubound(mtv,1) !nv
       exchem(iv2) = 0.d0
       do ir=2,ubound(gvv,1) !nr
          r = (ir-1)*dr
          exchemv = 0.d0
          do iv1=1,ubound(mtv,1) !nv
             ivv = jvv(iv1,iv2)
             hvv = gvv(ir,ivv) - 1.d0
             if (hvv >= 0.d0)  then
                h2c = - cvv(ir,ivv)
             else
                h2c = 0.5d0*hvv*hvv - cvv(ir,ivv)
             endif
             !                     exchemv = exchemv + rhov(iv1)*mtv(iv2) &
             exchemv = exchemv + rhotrgtv(iv1)*mtv(iv2) &
                  *(h2c - 0.5d0*hvv*cvv(ir,ivv))
          enddo
          exchem(iv2) = exchem(iv2) + exchemv*r**2
       enddo
    enddo
    exchem = exchem * 4.d0*pi*dr
end function rism1d_kh_exchem

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!Calculate the ionic(?) excess chemical potential in kT.  This seems to take
!!!the contribution only from the last solvent type.  I'm not sure what the point is
!!!here.
!!!IN:
!!!   this : the closure object
!!!   gvv  : site-site pair correlation function
!!!   cvv  : site-site direct correlation function
!!!   mtv  : multiplicity of each site
!!!   rhov : number density of each site
!!!   rhotrgt : (optional) The final, physical density for thermodynamics.  This
!!!             can be used as an effective correction for some closures that
!!!             over estimate the pressure (e.g. HNC and KH)
!!!   dr   : r-space grid spacing
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  function rism1d_kh_exchemIon(this, gvv, cvv, mtv, jvv, rhov, rhotrgtv, dr) result(exchem)
    use constants, only : pi
    implicit none
    type(rism1d_kh), intent(in) :: this
    _REAL_, intent(in) :: gvv(:,:),cvv(:,:), rhov(:), rhotrgtv(:), dr
    integer, intent(in) :: mtv(:),jvv(:,:)
    _REAL_ :: exchem(ubound(mtv,1))
    _REAL_ :: r, h2c,hvv, exchemv
    integer :: ivv, iv1, iv2, ir
    
    do iv2=1,ubound(mtv,1) !nv
       exchem(iv2) = 0.d0
       do ir=2,ubound(gvv,1) !nr
          r = (ir-1)*dr
          exchemv = 0.d0
!          do iv1=1,ubound(mtv,1) !nv
          iv1=ubound(mtv,1)
             ivv = jvv(iv1,iv2)
             hvv = gvv(ir,ivv) - 1.d0
             if (hvv >= 0.d0)  then
                h2c = - cvv(ir,ivv)
             else
                h2c = 0.5d0*hvv*hvv - cvv(ir,ivv)
             endif
             !                     exchemv = exchemv + rhov(iv1)*mtv(iv2) &
             exchemv = exchemv + rhotrgtv(iv2)*mtv(iv1) &
                  *(h2c - 0.5d0*hvv*cvv(ir,ivv))
 !         enddo
          exchem(iv2) = exchem(iv2) + exchemv*r**2
       enddo
    enddo
    exchem = exchem * 4.d0*pi*dr
  end function rism1d_kh_exchemIon

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!Calculate the solvation energy in kT
!!!IN:
!!!   this : the closure object
!!!   gvv  : site-site pair correlation function
!!!   cvv  : site-site direct correlation function
!!!   gvv_dT  : temperature derivative of site-site pair correlation function
!!!   cvv_dT  : temperature derivative of site-site direct correlation function
!!!   mtv  : multiplicity of each site
!!!   rhov : number density of each site
!!!   rhotrgt : (optional) The final, physical density for thermodynamics.  This
!!!             can be used as an effective correction for some closures that
!!!             over estimate the pressure (e.g. HNC and KH)
!!!   dr   : r-space grid spacing
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  function rism1d_kh_solvene(this, gvv, cvv, gvv_dT, cvv_dT, mtv, jvv, rhov, rhotrgtv, dr) result(solvene)
    use constants, only : pi
    implicit none
    type(rism1d_kh), intent(in) :: this
    _REAL_, intent(in) :: gvv(:,:),cvv(:,:), gvv_dT(:,:), cvv_dT(:,:), rhov(:), rhotrgtv(:), dr
    integer, intent(in) :: mtv(:),jvv(:,:)
    _REAL_ :: solvene(ubound(mtv,1))
    _REAL_ :: r, h2c,hvv, solvenev, hvv_dT
    integer :: ivv, iv1, iv2, ir
    
    do iv2=1,ubound(mtv,1) !nv
       solvene(iv2) = 0.d0
       do ir=2,ubound(gvv,1) !nr
          r = (ir-1)*dr
          solvenev = 0.d0
          do iv1=1,ubound(mtv,1) !nv
             ivv = jvv(iv1,iv2)
             hvv = gvv(ir,ivv) - 1.d0
             hvv_dT = gvv_dT(ir,ivv)
             if (hvv >= 0.d0)  then
                h2c = - cvv_dT(ir,ivv)
             else
                h2c = hvv*hvv_dT - cvv_dT(ir,ivv)
             endif
             solvenev = solvenev + rhotrgtv(iv1)*mtv(iv2) &
                  *(h2c - 0.5d0*hvv_dT*cvv(ir,ivv) - 0.5d0*hvv*cvv_dT(ir,ivv))
          enddo
          solvene(iv2) = solvene(iv2) + solvenev*r**2
       enddo
    enddo
    solvene = -1.d0 * solvene * 4.d0*pi*dr
end function rism1d_kh_solvene

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!Calculate the r-space contribution to the pressure in internal units
!!!using the free energy route
!!!IN:
!!!   this : the closure object
!!!   gvv  : site-site pair correlation function
!!!   cvv  : site-site direct correlation function
!!!   mtv  : multiplicity of each site
!!!   rhov : number density of each site
!!!   dr   : r-space grid spacing
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  function rism1d_kh_pressureFE(this, gvv, cvv, mtv, rhov, dr) result(pr)
    use constants, only : pi
    implicit none
    type(rism1d_kh), intent(in) :: this
    _REAL_, intent(in) :: gvv(:,:),cvv(:,:),rhov(:),dr
    integer, intent(in) :: mtv(:)
    _REAL_ :: pr
    _REAL_ :: prv
    _REAL_ :: r, hvv, h2c
    integer :: ir,ivv,iv1,iv2, cnt
    !r-space
    pr = 0.d0
    do ir=2,ubound(gvv,1) !nr
       r = (ir-1)*dr
       prv = 0.d0
       ivv = 0
       do iv2=1,ubound(mtv,1) !nv
          do iv1=1,iv2
             ivv = ivv + 1
             !if the sites are different, we need to double count the contribution
             if (iv1 == iv2)  then
                cnt = 1
             else
                cnt = 2
             endif
             hvv = gvv(ir,ivv) - 1.d0
             if (hvv >= 0.d0)  then
                h2c = - cvv(ir,ivv)
             else
                h2c = 0.5d0*hvv*hvv - cvv(ir,ivv)
             endif
             prv = prv + cnt*mtv(iv1)*rhov(iv2)*h2c
          enddo
       enddo
       pr = pr + prv*r**2
    enddo
    pr = pr * 2.d0*pi*dr
  end function rism1d_kh_pressureFE

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!Calculate the r-space contribution to the free energy in internal units
!!!IN:
!!!   this : the closure object
!!!   gvv  : site-site pair correlation function
!!!   cvv  : site-site direct correlation function
!!!   mtv  : multiplicity of each site
!!!   rhov : number density of each site
!!!   dr   : r-space grid spacing
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  function rism1d_kh_freeEnergy(this, gvv, cvv, mtv, rhov, dr) result(fe)
    use constants, only : pi
    implicit none
    type(rism1d_kh), intent(in) :: this
    _REAL_, intent(in) :: gvv(:,:),cvv(:,:),rhov(:),dr
    integer, intent(in) :: mtv(:)
    _REAL_ :: fe
    _REAL_ :: fev
    _REAL_ :: r, hvv, h2c
    integer :: ir,ivv,iv1,iv2, cnt
    !r-space
    fe = 0.d0
    do ir=2,ubound(gvv,1) !nr
       r = (ir-1)*dr
       fev = 0.d0
       ivv = 0
       do iv2=1,ubound(mtv,1) !nv
          do iv1=1,iv2
             ivv = ivv + 1
             !if the sites are different, we need to double count the contribution
             if (iv1 == iv2)  then
                cnt = 1
             else
                cnt = 2
             endif
             hvv = gvv(ir,ivv) - 1.d0
             if (hvv >= 0.d0)  then
                h2c = - cvv(ir,ivv)
             else
                h2c = 0.5d0*hvv*hvv - cvv(ir,ivv)
             endif
             fev = fev + cnt*rhov(iv1)*rhov(iv2)*h2c
          enddo
       enddo
       fe = fe + fev*r**2
    enddo
    fe = -fe * 2.d0*pi*dr
  end function rism1d_kh_freeEnergy

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!Frees memory and resets the KH closure
!!!IN:
!!!   this : KH object
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  subroutine rism1d_kh_destroy(this)
    implicit none
    type(rism1d_kh), intent(inout) :: this
  end subroutine rism1d_kh_destroy
end module rism1d_kh_c
