/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright (C) 2011 OpenFOAM Foundation
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "SprayCloud.H"
#include "AtomizationModel.H"
#include "BreakupModel.H"
#include "StochasticCollisionModel.H"

// * * * * * * * * * * * * * Protected Member Functions  * * * * * * * * * * //

template<class CloudType>
void Foam::SprayCloud<CloudType>::setModels()
{
    atomizationModel_.reset
    (
        AtomizationModel<SprayCloud<CloudType> >::New
        (
            this->subModelProperties(),
            *this
        ).ptr()
    );

    breakupModel_.reset
    (
        BreakupModel<SprayCloud<CloudType> >::New
        (
            this->subModelProperties(),
            *this
        ).ptr()
    );

    stochasticCollisionModel_.reset
    (
        StochasticCollisionModel<SprayCloud<CloudType> >::New
        (
            this->subModelProperties(),
            *this
        ).ptr()
    );
}


template<class CloudType>
void Foam::SprayCloud<CloudType>::cloudReset
(
    SprayCloud<CloudType>& c
)
{
    CloudType::cloudReset(c);

    atomizationModel_.reset(c.atomizationModel_.ptr());
    breakupModel_.reset(c.breakupModel_.ptr());
    stochasticCollisionModel_.reset(c.stochasticCollisionModel_.ptr());
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

template<class CloudType>
Foam::SprayCloud<CloudType>::SprayCloud
(
    const word& cloudName,
    const volScalarField& rho,
    const volVectorField& U,
    const dimensionedVector& g,
    const SLGThermo& thermo,
    bool readFields
)
:
    CloudType(cloudName, rho, U, g, thermo, false),
    sprayCloud(),
    cloudCopyPtr_(NULL),
    averageParcelMass_(0.0),
    atomizationModel_(NULL),
    breakupModel_(NULL),
    stochasticCollisionModel_(NULL)
{
    if (this->solution().active())
    {
        setModels();

        averageParcelMass_ = this->injection().averageParcelMass();

        if (readFields)
        {
            parcelType::readFields(*this, this->composition());
        }

        Info << "Average parcel mass: " << averageParcelMass_ << endl;
    }

    if (this->solution().resetSourcesOnStartup())
    {
        CloudType::resetSourceTerms();
    }
}


template<class CloudType>
Foam::SprayCloud<CloudType>::SprayCloud
(
    SprayCloud<CloudType>& c,
    const word& name
)
:
    CloudType(c, name),
    sprayCloud(),
    cloudCopyPtr_(NULL),
    averageParcelMass_(c.averageParcelMass_),
    atomizationModel_(c.atomizationModel_->clone()),
    breakupModel_(c.breakupModel_->clone()),
    stochasticCollisionModel_(c.stochasticCollisionModel_->clone())
{}


template<class CloudType>
Foam::SprayCloud<CloudType>::SprayCloud
(
    const fvMesh& mesh,
    const word& name,
    const SprayCloud<CloudType>& c
)
:
    CloudType(mesh, name, c),
    sprayCloud(),
    cloudCopyPtr_(NULL),
    averageParcelMass_(0.0),
    atomizationModel_(NULL),
    breakupModel_(NULL),
    stochasticCollisionModel_(NULL)
{}


// * * * * * * * * * * * * * * * * Destructor  * * * * * * * * * * * * * * * //

template<class CloudType>
Foam::SprayCloud<CloudType>::~SprayCloud()
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class CloudType>
void Foam::SprayCloud<CloudType>::setParcelThermoProperties
(
    parcelType& parcel,
    const scalar lagrangianDt
)
{
    CloudType::setParcelThermoProperties(parcel, lagrangianDt);

    const liquidMixtureProperties& liqMix = this->composition().liquids();

    const scalarField& Y(parcel.Y());
    scalarField X(liqMix.X(Y));

    // override rho and Cp from constantProperties
    parcel.Cp() = liqMix.Cp(parcel.pc(), parcel.T(), X);
    parcel.rho() = liqMix.rho(parcel.pc(), parcel.T(), X);
}


template<class CloudType>
void Foam::SprayCloud<CloudType>::checkParcelProperties
(
    parcelType& parcel,
    const scalar lagrangianDt,
    const bool fullyDescribed
)
{
    CloudType::checkParcelProperties(parcel, lagrangianDt, fullyDescribed);

    // store the injection position and initial drop size
    parcel.position0() = parcel.position();
    parcel.d0() = parcel.d();

    parcel.y() = breakup().y0();
    parcel.yDot() = breakup().yDot0();

    parcel.liquidCore() = atomization().initLiquidCore();
}


template<class CloudType>
void Foam::SprayCloud<CloudType>::storeState()
{
    cloudCopyPtr_.reset
    (
        static_cast<SprayCloud<CloudType>*>
        (
            clone(this->name() + "Copy").ptr()
        )
    );
}


template<class CloudType>
void Foam::SprayCloud<CloudType>::restoreState()
{
    cloudReset(cloudCopyPtr_());
    cloudCopyPtr_.clear();
}


template<class CloudType>
void Foam::SprayCloud<CloudType>::evolve()
{
    if (this->solution().canEvolve())
    {
        typename parcelType::template
            TrackingData<SprayCloud<CloudType> > td(*this);

        this->solve(td);
    }
}


template<class CloudType>
template<class TrackData>
void Foam::SprayCloud<CloudType>::motion(TrackData& td)
{
    const scalar dt = this->solution().trackTime();

    td.part() = TrackData::tpLinearTrack;
    CloudType::move(td, dt);

    this->updateCellOccupancy();

    if (stochasticCollision().active())
    {
        const liquidMixtureProperties& liqMix = this->composition().liquids();

        label i = 0;
        forAllIter(typename SprayCloud<CloudType>, *this, iter)
        {
            label j = 0;
            forAllIter(typename SprayCloud<CloudType>, *this, jter)
            {
                if (j > i)
                {
                    parcelType& p = iter();
                    scalar Vi = this->mesh().V()[p.cell()];
                    scalarField X1(liqMix.X(p.Y()));
                    scalar sigma1 = liqMix.sigma(p.pc(), p.T(), X1);
                    scalar mp = p.mass()*p.nParticle();

                    parcelType& q = jter();
                    scalar Vj = this->mesh().V()[q.cell()];
                    scalarField X2(liqMix.X(q.Y()));
                    scalar sigma2 = liqMix.sigma(q.pc(), q.T(), X2);
                    scalar mq = q.mass()*q.nParticle();

                    bool updateProperties = stochasticCollision().update
                    (
                        dt,
                        this->rndGen(),
                        p.position(),
                        mp,
                        p.d(),
                        p.nParticle(),
                        p.U(),
                        p.rho(),
                        p.T(),
                        p.Y(),
                        sigma1,
                        p.cell(),
                        Vi,
                        q.position(),
                        mq,
                        q.d(),
                        q.nParticle(),
                        q.U(),
                        q.rho(),
                        q.T(),
                        q.Y(),
                        sigma2,
                        q.cell(),
                        Vj
                    );

                    // for coalescence we need to update the density and
                    // the diameter cause of the temp/conc/mass-change
                    if (updateProperties)
                    {
                        if (mp > VSMALL)
                        {
                            scalarField Xp(liqMix.X(p.Y()));
                            p.rho() = liqMix.rho(p.pc(), p.T(), Xp);
                            p.Cp() = liqMix.Cp(p.pc(), p.T(), Xp);
                            p.d() =
                                cbrt
                                (
                                    6.0*mp
                                   /(
                                        p.nParticle()
                                       *p.rho()
                                       *constant::mathematical::pi
                                    )
                                );
                        }

                        if (mq > VSMALL)
                        {
                            scalarField Xq(liqMix.X(q.Y()));
                            q.rho() = liqMix.rho(q.pc(), q.T(), Xq);
                            q.Cp() = liqMix.Cp(q.pc(), q.T(), Xq);
                            q.d() =
                                cbrt
                                (
                                    6.0*mq
                                   /(
                                        q.nParticle()
                                       *q.rho()
                                       *constant::mathematical::pi
                                    )
                                );
                        }
                    }
                }
                j++;
            }
            i++;
        }

        // remove coalesced particles (diameter set to 0)
        forAllIter(typename SprayCloud<CloudType>, *this, iter)
        {
            parcelType& p = iter();
            if (p.mass() < VSMALL)
            {
                this->deleteParticle(p);
            }
        }
    }
}


template<class CloudType>
void Foam::SprayCloud<CloudType>::info()
{
    CloudType::info();
    scalar d32 = 1.0e+6*this->Dij(3, 2);
    scalar d10 = 1.0e+6*this->Dij(1, 0);
    scalar dMax = 1.0e+6*this->Dmax();
    scalar pen = this->penetration(0.95);

    Info << "    D10, D32, Dmax (mu)             = " << d10 << ", " << d32
         << ", " << dMax << nl
         << "    Liquid penetration 95% mass (m) = " << pen << endl;
}


// ************************************************************************* //
