/*
 *   SPECjEnterprise2010 - a benchmark for enterprise middleware
 *  Copyright 1995-2010 Standard Performance Evaluation Corporation
 *   All Rights Reserved
 *
 * This source code is provided as is, without any express or implied warranty.
 *
 *  History:
 *  Date        ID, Company               Description
 *  ----------  -----------------------   ---------------------------------------------------------------
 *  2007/10/02  Bernhard Riedhofer, SAP   Created, integration of loader into SPECjAppServer2007 application
 */
package org.spec.jent.loader;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;

/*
 * Provides a few convenience methods for db access and batch execution.
 */
public class DatabaseHelper {

    private static final String CLASSNAME = DatabaseHelper.class.getName();
    private static final Logger logger = Logger.getLogger(CLASSNAME);

    static final int BATCH_SIZE = 100;
    
    public static final String DS_JNDI_NAME_PREFIX = "jdbc/";
    public static final String SPECJ_DS_JNDI_NAME_PREFIX = DS_JNDI_NAME_PREFIX + "SPECj";
    public static final String DS_JNDI_NAME_POSTFIX = "DS";
    public static final String LOADER_DS_JNDI_NAME_POSTFIX = "Loader" + DS_JNDI_NAME_POSTFIX;

    public static final String ORDER_KEY = "Order";
    public static final String MFG_KEY = "Mfg";
    public static final String SUPPLIER_KEY = "Supplier";
    
    public static Connection getConnection(final String key) throws SQLException, NamingException {
        if (ORDER_KEY != key && MFG_KEY != key && SUPPLIER_KEY != key) { 
            throw new IllegalArgumentException("The " + key + " domain is unknown.");
        }
        
        ArrayList<String> dsNames = new ArrayList<String>();
        String dsName = SPECJ_DS_JNDI_NAME_PREFIX + key + LOADER_DS_JNDI_NAME_POSTFIX;
        dsNames.add(dsName);
        dsNames.add(DS_JNDI_NAME_PREFIX + dsName);
        dsName = SPECJ_DS_JNDI_NAME_PREFIX + key + DS_JNDI_NAME_POSTFIX;
        dsNames.add(dsName);
        dsNames.add(DS_JNDI_NAME_PREFIX + dsName);

        DataSource dataSource = null;
        for (String name : dsNames) {
            try {
                logger.log(Level.FINE, "Looking up data source " + name + " for loading SPECjEnterprise database.");
                dataSource = (DataSource) (new InitialContext()).lookup(name);
                return getConnection(dataSource);
            } catch (NamingException e) {
                logger.log(Level.FINE, "Lookup of data source " + name + " for loading SPECjEnterprise database failed.", e);
            }
        }

        // lookups failed
        StringBuilder names = new StringBuilder();
        for (String name : dsNames) {
            if (names.length() != 0) {
                names.append(", ");
            }
            names.append(name);
        }
        throw new NamingException("Cannot lookup data source for loading SPECjEnterprise database, tried the following names: " + names);
    }

    public static Connection getConnection(final DataSource dataSource) throws SQLException {
        final Connection connection = dataSource.getConnection();
        connection.setAutoCommit(false);
        return connection;
    }

    static void clean(final Connection dbConnection, final String tableName) throws SQLException,
            NamingException {
        try {
            final Statement deleteStatment = dbConnection.createStatement();
            try {
                deleteStatment.executeUpdate("truncate table " + tableName);
            } catch (SQLException e) {
                deleteStatment.executeUpdate("delete from " + tableName);
            } finally {
                deleteStatment.close();
            }
            dbConnection.commit();
        } finally {
            dbConnection.close();
        }
    }

    static void loadSequence(final Connection dbConnection, final String id, final long nextSeq) throws SQLException,
            NamingException {
        try {
            final Statement deleteStatement = dbConnection.createStatement();
            try {
                deleteStatement.executeUpdate("delete from U_SEQUENCES where S_ID = '" + id + "'");
            } finally {
                deleteStatement.close();
            }
            dbConnection.commit();
            final PreparedStatement cs = dbConnection
                    .prepareStatement("insert into U_SEQUENCES (S_ID, S_NEXTNUM) values (?,?)");
            try {
                cs.setString(1, id);
                cs.setLong(2, nextSeq);
                cs.executeUpdate();
            } finally {
                cs.close();
            }
            dbConnection.commit();
        } finally {
            dbConnection.close();
        }
    }

    static void executeBatch(final Connection dbConnection, final int count, final PreparedStatement ... pss) throws SQLException {
        if (count != 0) {
            for (PreparedStatement ps : pss) {
                ps.executeBatch();
            }
            dbConnection.commit();
        }
    }

    static int executeBatchIfFull(final Connection dbConnection, int count, final PreparedStatement ... pss) throws SQLException {
        if (count >= BATCH_SIZE) {
            executeBatch(dbConnection, count, pss);
            count = 0;
        }
        return count;
    }        

    static int addAndExecuteBatchIfFull(final Connection dbConnection, int count, final PreparedStatement ps) throws SQLException {
        ps.addBatch();
        count++;
        return executeBatchIfFull(dbConnection, count, ps);
    }        
}
