#!/usr/bin/python3

# Copyright 2020 VMware, Inc.
# All rights reserved. -- VMware Confidential

'''
    This python CLI can be used to introspect the contents of ConfigStore in
    the support bundle. It provides command to get current config, desired
    configs and default configs. It does not have any dependencies and can be
    run on any system with python.
'''

import argparse
import json
import os
import sqlite3
import sys

dbFileName = os.path.join(os.path.dirname(__file__),
    "./../etc/vmware/configstore/current-store-1")

groupHelpStr = "Value of group identifying the Configuration, e.g (system)"
keyHelpStr = " Value of key identifying the Configuration, e.g (keyboard)"
componentHelpStr = "Value of component identifying a Configuration, e.g (esx)"
idHelpStr = ("Value of instanceId identifying a specific instance of the  "
    "Configuration")

positionalsStr = "Available namespaces"

configHelpStr = "Operations pertaining to the %s configuration of the host."
defaultHelpStr = configHelpStr % "default"
currentHelpStr = configHelpStr % "current"
desiredHelpStr = configHelpStr % "desired"

GetHelpStr = ("Get Configuration for a specific object identified by "
                        "component, group, key and/or instanceid.")

configNsHelpStr = "Operations pertaining to the Configuration of the host."


def getFilter(params):
    if len(params) == 4:
        return ("Component = ? and Name = ? and ConfigGroup = ? and "
            "Identifier = ?")
    elif len(params) == 3:
        return "Component = ? and Name = ? and ConfigGroup = ? "
    else:
        return None

def setPathToDatabase():
    global dbFileName
    variableName = "CONFIG_STORE_DB_DIRECTORY"
    if variableName in os.environ:
        dbFileName = os.path.join(os.environ[variableName], "current-store-1")
    try:
        with sqlite3.connect(dbFileName) as conn:
            return
    except sqlite3.OperationalError:
        if variableName in os.environ:
            print("Error: Cannot open database file \"%s\" " % dbFileName)
        else:
            print("Error: ConfigStore was not be found. Please set "
                "environment variable CONFIG_STORE_DB_DIRECTORY to the "
                "directory of the database file before retrying the CLI.")
        sys.exit(1);

def NoneToEmptyDictStr(param):
    return param if param is not None else "{}"

def areAllNone(tupleParam):
    for item in tupleParam:
        if item is not None:
            return False
    return True

def printResultFormatted(result, params, isIdEmptyStr):
    if hasattr(params,"i") or isIdEmptyStr:
        print(json.dumps(result[0], indent = 3))
    else:
        print(json.dumps(result, indent = 3))

def getErrorStr(params):
    if len(params) == 3:
        return ("Configuration not found for component %s , key %s and"
            " group %s" % params)
    elif len(params) == 4:
        return ("Configuration not found for component %s , key %s ,"
            " group %s and id %s ." % params)

def ReadDatabase(query, params):
    listResultSQL = []
    with sqlite3.connect(dbFileName) as conn:
        cursor = conn.cursor()
        cursor.execute(query, params)
        listResultSQL = cursor.fetchall()

    if len(listResultSQL) == 0:
        print(getErrorStr(params))
        sys.exit(1)

    return listResultSQL

def GetCurrentConfig(params):
    '''
        Get the current configuration. Auto, User, Vital and Cached are merged
        together to produce the result. User overrides auto if present.
    '''

    sqlQuery = """
        SELECT AutoConfValue, UserValue, VitalValue, CachedValue, Identifier
        FROM Config
        WHERE %s and Success = 1 ;
        """ % getFilter(params)

    listResultSQL = ReadDatabase(sqlQuery,params)
    result = []
    colToIndex = {
        "AutoConfValue" : 0,
        "UserValue" : 1,
        "VitalValue" : 2,
        "CachedValue" : 3
    }
    isIdEmptyStr = True
    for rowTuple in listResultSQL:
        dataMerged = {}
        if rowTuple[-1] != "": #if Identifier is not an empty string
            isIdEmptyStr = False
        #check AutoConfValue, UserValue, VitalValue, CachedValue
        if areAllNone(rowTuple[:-1]):
            print(getErrorStr(params))
            sys.exit(1)
        for colName in colToIndex:
            dataMerged.update(
                json.loads(
                    NoneToEmptyDictStr(rowTuple[colToIndex[colName]])))
        result.append(dataMerged)
    printResultFormatted(result, params, isIdEmptyStr)

def GetDesiredConfig(params):
    '''
        Get the desired configuration.
    '''

    sqlQuery = """
         SELECT Version,
            CASE Success
                WHEN 0 THEN UserValue
                ELSE DesiredValue END
         as Value, Identifier
         FROM Config
         WHERE %s
         and Value is not NULL ;
         """ % getFilter(params)

    listResultSQL = ReadDatabase(sqlQuery,params)
    result = []
    isIdEmptyStr = True
    for rowTuple in listResultSQL:
        if rowTuple[1] != "":
            isIdEmptyStr = False
        # no nulls here, guaranteed from the query
        result.append(json.loads(rowTuple[0]))
    printResultFormatted(result, params, isIdEmptyStr)

def GetDefaultConfig(params):
    '''
        Get the default configuration.
    '''

    sqlQuery = """
         SELECT Value, Identifier
         FROM DefaultConfig
         WHERE %s ;
         """ % getFilter(params)

    listResultSQL = ReadDatabase(sqlQuery,params)
    result = []
    isIdEmptyStr = True
    for rowTuple in listResultSQL:
        if rowTuple[1] != "":
            isIdEmptyStr = False
        if rowTuple[0] is None:
            print(getErrorStr(params))
            sys.exit(1)
        result.append(json.loads(rowTuple[0]))
    printResultFormatted(result, params, isIdEmptyStr)

def addReqiredAndOptionalArgumentsToParser(parser):

    required = parser.add_argument_group("required arguments")

    required.add_argument("-g", metavar="group", help=groupHelpStr, \
        required=True)

    required.add_argument("-k", metavar="key", help=keyHelpStr, \
        required=True)

    required.add_argument("-c", metavar="component", \
        help=componentHelpStr, required=True)

    parser.add_argument("-i", metavar="id", help=idHelpStr, \
        required=False)

def createCustomParser():
    mainParser = argparse.ArgumentParser()
    mainParser._positionals.title = positionalsStr

    mainSubparser = mainParser.add_subparsers(dest="main")

    configParser = mainSubparser.add_parser("config", help=configNsHelpStr)

    configParser._positionals.title = positionalsStr

    configSubparser = configParser.add_subparsers(dest="config")

    defaultParser = configSubparser.add_parser("default", help=defaultHelpStr)
    defaultParser._positionals.title = positionalsStr
    defaultSubparser = defaultParser.add_subparsers(dest="get")
    defaultGetParser = defaultSubparser.add_parser("get",
        help=GetHelpStr)
    addReqiredAndOptionalArgumentsToParser(defaultGetParser)
    defaultGetParser.set_defaults(func=GetDefaultConfig)

    currentParser = configSubparser.add_parser("current", help=currentHelpStr)
    currentParser._positionals.title = positionalsStr
    currentSubparser = currentParser.add_subparsers(dest="get")
    currentGetParser = currentSubparser.add_parser("get",
        help=GetHelpStr)
    addReqiredAndOptionalArgumentsToParser(currentGetParser)
    currentGetParser.set_defaults(func=GetCurrentConfig)

    desiredParser = configSubparser.add_parser("desired", help=desiredHelpStr)
    desiredParser._positionals.title = positionalsStr
    desiredSubparser = desiredParser.add_subparsers(dest="get")
    desiredGetParser = desiredSubparser.add_parser("get",
        help=GetHelpStr)
    addReqiredAndOptionalArgumentsToParser(desiredGetParser)
    desiredGetParser.set_defaults(func=GetDesiredConfig)

    return mainParser

def makeTuple(args):
    if args.i is None:
        return (args.c, args.k, args.g)
    return (args.c, args.k, args.g, args.i)

if __name__ == '__main__':

    setPathToDatabase()
    parser = createCustomParser()
    args = parser.parse_args()
    if hasattr(args,"func"):
        args.func(makeTuple(args))
    else:
        print('Error: Namespace/Command not provided.')
        parser.print_help();
        sys.exit(1)
