#!/usr/bin/env python

#######################################################################
# Copyright (C) 2018-2020 VMWare, Inc.
# All Rights Reserved
#######################################################################

import os
import random
import string
import sys
import time
import socket
import itertools
import json
import threading
import shlex
import subprocess
from syslog import syslog, LOG_ERR, LOG_ALERT, LOG_WARNING

import pyVmomi
from pyVmomi import vim, vmodl
from pyVim import host
from pyVim.connect import SmartConnect, SetSi, Disconnect
from pyVim.account import CreateUser, GetAuthorizationManager, RemoveUser

from vmware.vapi.bindings.stub import ApiClient
from vmware.vapi.lib.connect import get_connector
from vmware.vapi.security.oauth import create_oauth_security_context
from vmware.vapi.security.user_password import create_user_password_security_context
from vmware.vapi.stdlib.client.factories import StubConfigurationFactory
import com.vmware.esx.authentication_client as authentication_client

# __ssl_hack__
import ssl
if sys.version_info >= (2,7,9):
    if (hasattr(ssl, '_create_unverified_context') and
        hasattr(ssl, '_create_default_https_context')):
        ssl._create_default_https_context = ssl._create_unverified_context

try:
    import httplib
except ImportError:
    # __python3_hack__
    import http.client as httplib

MAX_SLEEP = 5 * 60

def _getServiceInstance():
    syslog('trying to connect to host agent')

    SLEEP_TIME_HOST_AGENT = 0
    while True:
        try:
            try:
                si = SmartConnect(user='vpxuser')
            except:
                si = SmartConnect()
            return si
        except Exception as e:
            SLEEP_TIME_HOST_AGENT += 10

            if SLEEP_TIME_HOST_AGENT > MAX_SLEEP:
                syslog(
                    LOG_ERR,
                    'waiter-notify failed to connect to host agent, '
                    'error -- %s, exitting.' % e)
                raise SystemExit(1)

            syslog(
                LOG_ERR,
                'waiter-notify could not connect to host agent, error -- %s, '
                'retrying in %s seconds.' % (e, SLEEP_TIME_HOST_AGENT))

            time.sleep(SLEEP_TIME_HOST_AGENT)

def _exitIfStatefulInstalIsScheduledInStatelessMode(hostSystem):
    syslog('retrieving host state')

    isPxeBooted = False
    isStatefulInstall = False
    isInMaintenanceMode = True
    try:
        advancedOptionsManager = hostSystem.configManager.advancedOption

        try:
            isPxeBooted = advancedOptionsManager.QueryView(
                'UserVars.PXEBootEnabled')[0].value
        except vim.InvalidName:
            pass

        try:
            isStatefulInstall = advancedOptionsManager.QueryView(
                'UserVars.StatefulInstallScheduled')[0].value
        except vim.InvalidName:
            pass

        isInMaintenanceMode = hostSystem.summary.runtime.inMaintenanceMode
    except Exception as e:
        syslog(
            LOG_ERR,
            'waiternotify failed to retrieve host state, error -- %s.' % e)

    if isPxeBooted and isStatefulInstall and not isInMaintenanceMode:
        # If we have booted as a staless host but a stateful installation had
        # been scheduled and it succeeded (i.e. we are not in maintenance mode)
        # skip the host up event as it will be triggered again when we boot
        # from the device on which ESXi is installed.
        syslog(
            'Skipping host up notification because of the following values:'
            ' isPxeBooted = %s, isStatefulInstall = %s, isInMaintenanceMode'
            ' = %s' % (isPxeBooted, isStatefulInstall, isInMaintenanceMode))
        raise SystemExit

def _createWaiterUser(hostSystem, si, tmpUserPassAddHost):
    try:
        # Create "waiter" user,
        # update "vpxuser" password, needed to create security context for "vpxuser"
        syslog('Creating security context for "vpxuser"')
        accountSpec = vim.Host.LocalAccountManager.PosixAccountSpecification()
        accountSpec.SetId('vpxuser')
        # Retry updating "vpxuser" and "waiter" temp passwords, see PR2375427
        retries = 2
        while retries > 0:
            tmpPass = ''.join(random.sample(string.punctuation+string.ascii_letters, 20))
            try:
                accountSpec.SetPassword(tmpPass)
                si.content.accountManager.UpdateUser(accountSpec)
                CreateUser(name='waiter', password=tmpPass, description='Autodeploy user')
                break
            except vmodl.fault.SystemError as e:
                if "Authentication token manipulation error" in e.reason:
                    syslog(LOG_ERR, 'waiternotify failed to update vpxuser password '
                                    'or create waiter user, error -- %s. Retrying.' % e)
                    retries = retries - 1
                else:
                    syslog(LOG_ERR, 'waiternotify failed to update vpxuser password '
                                    'or create waiter user, error -- %s.' % e)
                    raise e

        # Create security context for "vpxuser",
        # needed for the client profile for "waiter" user after that
        connector = get_connector('http', 'json', url='http://localhost/token')
        stubConfig = StubConfigurationFactory.new_std_configuration(connector)
        connector.set_security_context(
            create_user_password_security_context('vpxuser', tmpPass))
        client = ApiClient(authentication_client.StubFactory(stubConfig))
        jwt = client.Token.create()

        # Create Client Profile and set entitlements for the 'waiter' user
        syslog('Creating client profile for user "waiter"')
        connector.set_security_context(create_oauth_security_context(jwt.access_token))
        clientProfile = client.ClientProfiles
        spec = clientProfile.CreateSpec(local_user_name='waiter', grants=[
           clientProfile.AccessGrant(clientProfile.ResourceType.ENTITLEMENT,
                                     clientProfile.Entitlement.IDENTITY_MGMT)])
        cpid = clientProfile.create(spec)

        hostAccessManager = hostSystem.configManager.hostAccessManager
        if hostAccessManager.GetLockdownMode() != 'lockdownDisabled':
            # During Lockdown mode, to use SetEntityPermissions here for
            # the 'waiter' user, and after that, for the addhost task from
            # the VC to the host, in ConfigureVimAccountHostd for the
            # 'vpxuser', we need to have them both in the
            # LockdownExceptions list. We remove these users from the list
            # when the host is added.

            lockdownUsers = hostAccessManager.QueryLockdownExceptions()
            lockdownUsers.append('waiter')
            lockdownUsers.append('vpxuser')
            hostAccessManager.UpdateLockdownExceptions(lockdownUsers)

        userPermission = vim.AuthorizationManager.Permission(
            principal='waiter',
            group=False,
            roleId=-1,
            propagate=True)
        authManager = GetAuthorizationManager()
        rootFolder = si.RetrieveContent().GetRootFolder()
        authManager.SetEntityPermissions(
            entity=rootFolder,
            permission=[userPermission])

        # Update 'waiter' password, stored encoded in
        # /etc/vmware/autodeploy/waiterNotify.json
        updatePasswdCmd = (
            'sed -i -e \'s;waiter:[^:]*:;waiter:%s:;\' /etc/shadow\n'
            % tmpUserPassAddHost)

        proc = subprocess.Popen(
            shlex.split(updatePasswdCmd),
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE)

        stdout, stderr = proc.communicate()
        if proc.returncode:
            raise Exception('waiter user password was not updated, '
                            'error -- %s.' % stderr)
    except Exception as e:
        syslog(
            LOG_ERR,
            'waiternotify failed to create waiter user, error -- %s.' % e)
    return clientProfile, cpid

def _notifyAutoDeploy(deployHostList, hostID):
    syslog('notifying autodeploy at {}'.format(deployHostList))

    SLEEP_TIME = 10
    AUTODEPLOY_CONTACTED = False
    try:
        # Try forever to signal that we are up.  We want to be resilient in
        # case the waiter or VC is down.
        deployHostIndex = 0
        while True:
            conn = None
            if SLEEP_TIME < MAX_SLEEP:
                try:
                    SLEEP_TIME += random.randint(8, 12)
                except:
                    syslog('failed to generate random number.')
                    SLEEP_TIME += 10

            try:
                conn = httplib.HTTPSConnection(
                    deployHostList[deployHostIndex],
                    timeout=30)

                conn.request('POST', '/vmw/rbd/host/{}/up'.format(hostID))
                response = conn.getresponse()

                if response.status == httplib.SERVICE_UNAVAILABLE:
                    if AUTODEPLOY_CONTACTED:
                        syslog('autodeploy successfully notified -- '
                               'add-host in progress -- '
                               'retrying in %s seconds' % SLEEP_TIME)
                    else:
                        AUTODEPLOY_CONTACTED = True
                        syslog('autodeploy successfully notified -- '
                               'add-host started -- '
                               'retrying in %s seconds' % SLEEP_TIME)
                else:
                    syslog('autodeploy notify response -- %s %s'
                        % (response.status, response.reason))

                    if response.status == httplib.OK:
                        syslog('autodeploy successfully notified -- '
                               'add-host finished.')
                        break
                    if response.status == httplib.NOT_FOUND:
                        syslog('autodeploy does not know about this host')
                        break
                    if response.status == httplib.UNAUTHORIZED:
                        syslog('autodeploy does not have valid credentials '
                               'for this host.')
                        break
                    if response.status == httplib.CONFLICT:
                        syslog('autodeploy could not add the host to vCenter.')
                        break
                    if response.status == httplib.GONE:
                        syslog('autodeploy could not find the addhost/reconnect'
                               ' task.')
                        break
            except socket.error as e:
                # Unable to reach autodeploy on this address, try another...
                syslog(
                    LOG_ERR,
                    'could not connect to autodeploy at %s: %s'
                        % (deployHostList[deployHostIndex], e))
                deployHostIndex = (deployHostIndex + 1) % len(deployHostList)
            finally:
                if conn:
                    conn.close()

            time.sleep(SLEEP_TIME)
    except Exception as e:
        syslog(LOG_ERR, 'autodeploy notify error -- %s' % e)

def _removeWaiterUser(hostSystem, clientProfile, cpid):
    try:
        hostAccessManager = hostSystem.configManager.hostAccessManager

        if hostAccessManager.GetLockdownMode() != 'lockdownDisabled':
            # Remove 'waiter' and 'vpxuser' from the LockdownExceptions list.
            lockdownUsers = hostAccessManager.QueryLockdownExceptions()
            if 'waiter' in lockdownUsers: lockdownUsers.remove('waiter')
            if 'vpxuser' in lockdownUsers: lockdownUsers.remove('vpxuser')
            hostAccessManager.UpdateLockdownExceptions(lockdownUsers)

        clientProfile.delete(cpid)
        RemoveUser('waiter')
    except Exception as e:
        syslog(
            LOG_ERR,
            'waiternotify failed to remove waiter user, error -- %s.' % e)

def main(deployHostList, hostID, tmpUserPassAddHost=None):

    try:
        random.seed()
    except:
        syslog('failed to initialize random number generator.')

    si = _getServiceInstance()
    hostSystem = host.GetHostSystem(si)

    _exitIfStatefulInstalIsScheduledInStatelessMode(hostSystem)

    clientProfile, cpid = _createWaiterUser(hostSystem, si, tmpUserPassAddHost)
    _notifyAutoDeploy(deployHostList, hostID)
    _removeWaiterUser(hostSystem, clientProfile, cpid)
    Disconnect(si)

if __name__ == '__main__':
    configFilePath = '/etc/vmware/autodeploy/waiterNotify.json'

    if os.path.isfile(configFilePath):
        with open(configFilePath, 'r') as cfg:
            config = json.load(cfg)

        options = ['deployHostList', 'hostID']
        options.append('waiter')

        for option in options:
            if not option in config:
                syslog(
                    LOG_ALERT,
                    'Could not find \'{}\' in {}.'
                        .format(option, configFilePath))
                raise SystemExit(1)

        ret = os.fork()

        if ret == 0:
            main(
                config['deployHostList'],
                config['hostID'],
                config['waiter'])
        elif ret < 0:
            syslog(LOG_ALERT, 'Failed to fork()')
