#!/usr/bin/python
########################################################################
# Copyright 2020 VMware, Inc.  All rights reserved.
# -- VMware Confidential
########################################################################
"""
   This script does post boot scrubbing of configstore for the following:
   1. NGUID devices which were missing during upgrade and their configuration is to be added to configstore. Also an opportunity to mark "need_update" true after upgrade.
"""
import vmkctl
from syslog import syslog, LOG_DEBUG, LOG_ERR
from esxutils import runCli
import os, sys, json, argparse
from esxutils import runCli
from syslog import *

################################################################################
# Prints to stdout and logs message into syslog.log file.
################################################################################
def print_log(level, message):
    print(str(message))
    if level == 0:
        syslog(LOG_ERR, str(message))
    elif level == 1:
        syslog(LOG_INFO, str(message))
    return

################################################################################
# Returns True if the config option is set for Nvme NGUID support.
################################################################################
def is_nvme_nguid_supported():
    cmd = "system settings advanced list -o /Scsi/EnableNVMeNGUIDSupport"
    res = execute_cli(cmd)
    if res != -1:
        res = json.loads(res)
        if res[0]['Int Value'] == 1:
            return True
    return False

################################################################################
# Returns True if the PSA configstore feature is enabled
################################################################################
def psaconfigstore_enabled():
    if not hasattr(psaconfigstore_enabled, "fss"):
        psaconfigstore_enabled.fss = None
    try:
        if psaconfigstore_enabled.fss is None:
            featureState.init(False)
            psaconfigstore_enabled.fss = featureState.ConfigStoreStage2 \
                                    and featureState.CoreStorage_ConfigStore
    except:
        log.warning("Except: PSA configstore feature state is not\
                             available")
        # Setting psaconfigstore_enabled.fss to true since it is an
        # enabled FSS
        psaconfigstore_enabled.fss = True
    return psaconfigstore_enabled.fss

################################################################################
# Runs a cli and returns the o/p on success and  -1 on failure.
################################################################################
def execute_cli(cmd):
    print_log(1, "Running the cli: " + cmd)
    try:
        res = runCli(cmd.split(), True)
        return json.dumps(res)
    except Exception as e:
        print_log(0, "Error running the cli: " + cmd)
        print_log(0, e)
        return -1

if __name__ == "__main__":
   try:
      # PSA post boot scrubbing.
      isNguidSupported = is_nvme_nguid_supported()
      storageInfo = vmkctl.StorageInfoImpl()
      if isNguidSupported:
         storageInfo.PsaPostBootScrub()
         sys.exit(0)
      else:
         print_log(1, "The config option EnableNVMeNGUIDSupport is not enabled")
         storageInfo.PsaHandleMissingDeviceConfig()

   except Exception as e:
      syslog(LOG_ERR, "Script failed to run. %s" % e)
      sys.exit(1)
