#!/usr/bin/perl -w
#
#  PowerOffAllTiles.pl
#
#  Shuts down all SVDC Workload VMs for SPECvirt Datacenter 2021
#
#  Returns 0 (SUCCESS) to PowerOffAllTiles.sh
#  or returns count of SVDC VM that did not shutdown
#
#  This script gets the names of all VMs matching ^svdc-t that are
#  connected and powered on the online hosts belonging to the SUT cluster.
#
#  ./PowerOffAllTiles.pl --server $mgmtServerIP --username $virtUser 
#    --password='[admin-passwd]' --cluster=[SUT-cluster] 
##
## Script used as starting point: VmHostDatastoreReport-Fast.pl
## Sample script to illustrate the performance of an optimized 
## VMware API entity and property collection.
## Created by Reuben Stump (rstump@vmware.com | http://www.virtuin.com)
##
 
use strict;
use warnings;
use 5.010;
 
use VMware::VILib; 
use VMware::VIRuntime;

my %opts = (
   cluster => {
      type => "=s",
      help => "Name of SUT cluster ",
      required => 1,
   },
);

my %h_ds_seen =();
my %seen =();

Opts::add_options(%opts); 
Opts::parse();
Opts::validate();
 
Util::connect();


my $cluster = Opts::get_option('cluster');
my ($cluster_view,$clusterhost_views,$vms,);

$cluster_view = Vim::find_entity_view(view_type => 'ClusterComputeResource', filter => { name => $cluster});

unless($cluster_view) {
  die "Unable to locate cluster name: \"$cluster\"!";
}


###
$clusterhost_views = Vim::get_views(mo_ref_array => $cluster_view->host, properties => ['name', 'runtime' ]);
my $i=0;
my $count=0;
my $VMcount=0;
my $Offcount=0;
my $sleepcycles=0;
my $vm_views;
my @clusterhostlist = ();

# Get list of Hosts on SUT cluster that are not in Maintenance Mode

my $clusterHostsOnline = scalar (@$clusterhost_views);
foreach(@$clusterhost_views) {
    $i++;
    ##print "Debug: SUT.Host.name[".$i."]=" . $_->name . "\n";
    if ($_->runtime->connectionState->val eq "connected") {
      if ($_->runtime->inMaintenanceMode) {
        $clusterHostsOnline--;
      } else {
        push (@clusterhostlist,$_->name);
      }
    }
}

unless($clusterHostsOnline >= 1) {
  die "SUT hosts in cluster: \"$cluster\" are offline!";
}

if ($clusterHostsOnline >= 1) {
# Fetch all VirtualMachines from Mgmtserver, limiting the property set
  $vm_views = Vim::find_entity_views(view_type => "VirtualMachine",
     properties => ['name', 'runtime.powerState', 'runtime.connectionState', 'runtime.host']) || 
     die "Failed to get VirtualMachines: $!";
 
# Fetch all HostSystems from Mgmtserver, limiting the property set
   my $host_views = Vim::find_entity_views(view_type => "HostSystem",
          properties => ['name']) ||
      die "Failed to get HostSystems: $!";

# Create hash tables with key = entity.mo_ref.value            
  my %host_map = map { $_->get_property('mo_ref.value') => $_ } @{ $host_views || [] };
 
  my $vmname;

# Enumerate VirtualMachines in SUT Cluster
  foreach my $vm ( @{$vm_views || []} ) {
   # Get HostSystem from the host map
   my $host_ref = $vm->get_property('runtime.host')->{'value'};
   my $host = $host_map{$host_ref};
   my $hostname = $host->get_property('name');
   $vmname = $vm->get_property('name');

   # Check that VM's host in the cluster host list and 
   # the name of VM starts with svdc-t and 3 digits and
   # VM is connected and powered on

   if (grep { $host->get_property('name') eq $_ } @clusterhostlist) {
     if ( $vm->get_property('name') =~ /^svdc-t\d\d\d-/ ) {
       $VMcount++;
       if ( $vm->get_property('runtime.powerState.val') eq 'poweredOn' && 
            $vm->get_property('runtime.connectionState.val') eq 'connected') {
         $count++;
         $vm->get_property('runtime.host.name');
         ##print "Debug: SUT.VM.name[".$count."]=".  $vm->get_property('name') . "\n";
         eval {
            $vm->ShutdownGuest();
            Util::trace (0,  "\nGuest $vmname on $hostname shutdown ");
         };
         if ($@) {
           if (ref($@) eq 'SoapFault') {
             Util::trace (0, "\nError in '" . $vmname . "' under host $hostname ");
             if (ref($@->detail) eq 'InvalidState') {
                Util::trace(0,"Current State of the "
                      ." VM is not supported for this operation \n");
             }
             elsif (ref($@->detail) eq 'InvalidPowerState') {
                Util::trace(0, "The attempted operation".
                          " cannot be performed in the current state \n" );
             }
             elsif (ref($@->detail) eq 'NotSupported') {
                Util::trace(0, "The operation is not supported on the object \n");
             }
             elsif(ref($@->detail) eq 'ToolsUnavailable') {
                Util::trace(0,"VMTools are not running in this VM \n");
                eval {
                   $vm->PowerOffVM();
                   Util::trace (0,  "\nGuest $vmname on $hostname force poweroff ");
                };
             }
             else {
               Util::trace(0, "VM '"  .$vmname. "' can't be shutdown \n"
                      . $@ . "" );
             }
          }
          else {
             Util::trace(0, "VM '"  .$vmname. "' can't be shutdown \n"
                      . $@ . "" );
          }
        }
	 
       }
     }
   }
  } #end foreach
#
# Check that all benchmark VMs have reached powered-off state
# before returning.
#


  while (1) {
    $vm_views = Vim::find_entity_views(view_type => "VirtualMachine",
     properties => ['name', 'runtime.powerState', 'runtime.connectionState', 'runtime.host']) || 
     die "Failed to get VirtualMachines: $!";
    foreach my $bvm ( @{$vm_views || []} ) {
     # Get HostSystem from the host map
     my $bhost_ref = $bvm->get_property('runtime.host')->{'value'};
     my $bhost = $host_map{$bhost_ref};
     my $bhostname = $bhost->get_property('name');
     $vmname = $bvm->get_property('name');

     # Check that VM's host in the cluster host list and 
     # the name of VM starts with svdc-t and 3 digits and
     # VM is powered off
  
     if (grep { $bhost->get_property('name') eq $_ } @clusterhostlist) {
       if ( $bvm->get_property('name') =~ /^svdc-t\d\d\d-/ ) {
         if ( $bvm->get_property('runtime.powerState.val') eq 'poweredOff') {
           $Offcount++;
         }
	 ##print "Debug: Benchmark VMs: " . $bvm->get_property('name') . " state: " . $bvm->get_property('runtime.powerState.val') . "\n";
       }
     }
    } #end foreach
    ##print "Debug: Offcount = " . $Offcount . " VMcount = " . $VMcount . "\n";
    if ( $VMcount == $Offcount ) {
      last;
    } elsif ( $sleepcycles > $VMcount*3 ) {
      last;
    } else {
      sleep 5;
      $sleepcycles++;
      $Offcount = 0;
    }
  }


} 


 
# Disable SSL hostname verification for vCenter self-signed certificate
BEGIN {
 $ENV{PERL_LWP_SSL_VERIFY_HOSTNAME} = 0;
}

Util::disconnect();

print "\nTotal Benchmark VMs: " .  $VMcount . ". VM's powered-off: ". $Offcount . "\n";
  if (  $VMcount == $Offcount ) {
    exit 0;  
  } else {
    exit ( $VMcount - $Offcount ); 
  }
