#!/usr/bin/perl -w
#
###########################################################################
#                                                                         #
#  Copyright (c) 2021 Standard Performance Evaluation Corporation (SPEC). #
#  All rights reserved.                                                   #
#                                                                         #
###########################################################################

# vSphere Event Collector for SPECvirt Datacenter 2021
# 
#  This script collects selected events needed for the test validaton.  
##
## Script used as starting point: vmeventmgr.pl
#
# Copyright 2007 RapidApp.  All rights reserved.
# Shawn Kearney <skearney@rapidapp.com> 
# 
# VMware Event Manager
#
# VMware Event Manager allows users to "subscribe" to specific VI3 events
# and create reports with those events from specific time periods.  The
# reports can be printed to screen, appended to a file, or emailed to a 
# list of addresses using an open SMTP server.
#
# VMware Event Manager is written in perl utilizing the VIperltoolkit
# for the VMware VI3 SDK.  Information on the VIperltoolkit can be found
# here:
# http://sourceforge.net/projects/viperltoolkit
#
# For instructions, run program with no options
#
# VERSION HISTORY
# v1.0	2007-01-08	Initial Release
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
###############################################################################

use strict;
use warnings;
use 5.010;
use Getopt::Long;
use VMware::VIRuntime;
use DateTime;
use DateTime::Format::ISO8601;
use Mail::Mailer;
use IO::File;

#use diagnostics;
use open qw/ :std :encoding(utf8) /;

$Util::script_version = "1.0";

$Data::Dumper::Sortkeys = 1; #Sort the keys in the output
$Data::Dumper::Deepcopy = 0; #Enable deep copies of structures
$Data::Dumper::Indent = 2;   #Output in a reasonable style (but no array indexes)
$Getopt::Long::order = $PERMUTE;

my %opts = (
   server => {
      type => "=s",
      help => "VC Management server address",
      required => 1,
   },
   service_url => {
      type => "=s",
      help => "SDK service URL, typically https://<VC Server>/sdk",
      required => 0,
   },
   categories => {
      type => "=s",
      help => "Optional: Name of file with VM event categories list",
      required => 0,
      default => "",
   },
   testbegin => {
      type => "=s",
      help => "Date and Time test began, format: yyyy-mm-ddThh:mm:ss (UTC)",
      required => 1,
   },
   testend => {
      type => "=s",
      help => "Date and Time test ended, format: yyyy-mm-ddThh:mm:ss (UTC)",
      required => 1,
   },
   max => {
      type => "=i",
      help => "Optional: Max page size ( <= 1000 ) ",
      required => 0,
      default => 1000,
   },
   diskaddfilter => {
      type => "=i",
      help => "Optional: Enable (1) capture only Add Hard Disk VmReconfiguredEvents  ",
      required => 0,
      default => 1,
   },
   debug => {
      type => "=i",
      help => "Optional: debug 0|1 ",
      required => 0,
      default => 0,
   },
   usage => {
      type => "",
      help => "Optional: Usage. ",
      required => 0,
      default => 0,
   },
);

Opts::add_options(%opts);
Opts::parse();
Opts::validate();

###Util::connect();

my $service_url = Opts::get_option('service_url');
my $server = Opts::get_option('server');
my $userid = Opts::get_option('username');
#print STDERR "Here 1 \n";
my $password = Opts::get_option('password');
#print STDERR "Here 2 \n";
my $testBegin = Opts::get_option('testbegin');
my $testEnd = Opts::get_option('testend');
my $max = Opts::get_option('max');
my $categories = Opts::get_option('categories');
#print STDERR "Here 3 \n";
my @defaultEvents=(
 'DrsVmMigratedEvent',
 'DrsVmPoweredOnEvent',
 'EnteredMaintenanceModeEvent',
 'ExitMaintenanceModeEvent',
 'VmCloneFailedEvent',
 'VmConnectedEvent',
 'VmGuestShutdownEvent',
 'VmMacChangedEvent',
 'VmMessageEvent',
 'VmMigratedEvent',
 'VmPoweredOffEvent',
 'VmPoweredOnEvent',
 'VmReconfiguredEvent',
 'VmRegisteredEvent',
 'VmRemovedEvent',
 'VmRenamedEvent',
 'VmResourcePoolMovedEvent');
my $diskaddfilter = Opts::get_option('diskaddfilter');
my $usage = Opts::get_option('usage');
my $debug = Opts::get_option('debug');
if ( $usage ) {
   UsageOut();
   exit (1);
}


# set the begin and end time to look for events
my %timePeriod;
$timePeriod{beginTime} = DateTime::Format::ISO8601->parse_datetime($testBegin);
$timePeriod{endTime} =   DateTime::Format::ISO8601->parse_datetime($testEnd);

#print STDERR "Here 4 \n";
# VI SDK Login
#Vim::login(service_url => $service_url, user_name => $userid, password => $password);
Util::connect();

#print STDERR "Here 5 \n";
# $events is hash of arrays event references with event types as the key
my $events = getEventCategories($categories);

# Collect and print events.
printEvents($events);

# VI SDK Logout
#Vim::logout();
Util::disconnect();

#### End Main Body ###

# Loop through events by category and print to file/screen
#
#              printEvents($events)
sub printEvents {
	my $events = shift @_;
	my $eventTime;
	my $eventCollector;
	foreach my $eventCategory (sort keys %$events) {
            $eventCollector = getEventCollector(\%timePeriod, ($eventCategory) );
            getEvents($events, $max, $eventCollector);
            if (defined ($$events{$eventCategory})) {
		my @eventList = @{$$events{$eventCategory}} ;
		if ( $debug ) { print STDERR "Detected eventCategory: $eventCategory (" . scalar @eventList .")\n" ;}
		
		my $reconfig1;
		my $reconfig2;
		my $part1;
		my $part2;
		###print $eventCategory.":\n";
		foreach my $event (@eventList) {
		    $eventTime = DateTime::Format::ISO8601->parse_datetime($$event{'createdTime'});
		    if ( $eventCategory eq "VmReconfiguredEvent" && $diskaddfilter ){
			$reconfig1 = $eventTime->strftime("[ %F %T %Z ] ") . "[".$$event{'userName'}."] ";
			$reconfig2 = $$event{'fullFormattedMessage'};
			my @lines = split /\n/, $reconfig2;
			foreach my $line (@lines) {
				if ($line =~ /^Reconfigured /) {
				   $part1 = $line;
				   chomp $part1;
				}
				if ($line =~ /(.*Hard disk.*B"\)),/) {
				   $part2 = $1;
			           print $reconfig1 . $part1 . $part2 . "\) \n";
				}
                        }
                    } else {
                        print $eventTime->strftime("[ %F %T %Z ] ");
                        print "[".$$event{'userName'}."] ";
                        print $$event{'fullFormattedMessage'}."\n";
                    }
		}
            } else {
		if ( $debug ) { print STDERR "Undetected eventCategory: $eventCategory \n" ;}
            } 
	}
}


# Get event categories from file
# 	my $events = getEventCategories($categories);
sub getEventCategories {
    my %events;
    if ($categories ne "") {
	open(EVENTCATEGORIES, "< $_[0]") or die "Couldn't open eventlist file $!\n";
	while ( <EVENTCATEGORIES> ) {
		chomp;
		$events{$_} = ();
	}
    } else {
	@events{@defaultEvents} = ();
    }
    \%events;
}

# Creates an event collector based on a begin and end time
# 
# my $eventCollector = getEventCollector(\%timePeriod, keys %$events);
sub getEventCollector {
	my $r_timePeriod = shift(@_);
	my %timePeriod = %$r_timePeriod;
	my $eventmgr_ref = Vim::get_service_content()->eventManager;
	my $eventmgr_view = Vim::get_view(mo_ref => $eventmgr_ref);

	my $event_filter_spec = EventFilterSpec->new(type => [@_],
		time => 
		      EventFilterSpecByTime->new(beginTime => 
                         $timePeriod{'beginTime'}->datetime,
	        endTime => $timePeriod{'endTime'}->datetime));

	my $eventCollector = $eventmgr_view->CreateCollectorForEvents(_this => $eventmgr_ref,
						filter => $event_filter_spec);
}

# Get all events (up to max) returned by our eventCollector
sub getEvents {
	
	my($events,$maxEvents,$eventCollector) = @_;
	
	my $eventCollectorView = Vim::get_view(mo_ref => $eventCollector);
	
	# Set page size so that all the events we want to get are on one page.
	# Max page size allowed is 1000, if number of events exceeds max value set
	# then the oldest events will be dropped

	$eventCollectorView->SetCollectorPageSize( _this => $eventCollector, maxCount => $maxEvents);
	# Move cursor to the beginning of that page.
	$eventCollectorView->ResetCollector(_this => $eventCollector);
	
	my $eventList = $eventCollectorView->ReadNextEvents(_this => $eventCollector,
    	      						    maxCount => $maxEvents);
	foreach(@{$eventList}) {
		push @{ $$events{ref($_)} }, $_;
	}
	
	$eventCollectorView->DestroyCollector(_this => $eventCollector);
}

	
# Usage Help Text
sub UsageOut {
   my $help_text = <<'END';

USAGE:
   getEvents.pl <parameters>
   
   REQUIRED PARAMETERS:
   	
      --service_url <SDK service URL>
   	 
   	 Specify the SDK service URL.  This is typically 
   	 https://<VC Server>/sdk	
   	
      --userid <userid>
   	
   	 User ID supplied to the SDK service.
   		
      --password <password>
   	
   	 Password for the user
   	
      --testbegin <yyyy-mm-ddThh:mm:ss> 

         Date and Time test began, format: yyyy-mm-ddThh:mm:ss (UTC)

      --testend <yyyy-mm-ddThh:mm:ss> 

         Date and Time test ended, format: yyyy-mm-ddThh:mm:ss (UTC)
   	 
   	
   OPTIONAL PARAMETERS:
      
      --max <number>
   	 
   	 Limits the maximum number of events returned.  The default value is
   	 1000.  If there are more events than the specified maximum, only the
   	 first 'n' events will be returned. Can be set <= 1000 only.
   	
      --categories <filename>
   	
   	 Filename should be a text file containing a list of the VMware events
   	 to monitor.  Each event must be listed on a separate line. Overides
	 default list of events.

      --diskaddfilter <0|1>
 
         Enables capture of only Add Hard Disk events for VmReconfiguredEvents,
	 uses single line compact format (default is 1 - enabled).
	 Set to 0 to get all VmReconfiguredEvents in usual multi-line format.
   	
EXAMPLES:
   
   To view the events from a SPEC VIRT DC 2020 test run, using the default VM events list.
   Note, times are UTC.

   getEvents.pl --service_url https://localhost/sdk --userid administrator 
   --password mypassword --testbegin 2020-04-21T03:48:12 --testend 2020-04-21T06:52:51
   
   
END
   print $help_text;
}
