#!/bin/sh 

#  Adapted for SPECvirt Datacenter 2021
#
#  reporter.sh - Run on the svdc-director VM to collect SUT host configuration info
#
#    This file is part of the SPECvirt Harness.
#
#    This file is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License, Version 2.0, 
#    as published by the Free Software Foundation. 
#
#    This file is is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with the this file; if not, write to the Free Software 
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA#
# --------------------------------------------------------------------------------
# 11032018
#
#     Collects various configuration and log files for benchmark disclosure 
#     with VMware vSphere. On ESX 6.x, this script, in combination with its 
#     companion script reporterHelper.sh, prepares output from the VMware 
#     vSphere support script vm-support.
#
##############################################################################
VERSION="05122020"

HOSTNAME=$1
runfolder=`ls -l /export/home/cp/results/specvirt|grep ^d|awk '{print $9}'|egrep '^[0-9]{4}-[0-9]{2}-[0-9]{2}_[0-9]{2}-[0-9]{2}-[0-9]{2}'|tail -1`
RESULTDIR=/datastore/tmp
#RESULTDIR=$runfolder/support
OUTPUT_FAIL=0
STARTDIR=$(pwd)

mkdir -p /datastore/tmp

postProcessingV65 () {
   
   VMSUPPORTPATH=$(cat ResultsPath.txt)
   printf "$(date '+%r'): Transferring Reporter file from $HOSTNAME\n"
   scp root@$HOSTNAME:"$VMSUPPORTPATH" $RESULTDIR >/dev/null 
   VMSUPPORTDIR=$(echo ${VMSUPPORTPATH%\.tgz})		# remove .tgz from end of zipfile name
   ssh -l root $HOSTNAME "rm -rf $VMSUPPORTPATH $VMSUPPORTDIR" 2>&1 >>$OUTPUT_WARN | tee -a $OUTPUT_ERR # Remove reporterHelper file from host after completed scp
   SUBMITFILE=$(basename $VMSUPPORTPATH)
   HOSTREPORTERDIR=$(dirname $VMSUPPORTPATH) # This variable will get passed to harness so it can clean up LUN/reporter directory on host.
}
	  
printUsage () {
   printf "\nUsage: sh reporterHelper HOSTNAME [dest_dir] [temp_dir] [config_file] [client | sut]"
   printf "\nRuns ReporterHelper script on host HOSTNAME.\n"
   printf "\n   HOSTNAME : Target ESXi host\n"
   printf "\n   dest_dir : Reporter results files <hostname>-<datestamp>.tgz and"
   printf "\n              ReporterResult_<hostname>.txt will be written to this"
   printf "\n              directory or the calling directory by default.\n"
   printf "\n   temp_dir : Temporary files will be created in this directory, or"
   printf "\n              /datastore/tmp by default.\n"
   printf "\n   config_file : Full path to Control.config file, or"
   printf "\n                 /export/home/cp/bin/Control.config by default.\n"
   printf "\n   client | sut : Specify whether the target ESXi host running the"
   printf "\n                  Reporter is a SUT host ('sut') or a Client host"
   printf "\n                  ('client'). Reporter will run as 'sut' by default."
   printf "\n                  Case sensitive.\n"
   printf "\n   reporter.sh must be run from a terminal on the svdc-director client."
   printf "\n   Passwordless SSH must be enabled on HOSTNAME.\n\n"
}

###############################################################################
# Error checking
###############################################################################
if [ -z "$1" ]; then
   printf "Missing HOSTNAME.\nPlease specify HOSTNAME as a target ESXi host.\n"
   printUsage; exit 1
fi
if [ "$1" = "localhost" ]; then
   printf "Please specify HOSTNAME as a target ESXi host.\n"
   printUsage; exit 1
fi
if [ -z "$2" ]; then
   RESULTDIR="/datastore/tmp"
fi
if [ ! -e /export/home/cp/config/workloads/specvirt/HV_Operations/vSphere/reporterHelper.sh ] ; then
   printf "Please ensure script reporterHelper.sh is located in /export/home/cp/config/workloads/specvirt/HV_Operations/vSphere/.\n"
   printUsage; exit 1
fi

# Find short hostname from host, set up path names 
HOSTNAME_SHORT=$(ssh -l root $HOSTNAME "hostname -s")
# If short hostname is localhost, revert to original hostname
if [ "$HOSTNAME_SHORT" = "localhost" ] ; then
  HOSTNAME_SHORT=$1
fi
# Append two random numbers to working directory name in case of hostname conflict. 
RAND=$(date +"%N" | cut -c 1-7)
DATE=$(date +"%Y%m%d-%H%M")$RAND
SUBMITFILE=$HOSTNAME_SHORT-$DATE.tgz
WORKDIR=/datastore/tmp/$HOSTNAME_SHORT-$DATE
# Set up temporary directory; sanitize temp directory argument $3
if [ ! -z "$3" ]; then
   WORKDIR=$3  
   # Replace any relative path (. at beginning of string) with the absolute path.
   WORKDIR=${WORKDIR/#\./$STARTDIR}
   # If there is a trailing '/', delete it
   WORKDIR=${WORKDIR/%\//}
   
   if [ ! -d $WORKDIR ] ; then 
      printf "Temporary output directory $WORKDIR does not exist.\n"
      printUsage; exit 1 
   else
   WORKDIR=$WORKDIR/$HOSTNAME_SHORT-$DATE
   fi
fi
# Determine config file (Control.config)
if [ -z "$4" ]; then
   CONFIGFILE=/export/home/cp/bin/Control.config
else
   CONFIGFILE=$4
fi
if [ ! -e "$CONFIGFILE" ] ; then
   printf "Config file $CONFIGFILE does not exist.\n"
   printUsage; exit 1
fi
# Determine host type (sut or client)
#if [ -z "$5" ]; then
   HOSTTYPE="sut"
#else
#   if [[ "$5" != "sut" && "$5" != "client" ]]; then
#      printf "Specify whether the target ESXi host running the Reporter is a SUT\n"
#      printf "host ('sut') or a Client host ('client').\n"
#      printUsage; exit 1
#   else    
#      HOSTTYPE="$5"
#   fi
#fi

OUTPUT_ERR="${WORKDIR}/output_err.txt"
OUTPUT_WARN="${WORKDIR}/output_warn.txt"

# Sanitize results directory argument
# Replace any relative path (. at beginning of string) with the absolute path.
RESULTDIR=${RESULTDIR/#\./$STARTDIR}
# If there is a trailing '/', delete it
RESULTDIR=${RESULTDIR/%\//}

if [ ! -d $RESULTDIR ] ; then 
    printf "Output directory $RESULTDIR does not exist.\n\n"
   printUsage; exit 1 
fi

# Error check hostname on manual execution of script.
TEMP=$(ssh -l root "$HOSTNAME" 'exit' 2>&1)
# if the host is not in the 'known hosts' list, this will return a benign 'Permanently added..' error which we can ignore.
if [[ "$TEMP" && ! "$TEMP" =~ "Permanently added" ]]; then 
   printf "The Reporter was unable to access $HOSTNAME.\nPlease verify that $HOSTNAME can be resolved.\n$TEMP\n"
   printUsage; exit 1
fi

# Find host version.
#VERSION=$(ssh -l root $HOSTNAME "vmware -v")
#
#if [[ "$VERSION" =~ ESXi*\ 4\. ]] ; then 
#   printf "Host $HOSTNAME is installed with $VERSION. \n"
#   # Supports 6.0 and later. 
#   printf "The Reporter supports ESXi 6.0 and later.\n"; exit 1 
#elif [[ "$VERSION" =~ ESXi*\ 5\. || "$VERSION" =~ ESXi*\ 6\.0 ]] ; then VERSION=5; # Designates codepath for ESXi 5.x and 6.0. 
#else 
VERSION=65; # Codepath for 6.5 or later
#fi
# Create working directory which is where all temporary files and activity will happen.
mkdir -pm 755 $WORKDIR
cd $WORKDIR
printf "$(date '+%r'): Starting Configuration Gathering Script for VMware vSphere\n"
printf "$(date '+%r'): reporterHelper.sh $VERSION\n"

# Copy reporterHelper script onto host.
printf "$(date '+%r'): Transferring reporterHelper.sh onto $HOSTNAME\n"
scp /export/home/cp/config/workloads/specvirt/HV_Operations/vSphere/reporterHelper.sh root@$HOSTNAME:/tmp 2>&1 >/dev/null | tee -a $OUTPUT_ERR
ssh -l root $HOSTNAME "chmod 755 /tmp/reporterHelper.sh"  2>>$OUTPUT_ERR 1>>$OUTPUT_WARN

printf "$(date '+%r'): Transferring $HOSTTYPE manifest onto $HOSTNAME\n"
scp /export/home/cp/config/workloads/specvirt/HV_Operations/vSphere/specvirt-$HOSTTYPE.mfx root@$HOSTNAME:/etc/vmware/vm-support 2>&1 >/dev/null | tee -a $OUTPUT_ERR

# Call Reporter script with ESX version, host type, host name as arguments.
# Send all stdout and stderr to temporary files output_err.txt and output_warn.txt
printf "$(date '+%r'): Running reporterHelper.sh on $HOSTNAME. This may take several minutes.\n"
ssh -l root $HOSTNAME "/tmp/reporterHelper.sh $VERSION $HOSTTYPE $HOSTNAME" 2>>$OUTPUT_ERR 1>>$OUTPUT_WARN

# Transfer standard output file from host.
scp root@$HOSTNAME:"/tmp/ResultsPath.txt" $WORKDIR 2>&1 >/dev/null | tee -a $OUTPUT_ERR
# Report error if the transfer failed
if [ ! -e $WORKDIR/ResultsPath.txt ] ; then
   printf "$(date '+%r'): reporterHelper.sh did not run correctly on $HOSTNAME.  Please rerun manually.\n"
   echo "reporterHelper.sh did not run correctly on $HOSTNAME.  Please rerun manually." | tee -a $OUTPUT_ERR
   exit 1
else
   # Remove temporary files from host.
   ssh -l root $HOSTNAME "rm -f /tmp/ResultsPath.txt" 2>&1 >>$OUTPUT_WARN | tee -a $OUTPUT_ERR
fi

###############################################################################
# Begin post processing
###############################################################################
printf "$(date '+%r'): Configuration-gathering script finished. Beginning post-processing.\n"
postProcessingV$VERSION
printf "$(date '+%r'): Post-processing finished. Starting cleanup.\n"

#
# Set output as 0 if no errors or warnings detected.
#
if [ "$(cat $OUTPUT_ERR)" = "" ] ; then
   echo "None" > $OUTPUT_ERR
else
   OUTPUT_FAIL=1
fi
if [ "$(cat $OUTPUT_WARN)" = "" ] ; then
   echo "None" > $OUTPUT_WARN
fi

#
# Publish errors, warnings and location of reporter tgz file for calling scripts & harness
REPORTERFILE=$RESULTDIR/ReporterResult-$HOSTNAME.txt

printf "$(date), SPECvirt Datacenter 2021 Reporter $VERSION\n" > $REPORTERFILE 2>/dev/null
printf "reporter_file:$SUBMITFILE\n" >> $REPORTERFILE 2>/dev/null
printf "reporter_full:$RESULTDIR/$SUBMITFILE\n" >> $REPORTERFILE 2>/dev/null
printf "reporter_dir:$HOSTREPORTERDIR\n" >> $REPORTERFILE 2>/dev/null
printf "failed:$OUTPUT_FAIL\n" >> $REPORTERFILE 2>/dev/null
printf "errors:$(cat $OUTPUT_ERR)\n" >> $REPORTERFILE 2>/dev/null 
printf "warnings:$(cat $OUTPUT_WARN)\n" >> $REPORTERFILE 2>/dev/null

# Remove non-ascii characters from $REPORTERFILE. Non-ascii characters cause STAX to crash.
perl -i -pe 's/[^[:ascii:]]//g' $REPORTERFILE 2>/dev/null

cd ../
# Delete working directory and all temporary files
rm -rf $WORKDIR 2>/dev/null

printf "$(date '+%r'): Cleanup finished.\n"
printf "$(date '+%r'): Include $RESULTDIR/$SUBMITFILE as part of your benchmark submission.\n"
printf "$(date '+%r'): View Reporter results at $RESULTDIR/ReporterResult-$HOSTNAME.txt\n"
printf "$(date '+%r'): Done!\n"
