#!/usr/bin/perl -w
#
###########################################################################
#                                                                         #
#  Copyright (c) 2021 Standard Performance Evaluation Corporation (SPEC). #
#  All rights reserved.                                                   #
#                                                                         #
###########################################################################

#  sutConfig.pl: Start/End Run Config script for SPECvirt Datacenter 2021
#
#  This script collects testbed configuration information at 
#  various levels: Datacenter, Cluster, Host, and VM and
#  generates the startRun.config and endRun.config files that
#  is used for validation of a VMware vSphere testbed at the begining 
#  and end of the test. Additional information is collected by other 
#  means for formal validation of a compliant test.
#
#  ./sutConfig.pl  --server=[hostname/ipaddr] --username=[admin-user] \
#    --password='[admin-passwd]' \
#    --cluster=[SUT-cluster] --clientcluster=[client-cluster]
##
## Script used as starting point: VmHostDatastoreReport-Fast.pl
## Sample script to illustrate the performance of an optimized 
## VMware SDK entity and property collection.
## Created by Reuben Stump (rstump@vmware.com | http://www.virtuin.com)
##
 
use strict;
use warnings;
use 5.010;
 
use VMware::VILib; 
use VMware::VIRuntime;

my %opts = (
   cluster => {
      type => "=s",
      help => "Name of SUT cluster ",
      required => 1,
   },
   clientcluster => {
      type => "=s",
      help => "Name of client cluster ",
      required => 1,
   },
);

my %h_ds_seen =();
my %seen =();

Opts::add_options(%opts); 
Opts::parse();
Opts::validate();
 
Util::connect();


my $cluster = Opts::get_option('cluster');
my ($cluster_view,$clusterhost_views,$vms,$cluster_datastores,);

my $clientcluster = Opts::get_option('clientcluster');
my ($clientcluster_view,$clienthost_views,$clientvms,);

$cluster_view = Vim::find_entity_view(view_type => 'ClusterComputeResource', filter => { name => $cluster});
$cluster_datastores = Vim::get_views(mo_ref_array => $cluster_view->datastore);

print "SUT.Cluster.name[1]=" . $cluster . "\n";
unless($cluster_view) {
  die "Unable to locate cluster name: \"$cluster\"!";
}

$clientcluster_view = Vim::find_entity_view(view_type => 'ClusterComputeResource', filter => { name => $clientcluster});


print "SUT.Cluster.name[2]=" . $clientcluster . "\n";
unless($clientcluster_view) {
  die "Unable to locate client cluster name: \"$clientcluster\"!";
}

my $datacenter_properties = [ 'name' ];

my $cluster_folder = Vim::get_view(mo_ref => $cluster_view->parent);
my $cluster_parent = Vim::get_view(mo_ref => $cluster_folder->parent);

my $datacenter_name = $cluster_parent->name;

print "SUT.Datacenter.name[1]=" . $datacenter_name . "\n";

###

$clusterhost_views = Vim::get_views(mo_ref_array => $cluster_view->host, properties => ['name', 'summary', 'vm', 'runtime' ]);
my $i=0;
my $mct=0;
my $uct=0;
my @clusterhostlist = ();
my $clusterHostsOnline = scalar (@$clusterhost_views);
foreach(@$clusterhost_views) {
    $i++;
    print "SUT.Host.name[".$i."]=" . $_->name . "\n";
    push (@clusterhostlist,$_->name);
    if ($_->runtime->connectionState->val eq "connected") {
      print "SUT.Host.state[".$i."]=" . ($_->runtime->inMaintenanceMode ? "maintenance" : "up") . "\n";
    ($_->runtime->inMaintenanceMode ? $mct++ : $uct++);
    } else {
      print "SUT.Host.state[".$i."]=" . "down" . "\n";
      $mct++;
      $clusterHostsOnline--;
    }
}

print "SUT.Host.offline.Total=". $mct .  "\n";
print "SUT.Host.online.Total=". $uct .  "\n";

#print ("Debug: ". "@clusterhostlist" , "\n" );


###
 
# Fetch all VirtualMachines from SDK, limiting the property set
my $vm_views = Vim::find_entity_views(view_type => "VirtualMachine",
          properties => ['name', 'runtime.powerState', 'runtime.connectionState', 'summary.config.numCpu', 'summary.config.memorySizeMB','runtime.host', 'datastore']) || 
  die "Failed to get VirtualMachines: $!";
 
# Fetch all HostSystems from SDK, limiting the property set
my $host_views = Vim::find_entity_views(view_type => "HostSystem",
          properties => ['name']) ||
  die "Failed to get HostSystems: $!";

# Create hash tables with key = entity.mo_ref.value            
my %host_map = map { $_->get_property('mo_ref.value') => $_ } @{ $host_views || [] };
 
my $vmname;
my %vmotions  = ();
my $numvMotions = 0;
my $eventMgr = Vim::get_view(mo_ref => Vim::get_service_content()->eventManager);
#my $vm_view = Vim::find_entity_view(view_type => 'VirtualMachine', filter => {"name" => $vmname});


# Enumerate VirtualMachines in SUT Cluster
my $count = 0;
my $ctoff = 0;
my $ctup = 0;
my $bctoff = 0;
my $bctup = 0;
foreach my $vm ( @{$vm_views || []} ) {
 # Get HostSystem from the host map
 my $host_ref = $vm->get_property('runtime.host')->{'value'};
 my $host = $host_map{$host_ref};
 $vmname = $vm->get_property('name');
 
 if (grep { $host->get_property('name') eq $_ } @clusterhostlist) {
   if ( $vm->get_property('runtime.connectionState.val') eq 'connected') {
     $count++;
     print "SUT.VM.name[".$count."]=".  $vm->get_property('name') . "\n";

     print "SUT.VM.state[".$count."]=". ($vm->get_property('runtime.powerState.val') eq 'poweredOff' ? "down" : "up" ) . "\n";
     ($vm->get_property('runtime.powerState.val') eq 'poweredOff' ? $ctoff++ : $ctup++ );
     if ( $vm->get_property('name') =~ /^svdc-t\d\d\d-/ ) {
        $vm->get_property('runtime.powerState.val') eq 'poweredOff' ? $bctoff++ : $bctup++;
     }
     print "SUT.VM.vCPU[".$count."]=".  $vm->get_property('summary.config.numCpu') . "\n";
     print "SUT.VM.memoryMB[".$count."]=". $vm->get_property('summary.config.memorySizeMB'). "\n";
     print "SUT.VM.host[".$count."]=". $host->get_property('name').  "\n";
   
     eval {
        my $recursion = EventFilterSpecRecursionOption->new("self");
        #my $entity = EventFilterSpecByEntity->new(entity => $vm_view, recursion => $recursion);
        my $entity = EventFilterSpecByEntity->new(entity => $vm, recursion => $recursion);
        my $filterSpec = EventFilterSpec->new(type => ["DrsVmMigratedEvent"], entity => $entity);
        my $events = $eventMgr->QueryEvents(filter => $filterSpec);
        $numvMotions = @$events;
        foreach(@$events) {
                if(defined($_->host)) {
                        $vmotions{$_->host->name} += 1;
                }
        }
     };
     if($@) {
        print "Error: " . $@ . "\n";
     }
   ###Debug## print "\n" . $vmname . " has a total of " . $numvMotions . " vMotions\n\n";
   }   
 }
 
}

if ($clusterHostsOnline >= 1) {
  my $k = 0;
  foreach(@$cluster_datastores) {
    if ($_->summary->accessible) {
      if  ($_->summary->multipleHostAccess) {
      $k++;
      print "SUT.Storage.Pool[$k]=". $_->summary->name . "\n";
      }
    }
  }
}


print "SUT.VM.running.total=". $ctup . "\n";
print "SUT.VM.down.total=". $ctoff . "\n";
print "Benchmark.VM.running.total=". $bctup . "\n";
print "Benchmark.VM.down.total=". $bctoff . "\n";

####SUT.VM.migration.total####
my $value=0;
my $ttl_vmotions=0;
for my $key ( sort keys %vmotions ) {
        $value = $vmotions{$key};
        ###print "$key => $value\n";
	$ttl_vmotions+= $value;
}

print "SUT.VM.migration.runningTotal=" . $ttl_vmotions . "\n";
 
# Disable SSL hostname verification for vCenter self-signed certificate
BEGIN {
 $ENV{PERL_LWP_SSL_VERIFY_HOSTNAME} = 0;
}

Util::disconnect();
