#!/bin/sh 
#  Copyright 2006-2020 VMware, Inc. All rights reserved. 
#
#
#   This file is part of the VMmark(r) Harness.
#
#    The VMmark Harness is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License, Version 2.0, 
#    as published by the Free Software Foundation. 
#
#    The VMmark Harness is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with the VMmark Harness; if not, write to the Free Software 
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA#
# --------------------------------------------------------------------------------
# 11032018
#
#     Collects various configuration and log files for benchmark disclosure 
#     with VMware vSphere. On ESX 6.x, this script, in combination with its 
#     companion script VMmark-Reporter.sh, prepares output from the 
#		VMware vSphere support script vm-support for benchmark submission.
#
#	For 5.x: Follow this step before running the reporting script 
#	(may require a reboot).
#
#	    Point the scratch location to a real disk to avoid inadvertent out of disk 
#	    space errors:
#
#   Log into the vSphere client.
#       
#      Click on the server then Configuration tab > Software > Advanced Settings.
#	    Click on ScratchConfig. Make sure ScratchConfig.ConfiguredScratchLocation 
#	    points to a writeable directory on the VMFS partition (/vmfs/volumes/...). 
#	    Requires a reboot to take effect.
#
#
##############################################################################
MASTERVERSION="11032018"

HOSTNAME=$1
RESULTDIR=$2
OUTPUT_FAIL=0
STARTDIR=$(pwd)

postProcessingV65 () {
   
   VMSUPPORTPATH=$(cat ResultsPath.txt)
   printf "$(date '+%r'): Transferring Reporter file from $HOSTNAME\n"
   scp root@$HOSTNAME:"$VMSUPPORTPATH" $RESULTDIR >/dev/null # For 6.5, copy reporter file directly into RESULTDIR. No postprocessing.
   VMSUPPORTDIR=$(echo ${VMSUPPORTPATH%\.tgz})		# remove .tgz from end of zipfile name
   ssh -l root $HOSTNAME "rm -rf $VMSUPPORTPATH $VMSUPPORTDIR" 2>&1 >>$OUTPUT_WARN | tee -a $OUTPUT_ERR # Remove reporter file from host after completed scp
   SUBMITFILE=$(basename $VMSUPPORTPATH)
   HOSTREPORTERDIR=$(dirname $VMSUPPORTPATH) # This variable will get passed to harness so it can clean up LUN/reporter directory on host.
}
	  
postProcessingV5 () {

   HOSTFILESPATH=$(tail -1 ResultsPath.txt)		# The location of the additional files (beyond vm-support) gathered by reporter on host.
   VMSUPPORTPATH=$(sed -e '$!{h;d;}' -e x ResultsPath.txt) # prints the next to last line.
   VMSUPPORTTEMPDIR=$(dirname $VMSUPPORTPATH) # Path to temporary directory on host for vm-support file to eliminate filename conflicts.
   HOSTREPORTERDIR=$(dirname $VMSUPPORTTEMPDIR) # This variable will get passed to harness so it can clean up LUN/reporter directory on host.
   sed -i 'N;$!P;$!D;$d' ResultsPath.txt 2>&1 >> $OUTPUT_WARN | tee -a $OUTPUT_ERR # Delete the last two lines of file ResultsPath.txt so it contains only rm commands.
  
   scp root@$HOSTNAME:"$VMSUPPORTPATH $HOSTFILESPATH" $WORKDIR 2>&1 >/dev/null | tee -a $OUTPUT_ERR #For 5.x, copy vm-support file to working directory for postprocessing.
   ssh -l root $HOSTNAME "rm -rf $VMSUPPORTTEMPDIR $HOSTFILESPATH" 2>&1 >>$OUTPUT_WARN | tee -a $OUTPUT_ERR # Remove reporter files from host after completed scp
   
   VMSUPPORTFILE=$(basename $VMSUPPORTPATH)		# Get zipfile name
   HOSTFILES=$(basename $HOSTFILESPATH)
   VMSUPPORTDIR=$(echo ${VMSUPPORTFILE%\.tgz})	# remove .tgz from end of zipfile name
   HOSTFILESDIR=$(echo ${HOSTFILES%_hostfiles\.tgz})
  
   printf "$(date '+%r'): Unzipping vm-support file\n"
   tar -xvzmf $VMSUPPORTFILE  2>&1 >/dev/null | tee -a $OUTPUT_ERR
   printf "$(date '+%r'): Processing vm-support file\n"
   rm $VMSUPPORTFILE 2>&1 >>$OUTPUT_WARN | tee -a $OUTPUT_ERR
   cd $VMSUPPORTDIR
   if [ -e reconstruct.sh ] ; then 
      sh reconstruct.sh 2>&1 >>$OUTPUT_WARN | tee -a $OUTPUT_ERR 	# required for 5.x vm-support: run reconstruct.sh to piece together lg files
      rm -rf ./reconstruction/ ./README ./reconstruct.sh >>$OUTPUT_WARN | tee -a $OUTPUT_ERR # Remove now-unneeded reconstruction files
   fi
   find . -type f -name "dump-vmdk-rdm-info.sh*.txt" 2> /dev/null | while read ORIG_FILENAME ; do # rename this file which often has error-producing long filename
      mv "$ORIG_FILENAME" "$(dirname "$ORIG_FILENAME")/dump-vmdk-rdm-info.sh.txt"
   done
   cd ../

   ############################################################################
   # Add additional host's files into vm-support directory
   ############################################################################
   printf "$(date '+%r'): Processing additional reporter files\n"
   tar -xvzmf $HOSTFILES 2>&1 >/dev/null | tee -a $OUTPUT_ERR # redirect stdout only to /dev/null. Errors still visible.
   # directory $HOSTFILESDIR gets created on unzip.
   cp -r $HOSTFILESDIR/* $VMSUPPORTDIR/	2>&1 >>$OUTPUT_WARN | tee -a $OUTPUT_ERR # Copy all files in HOSTFILESDIR to VMSUPPORTDIR. 
 
   # Change all UUID in vmfs/volumes directory names to Volume Name instead
   # parse Volume names vs UUID
   cd $VMSUPPORTDIR
   cat commands/localcli_storage-vmfs-extent-list.txt 2>>$OUTPUT_ERR | sed '1,2d' | while read LINE ; do
      VOLNAME=$(echo $LINE | awk {'print $1'})
      # sed command: make sure expected UUID matches 8-8-4-12 character pattern. Error checks in case volume name has a space.
      UUID=$(echo $LINE | awk {'print $2'} | sed '/^[0-9,a-f]\{8\}-[0-9,a-f]\{8\}-[0-9,a-f]\{4\}-[0-9,a-f]\{12\}/!d')      
	  # if UUID fits UUID format,
      if [ "$UUID" ] ; then
         ls vmfs/volumes 2> /dev/null | while read DIRNAME ; do
            if [ "$DIRNAME" = "$UUID" ] ; then
			      mkdir vmfs/volumes/$VOLNAME 2> /dev/null # if VOLNAME already exists, this will throw back an error, so ignore it.
               mv vmfs/volumes/$DIRNAME/* vmfs/volumes/$VOLNAME 2> /dev/null | tee -a $OUTPUT_WARN
			      rm -rf vmfs/volumes/$DIRNAME 2>&1 >>$OUTPUT_WARN | tee -a $OUTPUT_ERR
            fi
	     done
		 ls commands/vmkfstools_-P--v-10-vmfsvolumes* 2> /dev/null | while read FILENAME ; do
		    if [[ "$FILENAME" =~ "$UUID" ]] ; then 
			   mv $FILENAME commands/vmkfstools_-P--v-10-vmfsvolumes$VOLNAME.txt 2>&1 >>$OUTPUT_WARN | tee -a $OUTPUT_ERR
			fi
		 done
      fi
   done
   cd ../
 
   # This removes powered off VMs from the vmfs/volumes/* folder. Commands determined from on-host code.
   # Lines from ResultsPath will be executed. Extract only the lines that fit
   # the command pattern we want to execute.
   grep "rm -rf $VMSUPPORTDIR/vmfs/volumes/." ResultsPath.txt > Commands.txt 2> /dev/null
   bash Commands.txt 2>&1 >>$OUTPUT_WARN | tee -a $OUTPUT_ERR
   find "$VMSUPPORTDIR/vmfs/volumes/" -type d -empty -delete 2> /dev/null | tee -a $OUTPUT_WARN # remove empty volume directories
 
   FILES_WORKDIR=$(basename $WORKDIR)
   ############################################################################
   # Compress Reporter File and deposit into results directory
   ############################################################################
   printf "$(date '+%r'): Compressing final results file\n"
   mv $VMSUPPORTDIR $FILES_WORKDIR 2>&1 >>$OUTPUT_WARN | tee -a $OUTPUT_ERR # Rename the directory that holds our unzipped files to one matching our naming conventions.
   # Zip directory FILES_WORKDIR into a tgz named SUBMITFILE located in RESULTDIR (user specified)
   tar -czf $RESULTDIR/$SUBMITFILE $FILES_WORKDIR 2>&1 >>$OUTPUT_WARN | tee -a $OUTPUT_ERR 
}

printUsage () {
   printf "\nUsage: sh VMmark-ReporterMaster.sh HOSTNAME [dest_dir] [temp_dir] [config_file] [client | sut]"
   printf "\nRuns VMmark 3.1 Reporter script on host HOSTNAME.\n"
   printf "\n   HOSTNAME : Target ESXi host\n"
   printf "\n   dest_dir : Reporter results files <hostname>-<datestamp>.tgz and"
   printf "\n              ReporterResult_<hostname>.txt will be written to this"
   printf "\n              directory or the calling directory by default.\n"
   printf "\n   temp_dir : Temporary files will be created in this directory, or"
   printf "\n              /root/VMmark3/tmp by default.\n"
   printf "\n   config_file : Full path to VMmark3.properties file, or"
   printf "\n                 /root/VMmark3/VMmark3.properties by default.\n"
   printf "\n   client | sut : Specify whether the target ESXi host running the"
   printf "\n                  Reporter is a VMmark host ('sut') or a Client host"
   printf "\n                  ('client'). Reporter will run as 'sut' by default."
   printf "\n                  Case sensitive.\n"
   printf "\n   VMmark-ReporterMaster.sh must be run from a terminal on the Prime Client."
   printf "\n   Passwordless SSH should be enabled on HOSTNAME.\n\n"
}

###############################################################################
# Error checking
###############################################################################
if [ -z "$1" ]; then
   printf "Missing HOSTNAME.\nPlease specify HOSTNAME as a target ESXi host.\n"
   printUsage; exit 1
fi
if [ "$1" = "localhost" ]; then
   printf "Please specify HOSTNAME as a target ESXi host.\n"
   printUsage; exit 1
fi
if [ -z "$2" ]; then
   RESULTDIR="."
fi
if [ ! -e /root/VMmark3/tools/VMmark-Reporter.sh ] ; then
   printf "Please ensure helper script VMmark-Reporter.sh is located in /root/VMmark3/tools/.\n"
   printUsage; exit 1
fi

# Find short hostname from host, set up path names 
HOSTNAME_SHORT=$(ssh -l root $HOSTNAME "hostname -s")
# If short hostname is localhost, revert to original hostname
if [ "$HOSTNAME_SHORT" = "localhost" ] ; then
  HOSTNAME_SHORT=$1
fi
# Append two random numbers to working directory name in case of hostname conflict. 
RAND=$(date +"%N" | cut -c 1-7)
DATE=$(date +"%Y%m%d-%H%M")$RAND
SUBMITFILE=$HOSTNAME_SHORT-$DATE.tgz
WORKDIR=/root/VMmark3/tmp/$HOSTNAME_SHORT-$DATE
# Set up temporary directory; sanitize temp directory argument $3
if [ ! -z "$3" ]; then
   WORKDIR=$3  
   # Replace any relative path (. at beginning of string) with the absolute path.
   WORKDIR=${WORKDIR/#\./$STARTDIR}
   # If there is a trailing '/', delete it
   WORKDIR=${WORKDIR/%\//}
   
   if [ ! -d $WORKDIR ] ; then 
      printf "Temporary output directory $WORKDIR does not exist.\n"
      printUsage; exit 1 
   else
   WORKDIR=$WORKDIR/$HOSTNAME_SHORT-$DATE
   fi
fi
# Determine config file (VMmark3.properties)
if [ -z "$4" ]; then
   CONFIGFILE=/root/VMmark3/VMmark3.properties
else
   CONFIGFILE=$4
fi
if [ ! -e "$CONFIGFILE" ] ; then
   printf "Config file $CONFIGFILE does not exist.\n"
   printUsage; exit 1
fi
# Determine host type (sut or client)
if [ -z "$5" ]; then
   HOSTTYPE="sut"
else
   if [[ "$5" != "sut" && "$5" != "client" ]]; then
      printf "Specify whether the target ESXi host running the Reporter is a VMmark\n"
      printf "host ('sut') or a Client host ('client').\n"
      printUsage; exit 1
   else    
      HOSTTYPE="$5"
   fi
fi

OUTPUT_ERR="${WORKDIR}/output_err.txt"
OUTPUT_WARN="${WORKDIR}/output_warn.txt"

# Sanitize results directory argument
# Replace any relative path (. at beginning of string) with the absolute path.
RESULTDIR=${RESULTDIR/#\./$STARTDIR}
# If there is a trailing '/', delete it
RESULTDIR=${RESULTDIR/%\//}

if [ ! -d $RESULTDIR ] ; then 
    printf "Output directory $RESULTDIR does not exist.\n\n"
   printUsage; exit 1 
fi

# Error check hostname on manual execution of script.
TEMP=$(ssh -l root "$HOSTNAME" 'exit' 2>&1)
# if the host is not in the 'known hosts' list, this will return a benign 'Permanently added..' error which we can ignore.
if [[ "$TEMP" && ! "$TEMP" =~ "Permanently added" ]]; then 
   printf "The VMmark Reporter was unable to access $HOSTNAME.\nPlease verify that $HOSTNAME can be resolved from the Prime Client.\n$TEMP\n"
   printUsage; exit 1
fi

# Find host version.
VERSION=$(ssh -l root $HOSTNAME "vmware -v")

if [[ "$VERSION" =~ ESXi*\ 4\. ]] ; then 
   printf "Host $HOSTNAME is installed with $VERSION. \n"
   # VMmark 3.x supports 6.0 and later. However, VMmark-ReporterMaster supports 5.x for legacy purposes.
   printf "The VMmark Reporter supports ESXi 6.0 and later.\n"; exit 1 
elif [[ "$VERSION" =~ ESXi*\ 5\. || "$VERSION" =~ ESXi*\ 6\.0 ]] ; then VERSION=5; # Designates codepath for ESXi 5.x and 6.0. 
else 
   VERSION=65; # Codepath for 6.5 or later
fi
# Create working directory which is where all temporary files and activity will happen.
mkdir -pm 755 $WORKDIR
cd $WORKDIR
printf "$(date '+%r'): Starting Configuration Gathering Script for VMware vSphere\n"
printf "$(date '+%r'): VMmark-ReporterMaster.sh $MASTERVERSION\n"

# Copy VMmark-reporter script onto host.
printf "$(date '+%r'): Transferring VMmark-reporter.sh onto $HOSTNAME\n"
scp /root/VMmark3/tools/VMmark-Reporter.sh root@$HOSTNAME:/tmp 2>&1 >/dev/null | tee -a $OUTPUT_ERR
ssh -l root $HOSTNAME "chmod 755 /tmp/VMmark-Reporter.sh"  2>>$OUTPUT_ERR 1>>$OUTPUT_WARN

printf "$(date '+%r'): Transferring VMmark $HOSTTYPE manifest onto $HOSTNAME\n"
scp /root/VMmark3/tools/vmmark3-$HOSTTYPE.mfx root@$HOSTNAME:/etc/vmware/vm-support 2>&1 >/dev/null | tee -a $OUTPUT_ERR

# Call Reporter script with ESX version, host type, host name as arguments.
# Send all stdout and stderr to temporary files output_err.txt and output_warn.txt
printf "$(date '+%r'): Running VMmark-reporter.sh on $HOSTNAME. This may take several minutes.\n"
ssh -l root $HOSTNAME "/tmp/VMmark-Reporter.sh $VERSION $HOSTTYPE $HOSTNAME" 2>>$OUTPUT_ERR 1>>$OUTPUT_WARN

# Transfer standard output file from host.
scp root@$HOSTNAME:"/tmp/ResultsPath.txt" $WORKDIR 2>&1 >/dev/null | tee -a $OUTPUT_ERR
# Report error if the transfer failed
if [ ! -e $WORKDIR/ResultsPath.txt ] ; then
   printf "$(date '+%r'): VMmark-reporter.sh did not run correctly on $HOSTNAME.  Please rerun manually.\n"
   echo "VMmark-reporter.sh did not run correctly on $HOSTNAME.  Please rerun manually." | tee -a $OUTPUT_ERR
   exit 1
else
   # Remove temporary files from host.
   ssh -l root $HOSTNAME "rm -f /tmp/ResultsPath.txt" 2>&1 >>$OUTPUT_WARN | tee -a $OUTPUT_ERR
fi

###############################################################################
# Begin post processing
###############################################################################
printf "$(date '+%r'): Configuration-gathering script finished. Beginning post-processing.\n"
postProcessingV$VERSION
printf "$(date '+%r'): Post-processing finished. Starting cleanup.\n"

#
# Set output as 0 if no errors or warnings detected.
#
if [ "$(cat $OUTPUT_ERR)" = "" ] ; then
   echo "None" > $OUTPUT_ERR
else
   OUTPUT_FAIL=1
fi
if [ "$(cat $OUTPUT_WARN)" = "" ] ; then
   echo "None" > $OUTPUT_WARN
fi

#
# Publish errors, warnings and location of reporter tgz file for calling scripts & VMmark harness
REPORTERFILE=$RESULTDIR/ReporterResult-$HOSTNAME.txt

printf "$(date), VMmark 3.1 Reporter $MASTERVERSION\n" > $REPORTERFILE 2>/dev/null
printf "reporter_file:$RESULTDIR/$SUBMITFILE\n" >> $REPORTERFILE 2>/dev/null
printf "reporter_dir:$HOSTREPORTERDIR\n" >> $REPORTERFILE 2>/dev/null
printf "failed:$OUTPUT_FAIL\n" >> $REPORTERFILE 2>/dev/null
printf "errors:$(cat $OUTPUT_ERR)\n" >> $REPORTERFILE 2>/dev/null 
printf "warnings:$(cat $OUTPUT_WARN)\n" >> $REPORTERFILE 2>/dev/null

# Remove non-ascii characters from $REPORTERFILE. Non-ascii characters cause STAX to crash.
perl -i -pe 's/[^[:ascii:]]//g' $REPORTERFILE 2>/dev/null

cd ../
# Delete working directory and all temporary files
rm -rf $WORKDIR 2>/dev/null

printf "$(date '+%r'): Cleanup finished.\n"
printf "$(date '+%r'): Include $RESULTDIR/$SUBMITFILE as part of your benchmark submission.\n"
printf "$(date '+%r'): View VMmark 3.1 Reporter results at $RESULTDIR/ReporterResult-$HOSTNAME.txt\n"
printf "$(date '+%r'): Done!\n"
