#!/usr/bin/perl -w
# Kalpesh Patel
# https://www.linkedin.com/in/kalpeshpatel

use strict;
use warnings;

use FindBin;
use lib "$FindBin::Bin/../";

use VMware::VIRuntime;

$Util::script_version = "1.0";

sub vmdelete;

# Bug 222613
my %operations = (
   "delete", "",
);

my %opts = (
   'operation' => {
      type => "=s",
      help => "Operation to be performed: delete",
      required => 1,
   },
   'vmname' => {
      type => "=s",
      help => "Name of the virtual machine",
      required => 1,
   },
);


Opts::add_options(%opts);
Opts::parse();
Opts::validate(\&validate);

my $op = Opts::get_option('operation');
Util::connect();

if ($op eq "delete") {
   vmdelete();
} 

Util::disconnect();

# This is required to delete the VM
sub vmdelete() {
   my $vmname = Opts::get_option('vmname');
   my $vm_views = Vim::find_entity_views(view_type => 'VirtualMachine',
                                         filter => {'config.name' => $vmname});
   if (!@$vm_views) {
      Util::trace(0, "\nThere is no virtual machine with name '$vmname' registered\n");
      return;
   }
   foreach (@$vm_views){
      my $mor_host = $_->runtime->host;
      my $hostname = Vim::get_view(mo_ref => $mor_host)->name;
      eval {
         $_->Destroy();
         Util::trace(0,"\nDeletion of VM '$vmname' successfully"
                      . " completed under host $hostname\n");
      };
      if ($@) {
         if (ref($@) eq 'SoapFault') {
            if (ref($@->detail) eq 'InvalidPowerState') {
               Util::trace(0,"\nVM '$vmname' under host $hostname must be powered off"
                            ." for this operation\n");
            }
            elsif (ref($@->detail) eq 'HostNotConnected') {
               Util::trace(0,"\nUnable to communicate with the remote host,"
                            . " since it is disconnected\n");
            }
            else {
               Util::trace(0,"\nFault: " . $@ . "\n");
            }
         }
         else {
            Util::trace(0,"\nFault: " . $@ . "\n");
         }
      }
   }
}

sub validate {
   my $valid = 1;
   # bug 222613
   my $operation = Opts::get_option('operation');
   if ($operation) {
         if (!exists($operations{$operation})) {
         Util::trace(0, "Invalid operation: '$operation'\n");
         Util::trace(0, " List of valid operations:\n");
         map {print "  $_\n"; } sort keys %operations;
         $valid = 0;
      }
   }
   return $valid;
}


__END__

=head1 NAME

vmdelete.pl - Delete a virtual machine.

=head1 SYNOPSIS

vmdelete.pl  --operation <delete> [options]

=head1 DESCRIPTION

This VI Perl command-line utility allows users to perform delete
operations on a virtual machine. 

=head1 OPTIONS

=head2 GENERAL OPTIONS

=over

=item B<operation>

Required. Operation to be performed. Must be one of the following:

  <delete> (delete a virtual machine -- destructive and non-recoverable)

=back

=head2 DELETE OPTIONS

=over

=item B<vmname>

Required. Name of the virtual machine to register.

=head1 EXAMPLES

Deleting a virtual machine

vmdelete.pl  --url https://<ipaddress>:<port>/sdk/webService --username administrator
                --password mypassword --operation delete --vmname FC5

vmdelete.pl --url https://<ipaddress>:<port>/sdk/webService
               --username administrator --password mypassword
               --operation delete --vmname FC5

=head1 SUPPORTED PLATFORMS

All operations work with VMware VirtualCenter 2.0 or later.

All operations work with VMware ESX Server 3.0 or later.


